/* life.c */

#pragma pack(2)

#include <Pilot.h>

#include "life.h"
#include "life_rcp.h"
#include "PilotLib.h"

/*
 * types and macros
 */

#define MAX_LENGTH(max) (((max) - ((max) / 6 + 8192)) / sizeof(cell_t))

#define SCROLL_MARGIN 24
#define SCROLL_OFFSET 16

/*
 * global variables
 */

static Boolean rules[31] = {
    false, false,
    false, false,
    false, true,
    true,  true,
    false, false,
    false, false,
    false, false,
    false, false,
    false, false,
};

static cell_t adjust9[9] = {
    - (1L << 18) - (1 << 5),
    - (1L << 18),
    - (1L << 18) + (1 << 5),

    - (1 << 5),
    - 1,
    + (1 << 5),

    + (1L << 18) - (1 << 5),
    + (1L << 18),
    + (1L << 18) + (1 << 5),
};

static cell_t *tape = NULL;
static short max_length;
static short init_length;
static short life_length;
static short generation;

static short scroll_x;
static short scroll_y;
static Boolean running;

static char gens[6];
static char cells[6];

static DmOpenRef memo_db;
static UInt life_category;

/*
 * functions
 */

static void init_life()
{
    ULong free, max;

    MemHeapFreeBytes(0, &free, &max);
    max_length = MAX_LENGTH(max) * 2;

    while (tape == NULL) {
	max_length /= 2;
	tape = MemPtrNew(sizeof(cell_t) * max_length);
    }

    init_length = 0;
    life_length = 0;
    generation = 0;

    running = false;
}

static void free_life()
{
    if (tape != NULL) {
	MemPtrFree(tape);
    }
}

static void reset_life()
{
    for (life_length = 0; life_length < init_length; ++life_length) {
	tape[init_length + life_length] = tape[life_length];
    }
    generation = 0;

    scroll_x = gadget_x + gadget_width / 2;
    scroll_y = gadget_y + gadget_height / 2;
    running = false;
}

static void draw_list_item(Word item_no, RectanglePtr bounds, CharPtr *dummy)
{
    UInt index = 0;

    if (DmSeekRecordInCategory(
	    memo_db, &index, item_no, dmSeekForward, life_category) == 0) {
	VoidHand handle = DmQueryRecord(memo_db, index);
	Int width = bounds->extent.x;
	Int length = MemHandleSize(handle);
	CharPtr ptr = MemHandleLock(handle);
	Boolean fits;

	FntCharsInWidth(ptr, &width, &length, &fits);
	WinDrawChars(ptr, length, bounds->topLeft.x, bounds->topLeft.y);

	MemHandleUnlock(handle);
    } else {
	WinDrawChars("???", 3, bounds->topLeft.x, bounds->topLeft.y);
    }
}

static void load_life()
{
    UInt numRecords;
    FormPtr dialog;
    ListPtr list;
    Word button, selection;

    memo_db = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadOnly);
    if (memo_db == 0) {
	FrmAlert(load_alert_id);
	return;
    }

    life_category = FindCategory(memo_db, "Life", true);
    if (life_category >= dmRecNumCategories
     || (numRecords = DmNumRecordsInCategory(memo_db, life_category)) == 0) {
	DmCloseDatabase(memo_db);
	FrmAlert(load_alert_id);
	return;
    }

    dialog = FrmInitForm(load_form_id);
    list = (ListPtr) FrmGetObjectPtr(dialog,
	    FrmGetObjectIndex(dialog, load_list_id));
    LstSetDrawFunction(list, &draw_list_item);
    LstSetListChoices(list, NULL, numRecords);
    button = FrmDoDialog(dialog);
    selection = LstGetSelection(list);
    FrmDeleteForm(dialog);

    if (button == load_ok_id && selection >= 0) {
	UInt index = 0;

	if (DmSeekRecordInCategory(memo_db,
		&index, selection, dmSeekForward, life_category) == 0) {
	    VoidHand handle = DmQueryRecord(memo_db, index);
	    CharPtr ptr = MemHandleLock(handle);
	    short length, i;

	    length = load(tape + init_length + life_length,
		    max_length - init_length - life_length,
		    ptr, MemHandleSize(handle));

	    MemHandleUnlock(handle);

	    if (length <= 0) {
		FrmAlert(load_error_id);
	    } else {
		for (i = 0; i < length; ++i) {
		    tape[i] = tape[init_length + life_length + i];
		}
		init_length = length;
		reset_life();
	    }
	}
    }

    DmCloseDatabase(memo_db);
}

static void draw_cells()
{
    RectangleType r;
    short i;

    r.topLeft.x = gadget_x + 1;
    r.topLeft.y = gadget_y + 1;
    r.extent.x = gadget_width - 2;
    r.extent.y = gadget_height - 2;
    WinDrawRectangleFrame(rectangleFrame, &r);
    WinEraseRectangle(&r, 0);

    for (i = 0; i < life_length; ++i) {
	short x = GET_X(tape[init_length + i]) + scroll_x;
	short y = GET_Y(tape[init_length + i]) + scroll_y;
	if (x >= r.topLeft.x && x < r.topLeft.x + r.extent.x
	 && y >= r.topLeft.y && y < r.topLeft.y + r.extent.y) {
	    WinDrawLine(x, y, x, y);
	}
    }

    SetFieldNumber((FieldPtr) GetObjectPtr(main_gens_id), gens, generation);
    SetFieldNumber((FieldPtr) GetObjectPtr(main_cells_id), cells, life_length);
}

static void set_running(Boolean run)
{
    if (run != running) {
	ControlPtr go_btn = (ControlPtr) GetObjectPtr(main_go_id);
	ControlPtr stop_btn = (ControlPtr) GetObjectPtr(main_stop_id);

	CtlEraseControl(run ? go_btn : stop_btn);
	CtlSetUsable(go_btn, ! run);
	CtlSetUsable(stop_btn, run);
	CtlDrawControl(run ? stop_btn : go_btn);
	running = run;
    }
}

static void next_gen()
{
    cell_t *life = tape + init_length;
    short length, i;

    length = merge(life, life_length, 9, adjust9,
	    life + life_length, max_length - init_length - life_length, rules);
    if (length < 0) {
	set_running(false);
	return;
    }

    if (length == life_length) {
	for (i = 0; i < length && life[i] == life[life_length + i]; ++i) {}
	if (i == length) {
	    set_running(false);
	    return;
	}
    }

    for (i = 0; i < length; ++i) {
	life[i] = life[life_length + i];
    }
    life_length = length;

    ++generation;

    draw_cells();
}

static Boolean scroll(EventPtr event)
{
    short x = event->screenX;
    short y = event->screenY;

    if (x >= gadget_x && x < gadget_x + gadget_width
     && y >= gadget_y && y < gadget_y + gadget_height) {
	if (x < gadget_x + SCROLL_MARGIN) {
	    scroll_x += SCROLL_OFFSET;
	} else if (x >= gadget_x + gadget_width - SCROLL_MARGIN) {
	    scroll_x -= SCROLL_OFFSET;
	}
	if (y < gadget_y + SCROLL_MARGIN) {
	    scroll_y += SCROLL_OFFSET;
	} else if (y >= gadget_y + gadget_height - SCROLL_MARGIN) {
	    scroll_y -= SCROLL_OFFSET;
	}
	return true;
    }
    return false;
}

static Boolean handle_main_form_event(EventPtr event)
{
    switch (event->eType) {
    case frmOpenEvent:
	FrmDrawForm(FrmGetActiveForm());
	reset_life();
	draw_cells();
    	return true;

    case frmUpdateEvent:
	draw_cells();
    	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case main_go_id:
	    set_running(true);
	    return true;

	case main_stop_id:
	    set_running(false);
	    return true;

	case main_step_id:
	    set_running(false);
	    next_gen();
	    return true;

	case main_reset_id:
	    set_running(false);
	    reset_life();
	    draw_cells();
	    return true;

	case main_load_id:
	    set_running(false);
	    load_life();
	    draw_cells();
	    return true;
	}
	break;

    case penDownEvent:
	if (scroll(event)) {
	    draw_cells();
	    return true;
	}
	break;

    case nilEvent:
	if (running) {
	    next_gen();
	}
	return true;

    default:
	break;
    }

    return false;
}

static Boolean handle_app_event(EventPtr event)
{
    FormPtr form;

    if (event->eType == frmLoadEvent) {
	form = FrmInitForm(event->data.frmLoad.formID);
	FrmSetActiveForm(form);
	if (event->data.frmLoad.formID == main_form_id) {
	    FrmSetEventHandler(form, handle_main_form_event);
	}
	return true;
    }
    return false;
}

static void event_loop()
{
    EventType event;
    Word error;

    do {
	EvtGetEvent(&event, running ? 1 : 20);
	if (SysHandleEvent(&event) == false
	 && MenuHandleEvent(NULL, &event, &error) == false
	 && handle_app_event(&event) == false) {
	    FrmDispatchEvent(&event);
	}
    } while (event.eType != appStopEvent);
}

DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    if (cmd == sysAppLaunchCmdNormalLaunch) {
	init_life();
	FrmGotoForm(main_form_id);
	event_loop();
	free_life();
    }
    return 0;
}
