/*
 * This file is part of FDNPKG
 * Copyright (C) Mateusz Viste 2013. All rights reserved.
 * 
 * Test program for libunzip.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>     /* ctime() */
#include "crc32.h"
#include "libunzip.h"
#include "kinflate.h"
#include "lzmadec.h"


static void *SzAlloc(void *p, size_t size) {
  p = p; /* for gcc to not complain */
  if (size == 0) return(0);
  return(malloc(size));
}

static void SzFree(void *p, void *address) {
  p = p;  /* for gcc to not complain */
  free(address);
}



int main(int argc, char **argv) {
  char *zipfile, *filetoextract;
  struct ziplist *zipindex, *zipcursor;
  FILE *fd;

  if (argc != 3) {
    puts("Usage: test-unz file.zip filetoextract");
    return(1);
  }

  zipfile = argv[1];
  filetoextract = argv[2];

  fd = fopen(zipfile ,"rb");
  if (fd == NULL) {
    puts("zip file could not be opened!");
    return(3);
  }

  zipindex = zip_listfiles(fd);

  if (zipindex == NULL) {
    puts("Error: could not parse the zip file!");
    return(2);
  }

  for (zipcursor = zipindex; zipcursor != NULL; zipcursor = zipcursor->nextfile) {
    printf("%s [%ld->%ld]/meth #%d/CRC %08lX/offset 0x%lX/time %s", zipcursor->filename, zipcursor->filelen, zipcursor->compressedfilelen, zipcursor->compmethod, zipcursor->crc32, zipcursor->dataoffset, ctime(&zipcursor->timestamp));

    if (strcmp(zipcursor->filename, filetoextract) == 0) {
      int x;
      unsigned long crc32;
      FILE *xfd;
      printf("** found %s -> extracting to unz.out\n", filetoextract);
      xfd = fopen("unz.out", "wb");
      if (xfd != NULL) {
        if (zipcursor->compmethod == 8) { /* DEFLATE */
          kunzip_inflate_init();
          fseek(fd, zipcursor->dataoffset, SEEK_SET); /* place the cursor where data begins */
          x = kunzip_inflate(fd, xfd, &crc32);
          printf("inflate() returned %d (CRC: %08lX)\n", x, crc32);
          fclose(xfd);
          kunzip_inflate_free();
        } else if (zipcursor->compmethod == 14) {  /* LZMA */
          #define buffinsize 1024
          #define buffoutsize 8192 /* it's better for the output buffer to be significantly larger than the input buffer (we are decompressing here, remember?) */
          unsigned char *buffout;
          unsigned char *buffin;
          int bytesread, bytesreadtotal = 0, byteswritetotal = 0;
          SizeT buffoutreslen, bytesprocessed;
          ISzAlloc g_alloc;
          ELzmaStatus lzmastatus;
          SRes lzmaresult;
          CLzmaDec lzmahandle;
          unsigned char lzmahdr[LZMA_PROPS_SIZE]; /* 5 bytes of properties */
          unsigned long crc32;

          fseek(fd, zipcursor->dataoffset, SEEK_SET); /* place the cursor where data begins */

          fread(lzmahdr, 4, 1, fd); /* load the 4 bytes long 'zip-lzma header */
          if ((lzmahdr[2] != 5) || (lzmahdr[3] != 0)) puts("unsupported lzma variant"); /* lzma properties should be 5 bytes long. If it's not, it's either not valid lzma, or some version that wasn't existing yet when I wrote these words */
          bytesreadtotal = 4; /* remember we read 4 bytes already */

          buffin = malloc(buffinsize);
          buffout = malloc(buffoutsize);
          if ((buffin == NULL) || (buffout == NULL)) puts("MALLOC ERROR!");;

          crc32 = crc32_init();

          g_alloc.Alloc = SzAlloc;
          g_alloc.Free = SzFree;

          fread(lzmahdr, sizeof(lzmahdr), 1, fd); /* load the lzma header */
          bytesreadtotal += sizeof(lzmahdr);
          
          /* Note, that in a 'normal' lzma stream we would have now 8 bytes with the uncompressed length of the file. Here we don't. ZIP cut this information out, since it stores it already in its own header. */

          memset(&lzmahandle, 0, sizeof(lzmahandle)); /* reset the whole lzmahandle structure - not doing this leads to CRASHES!!! */
          LzmaDec_Init(&lzmahandle);
          RINOK(LzmaDec_Allocate(&lzmahandle, lzmahdr, LZMA_PROPS_SIZE, &g_alloc));

          printf("lzma stream offset at 0x%lX\n", zipcursor->dataoffset);

          for (;;) {
            bytesread = buffinsize;
            if (bytesread > zipcursor->compressedfilelen - bytesreadtotal) bytesread = zipcursor->compressedfilelen - bytesreadtotal;
            buffoutreslen = buffoutsize;
            bytesprocessed = bytesread;
            printf("Will read %d bytes from input stream\n", bytesread);
            fread(buffin, bytesread, 1, fd); /* read stuff from input stream */
            fseek(fd, 0 - bytesread, SEEK_CUR); /* get back to the position at the start of our chunk of data */
            lzmaresult = LzmaDec_DecodeToBuf(&lzmahandle, buffout, &buffoutreslen, buffin, &bytesprocessed, LZMA_FINISH_ANY, &lzmastatus);
            bytesreadtotal += bytesprocessed;
            printf("expanded %ld bytes into %ld (total read: %ld bytes)\n", (long)bytesprocessed, (long)buffoutreslen, (long)bytesreadtotal);
            fseek(fd, bytesprocessed, SEEK_CUR); /* go forward to the position next to the input we processed */
            if (lzmaresult != SZ_OK) {
              puts("lzmaresult != SZ_OK");
              if (lzmaresult == SZ_ERROR_DATA) puts("DATA ERROR");
              if (lzmaresult == SZ_ERROR_MEM) puts("MEMORY ALLOC ERROR");
              if (lzmaresult == SZ_ERROR_UNSUPPORTED) puts("UNSUPPORTED PROPERTY");
              if (lzmaresult == SZ_ERROR_INPUT_EOF) puts("NEED MORE INPUT");
              break;
            }
            byteswritetotal += buffoutreslen;
            fwrite(buffout, buffoutreslen, 1, xfd); /* write stuff to output file */
            crc32_feed(&crc32, buffout, buffoutreslen);
            if (lzmastatus == LZMA_STATUS_FINISHED_WITH_MARK) puts("lzma says we are done!");
            if ((lzmastatus == LZMA_STATUS_FINISHED_WITH_MARK) || (bytesreadtotal >= zipcursor->compressedfilelen)) break;
          }
          crc32_finish(&crc32);
          printf("Processed %d bytes of input into %d bytes of output. CRC32: %08lX\n", bytesreadtotal, byteswritetotal, crc32);
          fclose(xfd);
          free(buffin);
          free(buffout);
        } else { /* some unknown compression scheme OR stored file */
          int x;
          unsigned char bytebuff[16];
          fseek(fd, zipcursor->dataoffset, SEEK_SET); /* place the cursor where data begins */
          for (x = 0; x < zipcursor->compressedfilelen; x++) {
            fread(bytebuff, 1, 1, fd);
            fwrite(bytebuff, 1, 1, xfd);
          }
          fclose(xfd);
          printf("%d bytes dumped\n", x);
        }
      }
    }
  }

  fclose(fd);
  zip_freelist(&zipindex);

  return(0);
}
