
/*
 * This file is part of FDNPKG.
 * 
 * Loads the list of repositories from the config file specified in %FDNPKG%.
 * Returns the amount of repositories found (and loaded) on success, or -1 on failure.
 * 
 * Copyright (C) Mateusz Viste 2012, 2013
 */

#include <stdio.h>  /* printf(), fclose(), fopen()... */
#include <string.h> /* strcasecmp() */
#include <stdlib.h> /* malloc(), free() */
#include "crc32.h"  /* crc32() */
#include "kprintf.h"
#include "loadconf.h"


void freeconf(char **repolist, int repscount) {
  int x;
  for (x = 0; x < repscount; x++) free(repolist[x]);
}


static int checkfordoubledrepos(char **repolist, int repocount) {
  int x, y;
  for (x = 0; x < (repocount - 1); x++) {
    for (y = x + 1; y < repocount; y++) {
      if (strcmp(repolist[x], repolist[y]) == 0) {
        kitten_printf(7, 0, "Error: repository '%s' is listed twice!", repolist[x]);
        puts("");
        return(-1);
      }
    }
  }
  return(0);
}


int loadconf(char *cfgfile, char **repolist, int maxreps, unsigned long *crc32val, int *maxcachetime) {
  int bytebuff, parserstate = 0;
  FILE *fd;
  unsigned char *fbuff;
  #define maxtok 16
  char token[maxtok];
  #define maxval 1024
  char value[maxval];
  int curtok = 0, curval = 0, nline = 1;
  int repocount = 0;
  int filesize;
  fd = fopen(cfgfile, "r");
  if (fd == NULL) {
    kitten_printf(7, 1, "Error: Could not open config file '%s'!", cfgfile);
    puts("");
    return(-1);
  }

  /* compute the CRC32 of the configuration file */
  fseek(fd, 0L, SEEK_END);
  filesize = ftell(fd);
  rewind(fd);
  fbuff = malloc(filesize);
  if (fbuff == NULL) {
    fclose(fd);
    kitten_puts(7, 7, "Out of memory while reading repositories list!");
    puts("");
    return(-1);
  }
  fread(fbuff, sizeof(char), filesize, fd);
  rewind(fd);
  *crc32val = crc32(0L, fbuff, filesize);
  free(fbuff);

  /* read the config file line by line */
  do {
    bytebuff = fgetc(fd);
    if (bytebuff != '\r') {
      switch (parserstate) {
        case 0: /* Looking for start of line */
          if ((bytebuff == EOF) || (bytebuff == ' ') || (bytebuff == '\t')) break;
          if (bytebuff == '\n') {
            nline += 1;
            break;
          }
          if (bytebuff == '#') {
              parserstate = 9;
            } else {
              token[0] = bytebuff;
              curtok = 1;
              parserstate = 1;
          }
          break;
        case 1: /* Looking for token end */
          if ((bytebuff == EOF) || (bytebuff == '\n')) {
            kitten_printf(7, 2, "Warning: token without value on line #%d", nline);
            puts("");
            if (bytebuff == '\n') nline += 1;
            parserstate = 0;
            break;
          }
          if ((bytebuff == ' ') || (bytebuff == '\t')) {
              token[curtok] = 0;
              parserstate = 2;
            } else {
              token[curtok] = bytebuff;
              curtok += 1;
              if (curtok >= maxtok) {
                parserstate = 9; /* ignore the line */
                kitten_printf(7, 3, "Warning: Config file token overflow on line #%d", nline);
                puts("");
              }
          }
          break;
        case 2: /* Looking for value start */
          if ((bytebuff == EOF) || (bytebuff == '\n')) {
            kitten_printf(7, 4, "Warning: token with empty value on line #%d", nline);
            puts("");
            if (bytebuff == '\n') nline += 1;
            parserstate = 0;
            break;
          }
          if ((bytebuff != ' ') && (bytebuff != '\t')) {
            value[0] = bytebuff;
            curval = 1;
            parserstate = 3;
          }
          break;
        case 3: /* Looking for value end */
          if ((bytebuff == EOF) || (bytebuff == '\n')) {
              parserstate = 0;
              value[curval] = 0;
              if ((value[curval - 1] == ' ') || (value[curval - 1] == '\t')) {
                kitten_printf(7, 5, "Warning: trailing white-space(s) after value on line #%d", nline);
                puts("");
                while ((value[curval - 1] == ' ') || (value[curval - 1] == '\t')) value[--curval] = 0;
              }
              /* Interpret the token/value pair now! */
              /* printf("token='%s' ; value = '%s'\n", token, value); */
              if (strcasecmp(token, "REPO") == 0) { /* Repository declaration */
                  if (repocount >= maxreps) {
                      kitten_printf(7, 6, "Dropped a repository: too many configured (max=%d)", maxreps);
                      puts("");
                    } else {
                      repolist[repocount] = malloc(curval + 1); /* +1 for a trailing / if not specified */
                      if (repolist[repocount] == NULL) {
                        kitten_puts(7, 7, "Out of memory while reading repositories list!");
                        freeconf(repolist, repocount);
                        return(-1);
                      }
                      if (value[curval - 1] != '/') { /* add a trailing slash to the url if not there already */
                          sprintf(repolist[repocount], "%s/", value);
                        } else { 
                          sprintf(repolist[repocount], "%s", value);
                      }
                      repocount += 1;
                  }
                } else if (strcasecmp(token, "MAXCACHETIME") == 0) {
                  int tmpint = atoi(value);
                  if ((tmpint >= 0) && (tmpint < 1209600)) { /* min 0s / max 2 weeks */
                      *maxcachetime = tmpint;
                    } else {
                      kitten_printf(7, 10, "Warning: Ignored an illegal 'maxcachetime' value at line #%d", nline);
                      puts("");
                  }
                } else { /* unknown token */
                  kitten_printf(7, 8, "Warning: Unknown token '%s' at line #%d", token, nline);
                  puts("");
              }
              /* interpretation done */
              if (bytebuff == '\n') nline += 1;
            } else {
              value[curval] = bytebuff;
              curval += 1;
              if ((curval + 1) >= maxval) {
                parserstate = 9; /* ignore the line */
                kitten_printf(7, 9, "Warning: Config file value overflow on line #%d", nline);
                puts("");
              }
          }
          break;
        case 9: /* Got comment, ignoring the rest of line */
          if (bytebuff == EOF) break;
          if (bytebuff == '\n') {
            nline += 1;
            parserstate = 0;
          }
          break;
      }
    }
  } while (bytebuff != EOF);
  fclose(fd);

  /* Look out for doubled repositories */
  if (checkfordoubledrepos(repolist, repocount) != 0) return(-1);

  return(repocount);
}
