/*
 *   HTGET
 *
 *   Get a document via HTTP
 *
 *   This program was compiled under GNU C v4.3.2 (DJGPP v2.03)
 *   You will require the WATT32 libraries. A copy is included in the
 *   distribution. For the sources of WATTCP, search on the Watt32 project's
 *   homepage.
 *
 *   Please send bug fixes, ports and enhancements to the current maintainer
 *   for incorporation in newer versions: <mateusz(#)viste-family.net>
 *
 *   Copyright (C) 1996-1998 Ken Yap
 *
 *   This program is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU/GPL, a copy of which is included with this
 *   distribution.
 *
 *   THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 *   WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 *   MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <io.h>
#include <tcp.h>

#include "htget.h"

#ifdef __GNUC__
#define write _write
#define close _close
#endif

#if (defined(__SMALL__) && !defined(DOS386)) || defined(__LARGE__)
  #define SAVE_SPACE
#else
  #define perror perror_s  /* prevent duplicate symbols */
#endif

#define WINDOWSIZE  (16*1024)
#define BUFFERSIZE  2048
#define AUTHOR      "Copyright (C) 1996-1998 Ken Yap, 2009 Mateusz Viste"

#define HTTPVER     "HTTP/1.0"
#define HTTPVER10   "HTTP/1.0"
#define HTTPVER11   "HTTP/1.1"
#define strn(s)     s, sizeof(s)-1

#define debugMode 0
#define verboseMode 0


/* this is a wrapper around the wattcp lookup_host(), but with a small integrated cache */
static DWORD dnsresolve(const char *name) {
  DWORD hostaddr = 0;
  static DWORD cacheaddr[16];
  static char  cachename[16][64] = {{0},{0},{0},{0},{0},{0},{0},{0},{0},{0},{0},{0},{0},{0},{0},{0}};
  int x, freeentry = -1;
  int namelen = strlen(name);
  if (namelen < 64) {
    for (x = 0 ; x < 16 ; x++) {
      if (cachename[x][0] == 0) { /* empty (free) entry */
          if (freeentry == -1) freeentry = x; /* remember it for later */
        } else { /* else check if it's what we need */
          if (strcmp(cachename[x], name) == 0) return(cacheaddr[x]); /* if found in cache, stop here */
      }
    }
  }
  hostaddr = lookup_host(name,NULL);
  if (hostaddr == 0) return(0); /* dns resolving error */
  if ((namelen < 64) && (freeentry >= 0)) { /* if not longer than maxlen, and cache not full, then save it */
    strcpy(cachename[freeentry], name); /* save name in cache */
    cacheaddr[freeentry] = hostaddr; /* save addr in cache */
  }
  return(hostaddr);
}


static void base64encode (char *in, char *out) {
  int  c1, c2, c3;
  int  len = 0;
  static char basis_64[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
  while (*in) {
    c1 = *in++;
    if (*in == '\0') {
        c2 = c3 = 0;
      } else {
        c2 = *in++;
        if (*in == '\0') {
            c3 = 0;
          } else {
            c3 = *in++;
        }
    }
    *out++ = basis_64 [c1 >> 2];
    *out++ = basis_64 [((c1 & 0x3) << 4) | ((c2 & 0xF0) >> 4)];
    len += 2;
    if (c2 == 0 && c3 == 0) {
        *out++ = '=';
        *out++ = '=';
        len += 2;
      } else if (c3 == 0) {
        *out++ = basis_64 [((c2 & 0xF) << 2) | ((c3 & 0xC0) >> 6)];
        *out++ = '=';
        len += 2;
      } else {
        *out++ = basis_64 [((c2 & 0xF) << 2) | ((c3 & 0xC0) >> 6)];
        *out++ = basis_64 [c3 & 0x3F];
        len += 2;
    }
  }
  *out = '\0';
}

/*
 * WATTCP's sock_gets() doesn't do the right thing.
 * That's because sock isn't set to ASCII mode.
 */
static int sock_getline (void *sock, char *buf, int len) {
  int  i;
  char ch;
  for (i = 0, --len; i <= len && sock_read(sock,&ch,1) > 0; ) {
    if (ch == '\n') break;
    if (ch != '\r') {
      *buf++ = ch;
      ++i;
    }
  }
  *buf = '\0';
  return(i);
}

static long header(const char *path, char *buffer, tcp_Socket *sock) {
  int   i, len, response;
  long  contentlength;
  char *s;

  contentlength = LONG_MAX;     /* largest int, in case no CL header */
  if ((len = sock_getline(sock, buffer, BUFFERSIZE)) <= 0) {
    puts ("EOF from server");
    return (-1L);
  }
  if (strncmp(s = buffer,strn(HTTPVER10)) && strncmp(s,strn(HTTPVER11))) { /* no HTTP/1.[01]? */
    puts ("Not a HTTP/1.[01] server");
    return (-1L);
  }

  s += sizeof(HTTPVER10)-1;
  if ((i = strspn(s, " \t")) <= 0) {      /* no whitespace? */
    puts ("Malformed HTTP/1.[01] line");
    return (-1L);
  }
  s += i;
  response = 500;
  sscanf (s, "%3d", &response);
  if (response == 401) {
      printf ("%s: authorisation failed!\n", path);
      return (-1L);
    } else if (response != 200 && response != 301 && response != 302 && response != 304) {
      printf ("%s: %s\n", path, s);
      contentlength = -1L;
  }

  /* eat up the other header lines here */
  while ((len = sock_getline(sock, buffer, BUFFERSIZE)) > 0) {
    if (!strnicmp(s=buffer,strn("Content-Length:"))) {
        s += sizeof("Content-Length:")-1;
        contentlength = atol(s);
      } else if (!strnicmp(buffer, strn("Location:"))) {
        if (response == 301 || response == 302) printf ("At %s\n", buffer);
      } else if (strchr(" \t", buffer[0])) {
        printf ("Warning: continuation line encountered\n");
    }
  }
  return (response == 304 ? 0L : contentlength);
}


/* orgurl = your url, like 'http://www.example.org:81/file.htm
 * userPass = the user:pass http auth credentials. If none, then userPass must be NULL
 * proxy_host = do we use a proxy? If not, must be NULL
 * proxy_port = the port for the proxy (if used)
 * outputfile = the file where the data should be written (must be a valid path/filename - if NULL, then the data will be sent on stdout)
*/
int htget(char *orgurl, char *userPass, char *proxy_host, int proxy_port, char *outputfile) {
  struct in_addr a,b;
  DWORD hostaddr;
  int status = 0;
  int connected = 0;
  int completed = 0;
  int i, use_proxy = (proxy_host != NULL);
  long length, contentlength = 0L;
  const char *name;
  char *buffer;
  char *buf;
  char url[512];
  char *host, *path;
  char *s;
  int port = 80;
  int output;
  tcp_Socket *sock;
  FILE *of;

  /* parse the URL and explode it into host, port and path */
  if (!strnicmp(orgurl, strn("http://"))) orgurl += sizeof("http://") - 1;

  if ((path = strchr(orgurl,'/')) == NULL) { /* separate out the path */
      host = orgurl;
      path = "/";      /* top directory */
    } else {
      if ((host = calloc(path-orgurl+1,1)) == NULL) {
        printf("Out of memory\n");
        return(-1);
      }
      strncat(host, orgurl, path-orgurl);
  }

  /* do we have a port number? */
  if ((s = strchr(host,':')) != NULL) {
    *s++ = '\0';
    port = atoi(s);
  }

  /* Let's do serious stuff now */
  buffer = malloc (BUFFERSIZE);
  sock   = malloc (sizeof(*sock));
  if (!buffer || !sock) {
    puts ("No memory");
    return (-1);
  }
  buf = buffer;

  if (proxy_host != NULL) {
      name = proxy_host;
    } else {
      name = host;
  }

  if (debugMode != 0) {
    tcp_set_debug_state(1);
    dbug_init();
  }
  tzset();
  /* set_cnf_hook(); */
  sock_init();

  if ((hostaddr = dnsresolve(name)) == 0) {
    printf (dom_strerror(dom_errno));
    return (1);
  }

#ifndef SAVE_SPACE
  a.s_addr = intel (my_ip_addr);      /* A-side is me   */
  inet_aton (host, &b);               /* B-side is host */
  if (inet_netof(a) == inet_netof(b)) use_proxy = 0; /* on same network */
#endif


#if 0   /* test */
  {
    struct in_addr na, nb;
    na.s_addr = inet_netof (a);
    nb.s_addr = inet_netof (b);

    printf ("A [%s]: netof: %s\n", inet_ntoa(a), inet_ntoa(na));
    printf ("B [%s]: netof: %s\n", inet_ntoa(b), inet_ntoa(nb));

    na.s_addr = htonl (inet_lnaof(a));
    nb.s_addr = htonl (inet_lnaof(b));
    printf ("A [%s]: lnaof: %s\n", inet_ntoa(a), inet_ntoa(na));
    printf ("B [%s]: lnaof: %s\n", inet_ntoa(b), inet_ntoa(nb));

    printf ("use_proxy = %d\n", use_proxy);
  }
#endif

  if ((use_proxy != 0) && (proxy_port != 0) && (proxy_host != NULL)) port = proxy_port;
  if (debugMode) printf ("%s:%d%s\n", host, port, path);
  if (!tcp_open(sock, 0, hostaddr, port, NULL)) {
    printf ("Cannot connect to `%s'\n", name);
    return (1);
  }

  sock_setbuf (sock, malloc(WINDOWSIZE), WINDOWSIZE);
  sock_wait_established (sock, sock_delay, NULL, &status);
  connected = 1;
  completed = 1;
  sock_tick (sock, &status);      /* in case they sent reset */

  if (verboseMode) puts("Sending HTTP GET request");

  if (proxy_host != NULL) {
      sprintf(url, "http://%s:%d%s", host, port, path);
    } else {
      sprintf(url, "%s", path);
  }

  buf += sprintf (buf, "GET %s %s\r\nHost: %s\r\nUser-Agent: FDNPKG\r\n", url, HTTPVER, host);
  if (userPass != NULL) {
    char pass [100];
    base64encode(userPass, pass);
    if (debugMode) printf ("%s => %s\n", userPass, pass);
    buf += sprintf (buf, "Authorization: Basic %s\r\n", pass);
  }

  buf += sprintf(buf, "\r\n");
  sock_fastwrite(sock, buffer, buf-buffer);

  if ((contentlength = header(path, buffer, sock)) >= 0L) {
    /* We wait until the last moment to open the output file.
     * If any specified so that we won't overwrite the file
     * in case of error in contacting server.
     */
    if (outputfile != NULL) {
        if ((of = fopen(outputfile, "wb")) == NULL) {
          perror(outputfile);
          goto close_up;
        }
        output = fileno(of);
      } else { /* if no filename provided, we will send everything to stdout */
        output = fileno(stdout);
    }

    length = 0L;
    while ((i = sock_read(sock,buffer,BUFFERSIZE)) > 0) {
      write (output, buffer, i);
      length += i;
      if (verboseMode) {
        printf("Got %lu bytes\r", length);
        fflush(stdout);
      }
    }
    close(output);
    if ((contentlength != LONG_MAX) && (length != contentlength)) printf ("Warning, actual length = %ld, content length = %ld\n", length, contentlength);
  }

close_up:
  sock_close(sock);
  sock_wait_closed(sock, sock_delay, NULL, &status);

sock_err:
  if (status == -1) printf ("`%s' %s\n", name, sockerr(sock));
  if (!connected) puts ("Could not get connected");
  return (!completed || contentlength < 0L);
}


/*--------------------------------------------------------------*/

/*
static void Exit (char *str) {
  puts(str);
  exit(1);
}
*/

/*
void cnf_hook (const char *name, const char *value) {
  if (!strcmp(name,"HTTP.PROXY")) {
      if (sscanf(value,"%[^:]:%d",proxy_host,&proxy_port) != 2) Exit ("Config error: syntax is HTTP_PROXY=<host>:<port>");
    } else if (prev_hook) {
      (*prev_hook) (name, value);
  }
}

void set_cnf_hook (void) {
  prev_hook = usr_init;
  usr_init  = cnf_hook;
}
*/


/* int htgetmain() { */
  /* void     (*prev_hook) (const char*, const char*); */
/*
  int status;
  status = htget("www.wp.pl", NULL, NULL, 0, "c:\\temp\\htget.htm");
  return(status);
}
*/
