/*
   fdnpkg - FreeDOS Network Package manager
   Copyright (C) Mateusz Viste 2012, 2013
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see <http://www.gnu.org/licenses/>.
*/


#include <ctype.h>    /* tolower() */
#include <stdio.h>    /* printf() */
#include <stdlib.h>   /* getenv() */
#include <string.h>   /* strcasecmp() */
#include "htget.h"    /* htget() */
#include "kittendj.h" /* required for kitten subsystem */
#include "kprintf.h"  /* kitten_printf() */
#include "loadconf.h" /* loadconf() */
#include "pkgdb.h"    /* createdb(), findpkg()... */
#include "pkgsrch.h"  /* pkgsrch() */
#include "pkginst.h"  /* pkginstall() */
#include "pkgrem.h"   /* pkgrem() */

#define PVER "0.92"
#define PDATE "2012,2013"


extern char *wattcpVersion(); /* provided by wattcp to poll its version */

void printhelp() {
  puts("FDNPKG v" PVER " Copyright (C) Mateusz Viste " PDATE);
  kitten_puts(1, 0, "This is a network package manager for FreeDOS.");
  puts("");
  kitten_puts(1, 1, "Usage: FDNPKG action [parameters]");
  puts("");
  kitten_puts(1, 2,  "Where action is one of the following:");
  kitten_puts(1, 3,  " search [string]   - Search net repositories for package containing 'string'");
  kitten_puts(1, 4,  " vsearch [string]  - Same as 'search', but prints also source repositories");
  kitten_puts(1, 5,  " install pkg       - Install the package 'pkg' (or local zip file)");
  kitten_puts(1, 10, " install-nosrc pkg - Install the package 'pkg' (or local zip file) w/o sources");
  kitten_puts(1, 6,  " remove pkgname    - Remove the package 'pkgname'");
  kitten_puts(1, 7,  " listrep           - print out the list of configured repositories");
  kitten_puts(1, 8,  " license           - print out the license of this program");
  puts("");
  kitten_puts(1, 9, "FDNPKG is linked against WatTCP version below:");
  puts(wattcpVersion());
}


void printlic() {
  puts("FDNPKG - FreeDOS Network Package manager");
  puts("Copyright (C) Mateusz Viste 2012,2013");
  puts("");
  puts("This program is free software: you can redistribute it and/or modify it under");
  puts("the terms of the GNU General Public License as published by the Free Software");
  puts("Foundation, either version 3 of the License, or (at your option) any later");
  puts("version.");
  puts("");
  puts("This program is distributed in the hope that it will be useful, but WITHOUT ANY");
  puts("WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A");
  puts("PARTICULAR PURPOSE. See the GNU General Public License for more details.");
  puts("");
  puts("You should have received a copy of the GNU General Public License along with");
  puts("this program. If not, see <http://www.gnu.org/licenses/>.");
  puts("");
}


enum actions {
  ACTION_HELP,
  ACTION_SEARCH,
  ACTION_INSTALL,
  ACTION_INSTALL_LOCALFILE,
  ACTION_REMOVE,
  ACTION_LISTREP,
  ACTION_LICENSE
};


#define QUIT(x) \
  kittenclose(); \
  return(x);


int main(int argc, char **argv) {
  int htgetres;
  char *tempdir;
  char cfgfile[256];
  char *dosdir;
  #define MAXREPS 64
  char *repolist[MAXREPS];
  int repolistcount;
  int verbosemode = 0;
  int nosourceflag = 0;
  int maxcachetime = 7200;
  unsigned long cfgfilecrc;
  struct pkgdb *pkgdb = NULL;
  enum actions action = ACTION_HELP;

  /* First init KITTEN */
  kittenopen("FDNPKG");

  /* check if %TEMP% is set, and retrieve it */
  tempdir = getenv("TEMP");
  if (tempdir == NULL) {
    kitten_puts(2, 0, "%TEMP% not set! You should make it point to a writeable directory.");
    kitten_puts(2, 1, "Example: SET TEMP=C:\\TEMP");
    QUIT(0)
  }

  /* check if %DOSDIR% is set, and retrieve it */
  dosdir = getenv("DOSDIR");
  if (dosdir == NULL) {
    kitten_puts(2, 2, "%DOSDIR% not set! You should make it point to the FreeDOS main directory.");
    kitten_puts(2, 3, "Example: SET DOSDIR=C:\\FDOS");
    QUIT(0)
  }

  if (getenv("TZ") == NULL) {
    kitten_puts(2, 12, "Warning: %TZ% not set! timestamps on installed files might be inaccurate.");
    setenv("TZ", "UTC", 0); /* set TZ, if not set already (needed by UNZIP) */
  }

  sprintf(cfgfile, "%s\\bin\\fdnpkg.cfg", dosdir);

  /* parse cli parameters */
  if (argc > 1) { /* fdnpkg action [param] */
    if (strcasecmp(argv[1], "search") == 0) {
        if (argc > 3) {
            kitten_puts(2, 4, "Invalid number of arguments. Run fdnpkg without any parameter for help.");
            QUIT(0)
          } else {
            action = ACTION_SEARCH;
        }
      } else if (strcasecmp(argv[1], "vsearch") == 0) {
        if (argc > 3) {
            kitten_puts(2, 4, "Invalid number of arguments. Run fdnpkg without any parameter for help.");
            QUIT(0)
          } else {
            action = ACTION_SEARCH;
            verbosemode = 1;
        }
      } else if ((strcasecmp(argv[1], "install") == 0) || (strcasecmp(argv[1], "install-nosrc") == 0)) {
        if (strcasecmp(argv[1], "install-nosrc") == 0) nosourceflag = 1;
        if (argc != 3) {
            kitten_puts(2, 4, "Invalid number of arguments. Run fdnpkg without any parameter for help.");
            QUIT(0)
          } else {
            int arglen = strlen(argv[2]);
            action = ACTION_INSTALL;
            if (arglen > 4) {
              if ((argv[2][arglen - 4] == '.') && (tolower(argv[2][arglen - 3]) == 'z') && (tolower(argv[2][arglen - 2]) == 'i') && (tolower(argv[2][arglen - 1]) == 'p')) { /* if argument ends with '.zip', then it's a local package file */
                action = ACTION_INSTALL_LOCALFILE;
              }
            }
        }
      } else if (strcasecmp(argv[1], "remove") == 0) {
        if (argc != 3) {
            kitten_puts(2, 4, "Invalid number of arguments. Run fdnpkg without any parameter for help.");
          } else {
            pkgrem(argv[2], dosdir);
        }
        QUIT(0);
      } else if (strcasecmp(argv[1], "listrep") == 0) {
        if (argc != 2) {
            kitten_puts(2, 4, "Invalid number of arguments. Run fdnpkg without any parameter for help.");
            QUIT(0)
          } else {
            action = ACTION_LISTREP;
        }
      } else if (strcasecmp(argv[1], "license") == 0) {
        if (argc != 2) {
            kitten_puts(2, 4, "Invalid number of arguments. Run fdnpkg without any parameter for help.");
          } else {
            printlic();
        }
        QUIT(0)
    }
  }

  if (action == ACTION_HELP) {
    printhelp();
    QUIT(0)
  }

  /* if we install from a local file, do it and quit */
  if (action == ACTION_INSTALL_LOCALFILE) {
    char pkgname[64];
    int t, lastpathdelim = 0, u = 0;
    for (t = 0; argv[2][t] != 0; t++) {
      if ((argv[2][t] == '/') || (argv[2][t] == '\\')) lastpathdelim = t;
    }
    /* copy the filename into pkgname (without path elements) */
    for (t = lastpathdelim + 1; argv[2][t] != 0; t++) pkgname[u++] = argv[2][t];
    pkgname[u] = 0; /* terminate the string */
    /* truncate the file's extension (.zip) */
    for (t = u; t > 0; t--) {
      if (pkgname[t] == '.') {
        pkgname[t] = 0;
        break;
      }
    }
    pkginstall(NULL, pkgname, dosdir, tempdir, argv[2], nosourceflag, repolist);
    QUIT(0)
  }

  /* Load the list of package repositories */
  repolistcount = loadconf(cfgfile, repolist, MAXREPS, &cfgfilecrc, &maxcachetime);
  if (repolistcount < 0) return(5);
  if (repolistcount == 0) {
    kitten_puts(2, 5, "No repository is configured. You need at least one.");
    kitten_puts(2, 6, "You should place in your configuration file at least one entry of such form:");
    kitten_puts(2, 7, "REPO www.freedos.org/repo");
    puts("");
    QUIT(0)
  }

  if (action == ACTION_LISTREP) { /* if all we wanted was to print out repositories... */
      int x;
      kitten_puts(2, 8, "The list of configured fdnpkg repositories follows:");
      for (x = 0; x < repolistcount; x++) puts(repolist[x]);
      puts("");
    } else { /* other actions: search, install */
      pkgdb = createdb();
      if (pkgdb != NULL) {
        int x;
        char tempfile[512];
        char repoindex[512];
        sprintf(tempfile, "%s\\fdnpkg.db", tempdir);
        if (loaddb_fromcache(pkgdb, tempfile, cfgfilecrc, maxcachetime) == 0) { /* load db from cache (if not older than 2h) */
            kitten_puts(2, 13, "Package database loaded from local cache.");
          } else {
            freedb(&pkgdb);      /* recreate the db from scratch, because after */
            pkgdb = createdb();  /* a cache attempt it might contain garbage */
            sprintf(tempfile, "%s\\fdnpkg.tmp", tempdir);
            /* download every repo index into %TEMP% and load them in RAM */
            for (x = 0; x < repolistcount; x++) {
              kitten_printf(2, 9, "Refreshing %s...", repolist[x]);
              puts("");
              sprintf(repoindex, "%sindex.lst", repolist[x]);
              htgetres = htget(repoindex, NULL, NULL, 0, tempfile);
              if (htgetres != 0) {
                  kitten_puts(2, 10, "Repository download failed!");
                  maxcachetime = 0; /* disable cache writing this time */
                } else {
                  if (loaddb(pkgdb, tempfile, x) != 0) {
                    kitten_puts(2, 11, "An error occured while trying to load repository from tmp file...");
                    maxcachetime = 0; /* disable cache writing this time */
                  }
              }
            }
            /* save results into the (new) cache file db */
            if (maxcachetime > 0) {
              sprintf(tempfile, "%s\\fdnpkg.db", tempdir);
              dumpdb(pkgdb, tempfile, cfgfilecrc);
            }
        }
        if (action == ACTION_SEARCH) { /* for search: iterate through the sorted db, and print out all packages that match the pattern */
            if (argc == 3) { /* a search term has been provided */
                pkgsearch(pkgdb, argv[2], verbosemode, repolist);
              } else {
                pkgsearch(pkgdb, NULL, verbosemode, repolist);
            }
          } else if (action == ACTION_INSTALL) { /* for install: iterate through the sorted db until you find package pkgname. Then, install it or update it. */
            pkginstall(pkgdb, argv[2], dosdir, tempdir, NULL, nosourceflag, repolist);
        }
        /* free memory of the pkg database */
        freedb(&pkgdb);
      } /* pkgdb != NULL */
  } /* action == ACTION_LISTREP */

  freeconf(repolist, repolistcount);
  QUIT(0);
}
