/****************************************************************************

  TREE - Graphically displays the directory structure of a drive or path

  Written to work with FreeDOS (and other DOS variants)
  Win32(c) console and DOS with LFN support.

****************************************************************************/

#define VERSION "1.01"

/****************************************************************************

  Written by: Kenneth J. Davis
  Date:       August, 2000
  Updated:    September, 2000; October, 2000; November, 2000; January, 2001
  Contact:    jeremyd@computer.org


Copyright (c): Public Domain [United States Definition]

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR AUTHORS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT
OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

****************************************************************************/

/**
 * Define the appropriate target here or within your compiler 
 */
/* #define WIN32 */    /** Win32 console version **/
/* #define DOS */      /** DOS version           **/
/* #define UNIX */

/** 
 * Used to determine whether catgets (LGPL under Win32 & DOS) is used or not.
 * Undefine, ie comment out, to use hard coded strings only.
 */
/* #define USE_CATGETS */


/* Include files */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <direct.h>
#include <ctype.h>

#include "stack.h"


/* Platform (OS) specific definitions */

#ifdef WIN32       /* windows specific     */
#include <windows.h>
#include <winbase.h>
/* converts from Ansi to OEM (IBM extended )    */
#define charToDisplayChar(x) CharToOemA((x), (x))
/* For wide character conversion, charToDisplayChar
   must be a function that provides its own buffer
   to CharToOemW and copies the results back into x
   as CharToOemA is inplace safe, but CharToOemW is not.
*/
#else                   /* DOS specific         */
/* Win32 File compability stuff */
#include "w32fDOS.h"
/* currently no mapping required */
#define charToDisplayChar(x)
#endif                  

/* Define getdrive so it returns current drive, 0=A,1=B,...           */
#if defined _MSC_VER || defined __MSC /* MS Visual C/C++ 5 */
#define getdrive() (_getdrive() - 1)
#else /* #ifdef __BORLANDC__ || __TURBOC__ */
#define getdrive() getdisk()
#endif


/* include support for message files */
#ifdef USE_CATGETS
#include "catgets.h"
#endif /* USE_CATGETS */

/* End Platform (OS) specific sections */


/* The default extended forms of the lines used. */
#define VERTBAR_STR  "\xB3   "                 /* |    */
#define TBAR_HORZBAR_STR "\xC3\xC4\xC4\xC4"    /* +--- */
#define CBAR_HORZBAR_STR "\xC0\xC4\xC4\xC4"    /* \--- */


/* Global flags */
#define SHOWFILESON    1  /* Display names of files in directories       */
#define SHOWFILESOFF   0  /* Don't display names of files in directories */

#define ASCIICHARS     1  /* Use ASCII [7bit] characters                 */
#define EXTENDEDCHARS  0  /* Use extended ASCII [8bit] characters        */


/* Global variables */
short showFiles = SHOWFILESOFF;
short charSet = EXTENDEDCHARS;

#ifdef USE_CATGETS
  char *catsFile = "tree";   /* filename of our message catalog     */
#endif /* USE_CATGETS */


/* Global constants */
#define SERIALLEN 16      /* Defines max size of volume & serial number   */
#define VOLLEN 128

#define MAXBUF 1024       /* Must be larger than max file path length     */
char path[MAXBUF];        /* Path to begin search from, default=current   */

#define MAXPADLEN (MAXBUF*2) /* Must be large enough to hold the maximum padding */
/* (MAXBUF/2)*4 == (max path len / min 2chars dirs "?\") * 4chars per padding    */

/* The maximum size any line of text output can be, including room for '\0'*/
#define MAXLINE 160        /* Increased to fit two lines for translations  */


/* The hard coded strings used by the following show functions.            */

/* common to many functions [Set 1] */
char newLine[MAXLINE] = "\n";

/* showUsage [Set 2] - Each %c will be replaced with proper switch/option */
char treeDescription[MAXLINE] = "Graphically displays the directory structure of a drive or path.\n";
char treeUsage[MAXLINE] =       "TREE [drive:][path] [%c%c] [%c%c]\n";
char treeFOption[MAXLINE] =     "   %c%c   Display the names of the files in each directory.\n";
char treeAOption[MAXLINE] =     "   %c%c   Use ASCII instead of extended characters.\n";

/* showInvalidUsage [Set 3] */
char invalidOption[MAXLINE] = "Invalid switch - %s\n";  /* Must include the %s for option given. */
char useTreeHelp[MAXLINE] =   "Use TREE %c? for usage information.\n"; /* %c replaced with switch */

/* showVersionInfo [Set 4] */
/* also uses treeDescription */
char treeGoal[MAXLINE] =      "Written to work with FreeDOS\n";
char treePlatforms[MAXLINE] = "Win32(c) console and DOS with LFN support.\n";
char version[MAXLINE] =       "Version %s\n"; /* Must include the %s for version string. */
char writtenBy[MAXLINE] =     "Written by: Kenneth J. Davis\n";
char writtenDate[MAXLINE] =   "Date:       August/Sept/Oct/Nov, 2000; January 2001\n";
char contact[MAXLINE] =       "Contact:    jeremyd@computer.org\n";
char copyright[MAXLINE] =     "Copyright (c): Public Domain [United States Definition]\n";
#ifdef USE_CATGETS
char catsCopyright[MAXLINE] = "Uses Jim Hall's <jhall@freedos.org> Cats Library\n  version 3.8 Copyright (C) 1999,2000 Jim Hall\n";
#endif

/* showInvalidDrive [Set 5] */
char invalidDrive[MAXLINE] = "Invalid drive specification\n";

/* showInvalidPath [Set 6] */
char invalidPath[MAXLINE] = "Invalid path - %s\n"; /* Must include %s for the invalid path given. */

/* Misc Error messages [Set 7] */
/* showBufferOverrun */
/* %u required to show what the buffer's current size is. */
char bufferToSmall[MAXLINE] = "Error: File path specified exceeds maximum buffer = %u bytes\n";
/* showOutOfMemory */
/* %s required to display what directory we were processing when ran out of memory. */
char outOfMemory[MAXLINE] = "Out of memory on subdirectory: %s\n";

/* main [Set 1] */
char pathListingNoLabel[MAXLINE] = "Directory PATH listing\n";
char pathListingWithLabel[MAXLINE] = "Directory PATH listing for Volume %s\n"; /* %s for label */
char serialNumber[MAXLINE] = "Volume serial number is %s\n"; /* Must include %s for serial #   */
char noSubDirs[MAXLINE] = "No subdirectories exist\n\n";

/* Option Processing - parseArguments [Set 8]      */
char optionchar1 = '/';  /* Primary character used to determine option follows  */
char optionchar2 = '-';  /* Secondary character used to determine option follows  */
char FOptionU = 'F';  /* Show files */
char FOptionL = 'f';
char AOptionU = 'A';  /* Use ASCII only */
char AOptionL = 'a';
char VOptionU = 'V';  /* Version information */
char VOptionL = 'v';
char SOptionU = 'S';  /* Shortnames only (disable LFN support) */
char SOptionL = 's';


/* Procedures */

/* Displays to user valid options then exits program indicating no error */
void showUsage(void)
{
  printf("%s%s%s%s", treeDescription, newLine, treeUsage, newLine);
  printf("%s%s%s", treeFOption, treeAOption, newLine);
  exit(1);
}


/* Displays error message then exits indicating error */
void showInvalidUsage(char * badOption)
{
  printf(invalidOption, badOption);
  printf("%s%s", useTreeHelp, newLine);
  exit(1);
}


/* Displays author, copyright, etc info, then exits indicating no error. */
void showVersionInfo(void)
{
  printf("%s%s%s%s%s", treeDescription, newLine, treeGoal, treePlatforms, newLine);
  printf(version, VERSION);
  printf("%s%s%s%s%s", writtenBy, writtenDate, contact, newLine, newLine);
  printf("%s%s", copyright, newLine);
#ifdef USE_CATGETS
  printf("%s%s", catsCopyright, newLine);
#endif
  exit(1);
}


/* Displays error messge for invalid drives and exits */
void showInvalidDrive(void)
{
  printf(invalidDrive);
  exit(1);
}


/* Takes a fullpath, splits into drive (C:, or \\server\share) and path */
void splitpath(char *fullpath, char *drive, char *path);

/**
 * Takes a given path, strips any \ or / that may appear on the end.
 * Returns a pointer to its static buffer containing path
 * without trailing slash and any necessary display conversions.
 */
char *fixPathForDisplay(char *path);

/* Displays error message for invalid path; Does NOT exit */
void showInvalidPath(char *path)
{
  char partialPath[MAXBUF], dummy[MAXBUF];

  printf("%s\n", path);
  splitpath(path, dummy, partialPath);
  printf(invalidPath, fixPathForDisplay(partialPath));
}

/* Displays error message for out of memory; Does NOT exit */
void showOutOfMemory(char *path)
{
  printf(outOfMemory, path);
}

/* Displays buffer exceeded message and exits */
void showBufferOverrun(WORD maxSize)
{
  printf(bufferToSmall, maxSize);
  exit(1);
}


/**
 * Takes a fullpath, splits into drive (C:, or \\server\share) and path
 * It assumes a colon as the 2nd character means drive specified,
 * a double slash \\ (\\, //, \/, or /\) specifies network share.
 * If neither drive nor network share, then assumes whole fullpath
 * is path, and sets drive to "".
 * If drive specified, then drive to it and colone, eg "C:", with
 * the rest of fullpath being set in path.
 * If network share, the slash slash followed by the server name,
 * another slash and either the rest of fullpath or up to, but not
 * including, the next slash are placed in drive, eg "\\KJD\myshare";
 * the rest of the fullpath including the slash are placed in
 * path, eg "\mysubdir"; where fullpath is "\\KJD\myshare\mysubdir".
 * None of these may be NULL, and drive and path must be large
 * enough to hold fullpath.
 */
void splitpath(char *fullpath, char *drive, char *path)
{
  register char *src = fullpath;
  register char oldchar;

  /* If either network share or path only starting at root directory */
  if ( (*src == '\\') || (*src == '/') )
  {
    src++;

    if ( (*src == '\\') || (*src == '/') ) /* network share */
    {
      src++;

      /* skip past server name */
      while ( (*src != '\\') && (*src != '/') && (*src != '\0') )
        src++;

      /* skip past slash (\ or /) separating  server from share */
      if (*src != '\0') src++;

      /* skip past share name */
      while ( (*src != '\\') && (*src != '/') && (*src != '\0') )
        src++;

      /* src points to start of path, either a slash or '\0' */
      oldchar = *src;
      *src = '\0';

      /* copy server name to drive */
      strcpy(drive, fullpath);

      /* restore character used to mark end of server name */
      *src = oldchar;

      /* copy path */
      strcpy(path, src);
    }
    else /* path only starting at root directory */
    {
      /* no drive, so set path to same as fullpath */
      strcpy(drive, "");
      strcpy(path, fullpath);
    }
  }
  else
  {
    if (*src != '\0') src++;

    /* Either drive and path or path only */
    if (*src == ':')
    {
      /* copy drive specified */
      *drive = *fullpath;  drive++;
      *drive = ':';        drive++;
      *drive = '\0';

      /* copy path */
      src++;
      strcpy(path, src);
    }
    else
    {
      /* no drive, so set path to same as fullpath */
      strcpy(drive, "");
      strcpy(path, fullpath);
    }
  }
}


/* Converts given path to full path */
void getProperPath(char *fullpath)
{
  char drive[MAXBUF];
  char path[MAXBUF];

  splitpath(fullpath, drive, path);

  /* if no drive specified use current */
  if (drive[0] == '\0')
  {
    sprintf(fullpath, "%c:%s", 'A'+ getdrive(), path);
  }
  else if (path[0] == '\0') /* else if drive but no path specified */
  {
    if ((drive[0] == '\\') || (drive[0] == '/'))
    {
      /* if no path specified and network share, use root   */
      sprintf(fullpath, "%s%s", drive, "\\");
    }
    else
    {
      /* if no path specified and drive letter, use current path */
      sprintf(fullpath, "%s%s", drive, ".");
    }
  }
  /* else leave alone, it has both a drive and path specified */
}


/* Parses the command line and sets global variables. */
void parseArguments(int argc, char *argv[])
{
  register int i;     /* temp loop variable */

  /* if no drive specified on command line, use current */
  sprintf(path, "%c:.", 'A'+ getdrive());

  for (i = 1; i < argc; i++)
  {
    /* Check if user is giving an option or drive/path */
    if ((argv[i][0] == optionchar1) || (argv[i][0] == optionchar2) )
    {
      /* Any multicharacter options are invalid */
      if (argv[i][2] != '\0')
        showInvalidUsage(argv[i]);

      /* Must check both uppercase and lowercase                        */
      if ((argv[i][1] == FOptionU) || (argv[i][1] == FOptionL))
        showFiles = SHOWFILESON; /* set file display flag appropriately */
      else if ((argv[i][1] == AOptionU) || (argv[i][1] == AOptionL))
        charSet = ASCIICHARS;    /* set charset flag appropriately      */
      else if (argv[i][1] == '?')
        showUsage();             /* show usage info and exit            */
      else if ((argv[i][1] == VOptionU) || (argv[i][1] == VOptionL))
        showVersionInfo();       /* show version info and exit          */
#ifdef DOS
      else if ((argv[i][1] == SOptionU) || (argv[i][1] == SOptionL))
        LFN_Enable_Flag = LFN_DISABLE;         /* force shortnames only */
#endif
      else /* Invalid or unknown option */
        showInvalidUsage(argv[i]);
    }
    else /* should be a drive/path */
    {
      if (strlen(argv[i]) > MAXBUF)
        showBufferOverrun(MAXBUF);

      /* copy path over, making all caps to look prettier, can be strcpy */
      register char *dptr = path;
      for (register char *cptr = argv[i]; *cptr != '\0'; cptr++, dptr++)
        *dptr = toupper(*cptr);
      *dptr = '\0';

      /* Converts given path to full path */
      getProperPath(path);
    }
  }
}


/**
 * Fills in the serial and volume variables with the serial #
 * and volume found using path.
 * If there is an error getting the volume & serial#, then an 
 * error message is displayed and the program exits.
 * Volume and/or serial # returned may be blank if the path specified
 * does not contain them, or an error retrieving 
 * (ie UNC paths under DOS), but path is valid.
 */
void GetVolumeAndSerial(char *volume, char *serial, char *path)
{
  char rootPath[MAXBUF];
  char dummy[MAXBUF];
  union serialNumber {
    DWORD serialFull;
    struct {
      WORD a;
      WORD b;
    } serialParts;
  } serialNum;

  /* get drive letter or share server\name */
  splitpath(path, rootPath, dummy);
  strcat(rootPath, "\\");

  if (GetVolumeInformation(rootPath, volume, VOLLEN,
      &serialNum.serialFull, NULL, NULL, NULL, 0) == 0)
	showInvalidDrive();

  if (serialNum.serialFull == 0)
    serial[0] = '\0';
  else
    sprintf(serial, "%04X:%04X",
      serialNum.serialParts.b, serialNum.serialParts.a);
}

/**
 * Returns 0 if no subdirectories, count if has subdirs.
 * Path must end in slash \ or /
 * On error (invalid path) displays message and returns -1L.
 */
long hasSubdirectories(char *path)
{
  WIN32_FIND_DATA findData;
  HANDLE hnd;
  char buffer[MAXBUF];
  int hasSubdirs = 0;

  /* get the handle to start with (using wildcard spec) */
  strcpy(buffer, path);
  strcat(buffer, "*");
  hnd = FindFirstFile(buffer, &findData);
  if (hnd == INVALID_HANDLE_VALUE)
  {
    showInvalidPath(path); /* Display error message */
    return -1L;
  }

  /*  cycle through entries counting directories found until no more entries */
  do {
    if (((findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0) &&
	((findData.dwFileAttributes &
	 (FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM)) == 0) )
    {
      if ((strcmp(findData.cFileName, ".") != 0) &&
	  (strcmp(findData.cFileName, "..") != 0))
	hasSubdirs++;
    }
  } while(FindNextFile(hnd, &findData) != 0);

  /* prevent resource leaks, close the handle. */
  FindClose(hnd);

  return hasSubdirs;
}

/**
 * Contains the information stored in a Stack necessary to allow
 * non-recursive function to display directory tree.
 */
typedef struct SUBDIRINFO
{
  struct SUBDIRINFO * parent; /* points to parent subdirectory               */
  char *currentpath;   /* Stores the full path this structure represents     */
  char *subdir;        /* points to last subdir within currentpath           */
  long subdircnt;      /* Initially a count of how many subdirs in this dir  */
  HANDLE findnexthnd;  /* The handle returned by findfirst, used in findnext */
} SUBDIRINFO;

/**
 * Allocates memory and stores the necessary stuff to allow us to
 * come back to this subdirectory after handling its subdirectories.
 * parentpath must end in \ or / or be NULL, however
 * parent should only be NULL for initialpath
 * if subdir does not end in slash, one is added to stored subdir.
 */
SUBDIRINFO *newSubdirInfo(SUBDIRINFO *parent, char *subdir)
{
  register int parentLen, subdirLen;

  /* Get length of parent directory */
  if (parent == NULL)
    parentLen = 0;
  else
    parentLen = strlen(parent->currentpath);

  /* Get length of subdir, add 1 if does not end in slash */
  subdirLen = strlen(subdir);
  if ((subdirLen < 1) || ( (*(subdir+subdirLen-1) != '\\') && (*(subdir+subdirLen-1) != '/') ) )
    subdirLen++;

  SUBDIRINFO *temp = (SUBDIRINFO *)malloc(sizeof(SUBDIRINFO));
  if (temp == NULL) 
  {
    showOutOfMemory(subdir);
    return NULL;
  }
  if ((temp->currentpath = (char *)malloc(parentLen+subdirLen+1)) == NULL)
  {
    showOutOfMemory(subdir);
    free(temp);
    return NULL;
  }
  temp->parent = parent;
  if (parent == NULL)
    strcpy(temp->currentpath, "");
  else
    strcpy(temp->currentpath, parent->currentpath);
  strcat(temp->currentpath, subdir);
  /* if subdir[subdirLen-1] == '\0' then we must append a slash */
  if (*(subdir+subdirLen-1) == '\0')
    strcat(temp->currentpath, "\\");
  temp->subdir = temp->currentpath+parentLen;
  if ((temp->subdircnt = hasSubdirectories(temp->currentpath)) == -1L)
  {
    free (temp->currentpath);
    free(temp);
    return NULL;
  }
  temp->findnexthnd = INVALID_HANDLE_VALUE;

  return temp;
}

/**
 * Extends the padding with the necessary 4 characters.
 * Returns the pointer to the padding.
 * padding should be large enough to hold the additional 
 * characters and '\0', moreSubdirsFollow specifies if
 * this is the last subdirectory in a given directory
 * or if more follow (hence if a | is needed).
 * padding must not be NULL
 */
char * addPadding(char *padding, int moreSubdirsFollow)
{
    if (moreSubdirsFollow)
    {
      /* 1st char is | or a vertical bar */
      if (charSet == EXTENDEDCHARS)
        strcat(padding, VERTBAR_STR);
      else
        strcat(padding, "|   ");
    }
    else
      strcat(padding, "    ");

    return padding;
}

/**
 * Removes the last padding added (last 4 characters added).
 * Does nothing if less than 4 characters in string.
 * padding must not be NULL
 * Returns the pointer to padding.
 */
char * removePadding(char *padding)
{
  register size_t len = strlen(padding);
  if (len < 4) return padding;

  *(padding + len - 4) = '\0';

  return padding;
}

/**
 * Takes a given path, strips any \ or / that may appear on the end.
 * Returns a pointer to its static buffer containing path
 * without trailing slash and any necessary display conversions.
 */
char *fixPathForDisplay(char *path)
{
  static char buffer[MAXBUF];
  register int pathlen;

  strcpy(buffer, path);
  pathlen = strlen(buffer);
  if (pathlen > 1)
  {
    pathlen--;
    if ((buffer[pathlen] == '\\') || (buffer[pathlen] == '/'))
      buffer[pathlen] = '\0'; // strip off trailing slash on end
  }

  charToDisplayChar(buffer);

  return buffer;
}

/**
 * Displays the current path, with necessary padding before it.
 * A \ or / on end of currentpath is not shown.
 * moreSubdirsFollow should be nonzero if this is not the last
 * subdirectory to be displayed in current directory, else 0.
 */
void showCurrentPath(char *currentpath, char *padding, int moreSubdirsFollow)
{
  register char *buffer = fixPathForDisplay(currentpath);

  if (padding != NULL)
    printf("%s", padding);

  /* print padding followed by filename */
  if (charSet == EXTENDEDCHARS)
  {
    if (moreSubdirsFollow)
      printf("%s%s\n", TBAR_HORZBAR_STR, buffer);
    else
      printf("%s%s\n", CBAR_HORZBAR_STR, buffer);
  }
  else
  {
    if (moreSubdirsFollow)
      printf("+---%s\n", buffer);
    else
      printf("\\---%s\n", buffer);
  }
}

/** 
 * Displays files in directory specified by path.
 * Path must end in slash \ or /
 * Returns -1 on error,
 *          0 if no files, but no errors either,
 *      or  1 if files displayed, no errors.
 */
int displayFiles(char *path, char *padding, int hasMoreSubdirs)
{
  char buffer[MAXBUF];
  WIN32_FIND_DATA entry; /* current directory entry info    */
  HANDLE dir;         /* Current directory entry working with      */
  register int filesShown = 0;

  /* get handle for files in current directory (using wildcard spec) */
  strcpy(buffer, path);
  strcat(buffer, "*");
  dir = FindFirstFile(buffer, &entry);
  if (dir == INVALID_HANDLE_VALUE)
    return -1;

  addPadding(padding, hasMoreSubdirs);

  /* cycle through directory printing out files. */
  do 
  {
    /* print padding followed by filename */
    if ((entry.dwFileAttributes &
        (FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_HIDDEN |
         FILE_ATTRIBUTE_SYSTEM)) == 0)
    {
      printf("%s", padding);
      charToDisplayChar(entry.cFileName);
      printf("%s\n", entry.cFileName);
      filesShown = 1;
    }
  } while(FindNextFile(dir, &entry) != 0);

  if (filesShown)
  {
    printf("%s\n", padding);
  }

  /* cleanup directory search */
  FindClose(dir);
  /* dir = NULL; */

  removePadding(padding);

  return filesShown;
}

/**
 * Given the current path, find the 1st subdirectory.
 * The subdirectory found is stored in subdir.
 * subdir is cleared on error or no subdirectories.
 * Returns the findfirst search HANDLE, which should be passed to
 * findclose when directory has finished processing, and can be
 * passed to findnextsubdir to find subsequent subdirectories.
 * Returns INVALID_HANDLE_VALUE on error.
 * currentpath must end in \
 */
HANDLE findFirstSubdir(char *currentpath, char *subdir)
{
  char buffer[MAXBUF];
  WIN32_FIND_DATA entry; /* current directory entry info    */
  HANDLE dir;         /* Current directory entry working with      */

  /* get handle for files in current directory (using wildcard spec) */
  strcpy(buffer, currentpath);
  strcat(buffer, "*");

  if ((dir = FindFirstFile(buffer, &entry)) == INVALID_HANDLE_VALUE)
  {
    showInvalidPath(currentpath);
    return INVALID_HANDLE_VALUE;
  }

  /* clear result path */
  strcpy(subdir, "");

  /* cycle through directory until 1st non . or .. directory is found. */
  do
  {
    /* skip files & hidden or system directories */
    if ((((entry.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0) ||
         ((entry.dwFileAttributes &
          (FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM)) != 0) ) ||
        ((strcmp(entry.cFileName, ".") == 0) ||
         (strcmp(entry.cFileName, "..") == 0)) )
    {
      if (FindNextFile(dir, &entry) == 0)
        return INVALID_HANDLE_VALUE; // no subdirs found
    }
    else
    {
      strcpy(subdir, entry.cFileName);
      strcat(subdir, "\\");
    }
  } while (!*subdir); // while (subdir is still blank)

  return dir;
}

/**
 * Given a search HANDLE, will find the next subdirectory, 
 * setting subdir to the found directory name.
 * currentpath must end in \
 * If a subdirectory is found, returns 0, otherwise returns 1
 * (either error or no more files).
 */
int findNextSubdir(HANDLE findnexthnd, char *subdir)
{
  WIN32_FIND_DATA entry; /* current directory entry info    */

  /* clear result path */
  strcpy(subdir, "");

  if (FindNextFile(findnexthnd, &entry) == 0)
    return 1; // no subdirs found

  /* cycle through directory until 1st non . or .. directory is found. */
  do
  {
    /* skip files & hidden or system directories */
    if ((((entry.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0) ||
         ((entry.dwFileAttributes &
          (FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM)) != 0) ) ||
        ((strcmp(entry.cFileName, ".") == 0) ||
         (strcmp(entry.cFileName, "..") == 0)) )
    {
      if (FindNextFile(findnexthnd, &entry) == 0)
        return 1; // no subdirs found
    }
    else
    {
      strcpy(subdir, entry.cFileName);
      strcat(subdir, "\\");
    }
  } while (!*subdir); // while (subdir is still blank)

  return 0;
}

/**
 * Given an initial path, displays the directory tree with
 * a non-recursive function using a Stack.
 * initialpath must be large enough to hold an added slash \ or /
 * if it does not already end in one.
 * Returns the count of subdirs in initialpath.
 */
long traverseTree(char *initialpath)
{
  long subdirsInInitialpath;
  char padding[MAXPADLEN] = "";
  char subdir[MAXBUF];
  register SUBDIRINFO *sdi;

  STACK s;
  stackDefaults(&s);
  stackInit(&s);

  if ( (sdi = newSubdirInfo(NULL, initialpath)) == NULL)
    return 0L;
  stackPushItem(&s, sdi);

  /* Store count of subdirs in initial path so can display message if none. */
  subdirsInInitialpath = sdi->subdircnt;

  do
  {
    sdi = (SUBDIRINFO *)stackPopItem(&s);

    if (sdi->findnexthnd == INVALID_HANDLE_VALUE)  // findfirst not called yet
    {
      // 1st time this subdirectory processed, so display its name & possibly files
      if (sdi->parent == NULL) // if initial path
      {
        // display initial path
        charToDisplayChar(initialpath);
        printf("%s\n", initialpath);
      }
      else // normal processing (display path, add necessary padding)
      {
        showCurrentPath(sdi->subdir, padding, (sdi->parent->subdircnt > 0L)?1 : 0);
        addPadding(padding, (sdi->parent->subdircnt > 0L)?1 : 0);
      }

      if (showFiles == SHOWFILESON)  displayFiles(sdi->currentpath, padding, (sdi->subdircnt > 0L)?1 : 0);
    }

    if (sdi->subdircnt > 0) /* if (there are more subdirectories to process) */
    {
      sdi->subdircnt = sdi->subdircnt - 1L; /* decrement subdirs left count */

      if (sdi->findnexthnd == INVALID_HANDLE_VALUE)
        sdi->findnexthnd = findFirstSubdir(sdi->currentpath, subdir);
      else
        findNextSubdir(sdi->findnexthnd, subdir);

      stackPushItem(&s, sdi);

      /* store necessary information, validate subdir, and on error return. */
      if ((sdi = newSubdirInfo(sdi, subdir)) == NULL)
        return 0L; 

      stackPushItem(&s, sdi);
    }
    else /* this directory finished processing, so free resources */
    {
      /* Remove the padding for this directory, all but initial path. */
      if (sdi->parent != NULL)
        removePadding(padding);

      /* Prevent resource leaks, by ending findsearch and freeing memory. */
      FindClose(sdi->findnexthnd);
      if (sdi != NULL)
      {
        if (sdi->currentpath != NULL)
          free(sdi->currentpath);
        free(sdi);
      }
    }
  } while (stackTotalItems(&s)); /* while (stack is not empty) */

  stackTerm(&s);

  return subdirsInInitialpath;
}


/**
 * Process strings, converting \\, \n, \r, and \t to actual chars.
 * This method is used to allow the message catalog to use \\, \n, \r, and \t
 * returns a pointer to its internal buffer, so strcpy soon after use.
 * Can only handle lines up to MAXLINE chars.
 * This is required because most messages are passed as
 * string arguments to printf, and not actually parsed by it.
 */
char *processLine(char *line)
{
  static char buffer[MAXLINE+MAXLINE];
  register char *src = line, *dst = buffer;

  if (line == NULL) return NULL;

  /* cycle through copying characters, except when a \ is encountered. */
  for ( ; *src != '\0'; src++, dst++)
  {
    if (*src == '\\')
    {
      src++;
      switch (*src)
      {
	  case '\0': /* a slash ends a line, ignore the slash. */
		  src--; /* next time through will see the '\0'    */
		  break;
	  case '\\': /* a single slash */
		  *dst = '\\';
		  break;
	  case 'n': /* a newline */
		  *dst = '\n';
		  break;
	  case 'r': /* a carriage return */
		  *dst = '\r';
		  break;
	  case 't': /* a horizontal tab */
		  *dst = '\t';
		  break;
	  default: /* just copy over the letter */
		  *dst = *src;
		  break;
      }
    }
    else
      *dst = *src;
  }

  /* ensure '\0' terminated */
  *dst = '\0';

  return buffer;
}


/* Changes %c in output that displays options with proper option characters. */
void FixOptionText(void)
{
  char buffer[MAXLINE];  /* sprintf can have problems with src==dest */

  /* Handle %c for options within messages using Set 8 */
  strcpy(buffer, treeUsage);
  sprintf(treeUsage, buffer, optionchar1, FOptionU, optionchar1, AOptionU);
  strcpy(buffer, treeFOption);
  sprintf(treeFOption, buffer, optionchar1, FOptionU);
  strcpy(buffer, treeAOption);
  sprintf(treeAOption, buffer, optionchar1, AOptionU);
  strcpy(buffer, useTreeHelp);
  sprintf(useTreeHelp, buffer, optionchar1);
}


/* Loads all messages from the message catalog.
 * If USE_CATGETS is undefined or failure finding catalog then
 * hard coded strings are used
 */
void loadAllMessages(void)
{
  #ifdef USE_CATGETS
    nl_catd cat;              /* store id of our message catalog global       */
    char *bufPtr;

    /* Open the message catalog, keep hard coded values on error. */
    if ((cat = catopen (catsFile, MCLoadAll)) == -1) 
    {
      FixOptionText(); /* Changes %c in certain lines with default option characters. */
      return;
    }

    /* common to many functions [Set 1] */
    bufPtr = catgets (cat, 1, 1, newLine);
    if (bufPtr != newLine) strcpy(newLine, processLine(bufPtr));

    /* main [Set 1] */
    bufPtr = catgets (cat, 1, 2, pathListingNoLabel);
    if (bufPtr != pathListingNoLabel) strcpy(pathListingNoLabel, processLine(bufPtr));
    bufPtr = catgets (cat, 1, 3, pathListingWithLabel);
    if (bufPtr != pathListingWithLabel) strcpy(pathListingWithLabel, processLine(bufPtr));
    bufPtr = catgets (cat, 1, 4, serialNumber);
    if (bufPtr != serialNumber) strcpy(serialNumber, processLine(bufPtr));
    bufPtr = catgets (cat, 1, 5, noSubDirs);
    if (bufPtr != noSubDirs) strcpy(noSubDirs, processLine(bufPtr));

    /* showUsage [Set 2] */
    bufPtr = catgets (cat, 2, 1, treeDescription);
    if (bufPtr != treeDescription) strcpy(treeDescription, processLine(bufPtr));
    bufPtr = catgets (cat, 2, 2, treeUsage);
    if (bufPtr != treeUsage) strcpy(treeUsage, processLine(bufPtr));
    bufPtr = catgets (cat, 2, 3, treeFOption);
    if (bufPtr != treeFOption) strcpy(treeFOption, processLine(bufPtr));
    bufPtr = catgets (cat, 2, 4, treeAOption);
    if (bufPtr != treeAOption) strcpy(treeAOption, processLine(bufPtr));

    /* showInvalidUsage [Set 3] */
    bufPtr = catgets (cat, 3, 1, invalidOption);
    if (bufPtr != invalidOption) strcpy(invalidOption, processLine(bufPtr));
    bufPtr = catgets (cat, 3, 2, useTreeHelp);
    if (bufPtr != useTreeHelp) strcpy(useTreeHelp, processLine(bufPtr));

    /* showVersionInfo [Set 4] */
    /* also uses treeDescription from Set 2 */
    bufPtr = catgets (cat, 4, 1, treeGoal);
    if (bufPtr != treeGoal) strcpy(treeGoal, processLine(bufPtr));
    bufPtr = catgets (cat, 4, 2, treePlatforms);
    if (bufPtr != treePlatforms) strcpy(treePlatforms, processLine(bufPtr));
    bufPtr = catgets (cat, 4, 3, version);
    if (bufPtr != version) strcpy(version, processLine(bufPtr));
    bufPtr = catgets (cat, 4, 4, writtenBy);
    if (bufPtr != writtenBy) strcpy(writtenBy, processLine(bufPtr));
    bufPtr = catgets (cat, 4, 5, writtenDate);
    if (bufPtr != writtenDate) strcpy(writtenDate, processLine(bufPtr));
    bufPtr = catgets (cat, 4, 6, contact);
    if (bufPtr != contact) strcpy(contact, processLine(bufPtr));
    bufPtr = catgets (cat, 4, 7, copyright);
    if (bufPtr != copyright) strcpy(copyright, processLine(bufPtr));
//ifdef USE_CATGETS
    bufPtr = catgets (cat, 4, 8, catsCopyright);
    if (bufPtr != catsCopyright) strcpy(catsCopyright, processLine(bufPtr));
//endif

    /* showInvalidDrive [Set 5] */
    bufPtr = catgets (cat, 5, 1, invalidDrive);
    if (bufPtr != invalidDrive) strcpy(invalidDrive, processLine(bufPtr));

    /* showInvalidPath [Set 6] */
    bufPtr = catgets (cat, 6, 1, invalidPath);
    if (bufPtr != invalidPath) strcpy(invalidPath, processLine(bufPtr));

    /* Misc Error messages [Set 7] */
    /* showBufferOverrun */
    /* %u required to show what the buffer's current size is. */
    bufPtr = catgets (cat, 7, 1, bufferToSmall);
    if (bufPtr != bufferToSmall) strcpy(bufferToSmall, processLine(bufPtr));
    /* showOutOfMemory */
    /* %s required to display what directory we were processing when ran out of memory. */
    bufPtr = catgets (cat, 7, 2, outOfMemory);
    if (bufPtr != outOfMemory) strcpy(outOfMemory, processLine(bufPtr));

    /* parseArguments - options [Set 8] */
    /* Note all of these are single characters (only 1st character used) */
    bufPtr = catgets (cat, 8, 1, NULL);
    if (bufPtr != NULL) optionchar1 = bufPtr[0];
    bufPtr = catgets (cat, 8, 2, NULL);
    if (bufPtr != NULL) optionchar2 = bufPtr[0];
    bufPtr = catgets (cat, 8, 3, NULL);
    if (bufPtr != NULL) FOptionU = bufPtr[0];
    bufPtr = catgets (cat, 8, 4, NULL);
    if (bufPtr != NULL) FOptionL = bufPtr[0];
    bufPtr = catgets (cat, 8, 5, NULL);
    if (bufPtr != NULL) AOptionU = bufPtr[0];
    bufPtr = catgets (cat, 8, 6, NULL);
    if (bufPtr != NULL) AOptionL = bufPtr[0];
    bufPtr = catgets (cat, 8, 7, NULL);
    if (bufPtr != NULL) VOptionU = bufPtr[0];
    bufPtr = catgets (cat, 8, 8, NULL);
    if (bufPtr != NULL) VOptionL = bufPtr[0];
    bufPtr = catgets (cat, 8, 9, NULL);
    if (bufPtr != NULL) SOptionU = bufPtr[0];
    bufPtr = catgets (cat, 8, 10, NULL);
    if (bufPtr != NULL) SOptionL = bufPtr[0];

    /* close the message catalog */
    catclose (cat);
  #endif

  /* Changes %c in certain lines with proper option characters. */
  FixOptionText();
}


int main(int argc, char *argv[])
{
  char serial[SERIALLEN]; /* volume serial #  0000:0000 */
  char volume[VOLLEN];    /* volume name (label), possibly none */

  /* Load all text from message catalog (or uses hard coded text) */
  loadAllMessages();

  /* Parse any command line arguments, obtain path */
  parseArguments(argc, argv);

  /* Get Volume & Serial Number */
  GetVolumeAndSerial(volume, serial, path);
  if (strlen(volume) == 0)
    printf(pathListingNoLabel);
  else
    printf(pathListingWithLabel, volume);
  if (serial[0] != '\0')  /* Don't print anything if no serial# found */
    printf(serialNumber, serial);

  /* now traverse & print tree, returns nonzero if has subdirectories */
  if (traverseTree(path) == 0)
    printf(noSubDirs);

  return 0;
}
