/* more.c */

/* A program to display the contents of a text file, one screenful at
   a time */

/* Copyright (C) 1994-1999 Jim Hall, jhall1@isd.net */

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    See the file 'COPYING' for details.
*/


#include <stdio.h>
#include <stdlib.h>			/* for _splitpath, _makepath */
#include <dir.h>			/* for findfirst, findnext */
#include <dos.h>			/* for findfirst, findnext */
#include <bios.h>			/* for _bios_keybrd */


/* Symbolic constants */

#define LINES 25			/* lines per screen */
#define COLS  80			/* columns per screen */


/* Function prototypes */

void more (FILE *pfile, const char *descr);
unsigned keypress (void);
void usage (void);


/* Main program */

int
main (int argc, char **argv)
{
  char drive[_MAX_DRIVE];		/* for _splitpath, _makepath */
  char dir[_MAX_DIR];			/* for _splitpath, _makepath */
  char file[_MAX_FNAME];		/* for _splitpath, _makepath */
  char ext[_MAX_EXT];			/* for _splitpath, _makepath */
  char fullpath[_MAX_PATH];		/* the full path after _makepath */
  int i;
  int done;				/* flag for findfirst, findnext */
  int stat;				/* exit status */
  FILE *pfile;				/* file handle */
  struct ffblk ffblk;			/* findfirst, findnext block */

  /* Evaluate the files on the command line */

  if (argc == 1)
    {
      more (stdin, "<STDIN>");
      stat = 0;
    }

  for (i = 1; i < argc; i++)
    {
      /* Check if we are asking for help */

      if (argv[i][0] == '/')
	{
	  /* This is a flag character */

	  /* Add any code here that deals with special flag chars */

	  switch (argv[i][1])
	    {
	    case '?':
	      /* print usage and quit */

	      printf ("MORE - Display the contents of a text file, one screen at a time\n");
	      usage();
	      exit (0);
	      break;

	    default:
	      /* Not a recognized option */

	      printf ("MORE: Not a recognized option: %s\n", argv[i]);
	      usage();
	      exit (1);
	      break;
	    } /* switch */
	} /* if argv */

      /* Assume a file.  Open it and display it. */

      /* Note that findfirst, findnext only return the base file
         name in ff_name, and not the path.  The search still
         works with a path, but you don't get it back.  So we
         have to preserve the full path to the file using
         _splitpath and _makepath. */

      _splitpath (argv[i], drive, dir, file, ext);
      done = findfirst (argv[i], &ffblk, 0);

      if (done)
	{
	  /* We were not able to find a file. Display a message and
	     set the exit status. */

	  fprintf (stderr, "MORE: No such file %s\n", argv[i]);
	  stat = 1;
	}

      while (!done)
	{
	  /* We have found a file, so open it and display it.  Set
	     the exit status to 'successful' */

	  _makepath (fullpath, drive, dir, ffblk.ff_name, "");
	  pfile = fopen (fullpath, "r");

	  if (pfile)
	    {
	      more (pfile, ffblk.ff_name);
	      fclose (pfile);
	      stat = 0;
	    }

	  else
	    {
	      fprintf (stderr, "MORE: Cannot open file %s\n", ffblk.ff_name);
	      stat = 1;
	    }

	  /* find next file to match the filemask */

	  done = findnext (&ffblk);
	} /* while */
    } /* for */

  /* Done */

  exit (stat);
}

/* A function to display the contents of a text file, one screenful
   at a time. */

void
more (FILE *pfile, const char *descr)
{
  int ch;
  int nchars = 0;			/* no. of chars printed per line */
  int nlines = 0;			/* no. of lines printed per screen */

  while ((ch = fgetc (pfile)) != EOF)
    {
      putchar (ch);
      nchars++;

      /* Determine if we have caused a new line to be printed.
         Printing a CR/LF will do it, as will a screen wrap. */

      if ((ch == '\n') || (nchars == COLS))
	{
	  nlines++;
	  nchars = 0;

	  if (nlines == (LINES-1))
	    {
	      /* We have met the printable screen boundary.  Display
	         a prompt, and wait for a keypress before we show more. */

	      /* Since we don't necessarily know if the user is
	         redirecting to a file, always display on stderr. */

	      fprintf (stderr, "-- More -- %s --", descr);
	      keypress();
	      putchar ('\n');
	      nlines = 0;
	    }
	}
    } /* while */

  /* Since we can show more than one file at a time, we need to
     display a prompt when we are done with a file. */

  fprintf (stderr, "-- End -- %s --", descr);
  keypress();
  putchar ('\n');
}

/* Use BIOS services to retrieve the next keypress */

unsigned
keypress (void)
{
  while (! _bios_keybrd(_KEYBRD_READY)) /* wait */ ;
  return (_bios_keybrd(_KEYBRD_READ));
}

/* A function to display the program's usage */

void
usage (void)
{
  printf ("Usage:\n");
  printf ("  command | MORE\n");
  printf ("  MORE file..\n");
  printf ("  MORE < file\n");
}
