/****************************************************************/
/*                                                              */
/*                            partsize.c                        */
/*                                                              */
/*                      Copyright (c) 2001                      */
/*                      tom ehlert                              */
/*                      All Rights Reserved                     */
/*                                                              */
/* This file is part of DOS-C.                                  */
/*                                                              */
/* DOS-C is free software; you can redistribute it and/or       */
/* modify it under the terms of the GNU General Public License  */
/* as published by the Free Software Foundation; either version */
/* 2, or (at your option) any later version.                    */
/*                                                              */
/* DOS-C is distributed in the hope that it will be useful, but */
/* WITHOUT ANY WARRANTY; without even the implied warranty of   */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See    */
/* the GNU General Public License for more details.             */
/*                                                              */
/* You should have received a copy of the GNU General Public    */
/* License along with DOS-C; see the file COPYING.  If not,     */
/* write to the Free Software Foundation, 675 Mass Ave,         */
/* Cambridge, MA 02139, USA.                                    */
/****************************************************************/

/*
    PartSize.C
  
  how to determine size of a logical drive  
  
  this is portable and works on WinNT, FreeDOS > KE2024L, 
  and (nearly) on MSDOS 6.2
  
  DOS6.2 returns for some disks for anknown reasons 8040 MB through
  GetDosPartSizeRead(), which is simply largest possible non-LBA disksize
  
  
*/    

#include <stdio.h>
#include <ctype.h>
#include <dos.h>
#include <stdlib.h>

typedef unsigned long ULONG;
typedef unsigned long  __uint32;
typedef unsigned short __uint16;




/*
    TC absRead not functional on MSDOS 6.2, large disks
    MSDOS requires int25, CX=ffff for drives > 32MB
*/

#pragma warn -asc

int MyAbsRead(char DosDrive, int count, ULONG sector, void *buffer)
{
    struct {
        __uint32 sectorNumber;
        __uint16  count;
        void far *address;
        } diskReadPacket;
    int retval;

    diskReadPacket.sectorNumber = sector;
    diskReadPacket.count        = count;
    diskReadPacket.address      = buffer;
    
    asm {
        push si
        push di
        
        mov al, DosDrive
        lea bx, diskReadPacket
        mov cx, 0xffff
        int 0x25
        pop bx          // dummy pop
        
        pop di
        pop si

        mov ah,0xff        
        mov retval,ax

        jc error
        mov retval,word ptr 0
    }
error:        
    return retval;
}    


ULONG GetDosPartSizeRead(int drive);
ULONG GetDosPartSizeBoot(int drive);


main(int argc, char *argv[])
{
    int drive;
    ULONG byreadsize, bybootsize;
    
    if (argc > 1)
        {
        if (!isalpha(argv[1][0]))
            {
            printf("usage PARTSIZE [Drive = A:, B:, C:,...]\n");
            exit(1);
            }
        drive = toupper(argv[1][0]) - 'A';
        }
    else
        drive = bdos(0x19,0,0) & 0xff;
        
    printf("partitionsize for %c:\n", drive + 'A');

    printf(" size by reading -");
    byreadsize = GetDosPartSizeRead(drive);
    printf(" %lu sec = %lu MB\n", byreadsize, byreadsize/2048);

    printf(" size by bootsec -");
    bybootsize = GetDosPartSizeBoot( drive); 
    printf(" %lu sec = %lu MB\n", bybootsize, bybootsize/2048);
    return 0;       
}

char secbuffer[512];

ULONG GetDosPartSizeRead(int drive)
{
    ULONG ok, offset;
    
    ok = 0;
    offset  = 0x10000000l;

    while (offset != 0)
        {
/*        printf("ok = %lu=%lu,testing %lu=%lu, offset %lu\n", 
                ok, ok/2048,
                ok + offset,(ok + offset)/2048,offset);
*/            
        if (MyAbsRead(drive, 1, ok + offset, secbuffer) == 0)
            {
            ok += offset;    
            }                
        else {
            offset >>= 1;
            }
        }        
        
    if (ok)
        ok++;               /* return number of total sector */        
    
    return ok;
    
}    
ULONG GetDosPartSizeBoot(int drive)
{
    unsigned smallsize;
    ULONG   largesize;
    
    if (MyAbsRead(drive, 1, 0, secbuffer) != 0)
        return 0;

    smallsize = *(__uint16*)(secbuffer + 0x13);
    largesize = *(__uint32*)(secbuffer + 0x20);
    
    if (smallsize != 0) return smallsize;
    return largesize;
        
}    
