/* Functions to install a disk series, or a package */

/* Copyright (C) 1997, James Hall <jhall1@isd.net> */

/* This is free software.  See the file COPYING for details */

#include <stdio.h>
#include <stdlib.h>				/* for system(), free() */
#include "dat.h"
#include "file_io.h"
#include "fs.h"
#include "getkey.h"
#include "pause.h"
#include "showdesc.h"

/* Symbolic constants */

#define STRLEN 256


/* Functions */

/* Asks if you want to install a package, and returns true/false if
   the user responds y/n */

int
instyn (const char *pkg)
{
  int key;

  printf ("Install %s? (y/n) ", pkg);

  key = getkeyli ("yn");
  return ((key == 'y') || (key == 'Y'));
}

/* Installs a package file into dir "exdir" using "unzip".  Returns
   true if the package was successfully installed, or false if not.
   Files are located in the path "from" */

int
instpkg (const char *pkg, const char *from, const char *exdir)
{
  char cmd[STRLEN];
  char zipfile[STRLEN];

  /* Create the zip file name */

  sprintf (zipfile, "%s%c%s.ZIP", from, FS_DIR_SEP, pkg);

  /* Check if the file is there */

  if (!isfile (zipfile)) {
    /* Not there - return failure */
    printf ("install: %s not found\n", zipfile);
    return (0);
  }

  /* "-d exdir" extracs into directory exdir.  "-o" forces overwrite
     of same files. */

  /* I chose NOT to use -q or -qq because it may help the user to see
     these files being extracted, so they know the system isn't hung.
     Especially if we ever have a 1MB package. */

  /* In version 2.0, I will call the unzip code directly. */

  sprintf (cmd, "unzip -o %s -d%s", zipfile, exdir);
  return (system (cmd) == 0);
}

/* Installs all files from the named datfile, using the descfile to
   print descriptions.  Returns the number of files installed. */

int
instdisk (const char *datfile, const char *descfile, const char *from,
	  const char *exdir)
{
  dat_t *thisdisk;				/* the DAT array */
  int nfiles;					/* no. files installed */
  int i;

  /* Read the package list */

  if (!(thisdisk = dat_read (datfile))) {
    /* Failed */

    fprintf (stderr, "install: Can't find %s\n", datfile);
    return (0);
  }

  nfiles = dat_ary_size;

  /* Run the installation. */

  for (i = 0; i < nfiles; i++) {
    /* Show the description */

    printf ("\n\n");
    showdesc (thisdisk[i].fname, descfile);

    /* Find out which ones the user wants to install */

    switch (thisdisk[i].rank) {
    case 'n':
    case 'N':
      /* Do not install */

      printf ("%s [SKIPPED]\n", thisdisk[i].fname);
      break;

    case 'y':
    case 'Y':
      /* Always install */

      printf ("%s [REQUIRED]\n", thisdisk[i].fname);
      instpkg (thisdisk[i].fname, from, exdir);
      break;

    default:
      /* Optional */

      printf ("%s [OPTIONAL]\n", thisdisk[i].fname);

      /* Ask the user if you want to install it */

      if (instyn (thisdisk[i].fname)) {
	instpkg (thisdisk[i].fname, from, exdir);
      }
      break;

    } /* switch */
  } /* for */

  /* Return */

  free (thisdisk);
  return (nfiles);
}

/* Installs a disk series into dir "exdir", using the entry from the
   DAT file.  Files are located in the path "from".  Returns the
   number of disks installed from the series.  We know the series is
   finished when we find 'endfile' */

int
instser (const char *base, const char *from, const char *exdir)
{
  /* Variables */

  char endfile[STRLEN];				/* marks end of series */
  char descfile[STRLEN];			/* description file */
  char datfile[STRLEN];				/* current DAT file */
  int ndisks = 0;				/* no. disks installed */

  /* Create the filenames */

  sprintf (endfile, "%s%c%s.END", from, FS_DIR_SEP, base);
  sprintf (descfile, "%s%c%s.TXT", from, FS_DIR_SEP, base);

  /* Say hello */

  printf ("\n\n");
  printf (">> Installing series %s <<\n", base);

  /* Install while we have disks to work from */

  do {
    /* Set the DAT file name */

    sprintf (datfile, "%s%c%s.%d", from, FS_DIR_SEP, base, ++ndisks);

    /* Load the first disk */

    while (!isfile (datfile)) {
      printf ("\n\n");
      printf ("Load disk \'%s-%d\' in %s\n", base, ndisks, from);

      if (!pause ()) {
	/* Exit */

	return (ndisks);
      } /* if pause */
    } /* while */

    instdisk (datfile, descfile, from, exdir);
  } while (!isfile (endfile));

  /* Return the number of disks installed */

  printf ("\n%s found\n", endfile);
  return (ndisks);
}
