/*  FreeDOS HTML Help Viewer

    PES - Memory Manager for PES data structures

    Copyright (c) Express Software 2002-3
    See doc\htmlhelp\copying for licensing terms
    This file created by: Robert Platt.
*/

#include <string.h>
#include <malloc.h>
#include "pes.h"
#include "help_gui.h"
#include "readfile.h"

struct eventState **thePesList; /* list of pointers to pes structs */
long pesListSize = 0; /* size of the list */
long pesListCurr = 0; /* current position in the list */
long pesListBottom = 0;

int pesListInitialise(void)
{
  pesListSize = 0;
  pesListCurr = -1;
  pesListBottom = 0;

  thePesList = (struct eventState **)malloc(sizeof(struct eventState *));
  if (thePesList == NULL)
  {
    show_error("Cannot allocate enough memory");
    return FAIL;
  }

  return SUCCESS;
}

struct eventState * pesListAdd(char *filename)
{
    struct eventState * newpes;
    struct eventState ** thePesListResized;

    pesListDeleteForwards();

    /* Create a new process state structure */
    /* delete old ones if need be */
    while (
     (newpes = (struct eventState *)malloc (sizeof (struct eventState)))
	     == NULL)
       if (pesListDeleteOldest())
       {
	 show_error ("Cannot allocate enough memory.");
	 return NULL;
       }

    /* Create a buffer filled with the text */
    /* delete old pes structures if necessary */
    if ((newpes->text_buf = readFile (filename)) == NULL)
    {
       free (newpes);
       return NULL;
    }

    /* Create a buffer for screen information */
    /* delete old pes structures if necessary */
    /* The 512 bytes are for pes->text_holder */
    while ((newpes->screen_buf = (char *)malloc (H*W*2 + 512)) == NULL)
      if (pesListDeleteOldest())
      {
	 show_error ("Cannot allocate enough memory.");
	 free (newpes->text_buf);
	 free (newpes);
	 return NULL;
      }

    while ((newpes->filename = (char *)malloc (strlen(filename)+1)) == NULL)
      if (pesListDeleteOldest())
      {
	 show_error ("Cannot allocate enough memory.");
    free (newpes->screen_buf);
	 free (newpes->text_buf);
	 free (newpes);
	 return NULL;
      }
    strcpy(newpes->filename, filename);

    while ((thePesListResized = (struct eventState **)realloc(thePesList,
				sizeof(struct eventState *) * (pesListSize+1)))
				== NULL)
      if (pesListDeleteOldest())
      {
	 pesDelete(newpes);
	 return NULL;
      }

    thePesList = thePesListResized;

    /* update pes list information */
    ++pesListCurr;
    ++pesListSize;
    return (thePesList[pesListCurr] = newpes);
}



struct eventState * pesListBackward (void)
{
  --pesListCurr;

  if (pesListCurr < pesListBottom)
  {
    pesListCurr = -1;
    return NULL;
  }

  return thePesList[pesListCurr];
}


struct eventState * pesListForward (void)
{
  if (pesListCurr == pesListSize - 1)
    return NULL;

  return thePesList[++pesListCurr];
}

struct eventState * pesListCurrent (void)
{
  if (pesListCurr < pesListBottom || pesListSize <= 0)
    return NULL;

  return thePesList[pesListCurr];
}

struct eventState * pesResizeCurrentTextbuf(long newSize)
{
    char *newbuf;

    if (pesListSize <= 0)
       return NULL;

    if (newSize == 0)
       return thePesList[pesListCurr];

    while ((newbuf = (char *)realloc
            (thePesList[pesListCurr]->text_buf, newSize))
                   == NULL)
       if (pesListDeleteOldest())
       {
   	    show_error ("Cannot allocate enough memory.");
	       return NULL;
       }

   thePesList[pesListCurr]->text_buf = newbuf;
   return thePesList[pesListCurr];
}

void pesListDestroy (void)
{
  int i;

  for (i = pesListBottom; i < pesListSize; i++)
     pesDelete (thePesList[i]);

  free (thePesList);
}

int pesListDeleteOldest (void)
{
/* Semi-infinite browsing only
   at the moment. After a very
   long chain of browsing, there
   will be no more memory
   available. The true-infinite
   code, that will delete old
   files on the list completely,
   doesn't work yet - RP */

#define FUDGE_PES
#ifdef FUDGE_PES
   if (pesListBottom < pesListCurr)
   {
      pesDelete(thePesList[pesListBottom]);
      pesListBottom++;
      return SUCCESS;
   }
   return FAIL;
#else
   struct eventState ** thePesListResized;
   int i;


   /* Shift the remaining pes's back so that
      the array begins at thePesList[0] */
   for (i = 1; i < pesListSize; i++)
    thePesList[i-1] = thePesList[i];

   /* there is one less pes */
   --pesListSize;
   /* and the current pes is now less one in the list */
   --pesListCurr;

   /* Resize the list of pointers to the pes's to clear up the unused space */
   if ((thePesListResized = (struct eventState **)realloc(thePesList,
				sizeof(struct eventState *) * (pesListSize)))
				!= NULL)
      thePesList = thePesListResized;

   return SUCCESS;
   #endif
}

void pesListDeleteForwards (void)
{
   long i;
   struct eventState ** thePesListResized;
   long newPesListSize = pesListSize;

   /* Delete any pes's that are after the current pes */
   for (i = pesListCurr + 1; i < pesListSize; i++)
   {
     pesDelete (thePesList[i]);
     --newPesListSize;
   }

   if (pesListSize != newPesListSize)
   {
     /* Can't reallocate to zero, so just reinitialise
        This can happen if pesListDeleteForwards is
        called with pesListCurr = -1 */
     if (newPesListSize == 0)
     {
        pesListSize = 0;
	pesListDestroy();
	pesListInitialise();
	return;
     }

     if ((thePesListResized = (struct eventState **)realloc(thePesList,
				sizeof(struct eventState *) * (newPesListSize)))
				!= NULL)
     {
       thePesList = thePesListResized;
       pesListSize = newPesListSize;
     }
     else
       show_error("Internal Error: Resize Fail");
   }
}

void pesDelete (struct eventState *pes)
{
  if (pes == NULL)
     return;

  free(pes->screen_buf);
  free(pes->text_buf);
  free(pes->filename);
  free(pes);
}