/*
   This file is part of the CLib sub-project of the FreeDOS project
   Copyright (C) 1997 by the author see below

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/* $RCSfile: FFIND.C $
   $Locker:  $  $Name:  $ $State: Exp $

   int findfirst(char *name, struct ffblk *ff, int attr)
   int findnext(struct ffblk *ff)
   void findstop(struct ffblk *ff)

   DOS style scan of directories. findfirst() starts a directory scan,
   findnext() continues the scan. findstop() closes a scan.

   findstop() is Win95 specific, but is implicitly called by findnext()
   if findnext() does not find any further match. Though, for portably
   reason, one should invoke findstop() in normal DOS, too.

   The scan matches any files specified by name. The very last
   filename component of name can contain wildcards, name may contain
   a path specification.
   The attributes are matched in an "inclusive" manner, that means that
   if FA_HIDDEN is specified, find*() returns hidden and non-hidden files.
   attr may be constructed out of:
   FA_HIDDEN, FA_SYSTEM, FA_LABEL,& FA_DIREC.
   Any further bits are ignored.

   Input:
   ff != NULL
   name != NULL

   Return:
   0: OK, a match found
   else: DOS error code (also means: no further match found)

   Note:
   The ffblk structure is _not_ a copy of part of the DTA used by
   DOS for the find*() functions.

   Conforms to:
   DOS

   See also:
   opendir

   Target compilers:
   Any C compiler

   Origin:
   Steffen Kaiser (ska)

   Revised by:
   <none>

   File Revision:    Revision 1.2  1998/01/29 07:10:15  ska
 */

#include <_clib.h>              /* standard include, must be the first! */
#include <dir.h>
#include <dos.h>
#include <errno.h>
#include <string.h>

#define FIND_FIRST 0x4e00
#define FIND_NEXT 0x4f00

static int findfile();          /* function that performs any find*() action */

#ifdef RCS_Version
static char const rcsid[] =
"$Id: FFIND.C 1.2 1998/01/29 07:10:15 ska Exp $";
#endif

/*  All three functions have been placed into the same function, because:
   + findfirst() & findnext() are direct wrappers of findfile().
   + findfile() is not to be invoked stand-alone -> those files
   can be placed into the same file.
   + findstop() is referenced from findfile(), but there is no
   stand-alone useage thinkable.
   - 1997/10/22 ska */

#if 0 || !defined(LFN_WIN95)    /* Change 0->1 to have functions instead
                                   of macros */

#ifdef LFN_WIN95
#define dos_findfirst findfirst
#define dos_findnext findnext
#define ___DEF _CLibFunc
#else
#define ___DEF static
#endif

___DEF int dos_findfirst(char *name, struct ffblk *ff, int attr)
{
  return findfile(ff, FIND_FIRST, name, attr);
}

___DEF int dos_findnext(struct ffblk *ff)
{
  return findfile(ff, FIND_NEXT);
}

#else                           /* 0 */
#define dos_findfirst(name,ff,attr) findfile(ff,FIND_FIRST,name,attr)
#define dos_findnext(ff) findfile(ff,FIND_NEXT)

#endif                          /* 0 */

/******************** LFN API interface ************************/

#ifdef LFN_WIN95

/*
 *  Convert LFN style struct ffblk into DOS style
 */
static void convLFN(struct ffblk *ff)
{
  if (*ff->ff_lfn.ff_shortname) /* shortname present */
    memcpy(ff->ff_name, ff->ff_lfn.ff_shortname, sizeof(ff->ff_name));
  else                          /* no shortname present! extract from longname */
    strncpy(ff->ff_name, ff->ff_lfn.ff_longname, sizeof(ff->ff_name));
  _dosftime(ff->ff_lfn.ff_tmodification, &ff->ff_ftime, &ff->ff_fdate);
  ff->ff_fsize = ff->ff_lfn.ff_lowsize;
  ff->ff_attrib = (byte) ff->ff_lfn.ff_attr;
}

/*
 *  findfirst() for the LFN API
 *  On success the non-LFN portion is updated with the LFN data.
 */
static int findLFN(char *name
          ,struct ffblk *ff
          ,int attr)
{
  struct REGPACK r;
  REG int err;

  r.r_ax = 0x714e;
  r.r_cx = attr;
  r.r_si = 0;                   /* LFN style time */
  r.r_ds = FP_SEG(name);
  r.r_dx = FP_OFF(name);
  r.r_es = FP_SEG(&ff->ff_lfn);
  r.r_di = FP_OFF(&ff->ff_lfn);
  if ((err = _callDOS(&r)) == EZERO)
  {                             /* success */
    ff->ff_lfn.ff_status = r.r_ax;
    convLFN(ff);
  }
  else                          /* Failure */
    ff->ff_lfn.ff_status = _NOLFN;

  return err;
}

/*
 *  findnext() for the LFN API
 *  On success the non-LFN portion is updated with the LFN data.
 */
static int nextLFN(struct ffblk *ff)
{
  struct REGPACK r;
  REG int err;

  r.r_ax = 0x714f;
  r.r_bx = ff->ff_lfn.ff_status;
  r.r_si = 0;                   /* LFN style time/date */
  r.r_es = FP_SEG(&ff->ff_lfn);
  r.r_di = FP_OFF(&ff->ff_lfn);
  if ((err = _callDOS(&r)) == EZERO)    /* found */
    convLFN(ff);
  else                          /* failure -> end loop */
    findstop(ff);

  return err;
}

#endif

/*****************  findfirst/findnext interface  ********************/

#ifdef LFN_WIN95

/*
 *  Start a scan and fill the FF block, any original values of ff
 *  are ignored completely.
 *
 *  Return:
 *    == 0: OK a match found
 *    != 0: failure; OS error code
 */
_CLibFunc int findfirst(char *name
            ,struct ffblk *ff
            ,int attr)
{
  assert(name != NULL);
  assert(ff != NULL);

  if (findLFN(name, ff, attr & 0x7f) == 0)      /* try LFN API */
    return 0;                   /* OK */
  return dos_findfirst(name, ff, attr); /* try DOS API */
}

/*
 *  Continue a previously startet search.
 *  Return:
 *    == 0: OK a match found
 *    != 0: failure; OS error code
 */
_CLibFunc int
  findnext(struct ffblk *ff)
{
  assert(ff != NULL);

  return ff->ff_lfn.ff_status == _NOLFN ? dos_findnext(ff) : nextLFN(ff);
}

#endif

/*
 *  Stops a previously startet search.
 *  This call should be invoked in Win95 to ensure that all ressources
 *  are freed that were allocated to perform the request by the system.
 *  It is automatically performed, if nextLFN() returns not-OK
 */
_CLibFunc void
  findstop(struct ffblk *ff)
{
  assert(ff != NULL);

#ifdef LFN_WIN95
  if (ff->ff_lfn.ff_status != _NOLFN)
  {
    struct REGPACK r;

    r.r_ax = 0x71a1;
    r.r_bx = ff->ff_lfn.ff_status;
    ff->ff_lfn.ff_status = _NOLFN;      /* prevent further findstop() */
    _callDOS(&r);
  }
#endif
}

/******************** DOS API interface ************************/

/*
 *  Find first/next file using DOS API
 *
 *  The definition must be placed at the end of the file to allow
 *  to call this function with a variable sized parameter list without
 *  warnings from the compiler.
 *
 *  Return:
 *    0: if match found
 *    else: OS error code
 */
static int
  findfile(struct ffblk *ff
           ,int mode
           ,char_t * name
           ,int attr)
{
  struct REGPACK r;
  byte far *curDTA;
  REG int err;

  /* first swap the DTA */
  curDTA = getdta();
  setdta((byte far *) ff);

  /* second peform the desired command */
  if ((r.r_ax = mode) == FIND_FIRST)
  {
    r.r_ds = FP_SEG(name);
    r.r_dx = FP_OFF(name);
    r.r_cx = attr;
  }
  err = _callDOS(&r);

  /* third re-set previous DTA */
  setdta(curDTA);

  /* forth check for an error */
  return err;
}
