/*
 *  ALIAS.C - alias administration module
 *
 *
 *
 *  Comments:
 *
 *  02/02/1996 (Oliver Mueller)
 *    started.
 *
 *  02/03/1996 (Oliver Mueller)
 *    Added sorting algorithm and case sensitive substitution by using
 *    partstrupr().
 *
 * 27 Jul 1998  John P. Price
 * - added config.h include
 * - added ifdef's to disable aliases
 *
 */

#include "config.h"

#ifdef FEATURE_ALIASES

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <mem.h>
#include <ctype.h>

#include "tempfile.h"
#include "alias.h"


typedef struct TAlias
{
  char *name,
   *subst;
  unsigned short used;
  struct TAlias *next;
} TAlias;

static unsigned short useFlag = 0;
static TAlias *first = 0, *last = 0;
static FILE *tmp;

/* module internal functions */
void partstrupr(char *str)           /* strupr only for first word in string */
{
  char *c = str;
  while (*c && !isspace(*c))
  {
    *c = toupper(*c);
    c++;
  }
}

void aliasprint(void)
{
  TAlias *ptr = first;
  while (ptr)
  {
    printf("%s=%s\n", ptr->name, ptr->subst);
    ptr = ptr->next;
  }
}

void aliasdel(char *name)
{
  TAlias *ptr = first,
   *prev = 0;
  while (ptr)
  {
    if (!strcmp(ptr->name, name))
    {
      if (prev)
        prev->next = ptr->next;
      else
        first = ptr->next;
      free(ptr->name);
      free(ptr->subst);
      free(ptr);
    }
    prev = ptr;
    ptr = ptr->next;
  }
}

int aliasadd(char *name, char *subst)
{
  TAlias *ptr = first,
   *prev,
   *entry;
  char *s;

  while (ptr)
  {
    if (!strcmp(ptr->name, name))
    {
      s = (char *)malloc(strlen(subst) + 1);
      if (!s)
      {
        errno = ENOMEM;
        return -1;
      }
      free(ptr->subst);
      ptr->subst = s;
      strcpy(ptr->subst, subst);
      return 0;
    }
    ptr = ptr->next;
  }

  ptr = (TAlias *) malloc(sizeof(TAlias));
  if (!ptr)
  {
    errno = ENOMEM;
    return -1;
  }
  ptr->next = 0;

  ptr->name = (char *)malloc(strlen(name) + 1);
  if (!ptr->name)
  {
    free(ptr);
    errno = ENOMEM;
    return -1;
  }
  strcpy(ptr->name, name);

  ptr->subst = (char *)malloc(strlen(subst) + 1);
  if (!ptr->subst)
  {
    free(ptr->name);
    free(ptr);
    errno = ENOMEM;
    return -1;
  }
  strcpy(ptr->subst, subst);
  partstrupr(ptr->subst);       /* it's necessary for recursive substitution */

  ptr->used = 0;

  /* Alias table must be sorted!
   * Here a little example:
   *   command line = "ls -c"
   * If the entries are
   *   ls=dir
   *   ls -c=ls /w
   * command line will be expanded to "dir -c" which is not correct.
   * If the entries are sortet as
   *   ls -c=ls /w
   *   ls=dir
   * it will be expanded to "dir /w" which is a valid DOS command.
   */
  entry = first;
  prev = 0;
  while (entry)
  {
    if (strcmp(ptr->name, entry->name) > 0)
    {
      if (prev)
      {
        prev->next = ptr;
        ptr->next = entry;
      }
      else
      {
        ptr->next = entry;
        first = ptr;
      }
      return 0;
    }
    prev = entry;
    entry = entry->next;
  }

  /* The new entry is the smallest (or the first) and must be
   * added to the end of the list.
   */
  if (!first)
    first = ptr;
  else
    last->next = ptr;
  last = ptr;

  return 0;
}

/* specified routines */
void aliasexpand(char *cmd, int maxlen)
{
  unsigned n = 0,
    m,
    i,
    len;
  short d = 1;
  TAlias *ptr = first;

  useFlag++;
  if (useFlag == 0)
  {
    while (ptr)
      ptr->used = 0;
    ptr = first;
    useFlag = 1;
  }

  /* skipping white spaces */
  while (isspace(cmd[n]))
    n++;

  partstrupr(&cmd[n]);

  if (!strncmp(&cmd[n], "NOALIAS", 7) && (isspace(cmd[n + 7]) || cmd[n + 7] == '\0'))
  {
    memmove(cmd, &cmd[n + 7], strlen(&cmd[n + 7]) + 1);
    return;
  }

  /* substitution loop */
  while (d)
  {
    d = 0;
    while (ptr)
    {
      len = strlen(ptr->name);
      if (!strncmp(&cmd[n], ptr->name, len) && (isspace(cmd[n + len]) || cmd[n + len] == '\0') && ptr->used != useFlag)
      {
        m = strlen(ptr->subst);
        if (strlen(cmd) - len + m - n > maxlen)
        {
          fprintf(stderr, "Command line too long after alias expansion!\n");
          cmd[0] = '\0';        /* the parser won't cause any problems with an empty line */
        }
        else
        {
          memmove(&cmd[m], &cmd[n + len], strlen(&cmd[n + len]) + 1);
          for (i = 0; i < m; i++)
            cmd[i] = ptr->subst[i];
          ptr->used = useFlag;
          n = 0;                /* whitespaces are removed! */
          d = 1;
        }
      }
      ptr = ptr->next;
    }
  }
}

int aliasswapout(void)
{
  TAlias *ptr = first,
   *old;

  tmp = tempfile();
  if (!tmp)
    return -1;

  while (ptr)
  {
    if (fprintf(tmp, "%s=%s\n", ptr->name, ptr->subst) == EOF)
    {
      fclose(tmp);
      rmtmpfile();
      return -1;
    }
    ptr = ptr->next;
  }

  while (ptr)
  {
    old = ptr;
    ptr = ptr->next;
    free(old->name);
    free(old->subst);
    free(old);
  }

  return 0;
}

int aliasswapin(void)
{
  char buf[1024];

  fseek(tmp, 0, SEEK_SET);      /* reset file pointer */

  while (fgets(buf, sizeof(buf), tmp))
  {
    buf[strlen(buf) - 1] = '\0';
    if (cmd_alias(0, buf) == -1)
      return -1;
  }
  useFlag = 0;

  fclose(tmp);                  /* close and delete temporary file */
  rmtmpfile();

  return 0;
}

#pragma argsused
int cmd_alias(char *first, char *rest)
{
  char *ptr;
  int n = 0;

  if (rest[0] == '\0')
  {
    aliasprint();
    return 0;
  }

  if ((ptr = strchr(rest, '=')) == 0)
  {
    errno = EINVAL;
    return -1;
  }

  /* Split rest into name and substitute */
  *ptr++ = '\0';

  partstrupr(rest);

  if (ptr[0] == '\0')
    aliasdel(rest);
  else
    n = aliasadd(rest, ptr);

  return n;
}

#endif
