/*
 *  DIR.C - dir internal command
 *
 *  Comments:
 *
 *  01/29/97 (Tim Norman) ---------------------------------------------------
 *    started.
 *
 *  06/13/97 (Tim Norman) ---------------------------------------------------
 *    Fixed code.
 *
 *  07/12/97 (Tim Norman) ---------------------------------------------------
 *    Fixed bug that caused the root directory to be unlistable
 *
 *  07/12/97 (Marc Desrochers) ----------------------------------------------
 *    Changed to use maxx, maxy instead of findxy()
 *
 *  06/08/98 (Rob Lake)
 *    Added compatibility for /w in dir
 *
 * Compatibility for dir/s started 06/09/98 -- Rob Lake
 * 06/09/98 (Rob Lake)
 *  -Tested that program finds directories off root fine
 *       
 *
 * 06/10/98 (Rob Lake)
 *      -do_recurse saves the cwd and also stores it in Root
 *      -build_tree adds the cwd to the beginning of
 *    its' entries
 *      -Program runs fine, added print_tree -- works fine.. as EXE,
 *    program won't work properly as COM.
 *
 * 06/11/98 (Rob Lake)
 *  -Found problem that caused COM not to work
 *
 * 06/12/98 (Rob Lake)
 *      -debugged...
 *      -added free mem routine
 *
 * 06/13/98 (Rob Lake)
 *      -debugged the free mem routine
 *      -debugged whole thing some more
 *      Notes:
 *      -ReadDir stores Root name and _Read_Dir does the hard work
 *      -PrintDir prints Root and _Print_Dir does the hard work
 *      -KillDir kills Root _after_ _Kill_Dir does the hard work
 *      -Integrated program into DIR.C(this file) and made some same
 *       changes throughout
 *
 * 06/14/98 (Rob Lake)
 *      -Cleaned up code a bit, added comments
 *
 * 06/16/98 (Rob Lake)
 *  - Added error checking to my previously added routines
 *
 * 06/17/98 (Rob Lake)
 *      - Rewrote recursive functions, again! Most other recursive
 *      functions are now obsolete -- ReadDir, PrintDir, _Print_Dir,
 *      KillDir and _Kill_Dir.  do_recurse does what PrintDir did
 *      and _Read_Dir did what it did before along with what _Print_Dir
 *      did.  Makes /s a lot faster!
 *  - Reports 2 more files/dirs that MS-DOS actually reports
 *      when used in root directory(is this because dir defaults
 *      to look for read only files?)
 *      - Added support for /b, /a and /l
 *      - Made error message similar to DOS error messages
 *  - Added help screen
 *
 * 06/20/98 (Rob Lake)
 *  - Added check for /-(switch) to turn off previously defined
 *  switches
 *  - Added ability to check for DIRCMD in environment and
 *  process it
 *
 * 06/21/98 (Rob Lake)
 *  - Fixed up /B
 *  - Now can dir *.ext/X, no spaces!
 *
 * 06/29/98 (Rob Lake)
 *      - error message now found in command.h
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <dir.h>
#include <dirent.h>
#include <dos.h>
#include <io.h>
#include <conio.h>
#include <string.h>
#include <alloc.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "command.h"

/* useful macros */
#define MEM_ERR fprintf (stderr, "Memory error.\n"); return 1;

/* flag definitions */
/* Changed hex to decimal, hex wouldn't work
 * if > 8, Rob Lake 06/17/98.
 */
enum
{
  DIR_RECURSE = 1,
  DIR_PAGE = 2,
  DIR_WIDE = 4,                 /* Rob Lake */
  DIR_BARE = 8,                 /* Rob Lake */
  DIR_ALL = 16,                 /* Rob Lake */
  DIR_LWR = 32
};                              /* Rob Lake */

/* Globally save the # of dirs, files and bytes,
 * probabaly later pass them to functions. Rob Lake  */
long recurse_dir_cnt,
  recurse_file_cnt,
  recurse_bytes;

/*
 * help
 *
 * displays help screen for dir
 * Rob Lake
 */
void
help(void)
{
  printf("Displays a list of files and directories in a give subdirectory\n" \
         "\n" \
       "DIR [drive:][directory][filename] [/S] [/P] [/W] [/A] [/B] [/L]\n" \
         "\n" \
         "    /S -- Displays files in specified directory and all subdirectories\n" \
         "    /P -- Pauses after each screen full\n" \
         "    /W -- Prints in wide format\n" \
         "    /A -- Displays files with HIDDEN SYSTEM attributes\n" \
         "         default is ARCHIVE and READ ONLY\n" \
       "    /B -- Prints in bare format (no header or file information)\n" \
         "    /L -- Prints files in lower case\n" \
         "\n" \
      "Switches may be present in the DIRCMD environment variable.  Use\n" \
       "of the - (hyphen) can turn off defined swtiches.  Ex. /-W would\n" \
         "turn off printing in wide format.\n\n" \
      );
}

/*
 * dir_read_param
 *
 * read the parameters from the command line
 */
int
dir_read_param(char *line, char **param, unsigned *flags)
{
  int slash = 0;

  *param = NULL;

  /* scan the command line, processing switches */
  while (*line)
  {
    /* process switch */
    if (*line == '/' || slash)
    {
      if (!slash)
        line++;
      slash = 0;
/* begin Rob Lake 06/20/98 */
      if (*line == '-')
      {
        line++;
        if (toupper(*line) == 'S')
          *flags &= ~DIR_RECURSE;
        else if (toupper(*line) == 'P')
          *flags &= ~DIR_PAGE;
        else if (toupper(*line) == 'W')
          *flags &= ~DIR_WIDE;
        else if (toupper(*line) == 'B')
          *flags &= ~DIR_BARE;
        else if (toupper(*line) == 'A')
          *flags &= ~DIR_ALL;
        else if (toupper(*line) == 'L')
          *flags &= ~DIR_LWR;
        else
        {
          fprintf(stderr, INVALID_SWITCH, toupper(*line));
          return 1;
        }
        line++;
        continue;
      }
      else
      {
/* end Rob Lake 06/20/98 */
        if (toupper(*line) == 'S')
          *flags |= DIR_RECURSE;
        else if (toupper(*line) == 'P')
          *flags |= DIR_PAGE;
/* Rob Lake */
        else if (toupper(*line) == 'W')
          *flags |= DIR_WIDE;
        else if (toupper(*line) == 'B')
          *flags |= DIR_BARE;
        else if (toupper(*line) == 'A')
          *flags |= DIR_ALL;
        else if (toupper(*line) == 'L')
          *flags |= DIR_LWR;
        else if (*line == '?')
        {
          help();
          return 1;
        }
/* Rob Lake */
        else
        {
          fprintf(stderr, INVALID_SWITCH, toupper(*line));
          return 1;
        }
        line++;
        continue;
      }
    }

    /* process parameter */
    if (!isspace(*line))
    {
      if (*param)
      {
        fprintf(stderr, TOO_MANY_PARAMETERS, *param);
        return 1;
      }

      *param = line;

      /* skip to end of line or next whitespace or next / */
      while (*line && !isspace(*line) && *line != '/')
        line++;

      /* if end of line, return */
      if (!*line)
        return 0;

      /* if parameter, remember to process it later */
      if (*line == '/')
        slash = 1;

      *line++ = 0;
      continue;
    }

    line++;
  }

  if (slash)
  {
    fprintf(stderr, INVALID_SWITCH, *line);
    return 1;
  }

  return 0;
}

/*
 * extend_file
 *
 * extend the filespec, possibly adding wildcards
 */
void
extend_file(char **file)
{
  char *tmp;

  if (!*file)
    return;

  /* if no file spec, change to *.* */
  if (!**file)
  {
    free(*file);
    *file = strdup("*.*");
    return;
  }

  /* if starts with . add * in front */
  if (**file == '.')
  {
    tmp = malloc(strlen(*file) + 2);
    if (tmp)
    {
      *tmp = '*';
      strcpy(&tmp[1], *file);
    }
    free(*file);
    *file = tmp;
    return;
  }

  /* if no . add .* */
  if (strchr(*file, '.') == NULL)
  {
    tmp = malloc(strlen(*file) + 3);
    if (tmp)
    {
      strcpy(tmp, *file);
      strcat(tmp, ".*");
    }
    free(*file);
    *file = tmp;
    return;
  }
}

/*
 * dir_parse_pathspec
 *
 * split the pathspec into drive, directory, and filespec
 */
int
dir_parse_pathspec(char *pathspec, int *drive, char **dir, char **file)
{
  char *start,
   *tmp,
    orig_dir[128];
  int i,
    wildcards = 0;

  /* get the drive and change to it */
  if (pathspec[1] == ':')
  {
    *drive = toupper(pathspec[0]) - 'A';
    start = pathspec + 2;
    setdisk(*drive);
  }
  else
  {
    *drive = getdisk();
    start = pathspec;
  }

  getcwd(orig_dir, 128);

  /* check for wildcards */
  for (i = 0; pathspec[i]; i++)
    if (pathspec[i] == '*' || pathspec[i] == '?')
      wildcards = 1;

  /* check if this spec is a directory */
  if (!wildcards)
  {
    if (chdir(pathspec) == 0)
    {
      *file = strdup("*.*");
      if (!*file)
      {
        MEM_ERR
      }
      tmp = getcwd(NULL, 128);
      if (!tmp)
      {
        free(*file);
        chdir(orig_dir);
        MEM_ERR
      }
      *dir = strdup(&tmp[2]);
      free(tmp);
      if (!*dir)
      {
        free(*file);
        chdir(orig_dir);
        MEM_ERR
      }

      chdir(orig_dir);
      return 0;
    }
  }

  /* find the file spec */
  tmp = strrchr(start, '\\');

  /* if no path is specified */
  if (!tmp)
  {
    *file = strdup(start);
    extend_file(file);
    if (!*file)
    {
      MEM_ERR
    }

    tmp = getcwd(NULL, 128);
    if (!tmp)
    {
      free(*file);
      chdir(orig_dir);
      MEM_ERR
    }
    *dir = strdup(&tmp[2]);
    free(tmp);
    if (!*dir)
    {
      free(*file);
      chdir(orig_dir);
      MEM_ERR
    }

    return 0;
  }

  /* get the filename */
  *file = strdup(tmp + 1);
  extend_file(file);
  if (!*file)
  {
    MEM_ERR
  }

  *tmp = 0;

  /* change to this directory and get its full name */
  if (chdir(start) < 0)
  {
    fprintf(stderr, PATH_NOT_FOUND);
    *tmp = '\\';
    free(*file);
    chdir(orig_dir);
    return 1;
  }

  tmp = getcwd(NULL, 128);
  if (!tmp)
  {
    free(*file);
    MEM_ERR
  }
  *dir = strdup(&tmp[2]);
  free(tmp);
  if (!*dir)
  {
    free(*file);
    MEM_ERR
  }

  *tmp = '\\';

  chdir(orig_dir);
  return 0;
}

/*
 * pause
 *
 * pause until a key is pressed
 */
int
pause(void)
{
  int c;

  if ((isatty(0)) && (isatty(1)))
  {
    printf("Press any key to continue . . .");
    c = getch();
    printf("\n");
    if (c == 27 || c == 3)
    {
      return 1;
    }
    if (c == 0)
    {
      getch();
    }
  }
  return 0;
}

/*
 * incline
 *
 * increment our line if paginating, display message at end of screen
 */
int
incline(int *line, unsigned flags)
{
  if (!(flags & DIR_PAGE))
    return 0;

  (*line)++;

  if (*line >= *maxy)
  {
    *line = 0;
    return pause();
  }

  return 0;
}

/*
 * dir_print_header
 *
 * print the header for the dir command
 */
int
dir_print_header(int drive, int *line, unsigned flags)
{
  struct media_id
  {
    int info_level,
      serial1,
      serial2;
    char vol_id[11],
      file_sys[8];
  }
  media;
  struct ffblk f;
  struct SREGS s;
  union REGS r;

  if (flags & DIR_BARE)
    return 0;
  /* change to the drive */
  setdisk(drive);
  if (getdisk() != drive)
  {
    fprintf(stderr, "Invalid drive specification\n");
    return 1;
  }

  /* get the media ID of the drive */
  media.info_level = 0;
  r.x.bx = drive + 'A' - '@';
  r.x.cx = 0x866;
  s.ds = FP_SEG(&media);
  r.x.dx = FP_OFF(&media);
  r.x.ax = 0x440d;
  int86x(0x21, &r, &r, &s);
  media.vol_id[10] = NULL;

  /* print drive info */
  printf(" Volume in drive %c", drive + 'A');

  if (findfirst("\\*.*", &f, FA_LABEL) == 0)
  {
    printf(" is %s\n", f.ff_name);
  }
  else
  {
    printf(" has no label\n");
  }
  if (incline(line, flags) != 0)
    return 1;

  /* print the volume serial number if the return was successful */
  if (!r.x.cflag)
  {
    printf(" Volume Serial Number is %04X-%04X\n", media.serial2, media.serial1);
    if (incline(line, flags) != 0)
      return 1;
  }

  return 0;
}

/*
 * convert
 *
 * insert commas into a number
 */
int
convert(long num, char *des)
{
  char temp[32];
  int c = 0,
    n = 0;

  if (num == 0)
  {
    des[0] = 0x30;
    des[1] = 0;
    n = 1;
  }
  else
  {
    temp[31] = 0;
    while (num > 0)
    {
      if (((c + 1) % 4) == 0)
        temp[30 - c++] = ',';
      temp[30 - c++] = (char)(num % 10) + 0x30;
      num /= 10;
    }
    for (n = 0; n <= c; n++)
      des[n] = temp[31 - c + n];
  }
  return n;
}

/*
 *
 * print_summary: prints dir summary
 * Added by Rob Lake 06/17/98 to compact code
 * Just copied Tim's Code and patched it a bit
 *
 */
int
print_summary(int drive, long files, long dirs, long bytes,
              unsigned flags, int *line)
{
  char buffer[32];
  union REGS r;

  if (flags & DIR_BARE)
    return 0;

  convert(files, buffer);
  printf("   %6s file%c", buffer, files == 1 ? ' ' : 's');
  convert(bytes, buffer);
  printf("   %12s byte%c\n", buffer, bytes == 1 ? ' ' : 's');
  if (incline(line, flags) != 0)
    return 1;

  /* print number of dirs and bytes free */
  printf("%9d dirs", dirs);
  r.h.ah = 0x36;
  r.h.dl = drive + 'A' - '@';
  int86(0x21, &r, &r);
  convert((long)r.x.ax * r.x.bx * r.x.cx, buffer);
  if ((flags & DIR_RECURSE) == 0)
    printf(" %15s bytes free\n", buffer);
  if (incline(line, flags) != 0)
    return 1;
  return 0;
}

/*
 * dir_list
 *
 * list the files in the directory
 */
int
dir_list(int drive, char *directory, char *filespec, int *line,
         unsigned flags)
{
  char pathspec[128],
   *ext,
    buffer[32];
  struct ffblk file;
  long bytecount = 0,
    filecount = 0,
    dircount = 0;
  int time,
    count;
  unsigned mode = FA_RDONLY | FA_ARCH | FA_DIREC;

  /* if the user wants all files listed RL 06/17/98 */
  if (flags & DIR_ALL)
    mode |= FA_HIDDEN | FA_SYSTEM;

  if (directory[strlen(directory) - 1] == '\\')
    sprintf(pathspec, "%c:%s%s", drive + 'A', directory, filespec);
  else
    sprintf(pathspec, "%c:%s\\%s", drive + 'A', directory, filespec);

  if (findfirst(pathspec, &file, mode) != 0)
  {
    /* Don't want to print anything if scanning recursively
     * for a file. RL
     */
    if ((flags & DIR_RECURSE) == 0)
    {
      printf("File not found.\n");
      incline(line, flags);
      return 1;
    }
    return 0;
  }

  /* moved down here because if we are recursively searching and
   * don't find any files, we don't want just to print
   * Directory of C:\SOMEDIR
   * with nothing else
   * Rob Lake 06/13/98
   */
  if ((flags & DIR_BARE) == 0)
  {
    printf(" Directory of %c:%s\n", drive + 'A', directory);
    if (incline(line, flags) != 0)
      return 1;
    printf("\n");
    if (incline(line, flags) != 0)
      return 1;
  }

/* For counting columns of output */
  count = 0;

  do
  {
/* begin Rob Lake */
    if (flags & DIR_LWR)
      strlwr(file.ff_name);

    if (flags & DIR_WIDE && (flags & DIR_BARE) == 0)
    {
      if (file.ff_attrib & FA_DIREC)
      {
        sprintf(buffer, "[%s]", file.ff_name);
        dircount++;
      }
      else
      {
        sprintf(buffer, "%s", file.ff_name);
        filecount++;
      }
      printf("%-15s", buffer);
      count++;
      if (count == 5)
      {
        /* outputted 5 columns */
        printf("\n");
        if (incline(line, flags) != 0)
          return 1;
        count = 0;
      }
      bytecount += file.ff_fsize;
      /* next block 06/17/98 */
    }
    else if (flags & DIR_BARE)
    {
      if (strcmp(file.ff_name, ".") == 0 || strcmp(file.ff_name, "..") == 0)
        continue;
      if (flags & DIR_RECURSE)
      {
        char dir[128];
        sprintf(dir, "%c:%s\\", drive + 'A', directory);
        if (flags & DIR_LWR)
          strlwr(dir);
        printf(dir);
      }
      printf("%-13s\n", file.ff_name);
      if (file.ff_attrib & FA_DIREC)
        dircount++;
      else
        filecount++;
      if (incline(line, flags) != 0)
        return 1;
      bytecount += file.ff_fsize;
    }
    else
    {
/* end Rob Lake */
      if (file.ff_name[0] == '.')
        printf("%-13s", file.ff_name);
      else
      {
        ext = strchr(file.ff_name, '.');
        if (!ext)
          ext = "";
        else
          *ext++ = 0;

        printf("%-8s %-3s ", file.ff_name, ext);
      }

      if (file.ff_attrib & FA_DIREC)
      {
        printf("%-14s", "<DIR>");
        dircount++;
      }
      else
      {
        convert(file.ff_fsize, buffer);
        printf("   %10s ", buffer);
        bytecount += file.ff_fsize;
        filecount++;
      }

      printf("%.2d-%.2d-%d", ((file.ff_fdate >> 5) & 0x000f),
             (file.ff_fdate & 0x001f), ((file.ff_fdate >> 9) + 80));
      time = file.ff_ftime >> 5 >> 6;
      printf("  %2d:%.2u%c\n",
             (time == 0 ? 12 : (time <= 12 ? time : time - 12)),
             ((file.ff_ftime >> 5) & 0x003f),
             (time <= 11 ? 'a' : 'p'));

      if (incline(line, flags) != 0)
        return 1;

    }
  }
  while (findnext(&file) == 0);

/* Rob Lake, need to make clean output */
  if (flags & DIR_WIDE)
    printf("\n");

  if (filecount || dircount)
  {
    recurse_dir_cnt += dircount;
    recurse_file_cnt += filecount;
    recurse_bytes += bytecount;
    /* The code that was here is now in print_summary */
    if (print_summary(drive, filecount, dircount,
                      bytecount, flags, line) != 0)
      return 1;
  }
  else
  {
    printf("File not found\n");
    return 1;
  }
  if ((flags & DIR_BARE) == 0)
  {
    printf("\n");
    if (incline(line, flags) != 0)
      return 1;
  }

  return 0;
}

/*
 * _Read_Dir: Actual function that does recursive listing
 */
int
_Read_Dir(int drive, char *parent, char *filespec, int *lines,
          unsigned flags)
{
  DIR *dir;
  struct dirent *ent;

  if ((dir = opendir(parent)) == NULL)
    return 1;

  strupr(parent);
  if (parent[strlen(parent) - 1] == '\\')
    parent[strlen(parent) - 1] = '\0';

  while ((ent = readdir(dir)) != NULL)
  {
    char buffer[128];
    struct stat stbuf;
    if (strcmp(ent->d_name, ".") == 0)
      continue;
    if (strcmp(ent->d_name, "..") == 0)
      continue;
    sprintf(buffer, "%s\\%s", parent, ent->d_name);
    /* changed call to _chmod to a call to stat,
     * faster? or is it another interrupt call?
     */
    if (stat(buffer, &stbuf) == -1)
      return 1;
    if ((stbuf.st_mode & S_IFMT) == S_IFDIR)
    {
      if (dir_list(drive, buffer, filespec, lines, flags) != 0)
      {
        closedir(dir);
        return 1;
      }
      if ((flags & DIR_BARE) == 0)
      {
        printf("\n");
        if (incline(lines, flags) != 0)
          return 1;
      }
      if (chdir(buffer) == -1)
      {
        closedir(dir);
        return 1;
      }
      if (_Read_Dir(drive, buffer, filespec, lines, flags) == 1)
      {
        closedir(dir);
        return 1;
      }
      chdir("..");
    }
  }

  if (closedir(dir) != 0)
    return 1;
  return 0;
}

/*
 * do_recurse: Sets up for recursive directory listing
 */
int
do_recurse(int drive, char *directory, char *filespec,
           int *line, unsigned flags)
{
  char cur_dir[128];

  recurse_dir_cnt = recurse_file_cnt = recurse_bytes = 0L;
  setdisk(drive);
  getcwd(cur_dir, 128);

  if (chdir(directory) == -1)
    return 1;
  if (dir_print_header(drive, line, flags) != 0)
    return 1;
  if (dir_list(drive, directory, filespec, line, flags) != 0)
    return 1;
  if ((flags & DIR_BARE) == 0)
  {
    printf("\n");
    if (incline(line, flags) != 0)
      return 1;
  }
  if (_Read_Dir(drive, directory, filespec, line, flags) != 0)
    return 1;
  if ((flags & DIR_BARE) == 0)
    printf("Total files listed:\n");
  flags &= ~DIR_RECURSE;
  if (print_summary(drive, recurse_file_cnt,
                    recurse_dir_cnt, recurse_bytes, flags, line) != 0)
    return 1;
  if ((flags & DIR_BARE) == 0)
  {
    printf("\n");
    if (incline(line, flags) != 0)
      return 1;
  }
  printf("\n");
  if (incline(line, flags) != 0)
    return 1;
  chdir(cur_dir);
  return 0;
}

/*
 * dir
 *
 * internal dir command
 */
#pragma argsused
int
dir(char *first, char *rest)
{
  unsigned flags = 0;
  char *param,
   *dircmd;
  int line = 0;
  int drive,
    orig_drive;
  char *directory,
   *filespec,
    orig_dir[128];

  /* read the parameters from env */
  dircmd = getenv("DIRCMD");
  if (dir_read_param(dircmd, &param, &flags) != 0)
    return 1;
  /* read the parameters */
  if (dir_read_param(rest, &param, &flags) != 0)
    return 1;

  /* default to current directory */
  if (!param)
    param = ".";

  if (strchr(param, '/'))
    param = strtok(param, "/");

  /* save the current directory info */
  orig_drive = getdisk();
  getcwd(orig_dir, 128);

  /* parse the directory info */
  if (dir_parse_pathspec(param, &drive, &directory, &filespec) != 0)
  {
    setdisk(orig_drive);
    chdir(orig_dir);
    return 1;
  }
  if (flags & DIR_RECURSE)
  {
    incline(&line, flags);
    if (do_recurse(drive, directory, filespec, &line, flags) != 0)
      return 1;
    setdisk(orig_drive);
    chdir(orig_dir);
    return 0;
  }
  /* print the header */
  if (dir_print_header(drive, &line, flags) != 0)
    return 1;

  chdir(orig_dir);
  setdisk(orig_drive);

  if (dir_list(drive, directory, filespec, &line, flags) != 0)
    return 1;
  return 0;
}
