
#include "pDisplay.h"
#include "PhraseDefs.h"
#include "pEdit.h"

#include <app/Application.h>
#include <interface/TextView.h>
#include <interface/Menu.h>
#include <interface/MenuBar.h>
#include <interface/MenuItem.h>
#include <interface/Alert.h>
#include <interface/Screen.h>
#include <interface/ScrollBar.h>
#include <interface/Font.h>
#include <interface/PrintJob.h>
#include <storage/Entry.h>
#include <storage/File.h>
#include <storage/Path.h>
#include <storage/FilePanel.h>
#include <storage/NodeInfo.h>
#include <support/String.h>

#include <stdio.h>
#include <string.h>

static const BRect dummyFrame(0.0, 0.0, 0.0, 0.0); 
static const BPoint startPt(50.0,50.0);
static const BPoint incPt(175.0, 25.0);

BPoint	pDisplay::fNextWinPt(startPt);
int32	pDisplay::fUntitledCount(0);

pDisplay::pDisplay(const entry_ref *ref) :
	BWindow(dummyFrame, B_EMPTY_STRING, B_DOCUMENT_WINDOW_LOOK,
		B_NORMAL_WINDOW_FEEL, B_ASYNCHRONOUS_CONTROLS),
	fEdit(NULL),
	fSave(NULL),
	fQuitting(false)
{
	// create the phrase edit view
	fEdit = new pEdit("EditPhrase");
	// get the bounds of the text view to use in
	// layout calculations
	BRect editFrame = fEdit->Bounds();
	
	// build and add the menus
	BMenu *menu = BuildMenus();
	
	// make the window big enough to hold menu, edit view and scroll bars
	float menuHeight = menu->Frame().Height();
	float frameWidth = editFrame.Width() + B_V_SCROLL_BAR_WIDTH;
	float frameHeight = editFrame.Height() + B_H_SCROLL_BAR_HEIGHT + menuHeight + 1.0;
	
	ResizeTo(frameWidth, frameHeight);

	// Let's set some size limits:
	// minimum: 12 ems wide by 10 lines deep
	// maximum: screen size
	float minWidth = (12.0 * fEdit->EmWidth()) + B_V_SCROLL_BAR_WIDTH;
	float minHeight = (10.0 * fEdit->DefaultLineHeight()) + B_H_SCROLL_BAR_HEIGHT + menuHeight + 1.0;
	BScreen screen;
	BRect screenRect = screen.Frame();
	
	SetSizeLimits(minWidth, screenRect.Width(), minHeight,  screenRect.Height());
	
	// reposition edit view to just under the menu
	fEdit->MoveTo(0, menuHeight + 1.0);
	AddChild(fEdit);
	
	// reset the editFrame
	editFrame = fEdit->Frame();

	// build scroll bars based on the edit frame
	BScrollBar *scroll = NULL;
	BRect scrollRect;
	scrollRect.left = editFrame.right + 1.0;
	scrollRect.right = scrollRect.left + B_V_SCROLL_BAR_WIDTH;
	scrollRect.top = editFrame.top - 1.0;
	scrollRect.bottom = editFrame.bottom + 1.0;

	scroll = new BScrollBar(scrollRect, "VerticalScrollBar", NULL, 0.0, 100.0, B_VERTICAL);
	// target the edit view
	scroll->SetTarget(fEdit);
	AddChild(scroll);
	
	scrollRect.left = editFrame.left - 1.0;
	scrollRect.right = editFrame.right + 1.0;
	scrollRect.top = editFrame.bottom + 1.0;
	scrollRect.bottom = scrollRect.top + B_H_SCROLL_BAR_HEIGHT;

	scroll = new BScrollBar(scrollRect, "HorizontalScrollBar", NULL, 0.0, 100.0, B_HORIZONTAL);
	// target the edit view
	scroll->SetTarget(fEdit);
	AddChild(scroll);

	// target the menu items correctly
	TargetMenuItems(menu);

	// make sure the text view is the focus so it receives keyboard input
	fEdit->MakeFocus();

	// load the text of the file into the view
	LoadPhrase(ref);
	
	int32 num = atomic_add(&fUntitledCount, 1);
	
	if (strcmp(Title(), B_EMPTY_STRING) == 0)
	{
		BString title("Untitled ");
		title << num;
		SetTitle(title.String());
	}
	
	// position the window
	InitPosition();
}

pDisplay::~pDisplay()
{
	// note that BWindow will clean up all of its children
	// automatically for us.  There is no need (and it will cause problems)
	// if you delete the menu bar, menus, or the menu items created
	// in the constructor.
	fEdit = NULL;
	delete fSave; fSave = NULL;
	be_app->PostMessage(WINDOW_CLOSED);
}

bool 
pDisplay::QuitRequested()
{
	// don't close if the save panel is showing
	if(fSave && fSave->IsShowing())
	{
		fSave->Show();
		return false;
	}
	// save if dirty
	if (fEdit->IsDirty())
	{
		BString string("Would you like to save ");
		string += Title();
		string += " before closing?";
		BAlert *alert = new BAlert(B_EMPTY_STRING, string.String(), "Don't Close", "Don't Save", "Save",
			B_WIDTH_AS_USUAL, B_OFFSET_SPACING);
		int32 button = alert->Go();
		switch(button)
		{
			case 0:
				return false;
			case 2:
			{
				fQuitting = true;
				return Save();
			}
			case 1:
			default:
				return true;
		}
	}
	else return true;
}

void 
pDisplay::MessageReceived(BMessage *msg)
{
	switch(msg->what)
	{
		case SAVE_PHRASE:
		{
			Save();
			break;
		}	
		case SAVE_PHRASE_AS:
		{
			SaveTo(NULL);
			break;
		}	
		case B_SAVE_REQUESTED:
		{
			entry_ref ref;
			const char *name;
			msg->FindRef("directory", &ref);
			msg->FindString("name", &name);
			BDirectory dir(&ref);
			BEntry entry(&dir, name);
			SaveTo(&entry);
			break;
		}
		case B_CANCEL:
		{
			// always sent whenever the save panel closes
			// close down the window if previously quitting
			if (fQuitting)
				PostMessage(B_QUIT_REQUESTED);
			break;
		}
		default:
			BWindow::MessageReceived(msg);
			break;
	}
}

void 
pDisplay::LoadPhrase(const entry_ref *ref)
{
	BEntry entry(ref);
	if (!entry.Exists())
		return;

	// open the file
	BFile file(&entry, B_READ_ONLY);
	if (file.InitCheck() != B_OK)
	{
		// could not open so report an error
		char buf[256];
		BPath path(&entry);
		if (path.InitCheck() == B_OK)
			sprintf(buf, "Could not open %s\n", path.Path());		
		else
			sprintf(buf, "Could not open file");

		BAlert *alert = new BAlert("File Failure", buf, "Sorry!");
		alert->Go(NULL);
		return;
	}
	
	if (fEdit)
	{
		// read up to LONG_MAX bytes of the file into the text view
		off_t size = 0;
		file.GetSize(&size);
		int32 sizeToRead = min_c(size, (off_t)LONG_MAX);
		fEdit->SetText(&file, 0, sizeToRead);
		// set the title to the name of the file
		BPath path(&entry);
		if (path.InitCheck() == B_OK)
			SetTitle(path.Leaf());
	}

	fRef = *ref;
}


BMenu * 
pDisplay::BuildMenus()
{
	BRect bounds = Bounds();
	// build a menu bar
	BMenuBar *mBar = new BMenuBar(bounds, "MainMenu");
	// build the File menu
	BMenu *menu = new BMenu("File");
	menu->AddItem(new BMenuItem("New", new BMessage(NEW_PHRASE), 'N'));
	menu->AddItem(new BMenuItem("Open", new BMessage(OPEN_PHRASE), 'O'));
	menu->AddItem(new BSeparatorItem);
	menu->AddItem(new BMenuItem("Save", new BMessage(SAVE_PHRASE), 'S'));
	menu->AddItem(new BMenuItem("Save As", new BMessage(SAVE_PHRASE_AS), 'S', B_SHIFT_KEY));
	menu->AddItem(new BMenuItem("Close", new BMessage(B_QUIT_REQUESTED), 'W'));
	menu->AddItem(new BSeparatorItem);
	menu->AddItem(new BMenuItem("About Rephrase", new BMessage(B_ABOUT_REQUESTED)));
	menu->AddItem(new BSeparatorItem);
	menu->AddItem(new BMenuItem("Quit", new BMessage(B_QUIT_REQUESTED), 'Q'));
	mBar->AddItem(menu);
	//build the Edit menu
	menu = new BMenu("Edit");
	menu->AddItem(new BMenuItem("Cut", new BMessage(B_CUT), 'X'));
	menu->AddItem(new BMenuItem("Copy", new BMessage(B_COPY), 'C'));
	menu->AddItem(new BMenuItem("Paste", new BMessage(B_PASTE), 'V'));
	menu->AddItem(new BSeparatorItem);
	menu->AddItem(new BMenuItem("Select All", new BMessage(B_SELECT_ALL), 'A'));
	mBar->AddItem(menu);

	AddChild(mBar);
	return mBar;
}

void 
pDisplay::TargetMenuItems(BMenu *menu)
{
	BMenuItem *item = NULL;

	// File Menu
	item = menu->FindItem("New");
	if (item)
		item->SetTarget(be_app);
	
	item = menu->FindItem("Open");
	if (item)
		item->SetTarget(be_app);

	item = menu->FindItem("About Rephrase");
	if (item)
		item->SetTarget(be_app);

	item = menu->FindItem("Quit");
	if (item)
		item->SetTarget(be_app);

	// Edit Menu
	item = menu->FindItem("Edit");
	if (item)
		item->Submenu()->SetTargetForItems(fEdit);
}

void 
pDisplay::InitPosition()
{
	BRect frame(Frame());
	frame.OffsetTo(fNextWinPt);
	
	// check to see if the frame is out of bounds
	BScreen screen;
	BRect screenFrame = screen.Frame();
	if (frame.right > screenFrame.right)
	{
		// we are off the right edge so go down some
		fNextWinPt.x = startPt.x;
		fNextWinPt.y += incPt.y;
		frame.OffsetTo(fNextWinPt);
	}

	if (frame.top + 100.0 > screenFrame.bottom)
	{
		// we are off the bottom!
		// go back to the beginning
		fNextWinPt = startPt;
		frame.OffsetTo(fNextWinPt);
	}
	else fNextWinPt.x += incPt.x;

	MoveTo(frame.LeftTop());	
}

bool 
pDisplay::Save()
{
	BEntry entry(&fRef);
	if (entry.Exists())
	{
		// save to the specified file
		SaveTo(&entry);
		return true;
	}
	else
	{
		// a new file needs to be specified.
		SaveTo(NULL);
		return false;
	}
}

void 
pDisplay::SaveTo(const BEntry *entry)
{
	BPath path(entry);

	// if there is no entry, we need to show the save panel
	if (!entry)
	{
		// if no save panel, make one
		if (!fSave)
		{
			BMessenger msgr(this);
			fSave = new BFilePanel(B_SAVE_PANEL, &msgr);
			fSave->SetSaveText(Title());
			fSave->Window()->SetTitle("Rephrase: Save");
		}
		fSave->Show();
		return;
	}

	// open the file, creating it if necessary
	BFile file(entry, B_WRITE_ONLY | B_CREATE_FILE);
	status_t status = file.InitCheck();
	if (status != B_OK)
	{
		BString error("cannot open: ");
		error += path.Path();
		error += " ";
		error += strerror(status);
		error += "\n";
		
		BAlert *alert = new BAlert(B_EMPTY_STRING, error.String(), "Too Bad");
		alert->Go(NULL);
		return;
	}

	// write the contents of the phrase
	// fEdit->Text() returns a pointer to the text
	// fEdit->TextLength() returns that content's length
	ssize_t sizeWritten = file.Write(fEdit->Text(), fEdit->TextLength());
	if (sizeWritten < 0)
	{
		BString error("cannot write to: ");
		error += path.Path();
		error += " ";
		error += strerror(status);
		error += "\n";
		
		BAlert *alert = new BAlert(B_EMPTY_STRING, error.String(), "Too Bad");
		alert->Go(NULL);
		return;
	}
	else
	{
		// set the file size to avoid garbage when shortening a file
		file.SetSize(sizeWritten);
		
		// be sure that there is a mime type	
		BNodeInfo nInfo(&file);
		char type[B_MIME_TYPE_LENGTH];
		nInfo.GetType(type);
		if (strcmp(type, B_EMPTY_STRING) == 0)
			nInfo.SetType("text/plain");
		
		// set the window title to the name of the file
		char name[B_FILE_NAME_LENGTH];
		entry->GetName(name);
		SetTitle(name);
		
		// tell the file panel to use this new name
		fSave->SetSaveText(name);
		
		// mark the text view as clean
		fEdit->SetDirty(false);
		
		// mark fRef accordingly
		entry_ref ref;
		entry->GetRef(&ref);
		if (fRef != ref)
			fRef = ref;
	}
}
