
#include "pDisplay.h"

#include <app/Application.h>
#include <interface/TextView.h>
#include <interface/Menu.h>
#include <interface/MenuBar.h>
#include <interface/MenuItem.h>
#include <interface/Alert.h>
#include <interface/Screen.h>
#include <interface/ScrollBar.h>
#include <interface/Font.h>
#include <interface/PrintJob.h>
#include <storage/Entry.h>
#include <storage/File.h>
#include <storage/Path.h>

#include <stdio.h>

static const BRect dummyFrame(0.0, 0.0, 0.0, 0.0); 
static const BPoint startPt(50.0, 50.0);
static const BPoint incPt(175.0, 25.0);

BPoint pDisplay::fNextWinPt(startPt);

pDisplay::pDisplay(const entry_ref *ref) :
	BWindow(dummyFrame, "Rephrase", B_DOCUMENT_WINDOW_LOOK,
		B_NORMAL_WINDOW_FEEL, B_ASYNCHRONOUS_CONTROLS),
	fEdit(NULL)
{
	// base window size on the 
	// create a text view that has a default text rect of
	// at least 45 characters by 40 lines
	
	// use the font to determine line height and character width
	font_height height;
	be_plain_font->GetHeight(&height);
	float lineHeight = height.ascent + height.descent + height.leading;
	// typographically, the em (the width of M) is a standard unit of width
	// for a font with a roman character set
	float em = be_plain_font->StringWidth("M");

	// if a printer has been set up, use its printable width
	// as the lineWidth
	float lineWidth = 0.0;
	BPrintJob job("job");
	BRect printRect = job.PrintableRect();
	if (printRect.IsValid())
		lineWidth = printRect.Width();
	else
		// no printer set up - use 45 ems
		lineWidth = 45.0 * em;
		
	// build the text rect: lineWidth wide and 40 lines deep
	BRect textFrame(0.0, 0.0, lineWidth, 40.0 * lineHeight);
	// build a frame for the text view that is 4 pixels wider on each side
	BRect editFrame = textFrame;
	editFrame.InsetBy(-4.0, -4.0);
	editFrame.OffsetTo(B_ORIGIN);
	
	// create the phrase edit view
	fEdit = new BTextView(editFrame, "EditPhrase", textFrame, B_FOLLOW_ALL_SIDES);
	
	// build and add the menus
	BMenu *menu = BuildMenus();
	
	// make the window big enough to hold menu, edit view and scroll bars
	float menuHeight = menu->Frame().Height();
	float frameWidth = editFrame.Width() + B_V_SCROLL_BAR_WIDTH;
	float frameHeight = editFrame.Height() + B_H_SCROLL_BAR_HEIGHT + menuHeight + 1.0;
	
	ResizeTo(frameWidth, frameHeight);

	// Let's set some size limits:
	// minimum: 12 ems wide by 10 lines deep
	// maximum: screen size
	float minWidth = (12.0 * em) + B_V_SCROLL_BAR_WIDTH;
	float minHeight = (10.0 * lineHeight) + B_H_SCROLL_BAR_HEIGHT + menuHeight + 1.0;
	BScreen screen;
	BRect screenRect = screen.Frame();
	
	SetSizeLimits(minWidth, screenRect.Width(), minHeight,  screenRect.Height());
	
	// reposition edit view to just under the menu
	fEdit->MoveTo(0, menuHeight + 1.0);
	AddChild(fEdit);
	
	// reset the editFrame
	editFrame = fEdit->Frame();

	// build scroll bars based on the edit frame
	BScrollBar *scroll = NULL;
	BRect scrollRect;
	scrollRect.left = editFrame.right + 1.0;
	scrollRect.right = scrollRect.left + B_V_SCROLL_BAR_WIDTH;
	scrollRect.top = editFrame.top - 1.0;
	scrollRect.bottom = editFrame.bottom + 1.0;

	scroll = new BScrollBar(scrollRect, "VerticalScrollBar", NULL, 0.0, 100.0, B_VERTICAL);
	// target the edit view
	scroll->SetTarget(fEdit);
	AddChild(scroll);
	
	scrollRect.left = editFrame.left - 1.0;
	scrollRect.right = editFrame.right + 1.0;
	scrollRect.top = editFrame.bottom + 1.0;
	scrollRect.bottom = scrollRect.top + B_H_SCROLL_BAR_HEIGHT;

	scroll = new BScrollBar(scrollRect, "HorizontalScrollBar", NULL, 0.0, 100.0, B_HORIZONTAL);
	// target the edit view
	scroll->SetTarget(fEdit);
	AddChild(scroll);

	// target the menu items correctly
	BMenuItem *item = menu->FindItem("Edit");
	if (item)
		item->Submenu()->SetTargetForItems(fEdit);
		
	item = menu->FindItem("About Rephrase");
	if (item)
		item->SetTarget(be_app);

	// make sure the text view is the focus so it receives keyboard input
	fEdit->MakeFocus();

	// load the text of the file into the view
	LoadPhrase(ref);
	
	// position the window
	InitPosition();

	// show the window
	Show();
}

pDisplay::~pDisplay()
{
	// note that BWindow will clean up all of its children
	// automatically for us.  There is no need (and it will cause problems)
	// if you delete the menu bar, menus, or the menu items created
	// in the constructor.
	fEdit = NULL;
}

bool 
pDisplay::QuitRequested()
{
	// if we are the only window tell the app to quit
	if (be_app->CountWindows() == 1)
		be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

void 
pDisplay::FrameResized(float new_width, float new_height)
{
	// make sure that the text wraps to the window properly
	// this flickers a lot!  In the future we should try and improve this
	if (fEdit)
	{
		BRect editRect = fEdit->Bounds();
		editRect.InsetBy(4.0, 4.0);
		fEdit->SetTextRect(editRect);
	}
}

void 
pDisplay::LoadPhrase(const entry_ref *ref)
{
	BEntry entry(ref);
	if (!entry.Exists())
		return;

	// open the file
	BFile file(&entry, B_READ_ONLY);
	if (file.InitCheck() != B_OK)
	{
		// could not open so report an error
		char buf[256];
		BPath path(&entry);
		if (path.InitCheck() == B_OK)
			sprintf(buf, "Could not open %s\n", path.Path());		
		else
			sprintf(buf, "Could not open file");

		BAlert *alert = new BAlert("File Failure", buf, "Sorry!");
		alert->Go(NULL);
		return;
	}
	
	if (fEdit)
	{
		// read up to LONG_MAX bytes of the file into the text view
		off_t size = 0;
		file.GetSize(&size);
		int32 sizeToRead = 0;
		(size > (off_t)LONG_MAX) ? sizeToRead = LONG_MAX : sizeToRead = (int32)size;
		fEdit->SetText(&file, 0, sizeToRead);
		// set the title to the name of the file
		BPath path(&entry);
		if (path.InitCheck() == B_OK)
			SetTitle(path.Leaf());
	}
}


BMenu * 
pDisplay::BuildMenus()
{
	BRect bounds = Bounds();
	// build a menu bar
	BMenuBar *mBar = new BMenuBar(bounds, "MainMenu");
	// build the File menu
	BMenu *menu = new BMenu("File");
	menu->AddItem(new BMenuItem("About Rephrase", new BMessage(B_ABOUT_REQUESTED)));
	menu->AddItem(new BSeparatorItem);
	menu->AddItem(new BMenuItem("Quit", new BMessage(B_QUIT_REQUESTED), 'Q', B_COMMAND_KEY));
	mBar->AddItem(menu);
	//build the Edit menu
	menu = new BMenu("Edit");
	menu->AddItem(new BMenuItem("Cut", new BMessage(B_CUT), 'X', B_COMMAND_KEY));
	menu->AddItem(new BMenuItem("Copy", new BMessage(B_COPY), 'C', B_COMMAND_KEY));
	menu->AddItem(new BMenuItem("Paste", new BMessage(B_PASTE), 'V', B_COMMAND_KEY));
	menu->AddItem(new BSeparatorItem);
	menu->AddItem(new BMenuItem("Select All", new BMessage(B_SELECT_ALL), 'A', B_COMMAND_KEY));
	mBar->AddItem(menu);

	AddChild(mBar);
	return mBar;
}

void 
pDisplay::InitPosition()
{
	BRect frame(Frame());
	frame.OffsetTo(fNextWinPt);
	
	// check to see if the frame is out of bounds
	BScreen screen;
	BRect screenFrame = screen.Frame();
	if (frame.right > screenFrame.right)
	{
		// we are off the right edge so go down some
		fNextWinPt.x = startPt.x;
		fNextWinPt.y += incPt.y;
		frame.OffsetTo(fNextWinPt);
	}

	if (frame.top + 100.0 > screenFrame.bottom)
	{
		// we are off the bottom!
		// go back to the beginning
		fNextWinPt = startPt;
		frame.OffsetTo(fNextWinPt);
	}
	else fNextWinPt.x += incPt.x;

	MoveTo(frame.LeftTop());	
}
