
#include "pDisplay.h"

#include <app/Application.h>
#include <interface/TextView.h>
#include <interface/Menu.h>
#include <interface/MenuBar.h>
#include <interface/MenuItem.h>
#include <interface/Alert.h>
#include <interface/Screen.h>
#include <storage/Entry.h>
#include <storage/File.h>
#include <storage/Path.h>

#include <stdio.h>

static const BRect displayFrame(0,0, 349, 499);
static const BPoint startPt(50,50);
static const BPoint incPt(175, 25);

BPoint pDisplay::fNextWinPt(startPt);

pDisplay::pDisplay(const entry_ref *ref) :
	BWindow(displayFrame, "Rephrase", B_TITLED_WINDOW_LOOK,
		B_NORMAL_WINDOW_FEEL, B_NOT_ZOOMABLE | B_NOT_RESIZABLE)
{
	BRect bounds = Bounds();
	// build a menu bar
	BMenuBar *mBar = new BMenuBar(bounds, "MainMenu");
	// build the File menu
	BMenu *menu = new BMenu("File");
	menu->AddItem(new BMenuItem("About Rephrase", new BMessage(B_ABOUT_REQUESTED)));
	menu->AddItem(new BSeparatorItem);
	menu->AddItem(new BMenuItem("Quit", new BMessage(B_QUIT_REQUESTED), 'Q', B_COMMAND_KEY));
	mBar->AddItem(menu);
	//build the Edit menu
	menu = new BMenu("Edit");
	menu->AddItem(new BMenuItem("Cut", new BMessage(B_CUT), 'X', B_COMMAND_KEY));
	menu->AddItem(new BMenuItem("Copy", new BMessage(B_COPY), 'C', B_COMMAND_KEY));
	menu->AddItem(new BMenuItem("Paste", new BMessage(B_PASTE), 'V', B_COMMAND_KEY));
	menu->AddItem(new BSeparatorItem);
	menu->AddItem(new BMenuItem("Select All", new BMessage(B_SELECT_ALL), 'A', B_COMMAND_KEY));
	mBar->AddItem(menu);
	AddChild(mBar);

	// leave room for the menu bar (fixing off by one error from 0.1d1)
	bounds.top += mBar->Frame().Height() + 1;
		
	// inset the text by 4 in the text view's coordinate space
	BRect text(bounds);
	text.OffsetTo(B_ORIGIN);
	text.InsetBy(4,4);
	BTextView *edit = new BTextView(bounds, "EditPhrase", text, B_FOLLOW_ALL_SIDES);	
	AddChild(edit);

	// make sure the text view is the focus so it receives keyboard input
	edit->MakeFocus();
	
	// target the menu items correctly
	BMenuItem *item = mBar->FindItem("Edit");
	if (item)
		item->Submenu()->SetTargetForItems(edit);

	item = mBar->FindItem("About Rephrase");
	if (item)
		item->SetTarget(be_app);
	
	// load the text of the file into the view
	LoadPhrase(ref);
	
	// move to the initial position
	InitPosition();
	
	// show the window
	Show();
}


pDisplay::~pDisplay()
{
	// no cleanup
	// note that BWindow will clean up all of its children
	// automatically for us.  There is no need (and it will cause problems)
	// if you delete the menu bar, menus, menu items or text view created
	// in the constructor.  This is why we do not stash
	// any pointers to these items in the class
}

bool 
pDisplay::QuitRequested()
{
	// if we are the only window tell the app to quit
	if (be_app->CountWindows() == 1)
		be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

void 
pDisplay::LoadPhrase(const entry_ref *ref)
{
	BEntry entry(ref);
	if (!entry.Exists())
		return;

	// open the file
	BFile file(&entry, B_READ_ONLY);
	if (file.InitCheck() != B_OK)
	{
		// could not open so report an error
		char buf[256];
		BPath path(&entry);
		if (path.InitCheck() == B_OK)
			sprintf(buf, "Could not open %s\n", path.Path());		
		else
			sprintf(buf, "Could not open file");

		BAlert *alert = new BAlert("File Failure", buf, "Sorry!");
		alert->Go(NULL);
		return;
	}
	
	// find the text view	
	BTextView *view = dynamic_cast<BTextView *>(FindView("EditPhrase"));
	if (view)
	{
		// read up to LONG_MAX bytes of the file into the text view
		off_t size = 0;
		file.GetSize(&size);
		int32 sizeToRead = 0;
		(size > LONG_MAX) ? sizeToRead = LONG_MAX : sizeToRead = (int32)size;
		view->SetText(&file, 0, sizeToRead);
		// set the title to the name of the file
		BPath path(&entry);
		if (path.InitCheck() == B_OK)
			SetTitle(path.Leaf());
	}
}


void 
pDisplay::InitPosition()
{
	BRect frame(Frame());
	frame.OffsetTo(fNextWinPt);
	
	// check to see if the frame is out of bounds
	BScreen screen;
	BRect screenFrame = screen.Frame();
	if (frame.right > screenFrame.right)
	{
		// we are off the right edge so go down some
		fNextWinPt.x = startPt.x;
		fNextWinPt.y += incPt.y;
		frame.OffsetTo(fNextWinPt);
	}

	if (frame.top + 100 > screenFrame.bottom)
	{
		// we are off the bottom!
		// go back to the beginning
		fNextWinPt = startPt;
		frame.OffsetTo(fNextWinPt);
	}
	else fNextWinPt.x += incPt.x;

	MoveTo(frame.LeftTop());	
}

