#ifndef MEASURES_H
#define MEASURES_H

/*
** measures.h
**
** contains all sorts of measures and measure converters.
*/

#include "numbers.h"

//=======================================================
//	SI, the abbreviation for the French words:
//	Systeme Internationale d'Unites
//	Known as the metric system.
//
//  Quantity				Unit			Symbol
//	Measured
//	Length					Meter			m
//	Mass					Kilogram		kg
//	Time					Second			s
//	Electric Current		Ampere			A	
//	Temperature				Degrees Kelvin	K
//	Luminous intensity		Candela			cd
//	Amount of substance		Mole			mol
//
// The following classes encapsulate the metric system
// as well as conversions to other systems.
//=======================================================


//=======================================================
// class: ALength
//
//	This is the base class for any measure of length.  The
//  base representation of length is a meter.  Thus whenever
//	one length is constructed from another, the data is 
//	converted to meters when the base() method is called.
//
//=======================================================
class ALength
{
public:
		ALength():fNative(0.0){};
		ALength(const double aValue):fNative(aValue){};
		ALength(const ALength& other):fNative(other.base()){};
	virtual	~ALength();
	
	virtual operator double() { return fNative; };
	virtual double base() const;

protected:
	
	double	fNative;
	
private:
};


class meters : public ALength
{
public:
		meters(const double aValue):ALength(aValue){};
		meters(const ALength& other) {fNative = other.base();};
		meters(const meters& other) {fNative = other.fNative;};
	
	virtual double base() const;

protected:

private:
};

class kilometers : public ALength
{
public:
		kilometers(const double aValue):ALength(aValue){};
		kilometers(const ALength& other) {fNative = other.base()*kmilli;};
		kilometers(const kilometers& other) {fNative = other.fNative;};
	
	virtual double base() const;

protected:

private:
};

class centimeters : public ALength
{
public:
		centimeters(const double aValue):ALength(aValue){};
		centimeters(const ALength& other) {fNative = other.base()*khecto;};
		centimeters(const centimeters& other) {fNative = other.fNative;};
	
	virtual double base() const;

protected:

private:
};

class millimeters : public ALength
{
public:
		millimeters(const double aValue):ALength(aValue){};
		millimeters(const ALength& other) {fNative = other.base()*kkilo;};
		millimeters(const millimeters& other) {fNative = other.fNative;};
	
	virtual double base() const;

protected:

private:
};

class micrometers : public ALength
{
public:
		micrometers(const double aValue):ALength(aValue){};
		micrometers(const ALength& other) {fNative = other.base()*kmega;};
		micrometers(const micrometers& other) {fNative = other.fNative;};
	
	virtual double base() const;

protected:

private:
};

class nanometers : public ALength
{
public:
		nanometers(const double aValue):ALength(aValue){};
		nanometers(const ALength& other) {fNative = other.base()*kgiga;};
		nanometers(const nanometers& other) {fNative = other.fNative;};
	
	virtual double base() const;

protected:

private:
};


//=======================================================
// class: AMass
//
//	This is the base class for any measure of mass.  The
//  base representation of mass is a Kilogram.
//
//=======================================================

//=======================================================
// class: ACurrent
//
//	This is the base class for any measure of electric current.
//  The base representation of current is a Ampere.
//
//=======================================================

//=======================================================
// class: ATemperature
//
//	This is the base class for any measure of temperature.
//  The base representation of temperature is degrees Kelvin.
//
//=======================================================

//=======================================================
// class: ALumin
//
//	This is the base class for any measure of Luminous intensity.
//  The base representation of luminous intensity is the Candela.
//
//=======================================================

//=======================================================
// class: AAmount
//
//	This is the base class for any measure of amount of substance.
//  The base representation of amount of substance is a Mole.
//
//=======================================================

#endif
