#include "random.h"
#include <limits.h>
#include <math.h>

static const long A = 48271L;
static const long M = 2147483647L;
static const long Q = M/A;
static const long R = M%A;

// Uniform distributions
U32 	
Random::RandomLong()
{
	long TmpSeed = A*(fSeed%Q) - R * (fSeed/Q);
	if (TmpSeed >= 0)
		fSeed = TmpSeed;
	else
		fSeed = TmpSeed + M;
	
	return fSeed;
}

double	
Random::RandomReal()
{
	return (RandomLong() / ULONG_MAX);
}

long
Random::RandLong(long low, long high)
{
	unsigned long diff = high - low;
	return (low + RandomReal()*diff);
}

	
// Non-uniform distributions
U32 
Random::Poisson(double expectedVal)
{
	double Limit = exp(-expectedVal);
	double Product = RandomReal();
	
	int Count;
	for (Count = 0; Product > Limit; Count++)
		Product *= RandomReal();
		
	return Count;
}

double	
Random::NegExp(double expectedVal)
{
	return - expectedVal * log(RandomReal());
}



/******************************************************************************
*  Module:  r250.cpp
*  Description: implements R250 random number generator, from 
*  S. Kirkpatrick and E.  Stoll, 
*  Journal of Computational Physics, 40, p. 517 (1981).
*  Written by:    W. L. Maier
*
*	METHOD...
*		16 parallel copies of a linear shift register with
*		period 2^250 - 1.  FAR longer period than the usual
*		linear congruent generator, and commonly faster as
*		well.  (For details see the above paper, and the
*		article in DDJ referenced below.)
*
*	HISTORY...
*		Sep 92: Number returned by dr250() is in range [0,1) instead 
*			of [0,1], so for example a random angle in the
*			interval [0, 2*PI) can be calculated
*			conveniently.  (J. R. Van Zandt <jrv@mbunix.mitre.org>)
*		Aug 92: Initialization is optional.  Default condition is 
*			equivalent to initializing with the seed 12345,
*			so that the sequence of random numbers begins:
*			1173, 53403, 52352, 35341...  (9996 numbers
*			skipped) ...57769, 14511, 46930, 11942, 7978,
*			56163, 46506, 45768, 21162, 43113...  Using ^=
*			operator rather than two separate statements. 
*			Initializing with own linear congruent
*			pseudorandom number generator for portability. 
*			Function prototypes moved to a header file. 
*			Implemented r250n() to generate numbers
*			uniformly distributed in a specific range
*			[0,n), because the common expedient rand()%n is
*			incorrect.  (J. R. Van Zandt <jrv@mbunix.mitre.org>)
*		May 91: Published by W. L. Maier, "A Fast Pseudo Random Number 
*			Generator", Dr. Dobb's Journal #176.
******************************************************************************/
//=================================================================
// Default Constructor
//=================================================================

r250::r250()
{
	r250_init(12345);
}

//=================================================================
// Constructor
//
// Gives generator a initial seed value
//=================================================================

r250::r250(UINT seed)
{
	r250_init(seed);
}
void
r250::r250_init(UINT seed)
{
    INT  j, k;
    UINT mask;
    UINT msb;

	lcgseed = seed;
	r250_index = 0;

    for (j = 0; j < 250; j++)     /* Fill the r250 buffer with 15-bit values */
        r250_buffer[j] = myrand();
    for (j = 0; j < 250; j++)     /* Set some of the MS bits to 1 */
        if (myrand() > 16384)
            r250_buffer[j] |= 0x8000;
	

    msb = 0x8000;       /* To turn on the diagonal bit   */
    mask = 0xffff;      /* To turn off the leftmost bits */
    for (j = 0; j < 16; j++)
	{
        k = 11 * j + 3;             /* Select a word to operate on        */
        r250_buffer[k] &= mask;     /* Turn off bits left of the diagonal */
        r250_buffer[k] |= msb;      /* Turn on the diagonal bit           */
        mask >>= 1;
        msb >>= 1;
	}
}

//=================================================================
//	Function: r250 Description: returns a random unsigned integer k
//		uniformly distributed in the interval 0 <= k < 65536.  
//=================================================================

UINT r250::NextInt()
{
    register INT    j;
    register UINT new_rand;

    if (r250_index >= 147)
        j = r250_index - 147;      /* Wrap pointer around */
    else
        j = r250_index + 103;
    new_rand = r250_buffer[r250_index] ^= r250_buffer[j];
    if (r250_index >= 249)      /* Increment pointer for next time */
        r250_index = 0;
    else
        r250_index++;
    return new_rand;
}

//=================================================================
//	Function: r250n Description: returns a random unsigned integer k
//					uniformly distributed in the interval 0 <= k < n 
//=================================================================

UINT r250::Between(UINT n)
{
	register INT    j;
	register UINT new_rand, limit;
	limit = (65535U/n)*n;
	do {
		new_rand = NextInt();
		
		if (r250_index >= 147)
            j = r250_index - 147;      /* Wrap pointer around */
        else
            j = r250_index + 103;
        new_rand = r250_buffer[r250_index] ^= r250_buffer[j];
        if (r250_index >= 249)      /* Increment pointer for next time */
            r250_index = 0;
        else
            r250_index++;
        } while(new_rand >= limit);
    return new_rand%n;
}

/**** 
	Function: dr250 
		Description: returns a random double z in range 0 <= z < 1.  
****/
double r250::NextDouble()
{
    register INT    j;
    register UINT new_rand;
    if (r250_index >= 147)
        j = r250_index - 147;     /* Wrap pointer around */
    else
        j = r250_index + 103;
    new_rand = r250_buffer[r250_index] ^= r250_buffer[j];
    if (r250_index >= 249)      /* Increment pointer for next time */
        r250_index = 0;
    else
        r250_index++;
    return new_rand / 65536.;   /* Return a number in [0.0 to 1.0) */
}


/***  linear congruent pseudorandom number generator for initialization ***/

/*	
	Return a pseudorandom number in the interval 0 <= n < 32768.
	This produces the following sequence of pseudorandom 
	numbers:
		346, 130, 10982, 1090...  (9996 numbers skipped) ...23369,
		2020, 5703, 12762, 10828, 16252, 28648, 27041, 23444, 6604... 
*/ 
UINT 
r250::myrand()
{
	lcgseed = lcgseed*0x015a4e35L + 1;
	return (lcgseed>>16)&0x7fff;
}

