//--------------------------------------------------------------------
//	
//	Mail.cpp
//
//	Written by: Robert Polic
//	
//	Copyright 1997 Be, Inc. All Rights Reserved.
//	
//--------------------------------------------------------------------

#pragma once
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <Screen.h>

#include "Mail.h"
#include "Header.h"
#include "Content.h"
#include "Enclosures.h"
#include "Signature.h"
#include "Status.h"
#include "Utilities.h"

int32		level = L_BEGINNER;
bool		header = FALSE;
entry_ref	open_dir;
BMessage	*print_settings = NULL;
BRect		signature_window;
BRect		mail_window;


//====================================================================

int main(void)
{	
	TMailApp	*myApp;

	myApp = new TMailApp();
	myApp->Run();

	delete myApp;
	return B_NO_ERROR;
}

//--------------------------------------------------------------------

TMailApp::TMailApp(void)
		 :BApplication("application/x-vnd.Be-MAIL")
{
	char		f_size[16];
	int32		family_count;
	int32		family_loop;
	int32		loop;
	int32		style_count;
	int32		style_loop;
	float		size;
	BDirectory	dir;
	BEntry		entry;
	BFont		m_font = *be_plain_font;
	BMenu		*font;
	BMenu		*style;
	BMenu		*sub_menu;
	BMenuItem	*item;
	BPath		path;
	font_family	f_family;
	font_style	f_style;
	font_family	def_family;
	font_style	def_style;

	fWindowCount = 0;
	fFont = *be_plain_font;
	fFont.SetSize(FONT_SIZE);
	fSigWindow = NULL;
	mail_window.Set(0, 0, 0, 0);
	signature_window.Set(6, TITLE_BAR_HEIGHT, 6 + SIG_WIDTH, TITLE_BAR_HEIGHT + SIG_HEIGHT);

	find_directory(B_USER_SETTINGS_DIRECTORY, &path, true);
	dir.SetTo(path.Path());
	if (dir.FindEntry("Mail_data", &entry) == B_NO_ERROR) {
		fPrefs = new BFile(&entry, O_RDWR);
		if (fPrefs->InitCheck() == B_NO_ERROR) {
			fPrefs->Read(&mail_window, sizeof(BRect));
			fPrefs->Read(&level, sizeof(level));
			fPrefs->Read(&f_family, sizeof(font_family));
			fPrefs->Read(&f_style, sizeof(font_style));
			fPrefs->Read(&size, sizeof(float));
			fPrefs->Read(&signature_window, sizeof(BRect));
			fPrefs->Read(&header, sizeof(bool));

			if ((strlen(f_family)) && (strlen(f_style)))
				fFont.SetFamilyAndStyle(f_family, f_style);
			if (size >= 9)
				fFont.SetSize(size);
		}
		else {
			delete fPrefs;
			fPrefs = NULL;
		}
	}
	else {
		fPrefs = new BFile();
		if (dir.CreateFile("Mail_data", fPrefs) != B_NO_ERROR) {
			delete fPrefs;
			fPrefs = NULL;
		}
	}

	fFont.SetSpacing(B_CHAR_SPACING);
	if (fFont.StringWidth("M") == fFont.StringWidth("I"))
		fFont.SetSpacing(B_FIXED_SPACING);
	else
		fFont.SetSpacing(B_BITMAP_SPACING);

	m_font.SetSize(10);
	fMenu = new BPopUpMenu("BeMail", FALSE, FALSE);
	fMenu->SetFont(&m_font);
	fMenu->AddItem(item = new BMenuItem("About BeMail"B_UTF8_ELLIPSIS, new BMessage(B_ABOUT_REQUESTED)));
	item->SetTarget(be_app);
	fMenu->AddSeparatorItem();
				
	sub_menu = new BMenu("Preferences");
	sub_menu->SetFont(&m_font);
	font = new BMenu("Font");
	font->SetFont(&m_font);
	fFont.GetFamilyAndStyle(&def_family, &def_style);
	family_count = count_font_families();
	for (family_loop = 0; family_loop < family_count; family_loop++) {
		get_font_family(family_loop, &f_family);
		style = new BMenu(f_family);
		style->SetFont(&m_font);
		style_count = count_font_styles(f_family);
		for (style_loop = 0; style_loop < style_count; style_loop++) {
			get_font_style(f_family, style_loop, &f_style);
			style->AddItem(item = new BMenuItem(f_style, new BMessage(M_STYLE)));
			item->SetTarget(be_app);
			if ((!strcmp(def_family, f_family)) && (!strcmp(def_style, f_style)))
				item->SetMarked(TRUE);
		}
		font->AddItem(item = new BMenuItem(style, new BMessage(M_FONT)));
		item->SetTarget(be_app);
		if (!strcmp(def_family, f_family))
			item->SetMarked(TRUE);
	}
	sub_menu->AddItem(font);

	font = new BMenu("Size");
	font->SetFont(&m_font);
	for (loop = 9; loop <= 14; loop++) {
		sprintf(f_size, "%d", loop);
		font->AddItem(item = new BMenuItem(f_size, new BMessage(M_SIZE)));
		item->SetTarget(be_app);
		if (fFont.Size() == loop)
			item->SetMarked(TRUE);
	}
	font->SetRadioMode(TRUE);
	sub_menu->AddItem(font);

	fLevelMenu = new BMenu("User Level");
	fLevelMenu->SetFont(&m_font);
	fLevelMenu->SetRadioMode(TRUE);
	fLevelMenu->AddItem(item = new BMenuItem("Beginner",
						new BMessage(M_BEGINNER)));
	item->SetTarget(be_app);
	if (level == L_BEGINNER)
		item->SetMarked(TRUE);
	fLevelMenu->AddItem(item = new BMenuItem("Expert",
						new BMessage(M_EXPERT)));
	item->SetTarget(be_app);
	if (level == L_EXPERT)
		item->SetMarked(TRUE);
	sub_menu->AddItem(fLevelMenu);

	sub_menu->AddItem(item = new BMenuItem("Signatures"B_UTF8_ELLIPSIS,
						new BMessage(M_EDIT_SIGNATURE)));
	item->SetTarget(be_app);
	fMenu->AddItem(sub_menu);

	fMenu->AddSeparatorItem();
	fMenu->AddItem(item = new BMenuItem("Quit BeMail", new BMessage(B_QUIT_REQUESTED), 'Q'));
	item->SetTarget(be_app);
}

//--------------------------------------------------------------------

TMailApp::~TMailApp(void)
{
	if (fPrefs)
		delete fPrefs;
}

//--------------------------------------------------------------------

void TMailApp::AboutRequested(void)
{
	(new BAlert("", B_UTF8_ELLIPSIS"by Robert Polic", "Big Deal"))->Go();
}

//--------------------------------------------------------------------

void TMailApp::ArgvReceived(int32 argc, char **argv)
{
	char		*addr;
	char		*names = NULL;
	int32		loop;
	entry_ref	ref;
	BEntry		entry;
	BMessage	msg(B_REFS_RECEIVED);

	for (loop = 1; loop < argc; loop++) {
		if (strncmp(argv[loop], "mailto:", 7) == 0) {
			addr = argv[loop] + 7;
			if (!names)
				names = strdup(addr);
			else {
				names = (char *)realloc(names, strlen(names) + 1 +
										strlen(addr) + 1 + 2);
				strcat(names, ", ");
				strcat(names, addr);
			}
		}
		else if (entry.SetTo(argv[loop]) == B_NO_ERROR) {
			entry.GetRef(&ref);
			msg.AddRef("refs", &ref);
			RefsReceived(&msg);
		}
	}

	if (names) {
		TMailWindow	*window = NewWindow(NULL, names);
		free(names);
		window->Show();
	}
}

//--------------------------------------------------------------------

void TMailApp::MessageReceived(BMessage* msg)
{
	bool		all = FALSE;
	int32		size;
	int32		type;
	BMenuItem	*item;
	BMessage	*message;
	entry_ref	ref;
	TMailWindow	*window = NULL;
	TMailWindow	*src_window;

	switch (msg->what) {
		case M_NEW:
			msg->FindInt32("type", &type);
			switch (type) {
				case M_NEW:
					window = NewWindow();
					break;

				case M_RESEND:
					msg->FindRef("ref", &ref);
					window = NewWindow(&ref, "", TRUE);
					break;

				case M_FORWARD:
					msg->FindRef("ref", &ref);
					window = NewWindow();
					window->Lock();
					window->Forward(&ref);
					window->Unlock();
					break;

				case M_REPLY:
				case M_REPLY_ALL:
					msg->FindPointer("window", &src_window);
					if (!src_window->Lock())
						break;
					msg->FindRef("ref", &ref);
					window = NewWindow();
					window->Lock();
					window->Reply(&ref, src_window, type == M_REPLY_ALL);
					window->Unlock();
					src_window->Unlock();
					break;
			}
			if (window)
				window->Show();
			break;

		case M_EDIT_SIGNATURE:
			if (fSigWindow)
				fSigWindow->Activate(TRUE);
			else {
				fSigWindow = new TSignatureWindow(signature_window, &fFont);
				fSigWindow->Show();
			}
			break;

		case M_FONT:
			msg->FindPointer("source", &item);
			FontChange((char *)item->Label(), NULL, 0);
			break;

		case M_STYLE:
			msg->FindPointer("source", &item);
			FontChange((char *)item->Menu()->Superitem()->Label(),
					   (char *)item->Label(), 0);
			break;

		case M_SIZE:
			msg->FindInt32("index", &size);
			FontChange(NULL, NULL, size + 9);
			break;

		case M_BEGINNER:
		case M_EXPERT:
			level = msg->what - M_BEGINNER;
			break;

		case REFS_RECEIVED:
			if (msg->HasPointer("window")) {
				msg->FindPointer("window", &window);
				message = new BMessage(msg);
				window->PostMessage(message, window);
				delete message;
			}
			break;

		case WINDOW_CLOSED:
			switch (msg->FindInt32("kind")) {
				case MAIL_WINDOW:
					fWindowCount--;
					break;

				case SIG_WINDOW:
					fSigWindow = NULL;
					break;
			}

			if ((!fWindowCount) && (!fSigWindow))
				be_app->PostMessage(B_QUIT_REQUESTED);
			break;

		case B_REFS_RECEIVED:
			RefsReceived(msg);
			break;

		default:
			BApplication::MessageReceived(msg);
	}
}

//--------------------------------------------------------------------

bool TMailApp::QuitRequested(void)
{
	float		size;
	font_family	f_family;
	font_style	f_style;

	if (BApplication::QuitRequested()) {
		if (fPrefs) {
			fFont.GetFamilyAndStyle(&f_family, &f_style);
			size = fFont.Size();

			fPrefs->Seek(0, 0);
			fPrefs->Write(&mail_window, sizeof(BRect));
			fPrefs->Write(&level, sizeof(level));
			fPrefs->Write(&f_family, sizeof(font_family));
			fPrefs->Write(&f_style, sizeof(font_style));
			fPrefs->Write(&size, sizeof(float));
			fPrefs->Write(&signature_window, sizeof(BRect));
			fPrefs->Write(&header, sizeof(bool));
		}
	}
	return TRUE;
}

//--------------------------------------------------------------------

void TMailApp::ReadyToRun(void)
{
	TMailWindow	*window;

	if (!fWindowCount) {
		window = NewWindow();
		window->Show();
	}
}

//--------------------------------------------------------------------

void TMailApp::RefsReceived(BMessage *msg)
{
	bool		have_names = FALSE;
	char		*name;
	char		*names = NULL;
	char		type[B_FILE_NAME_LENGTH];
	int32		item = 0;
	BFile		file;
	BNodeInfo	*node;
	TMailWindow	*window;
	entry_ref	ref;
	attr_info	info;

	while (msg->HasRef("refs", item)) {
		msg->FindRef("refs", item++, &ref);
		if (window = FindWindow(ref))
			window->Activate(TRUE);
		else {
			file.SetTo(&ref, O_RDONLY);
			if (file.InitCheck() == B_NO_ERROR) {
				node = new BNodeInfo(&file);
				node->GetType(type);
				delete node;
				if (!strcmp(type, B_MAIL_TYPE)) {
					window = NewWindow(&ref);
					window->Show();
				}
				else if (!strcmp(type, "application/x-person")) {
					if (file.GetAttrInfo("META:email", &info) == B_NO_ERROR) {
						name = (char *)malloc(info.size);
						file.ReadAttr("META:email", B_STRING_TYPE, 0, name, info.size);
						if (strlen(name)) {
							if (!names) {
								names = (char *)malloc(strlen(name) + 1);
								strcpy(names, name);
							}
							else {
								names = (char *)realloc(names, strlen(names) + 1 + 1 + strlen(name));
								strcat(names, ", ");
								strcat(names, name);
							}
							have_names = TRUE;
						}
						free(name);
					}
				}
			}
		}
	}

	if (have_names) {
		window = NewWindow(NULL, names);
		free(names);
		window->Show();
	}
}

//--------------------------------------------------------------------

TMailWindow* TMailApp::FindWindow(entry_ref ref)
{
	int32		index = 0;
	TMailWindow	*window;

	while (window = (TMailWindow *)WindowAt(index++)) {
		if ((window->FindView("m_header")) && (window->fRef) && (*(window->fRef) == ref))
			return window;
	}
	return NULL;
}

//--------------------------------------------------------------------

void TMailApp::FontChange(font_family family, font_style style, float size)
{
	bool		change = FALSE;
	int32		index = 0;
	BMenuItem	*item;
	BMessage	message;
	BWindow		*window;
	font_family	orig_family;
	font_style	orig_style;
	font_family	new_family;
	font_style	new_style;

	fFont.GetFamilyAndStyle(&orig_family, &orig_style);
	if ((family) && (strcmp(family, orig_family)))
		change = TRUE;
	else if ((style) && (strcmp(style, orig_style)))
		change = TRUE;
	else if ((size) && (size != fFont.Size()))
		change = TRUE;

	if (change) {
		if (family)
			strcpy(new_family, family);
		else
			strcpy(new_family, orig_family);

		if (style)
			strcpy(new_style, style);
		else
			strcpy(new_style, orig_style);

		item = fMenu->FindItem(orig_family);
		item->SetMarked(FALSE);
		item->Submenu()->FindItem(orig_style)->SetMarked(FALSE);

		item = fMenu->FindItem(new_family);
		item->SetMarked(TRUE);
		if (!item->Submenu()->FindItem(new_style))
			get_font_style(new_family, 0, &new_style);
		item->Submenu()->FindItem(new_style)->SetMarked(TRUE);

		fFont.SetFamilyAndStyle(new_family, new_style);

		if (size)
			fFont.SetSize(size);

		fFont.SetSpacing(B_CHAR_SPACING);
		if (fFont.StringWidth("M") == fFont.StringWidth("I"))
			fFont.SetSpacing(B_FIXED_SPACING);
		else
			fFont.SetSpacing(B_BITMAP_SPACING);

		message.what = CHANGE_FONT;
		message.AddPointer("font", &fFont);

		for (;;) {
			window = WindowAt(index++);
			if (!window)
				break;
			window->PostMessage(&message);
		}
	}
}

//--------------------------------------------------------------------

TMailWindow* TMailApp::NewWindow(entry_ref *ref, char *to, bool resend)
{
	char			*str1;
	char			*str2;
	char			*title = NULL;
	int32			len;
	BFile			file;
	BPoint			win_pos;
	BRect			r;
	TMailWindow		*window;
	attr_info		info;

	BScreen screen( B_MAIN_SCREEN_ID );
	BRect screen_frame = screen.Frame();

	if ((mail_window.Width()) && (mail_window.Height()))
		r = mail_window;
	else
		r.Set(6, TITLE_BAR_HEIGHT,
			  6 + WIND_WIDTH, TITLE_BAR_HEIGHT + WIND_HEIGHT);
	r.OffsetBy(fWindowCount * 20, fWindowCount * 20);
	if ((r.left - 6) < screen_frame.left)
		r.OffsetTo(screen_frame.left + 8, r.top);
	if ((r.left + 20) > screen_frame.right)
		r.OffsetTo(6, r.top);
	if ((r.top - 26) < screen_frame.top)
		r.OffsetTo(r.left, screen_frame.top + 26);
	if ((r.top + 20) > screen_frame.bottom)
		r.OffsetTo(r.left, TITLE_BAR_HEIGHT);
	if (r.Width() < WIND_WIDTH)
		r.right = r.left + WIND_WIDTH;
	fWindowCount++;

	if (ref) {
		file.SetTo(ref, O_RDONLY);
		if (file.InitCheck() == B_NO_ERROR) {
			if (file.GetAttrInfo(B_MAIL_ATTR_NAME, &info) == B_NO_ERROR) {
				str1 = (char *)malloc(info.size);
				file.ReadAttr(B_MAIL_ATTR_NAME, B_STRING_TYPE, 0, str1, info.size);
				if (file.GetAttrInfo(B_MAIL_ATTR_SUBJECT, &info) == B_NO_ERROR) {
					str2 = (char *)malloc(info.size);
					file.ReadAttr(B_MAIL_ATTR_SUBJECT, B_STRING_TYPE, 0, str2, info.size);
					title = (char *)malloc(strlen(str1) + strlen(str2) + 3);
					sprintf(title, "%s->%s", str1, str2);
					free(str1);
					free(str2);
				}
				else
					title = str1;
			}
		}
	}
	if (!title) {
		title = (char *)malloc(strlen("BeMail"));
		sprintf(title, "BeMail");
	}
	window = new TMailWindow(r, title, ref, to, &fFont, resend);
	free(title);
	return window;
}


//====================================================================

TMailWindow::TMailWindow(BRect rect, char *title, entry_ref *ref, char *to,
						 BFont *font, bool resending)
			:BWindow(rect, title, B_TITLED_WINDOW, 0)
{
	char		str[256];
	char		status[272];
	char		*type;
	int32		len;
	uint32		message;
	float		height;
	BMenu		*menu;
	BMenu		*sub_menu;
	BMenuBar	*menu_bar;
	BMenuItem	*item;
	BRect		r;
	attr_info	info;

	fDelete = FALSE;
	fReplying = FALSE;
	fResending = resending;
	if (ref) {
		fRef = new entry_ref(*ref);
		fFile = new BFile(fRef, O_RDONLY);
		fIncoming = TRUE;
	}
	else {
		fRef = NULL;
		fFile = NULL;
		fIncoming = FALSE;
	}

	r.Set(0, 0, 32767, 15);
	menu_bar = new BMenuBar(r, "");
	menu = new BMenu("Message");

	BMessage *msg = new BMessage(M_NEW);
	msg->AddInt32("type", M_NEW);
	menu->AddItem(item = new BMenuItem("New Mail Message", msg, 'N'));
	item->SetTarget(be_app);
	menu->AddSeparatorItem();
	
	if ((!resending) && (fIncoming)) {
		menu->AddItem(new BMenuItem("Reply to Sender",
						new BMessage(M_REPLY), 'R'));
		menu->AddItem(new BMenuItem("Reply to All",
						new BMessage(M_REPLY_ALL), 'R', B_SHIFT_KEY));
		menu->AddItem(new BMenuItem("Forward", new BMessage(M_FORWARD), 'F'));
		menu->AddItem(new BMenuItem("Resend", new BMessage(M_RESEND)));
		menu->AddSeparatorItem();
		menu->AddItem(fHeader = new BMenuItem("Show Header",
								new BMessage(M_HEADER), 'H'));
		if (header)
			fHeader->SetMarked(TRUE);
		menu->AddItem(fRaw = new BMenuItem("Show Raw Message",
								new BMessage(M_RAW)));
	}
	else {
		menu->AddItem(fSendNow = new BMenuItem("Send Now",
								new BMessage(M_SEND_NOW), 'M', B_SHIFT_KEY));
		menu->AddItem(fSendLater = new BMenuItem("Send Later",
								new BMessage(M_SEND_LATER), 'M'));
	}
	menu->AddSeparatorItem();
	menu->AddItem(fPrint = new BMenuItem("Page Setup"B_UTF8_ELLIPSIS,
								new BMessage(M_PRINT_SETUP)));
	menu->AddItem(fPrint = new BMenuItem("Print"B_UTF8_ELLIPSIS,
								new BMessage(M_PRINT), 'P'));
	menu->AddSeparatorItem();
	if ((!resending) && (fIncoming)) {
		menu->AddItem(new BMenuItem("Move to Trash", new BMessage(M_DELETE), 'T'));
		sub_menu = new BMenu("Close");
		if (fFile->GetAttrInfo(B_MAIL_ATTR_STATUS, &info) == B_NO_ERROR)
			fFile->ReadAttr(B_MAIL_ATTR_STATUS, B_STRING_TYPE, 0, str, info.size);
		else
			str[0] = 0;
		if (!strcmp(str, "New")) {
			sub_menu->AddItem(item = new BMenuItem("Set To 'Read'",
							new BMessage(M_CLOSE_READ), 'W'));
			sub_menu->AddItem(item = new BMenuItem("Leave As 'New'",
							new BMessage(M_CLOSE_SAME), 'W', B_SHIFT_KEY));
			message = M_CLOSE_READ;
		}
		else {
			if (strlen(str))
				sprintf(status, "Leave As '%s'", str);
			else
				sprintf(status, "Leave same");
			sub_menu->AddItem(item = new BMenuItem(status,
							new BMessage(M_CLOSE_SAME), 'W'));
			message = M_CLOSE_SAME;
			AddShortcut('W', B_COMMAND_KEY | B_SHIFT_KEY, new BMessage(M_CLOSE_SAME));
		}
		sub_menu->AddItem(item = new BMenuItem("Set To"B_UTF8_ELLIPSIS,
							new BMessage(M_CLOSE_CUSTOM), 'W',
							B_SHIFT_KEY | B_CONTROL_KEY));
		menu->AddItem(new BMenuItem(sub_menu, new BMessage(message)));
	}
	else
		menu->AddItem(new BMenuItem("Close",
								new BMessage(B_CLOSE_REQUESTED), 'W'));
	menu_bar->AddItem(menu);

	menu = new BMenu("Edit");
	menu->AddItem(fUndo = new BMenuItem("Undo", new BMessage(M_UNDO), 'Z'));
	menu->AddSeparatorItem();
	menu->AddItem(fCut = new BMenuItem("Cut", new BMessage(B_CUT), 'X'));
	fCut->SetTarget(NULL, this);
	menu->AddItem(fCopy = new BMenuItem("Copy", new BMessage(B_COPY), 'C'));
	fCopy->SetTarget(NULL, this);
	menu->AddItem(fPaste = new BMenuItem("Paste", new BMessage(B_PASTE), 'V'));
	fPaste->SetTarget(NULL, this);
	menu->AddItem(item = new BMenuItem("Select All",
								new BMessage(M_SELECT), 'A'));
	item->SetTarget(NULL, this);
	if (!fIncoming) {
		menu->AddSeparatorItem();
		menu->AddItem(fQuote = new BMenuItem("Quote", new BMessage(M_QUOTE),
								B_RIGHT_ARROW));
		menu->AddItem(fRemoveQuote = new BMenuItem("Remove Quote",
								new BMessage(M_REMOVE_QUOTE), B_LEFT_ARROW));
		fSignature = new TMenu("Add Signature");
		menu->AddItem(new BMenuItem(fSignature));
	}
	menu_bar->AddItem(menu);

	if (!fIncoming) {
		menu = new BMenu("Enclosures");
		menu->AddItem(fAdd = new BMenuItem("Add"B_UTF8_ELLIPSIS, new BMessage(M_ADD), 'O'));
		menu->AddItem(fRemove = new BMenuItem("Remove",
									new BMessage(M_REMOVE), 'D'));
		menu_bar->AddItem(menu);
	}
	Lock();
	AddChild(menu_bar);
	height = menu_bar->Bounds().bottom + 1;
	Unlock();

	r.top = height;
	r.left -= 1;
	if (!fIncoming)
		r.bottom = height + HEADER_HEIGHT;
	else
		r.bottom = height + MIN_HEADER_HEIGHT;
	fHeaderView = new THeaderView(r, rect, fIncoming, fFile, resending);

	r = Frame();
	r.OffsetTo(0, 0);
	r.right = 32767;
	r.left -= 1;
	if (fIncoming) {
		r.bottom += ENCLOSURES_HEIGHT;
		fEnclosuresView = NULL;
	}
	else {
		r.bottom++;
		r.top = r.bottom - ENCLOSURES_HEIGHT;
		fEnclosuresView = new TEnclosuresView(r, rect);
	}

	r.right = Frame().right - Frame().left + 2;
	if (fIncoming)
		r.top = height + MIN_HEADER_HEIGHT - 1;
	else
		r.top = height + HEADER_HEIGHT - 1;
	r.bottom -= ENCLOSURES_HEIGHT - 1;
	fContentView = new TContentView(r, fIncoming, fFile, font);

	Lock();
	AddChild(fHeaderView);
	AddChild(fContentView);
	if (fEnclosuresView)
		AddChild(fEnclosuresView);
	Unlock();

	if (to) {
		Lock();
		fHeaderView->fTo->SetText(to);
		Unlock();
	}

	SetSizeLimits(WIND_WIDTH, 32767,
				  HEADER_HEIGHT + ENCLOSURES_HEIGHT + height + 50, 32767);

	Lock();
	AddCommonFilter(new TFilter());
	Unlock();
	AddShortcut('n', B_COMMAND_KEY, new BMessage(M_NEW));
}

//--------------------------------------------------------------------

TMailWindow::~TMailWindow(void)
{
	void		*header;
	int32		flags;
	BMessage	msg('Ttrs');
	BMessenger	*tracker;

	if (fFile) {
		if (fDelete) {
			tracker = new BMessenger("application/x-vnd.Be-TRAK", -1, NULL);
			if (tracker->IsValid()) {
				msg.AddRef("refs", fRef);
				tracker->SendMessage(&msg);
			}
			else
				(new BAlert("", "Need Tracker to move items to Trash.", "Sorry"))->Go();
		}
		delete fFile;
		delete fRef;
	}
	mail_window = Frame();
}

//--------------------------------------------------------------------

void TMailWindow::FrameResized(float width, float height)
{
	fContentView->FrameResized(width, height);
}

//--------------------------------------------------------------------

void TMailWindow::MenusBeginning(void)
{
	bool		enable;
	int32		finish = 0;
	int32		len = 0;
	int32		start = 0;
	BTextView	*text_view;

	if (!fIncoming) {
		enable = strlen(fHeaderView->fTo->Text()) ||
				 strlen(fHeaderView->fBcc->Text());
		fSendNow->SetEnabled(enable);
		fSendLater->SetEnabled(enable);

		fUndo->SetEnabled(FALSE);		// ***TODO***

		be_clipboard->Lock();
		fPaste->SetEnabled(be_clipboard->Data()->HasData("text/plain", B_MIME_TYPE) &&
						   !fEnclosuresView->fList->IsFocus());
		be_clipboard->Unlock();

		fQuote->SetEnabled(FALSE);
		fRemoveQuote->SetEnabled(FALSE);

		fAdd->SetEnabled(TRUE);
		fRemove->SetEnabled(fEnclosuresView->fList->CurrentSelection() >= 0);
	}
	else {
		fUndo->SetEnabled(FALSE);
		if (fResending) {
			enable = strlen(fHeaderView->fTo->Text());
			fSendNow->SetEnabled(enable);
			fSendLater->SetEnabled(enable);
			text_view = (BTextView *)fHeaderView->fTo->ChildAt(0);
			if (text_view->IsFocus()) {
				text_view->GetSelection(&start, &finish);
				fCut->SetEnabled(start != finish);
				be_clipboard->Lock();
				fPaste->SetEnabled(be_clipboard->Data()->HasData("text/plain", B_MIME_TYPE));
				be_clipboard->Unlock();
			}
			else {
				fCut->SetEnabled(FALSE);
				fPaste->SetEnabled(FALSE);
			}
		}
		else {
			fCut->SetEnabled(FALSE);
			fPaste->SetEnabled(FALSE);
		}
	}

	fPrint->SetEnabled(fContentView->fTextView->TextLength());

	text_view = (BTextView *)fHeaderView->fTo->ChildAt(0);
	if (text_view->IsFocus())
		text_view->GetSelection(&start, &finish);
	else {
		text_view = (BTextView *)fHeaderView->fSubject->ChildAt(0);
		if (text_view->IsFocus())
			text_view->GetSelection(&start, &finish);
		else {
			if (fContentView->fTextView->IsFocus()) {
				fContentView->fTextView->GetSelection(&start, &finish);
				if (!fIncoming) {
					fQuote->SetEnabled(TRUE);
					fRemoveQuote->SetEnabled(TRUE);
				}
			}
			else if (!fIncoming) {
				text_view = (BTextView *)fHeaderView->fCc->ChildAt(0);
				if (text_view->IsFocus())
					text_view->GetSelection(&start, &finish);
				else {
					text_view = (BTextView *)fHeaderView->fBcc->ChildAt(0);
					if (text_view->IsFocus())
						text_view->GetSelection(&start, &finish);
				}
			}
		}
	}
	fCopy->SetEnabled(start != finish);
	if (!fIncoming)
		fCut->SetEnabled(start != finish);
}

//--------------------------------------------------------------------

void TMailWindow::MessageReceived(BMessage* msg)
{
	bool		enable;
	bool		now = FALSE;
	bool		raw;
	char		*str;
	int32		item = 0;
	BEntry		entry;
	BFilePanel	*panel;
	BMessage	*message;
	BMessenger	*me;
	entry_ref	ref;
	BRect		r;
	attr_info	info;

	switch(msg->what) {
		case LIST_INVOKED:
			PostMessage(msg, fEnclosuresView);
			break;

		case CHANGE_FONT:
			PostMessage(msg, fContentView);
			break;

		case M_NEW:
			message = new BMessage(M_NEW);
			message->AddInt32("type", msg->what);
			be_app->PostMessage(message);
			delete message;
			break;

		case M_REPLY:
		case M_REPLY_ALL:
		case M_FORWARD:
		case M_RESEND:
			message = new BMessage(M_NEW);
			message->AddRef("ref", fRef);
			message->AddPointer("window", this);
			message->AddInt32("type", msg->what);
			be_app->PostMessage(message);
			delete message;
			break;

		case M_DELETE:
			if (level == L_BEGINNER) {
				beep();
				if (!(new BAlert("",
						"Are you sure you want to move this message to the trash?",
						"Cancel", "Trash", NULL, B_WIDTH_AS_USUAL,
						B_WARNING_ALERT))->Go())
					break;
			}
			fDelete = TRUE;
			PostMessage(message = new BMessage(B_CLOSE_REQUESTED));
			delete message;
			break;

		case M_CLOSE_READ:
			message = new BMessage(B_CLOSE_REQUESTED);
			message->AddString("status", "Read");
			PostMessage(message);
			delete message;
			break;

		case M_CLOSE_SAME:
			message = new BMessage(B_CLOSE_REQUESTED);
			message->AddString("status", "");
			message->AddString("same", "");
			PostMessage(message);
			delete message;
			break;

		case M_CLOSE_CUSTOM:
			if (msg->HasString("status")) {
				msg->FindString("status", &str);
				message = new BMessage(B_CLOSE_REQUESTED);
				message->AddString("status", str);
				PostMessage(message);
				delete message;
			}
			else {
				r = Frame();
				r.left += ((r.Width() - STATUS_WIDTH) / 2);
				r.right = r.left + STATUS_WIDTH;
				r.top += 40;
				r.bottom = r.top + STATUS_HEIGHT;
				if (fFile->GetAttrInfo(B_MAIL_ATTR_STATUS, &info) == B_NO_ERROR) {
					str = (char *)malloc(info.size);
					fFile->ReadAttr(B_MAIL_ATTR_STATUS, B_STRING_TYPE, 0, str, info.size);
				}
				else {
					str = (char *)malloc(1);
					str[0] = 0;
				}
				new TStatusWindow(r, this, str);
				free(str);
			}
			break;

		case M_HEADER:
			header = !(fHeader->IsMarked());
			fHeader->SetMarked(header);
			message = new BMessage(M_HEADER);
			message->AddBool("header", header);
			PostMessage(message, fContentView->fTextView);
			delete message;
			break;

		case M_RAW:
			raw = !(fRaw->IsMarked());
			fRaw->SetMarked(raw);
			message = new BMessage(M_RAW);
			message->AddBool("raw", raw);
			PostMessage(message, fContentView->fTextView);
			delete message;
			break;

		case M_SEND_NOW:
			now = TRUE;
			// yes, we are suppose to fall through
		case M_SEND_LATER:
			Send(now);
			break;

		case M_PRINT_SETUP:
			PrintSetup();
			break;

		case M_PRINT:
			Print();
			break;

		case M_UNDO:
			// ***TODO***
			break;

		case M_SELECT:
			break;

		case M_QUOTE:
		case M_REMOVE_QUOTE:
			PostMessage(msg->what, fContentView);
			break;

		case M_SIGNATURE:
			message = new BMessage(msg);
			PostMessage(message, fContentView);
			delete message;
			break;

		case M_ADD:
			me = new BMessenger(this);
			panel = new BFilePanel(B_OPEN_PANEL, me, &open_dir, 
								   FALSE, TRUE, &BMessage(REFS_RECEIVED));
			panel->Window()->Show();
			delete me;
			break;

		case M_REMOVE:
			PostMessage(msg->what, fEnclosuresView);
			break;

		case REFS_RECEIVED:
			if ((fEnclosuresView) && (msg->HasRef("refs"))) {
				PostMessage(msg, fEnclosuresView);

				msg->FindRef("refs", &ref);
				entry.SetTo(&ref);
				entry.GetParent(&entry);
				entry.GetRef(&open_dir);
			}
			break;

		default:
			BWindow::MessageReceived(msg);
	}
}

//--------------------------------------------------------------------

bool TMailWindow::QuitRequested(void)
{
	char		status[256];
	const char	*str = NULL;
	BFile		file;
	BMessage	message(WINDOW_CLOSED);
	attr_info	info;

	message.AddInt32("kind", MAIL_WINDOW);
	be_app->PostMessage(&message);

	if ((CurrentMessage()) && (CurrentMessage()->HasString("status"))) {
		if (!CurrentMessage()->HasString("same"))
			str = CurrentMessage()->FindString("status");
	}
	else if (fFile) {
		if (fFile->GetAttrInfo(B_MAIL_ATTR_STATUS, &info) == B_NO_ERROR) {
			fFile->ReadAttr(B_MAIL_ATTR_STATUS, B_STRING_TYPE, 0, status, info.size);
			if (strcmp(status, "New") == 0)
				strcpy(status, "Read");
			str = status;
		}
	}
	if (str) {
		file.SetTo(fRef, O_RDWR);
		if (file.InitCheck() == B_NO_ERROR) {
			file.RemoveAttr(B_MAIL_ATTR_STATUS);
			file.WriteAttr(B_MAIL_ATTR_STATUS, B_STRING_TYPE, 0, str, strlen(str) + 1);
		}
	}
	return TRUE;
}

//--------------------------------------------------------------------

void TMailWindow::Show(void)
{
	BTextView	*text_view;

	Lock();
	if ((!fResending) && ((fIncoming) || (fReplying)))
		fContentView->fTextView->MakeFocus(TRUE);
	else {
		text_view = (BTextView *)fHeaderView->fTo->ChildAt(0);
		fHeaderView->fTo->MakeFocus(TRUE);
		text_view->Select(0, text_view->TextLength());
	}
	Unlock();
	BWindow::Show();
}

//--------------------------------------------------------------------

void TMailWindow::Forward(entry_ref *ref)
{
	char		*str;
	char		*str1;
	BFile		*file;
	attr_info	info;

	file = new BFile(ref, O_RDONLY);
	if (file->InitCheck() == B_NO_ERROR) {
		if (file->GetAttrInfo(B_MAIL_ATTR_SUBJECT, &info) == B_NO_ERROR) {
			str = (char *)malloc(info.size);
			file->ReadAttr(B_MAIL_ATTR_SUBJECT, B_STRING_TYPE, 0, str, info.size);
			if ((strstr(str, "fwd")) || (strstr(str, "forward")) ||
				(strstr(str, "FW")) || (strstr(str, "FORWARD")))
				fHeaderView->fSubject->SetText(str);
			else {
				str1 = (char *)malloc(strlen(str) + 1 + 6);
				sprintf(str1, "%s (fwd)", str);
				fHeaderView->fSubject->SetText(str1);
				free(str1);
			}
			free(str);
		}
		fContentView->fTextView->fHeader = TRUE;
		fContentView->fTextView->LoadMessage(file, FALSE, TRUE, "Forwarded message:\n");
	}
	else
		delete file;
}

//--------------------------------------------------------------------

void TMailWindow::Print(void)
{
	int32			lines;
	int32			lines_page;
	int32			loop;
	int32			pages;
	float			line_height;
	BPrintJob		print("mail_print");
	BRect			r;

	if (print_settings)
		print.SetSettings(new BMessage(print_settings));
	else {
		PrintSetup();
		if (!print_settings)
			return;
	}

	lines = fContentView->fTextView->CountLines();
	line_height = fContentView->fTextView->LineHeight();
	if ((lines) && ((int)line_height) && (print.ConfigJob() == B_NO_ERROR)) {
		r = print.PrintableRect();
		lines_page = r.Height() / line_height;
		pages = lines / lines_page;
		r.top = 0;
		r.bottom = line_height * lines_page;
		r.right -= r.left;
		r.left = 0;

		print.BeginJob();
		if (!print.CanContinue())
			goto out;
		for (loop = 0; loop <= pages; loop++) {
			print.DrawView(fContentView->fTextView, r, BPoint(0, 0));
			print.SpoolPage();
			r.top += (line_height * lines_page);
			r.bottom += (line_height * lines_page);
			if (!print.CanContinue())
				goto out;
		}
		print.CommitJob();
out:;
	}
}

//--------------------------------------------------------------------

void TMailWindow::PrintSetup(void)
{
	BPrintJob	print("mail_print");
	status_t	result;

	if (print_settings)
		print.SetSettings(new BMessage(print_settings));

	if ((result = print.ConfigPage()) == B_NO_ERROR) {
		delete print_settings;
		print_settings = print.Settings();
	}
}

//--------------------------------------------------------------------

void TMailWindow::Reply(entry_ref *ref, TMailWindow *wind, bool all)
{
	bool		have_to = FALSE;
	bool		have_cc = FALSE;
	char		*to = NULL;
	char		*cc;
	char		*header;
	char		*str;
	char		*str1;
	int32		finish;
	int32		len;
	int32		loop;
	int32		start;
	BFile		*file = NULL;
	attr_info	info;

	file = new BFile(ref, O_RDONLY);
	if (file->InitCheck() == B_NO_ERROR) {
		if (file->GetAttrInfo(B_MAIL_ATTR_REPLY, &info) == B_NO_ERROR) {
			to = (char *)malloc(info.size);
			file->ReadAttr(B_MAIL_ATTR_REPLY, B_STRING_TYPE, 0, to, info.size);
			fHeaderView->fTo->SetText(to);
		}
		else if (file->GetAttrInfo(B_MAIL_ATTR_FROM, &info) == B_NO_ERROR) {
			to = (char *)malloc(info.size);
			file->ReadAttr(B_MAIL_ATTR_FROM, B_STRING_TYPE, 0, to, info.size);
			fHeaderView->fTo->SetText(to);
		}

		if (file->GetAttrInfo(B_MAIL_ATTR_SUBJECT, &info) == B_NO_ERROR) {
			str = (char *)malloc(info.size);
			file->ReadAttr(B_MAIL_ATTR_SUBJECT, B_STRING_TYPE, 0, str, info.size);
			if (cistrncmp(str, "re:", 3) != 0) {
				str1 = (char *)malloc(strlen(str) + 1 + 4);
				sprintf(str1, "Re: %s", str);
				fHeaderView->fSubject->SetText(str1);
				free(str1);
			}
			else
				fHeaderView->fSubject->SetText(str);
			free(str);
		}

		wind->fContentView->fTextView->GetSelection(&start, &finish);
		if (start != finish) {
			str = (char *)malloc(finish - start + 1);
			wind->fContentView->fTextView->GetText(start, finish - start, str);
			if (str[strlen(str) - 1] != '\n') {
				str[strlen(str)] = '\n';
				finish++;
			}
			fContentView->fTextView->SetText(str, finish - start);
			free(str);

			finish = fContentView->fTextView->CountLines() - 1;
			for (loop = 0; loop < finish; loop++) {
				fContentView->fTextView->GoToLine(loop);
				fContentView->fTextView->Insert((const char *)QUOTE);
			}
			fContentView->fTextView->GoToLine(0);
		}
		else if (file)
			fContentView->fTextView->LoadMessage(file, TRUE, TRUE, NULL);

		if (all) {
			file->ReadAttr(B_MAIL_ATTR_HEADER, B_INT32_TYPE, 0, &len, sizeof(int32));
			header = (char *)malloc(len);
			file->Seek(0, 0);
			file->Read(header, len);
			start = 0;
			cc = (char *)malloc(1);
			cc[0] = 0;
			for (;;) {
				if ((!have_to) &&
					!(cistrncmp(B_MAIL_TO, &header[start], strlen(B_MAIL_TO) - 1))) {
					have_to = TRUE;
					extract(&cc, &header[start + strlen(B_MAIL_TO)]);
					if (have_cc)
						break;
				}
				else if ((!have_cc) &&
					!(cistrncmp(B_MAIL_CC, &header[start], strlen(B_MAIL_CC) - 1))) {
					have_cc = TRUE;
					extract(&cc, &header[start + strlen(B_MAIL_CC)]);
					if (have_to)
						break;
				}
				start += linelen(&header[start], len - start, TRUE);
				if (start >= len)
					break;
			}
			if (have_to || have_cc) {
				if (to) {
					if (str = cistrstr(cc, to)) {
						len = 0;
						if (str == cc) {
							while ((strlen(to) + len < strlen(cc)) &&
								   ((str[strlen(to) + len] == ' ') ||
									(str[strlen(to) + len] == ','))) {
								len++;
							}
						}
						else {
							while ((str > cc) && ((str[-1] == ' ') || (str[-1] == ','))) {
								str--;
								len++;
							}
						}
						memmove(str, &str[strlen(to) + len], &cc[strlen(cc)] - 
														 &str[strlen(to) + len] + 1);
					}
				}
				fHeaderView->fCc->SetText(cc);
			}
			free(cc);
			free(header);
		}
		if (to)
			free(to);
		fReplying = TRUE;
	}
}

//--------------------------------------------------------------------

void TMailWindow::Send(bool now)
{
	bool			close = FALSE;
	char			mime[256];
	int32			index = 0;
	int32			len;
	int32			result;
	BMailMessage	*mail;
	entry_ref		*ref;
	TListItem		*item;

	if (fResending)
		result = forward_mail(fRef, fHeaderView->fTo->Text(), now);
	else {
		mail = new BMailMessage();
		if (len = strlen(fHeaderView->fTo->Text()))
			mail->AddHeaderField(B_MAIL_TO, fHeaderView->fTo->Text());

		if (len = strlen(fHeaderView->fSubject->Text()))
			mail->AddHeaderField(B_MAIL_SUBJECT, fHeaderView->fSubject->Text());

		if (len = strlen(fHeaderView->fCc->Text()))
			mail->AddHeaderField(B_MAIL_CC, fHeaderView->fCc->Text());

		if (len = strlen(fHeaderView->fBcc->Text()))
			mail->AddHeaderField(B_MAIL_BCC, fHeaderView->fBcc->Text());

		if (len = fContentView->fTextView->TextLength())
			mail->AddContent(fContentView->fTextView->Text(), len);

		while (item = (TListItem *)fEnclosuresView->fList->ItemAt(index++)) {
			mail->AddEnclosure(item->Ref());
		}
		result = mail->Send(now);
		delete mail;
	}

	switch (result) {
		case B_NO_ERROR:
			close = TRUE;
			break;

		case B_MAIL_NO_DAEMON:
			close = TRUE;
			sprintf(mime, "The mail_daemon is not running.  The message is \
queued and will be sent when the mail_daemon is started.");
			break;

		case B_MAIL_UNKNOWN_HOST:
		case B_MAIL_ACCESS_ERROR:
			sprintf(mime, "An error occurred trying to connect with the SMTP \
host.  Check your SMTP host name.");
			break;

		case B_MAIL_NO_RECIPIENT:
			sprintf(mime, "You must have either a \"To\" or \"Bcc\" recipient.");
			break;

		default:
			sprintf(mime, "An error occurred trying to send mail (0x%.8x).",
							result);
	}
	if (result != B_NO_ERROR) {
		beep();
		(new BAlert("", mime, "OK"))->Go();
	}
	if (close)
		PostMessage(B_QUIT_REQUESTED);
}


//====================================================================

TMenu::TMenu(const char *name)
	  :BMenu(name)
{
	BuildMenu();
}

//--------------------------------------------------------------------

void TMenu::AttachedToWindow(void)
{
	BuildMenu();
	BMenu::AttachedToWindow();
}

//--------------------------------------------------------------------

void TMenu::BuildMenu(void)
{
	char			name[B_FILE_NAME_LENGTH];
	int32			index = 0;
	BEntry			entry;
	BFile			file;
	BMenuItem		*item;
	BMessage		*msg;
	BQuery			query;
	BVolume			vol;
	BVolumeRoster	volume;
	entry_ref		ref;

	while (item = RemoveItem((int32)0)) {
		free(item);
	}
	volume.GetBootVolume(&vol);
	query.SetVolume(&vol);
	query.SetPredicate("_signature = *");
	query.Fetch();

	while (query.GetNextEntry(&entry) == B_NO_ERROR) {
		file.SetTo(&entry, O_RDONLY);
		if (file.InitCheck() == B_NO_ERROR) {
			msg = new BMessage(M_SIGNATURE);
			entry.GetRef(&ref);
			msg->AddRef("ref", &ref);
			file.ReadAttr("_signature", B_STRING_TYPE, 0, name, sizeof(name));
			if (index < 9)
				AddItem(new BMenuItem(name, msg, '1' + index));
			else
				AddItem(new BMenuItem(name, msg));
			index++;
		}
	}
}

//====================================================================

TFilter::TFilter(void)
		:BMessageFilter(B_ANY_DELIVERY, B_ANY_SOURCE, B_MOUSE_DOWN)
{
}

//--------------------------------------------------------------------

filter_result TFilter::Filter(BMessage *msg, BHandler **target)
{
	uint32	buttons;
	BPoint	where;

	msg->FindInt32("buttons", (int32*)&buttons);
	if (buttons > 1) {
		be_app->SetCursor(B_HAND_CURSOR);
		msg->FindPoint("where", &where);
		((BView*)*target)->ConvertToScreen(&where);
		((TMailApp *)be_app)->fMenu->Go(where, TRUE);
		return B_SKIP_MESSAGE;
	}
	return B_DISPATCH_MESSAGE;
}
