/* ++++++++++

   FILE:  HP_Setup.cpp
   REVS:  $Revision: 1.13 $
   NAME:  Robert Polic

   Copyright (c) 1996 by Be Incorporated.  All Rights Reserved.

+++++ */

#include <fcntl.h>  
#include <unistd.h>
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>

#include <PrintJob.h>
#include <HP_Setup.h>


//====================================================================

BSetup::BSetup(BMessage *msg, char *name)
	   :BWindow(BRect(100, 100, 100 + SETUP_WIDTH, 100 + SETUP_HEIGHT),
				name, B_TITLED_WINDOW, B_NOT_RESIZABLE | B_NOT_CLOSABLE |
									   B_NOT_MINIMIZABLE | B_NOT_ZOOMABLE)
{
	char	print_name[256];

	fSetupMessage = msg;
	fResult = 0;

	sprintf(print_name, "%s Setup", name);
	SetTitle(print_name);

	Lock();
	fView = new TSetupView(Bounds(), msg);
	AddChild(fView);

	fSetupSem = create_sem(0, "SetupSem");

	fFilter = new BMessageFilter(B_ANY_DELIVERY, B_ANY_SOURCE, B_KEY_DOWN,
								 &SetupKeyFilter);
	AddCommonFilter(fFilter);
	Unlock();
}

//--------------------------------------------------------------------

BSetup::~BSetup()
{
	Lock();
	RemoveCommonFilter(fFilter);
	Unlock();
	delete fFilter;
}

//--------------------------------------------------------------------

void BSetup::MessageReceived(BMessage *msg)
{
	switch (msg->what) {
		case M_OK:
			fView->UpdateMessage(fSetupMessage);
			fResult = B_NO_ERROR;
			// by releasing this semaphore, the Go() method will unblock.
			release_sem(fSetupSem);
			break;

		case M_CANCEL:
			fResult = B_ERROR;
			// by releasing this semaphore, the Go() method will unblock.
			release_sem(fSetupSem);
			break;

		case PAGE_LETTER:
		case PAGE_LEGAL:
		case PAGE_A4:
		case PAGE_B5:
		case PAGE_TABLOID:
		case PAGE_LEDGER:
		case PAGE_ENVELOPE:
		case PAGE_OTHER:
		case SIZE_WIDTH:
		case SIZE_HEIGHT:
		case SIZE_SCALING:
		case ORIENT_PORTRAIT:
		case ORIENT_LANDSCAPE:
			PostMessage(msg, fView);
			break;

		default:
			BWindow::MessageReceived(msg);
			break;
	}
}

//--------------------------------------------------------------------

long BSetup::Go(void)
{
	long value;

	Show();

	acquire_sem(fSetupSem);
	delete_sem(fSetupSem);

	// synchronous call to close the alert window. Remember that this will
	// 'delete' the object that we're in. That's why the return value is
	// saved on the stack.
	value = fResult;
	Lock();
	Quit();
	return(value);
}

//--------------------------------------------------------------------

filter_result SetupKeyFilter(BMessage *msg, BHandler **target, BMessageFilter *filter)
{
	const char	*text;
	long		key;
	long		len;
	long		loop;
	ulong		mods = 0;
	TSetupView	*view;

	BLooper *looper = filter->Looper();
	view = ((BSetup *)looper)->fView;
	if ((*target != view->fWidth->ChildAt(0)) &&
		(*target != view->fHeight->ChildAt(0)) &&
		(*target != view->fScaling->ChildAt(0)))
		return B_DISPATCH_MESSAGE;

	mods = msg->FindInt32("modifiers");
	if (mods & B_COMMAND_KEY)
		return B_DISPATCH_MESSAGE;

	const uchar *bytes = NULL;
	if (msg->FindString("bytes", (char **)&bytes) != B_NO_ERROR)
		return B_DISPATCH_MESSAGE;

	key = bytes[0];
	if (key < '0') {
		if ((key != B_TAB) && (key != B_BACKSPACE) && (key != B_ENTER) &&
			(key != B_LEFT_ARROW) && (key != B_RIGHT_ARROW) &&
			(key != B_UP_ARROW) && (key != B_DOWN_ARROW) && (key != '.')) 
			return B_SKIP_MESSAGE;
		if (key == '.') {
			((BTextView *)*target)->Delete();
			text = ((BTextView *)*target)->Text();
			len = ((BTextView *)*target)->TextLength();
			for (loop = 0; loop < len; loop++)
				if (text[loop] == '.')
					return B_SKIP_MESSAGE;
		}
	}
	if (key > '9')
		return B_SKIP_MESSAGE;
	return B_DISPATCH_MESSAGE;
}


//====================================================================

TSetupView::TSetupView(BRect frame, BMessage *msg)
		   :BView(frame, "", B_FOLLOW_ALL, B_WILL_DRAW)
{
	rgb_color	c;
	BRect		r;

	if (msg->HasInt32("orientation"))
		fOrient = msg->FindInt32("orientation") + ORIENT_PORTRAIT;
	else
		fOrient = ORIENT_PORTRAIT;

	if (msg->HasFloat("scaling"))
		fScale = msg->FindFloat("scaling");
	else
		fScale = 100.0;

	if (msg->HasRect("paper_rect")) {
		r = msg->FindRect("paper_rect");
		fPageWidth = r.Width() / 72.0;
		fPageHeight = r.Height() / 72.0;
	}
	else {
		fPageWidth = 8.5;
		fPageHeight = 11.0;
	}

	c.red = c.green = c.blue = VIEW_COLOR;
	SetViewColor(c);
	SetDrawingMode(B_OP_OVER);
	SetFontSize(12);
}

//--------------------------------------------------------------------

void TSetupView::AttachedToWindow(void)
{
	bool		marked = FALSE;
	char		size[16];
	long		loop;
	BButton		*button;
	BMenuItem	*item;
	BPopUpMenu	*menu;
	BRect		r;

	menu = new BPopUpMenu("");
	menu->SetRadioMode(TRUE);
	for (loop = 0; loop < sizeof(sizes) / sizeof(PAGESIZE); loop++) {
		menu->AddItem(item = new BMenuItem(sizes[loop].name,
							 new BMessage(sizes[loop].message)));
		if ((sizes[loop].width == fPageWidth) &&
			(sizes[loop].height == fPageHeight)) {
			item->SetMarked(TRUE);
			marked = TRUE;
		}
	}
	if (!marked)
		item->SetMarked(TRUE);

	r.Set(PAPER_H, PAPER_V, PAPER_H + 150, PAPER_V + MENU_HEIGHT);
	fPaper = new BMenuField(r, "", PAPER_TEXT, menu);
	fPaper->SetDivider(StringWidth(PAPER_TEXT) + 7);
	AddChild(fPaper);

	NiceString(size, fPageWidth);
	r.Set(WIDTH_H, WIDTH_V, WIDTH_H + WIDTH_WIDTH, WIDTH_V + WIDTH_HEIGHT);
	fWidth = new BTextControl(r, "", WIDTH_TEXT, size, new BMessage(SIZE_WIDTH));
	AddChild(fWidth);
	fWidth->SetAlignment(B_ALIGN_LEFT, B_ALIGN_RIGHT);
	fWidth->SetDivider(StringWidth(WIDTH_TEXT) + 7);

	NiceString(size, fPageHeight);
	r.Set(HEIGHT_H, HEIGHT_V, HEIGHT_H + HEIGHT_WIDTH, HEIGHT_V + HEIGHT_HEIGHT);
	fHeight = new BTextControl(r, "", HEIGHT_TEXT, size, new BMessage(SIZE_HEIGHT));
	AddChild(fHeight);
	fHeight->SetAlignment(B_ALIGN_LEFT, B_ALIGN_RIGHT);
	fHeight->SetDivider(StringWidth(HEIGHT_TEXT) + 7);

	NiceString(size, fScale);
	r.Set(SCALE_H, SCALE_V, SCALE_H + SCALE_WIDTH + 60, SCALE_V + SCALE_HEIGHT);
	fScaling = new BTextControl(r, "", SCALE_TEXT, size, new BMessage(SIZE_SCALING));
	AddChild(fScaling);
	fScaling->SetAlignment(B_ALIGN_LEFT, B_ALIGN_RIGHT);
	fScaling->SetDivider(StringWidth(SCALE_TEXT) + 7);

	menu = new BPopUpMenu("");
	menu->SetRadioMode(TRUE);
	menu->AddItem(item = new BMenuItem("Portrait", new BMessage(ORIENT_PORTRAIT)));
	if (fOrient == ORIENT_PORTRAIT)
		item->SetMarked(TRUE);
	menu->AddItem(item = new BMenuItem("Landscape", new BMessage(ORIENT_LANDSCAPE)));
	if (fOrient == ORIENT_LANDSCAPE)
		item->SetMarked(TRUE);

	r.Set(ORIENT_H, ORIENT_V, ORIENT_H + 150, ORIENT_V + MENU_HEIGHT);
	fOrientation = new BMenuField(r, "", ORIENT_TEXT, menu);
	fOrientation->SetDivider(StringWidth(ORIENT_TEXT) + 7);
	AddChild(fOrientation);

	r.Set(SAMPLE_H, SAMPLE_V,
		  SAMPLE_H + SAMPLE_WIDTH, SAMPLE_V + SAMPLE_HEIGHT);
	fView = new TSampleView(r, fOrient, fPageWidth, fPageHeight);
	AddChild(fView);

	r.Set(PRINT_BUTTON_H, PRINT_BUTTON_V,
		  PRINT_BUTTON_H + BUTTON_WIDTH,
		  PRINT_BUTTON_V + BUTTON_HEIGHT);
	button = new BButton(r, "", PRINT_BUTTON_TEXT, new BMessage(M_PRINT));
	// AddChild(button);

	r.Set(SET_CANCEL_BUTTON_H, SET_CANCEL_BUTTON_V,
		  SET_CANCEL_BUTTON_H + BUTTON_WIDTH,
		  SET_CANCEL_BUTTON_V + BUTTON_HEIGHT);
	button = new BButton(r, "", SET_CANCEL_BUTTON_TEXT, new BMessage(M_CANCEL));
	AddChild(button);

	r.Set(SET_OK_BUTTON_H, SET_OK_BUTTON_V,
		  SET_OK_BUTTON_H + BUTTON_WIDTH,
		  SET_OK_BUTTON_V + BUTTON_HEIGHT);
	button = new BButton(r, "", SET_OK_BUTTON_TEXT, new BMessage(M_OK));
	AddChild(button);
	button->MakeDefault(TRUE);
}

//--------------------------------------------------------------------

void TSetupView::Draw(BRect rect)
{
	BRect	r;

	r = Bounds();

	SetHighColor(255, 255, 255);
	StrokeLine(BPoint(r.left, r.top), BPoint(r.right, r.top));
	StrokeLine(BPoint(r.left, r.top + 1), BPoint(r.left, r.bottom - 1));
	StrokeLine(BPoint(r.left + 3, LINE_V + 4),
			   BPoint(r.right - 3, LINE_V + 4));

	SetHighColor(120, 120, 120);
	StrokeLine(BPoint(r.right, r.top + 1), BPoint(r.right, r.bottom));
	StrokeLine(BPoint(r.right - 1, r.bottom), BPoint(r.left, r.bottom));
	StrokeLine(BPoint(r.left + 3, LINE_V + 3),
			   BPoint(r.right - 3, LINE_V + 3));
}

//--------------------------------------------------------------------

void TSetupView::MessageReceived(BMessage *msg)
{
	char	size[16];
	long	orient;
	float	height;
	float	temp;
	float	width;

	switch (msg->what) {
		case PAGE_LETTER:
		case PAGE_LEGAL:
		case PAGE_A4:
		case PAGE_B5:
		case PAGE_TABLOID:
		case PAGE_LEDGER:
		case PAGE_ENVELOPE:
		case PAGE_OTHER:
			width = sizes[msg->what - PAGE_LETTER].width;
			height = sizes[msg->what - PAGE_LETTER].height;
			if (fOrient == ORIENT_LANDSCAPE) {
				temp = width;
				width = height;
				height = temp;
			}
			if (width != fPageWidth) {
				fPageWidth = width;
				NiceString(size, fPageWidth);
				fWidth->SetText(size);
			}
			if (height != fPageHeight) {
				fPageHeight = height;
				NiceString(size, fPageHeight);
				fHeight->SetText(size);
			}
			fView->SetPageSize(fPageWidth, fPageHeight);
			break;

		case SIZE_WIDTH:
			width = atof(fWidth->Text());
			if (width != fPageWidth) {
				fPageWidth = width;
				FindMenu();
				fView->SetPageSize(fPageWidth, fPageHeight);
			}
			break;

		case SIZE_HEIGHT:
			height = atof(fHeight->Text());
			if (height != fPageHeight) {
				fPageHeight = height;
				FindMenu();
				fView->SetPageSize(fPageWidth, fPageHeight);
			}
			break;

		case SIZE_SCALING:
			break;

		case ORIENT_PORTRAIT:
		case ORIENT_LANDSCAPE:
			orient = msg->what;
			if (orient != fOrient) {
				fOrient = orient;
				fView->SetOrientation(fOrient);
				temp = fPageWidth;
				fPageWidth = fPageHeight;
				fPageHeight = temp;
				NiceString(size, fPageWidth);
				fWidth->SetText(size);
				NiceString(size, fPageHeight);
				fHeight->SetText(size);
			}
			break;
	}
}

//--------------------------------------------------------------------

void TSetupView::FindMenu(void)
{
	long	loop;
	float	width;
	float	height;

	if (fOrient == ORIENT_PORTRAIT) {
		width = fPageWidth;
		height = fPageHeight;
	}
	else {
		width = fPageHeight;
		height = fPageWidth;
	}

	for (loop = 0; loop < sizeof(sizes) / sizeof(PAGESIZE); loop++) {
		if ((sizes[loop].width == width) &&
			(sizes[loop].height == height)) {
			fPaper->Menu()->ItemAt(loop)->SetMarked(TRUE);
			return;
		}
	}
	fPaper->Menu()->ItemAt(loop - 1)->SetMarked(TRUE);
}

//--------------------------------------------------------------------

void TSetupView::NiceString(char *str, float n)
{
	long	size;

	sprintf(str, "%f", n);
	size = strlen(str);
	while (size > 1) {
		if ((str[size - 1] == '0') && (str[size - 2] != '.')) {
			size--;
			str[size] = 0;
		}
		else
			break;
	}
}

//--------------------------------------------------------------------

void TSetupView::UpdateMessage(BMessage *msg)
{
	float	n;
	BRect	r;

	if (msg->HasInt32("orientation"))
		msg->ReplaceInt32("orientation", fOrient - ORIENT_PORTRAIT);
	else
		msg->AddInt32("orientation", fOrient - ORIENT_PORTRAIT);

	n = atof(fScaling->Text());
	if (msg->HasFloat("scale"))
		msg->ReplaceFloat("scale", n);
	else
		msg->AddFloat("scale", n);

	r.left = 0.0;
	r.top = 0.0;
	r.right = atof(fWidth->Text()) * 72;
	r.bottom = atof(fHeight->Text()) * 72;
	if (msg->HasRect("paper_rect"))
		msg->ReplaceRect("paper_rect", r);
	else
		msg->AddRect("paper_rect", r);
}


//====================================================================

TSampleView::TSampleView(BRect frame, long orient, float width, float height)
			:BView(frame, "", B_FOLLOW_ALL, B_WILL_DRAW)
{
	BRect		r;
	rgb_color	c;

	fOrient = orient;
	fWidth = width;
	fHeight = height;

	c.red = c.green = c.blue = VIEW_COLOR;
	SetViewColor(c);

	r.Set(0, 0, ((PORT_WIDTH + 8) & 0xfff8) - 1, PORT_HEIGHT);
	fPortrait = new BBitmap(r, B_COLOR_8_BIT);
	fPortrait->SetBits(PORT_PICT, fPortrait->BitsLength(), 0, B_COLOR_8_BIT);

	r.Set(0, 0, ((LAND_WIDTH + 8) & 0xfff8) - 1, LAND_HEIGHT);
	fLandscape = new BBitmap(r, B_COLOR_8_BIT);
	fLandscape->SetBits(LAND_PICT, fLandscape->BitsLength(), 0, B_COLOR_8_BIT);
}

//--------------------------------------------------------------------

TSampleView::~TSampleView(void)
{
	delete fPortrait;
	delete fLandscape;
}

//--------------------------------------------------------------------

void TSampleView::Draw(BRect rect)
{
	float	n;
	BBitmap	*bits;
	BRect	r;
	BRect	sr;
	BRect	dr;

	r = Bounds();

	if (fOrient == ORIENT_PORTRAIT) {
		sr.Set(0, 0, PORT_WIDTH, PORT_HEIGHT);
		bits = fPortrait;
		if (fWidth > fHeight) {
			n = PORT_WIDTH * (fHeight / fWidth);
			dr.top = (r.Height() - n) / 2;
			dr.bottom = dr.top + n;
			dr.left = (r.Width() - PORT_WIDTH) / 2;
			dr.right = dr.left + PORT_WIDTH;
		}
		else {
			n = PORT_HEIGHT * (fWidth / fHeight);
			dr.left = (r.Width() - n) / 2;
			dr.right = dr.left + n;
			dr.top = (r.Height() - PORT_HEIGHT) / 2;
			dr.bottom = dr.top + PORT_HEIGHT;
		}
	}
	else {
		sr.Set(0, 0, LAND_WIDTH, LAND_HEIGHT);
		bits = fLandscape;
		if (fWidth > fHeight) {
			n = LAND_WIDTH * (fHeight / fWidth);
			dr.top = (r.Height() - n) / 2;
			dr.bottom = dr.top + n;
			dr.left = (r.Width() - LAND_WIDTH) / 2;
			dr.right = dr.left + LAND_WIDTH;
		}
		else {
			n = LAND_HEIGHT * (fWidth / fHeight);
			dr.left = (r.Width() - n) / 2;
			dr.right = dr.left + n;
			dr.top = (r.Height() - LAND_HEIGHT) / 2;
			dr.bottom = dr.top + LAND_HEIGHT;
		}
	}

	SetHighColor(255, 255, 255);
	dr.InsetBy(-4, -4);
	FillRect(dr);
	SetHighColor(0, 0, 0);
	StrokeRect(dr);
	dr.InsetBy(4, 4);
	DrawBitmap(bits, sr, dr);
}

//--------------------------------------------------------------------

void TSampleView::SetOrientation(long orient)
{
	float	temp;
	BRect	r;

	if (orient != fOrient) {
		fOrient = orient;
		temp = fWidth;
		fWidth = fHeight;
		fHeight = temp;
		r = Bounds();
		SetHighColor(VIEW_COLOR, VIEW_COLOR, VIEW_COLOR);
		FillRect(r);
		Draw(r);
	}
}

//--------------------------------------------------------------------

void TSampleView::SetPageSize(float width, float height)
{
	BRect	r;

	if ((fWidth != width) || (fHeight != height)) {
		fWidth = width;
		fHeight = height;
		r = Bounds();
		SetHighColor(VIEW_COLOR, VIEW_COLOR, VIEW_COLOR);
		FillRect(r);
		Draw(r);
	}
}
