/* ++++++++++

   FILE:  HP_Print.cpp
   REVS:  $Revision: 1.7 $
   NAME:  Robert Polic

   Copyright (c) 1996 by Be Incorporated.  All Rights Reserved.

+++++ */

#include <fcntl.h>  
#include <unistd.h>
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>

#include <PrintJob.h>
#include <HP_Print.h>


//====================================================================

BPrint::BPrint(BMessage *msg, char *name)
	   :BWindow(BRect(100, 100, 100 + PRINT_WIDTH, 100 + PRINT_HEIGHT),
				name, B_TITLED_WINDOW, B_NOT_RESIZABLE | B_NOT_CLOSABLE |
									   B_NOT_MINIMIZABLE | B_NOT_ZOOMABLE)
{
	char	print_name[256];

	fPrintMessage = msg;
	fResult = 0;

	sprintf(print_name, "%s Print", name);
	SetTitle(print_name);

	Lock();
	fView = new TPrintView(Bounds(), msg);
	AddChild(fView);

	fPrintSem = create_sem(0, "PrintSem");

	fFilter = new BMessageFilter(B_ANY_DELIVERY, B_ANY_SOURCE, B_KEY_DOWN,
								 &PrintKeyFilter);
	AddCommonFilter(fFilter);
	Unlock();
}

//--------------------------------------------------------------------

BPrint::~BPrint()
{
	Lock();
	RemoveCommonFilter(fFilter);
	Unlock();
	delete fFilter;
}

//--------------------------------------------------------------------

void BPrint::MessageReceived(BMessage *msg)
{
	switch (msg->what) {
		case PRINT_OK:
			fView->UpdateMessage(fPrintMessage);
			fResult = B_NO_ERROR;
			// by releasing this semaphore, the Go() method will unblock.
			release_sem(fPrintSem);
			break;

		case PRINT_CANCEL:
			fResult = B_ERROR;
			// by releasing this semaphore, the Go() method will unblock.
			release_sem(fPrintSem);
			break;

		case PRINT_COPIES:
		case RANGE_ALL:
		case RANGE_SELECTION:
		case RANGE_FROM:
		case RANGE_TO:
		case QUALITY_DRAFT:
		case QUALITY_GOOD:
		case QUALITY_BEST:
			PostMessage(msg, fView);
			break;

		default:
			BWindow::MessageReceived(msg);
			break;
	}
}

//--------------------------------------------------------------------

long BPrint::Go(void)
{
	long value;

	Show();

	acquire_sem(fPrintSem);
	delete_sem(fPrintSem);

	// synchronous call to close the alert window. Remember that this will
	// 'delete' the object that we're in. That's why the return value is
	// saved on the stack.
	value = fResult;
	Quit();
	return(value);
}

//--------------------------------------------------------------------

filter_result PrintKeyFilter(BMessage *msg, BHandler **target, BMessageFilter *filter)
{
	const char	*text;
	long		key;
	long		len;
	long		loop;
	ulong		mods = 0;
	TPrintView	*view;

	BLooper *looper = filter->Looper();
	view = ((BPrint *)looper)->fView;

	if ((*target != view->fCopies->ChildAt(0)) &&
		(*target != view->fFrom->ChildAt(0)) &&
		(*target != view->fTo->ChildAt(0)))
		return B_DISPATCH_MESSAGE;

	mods = msg->FindInt32("modifiers");
	if (mods & B_COMMAND_KEY)
		return B_DISPATCH_MESSAGE;

	const uchar *bytes = NULL;
	if (msg->FindString("bytes", (char **)&bytes) != B_NO_ERROR)
		return B_DISPATCH_MESSAGE;

	key =  bytes[0];
	if (key < '0') {
		if ((key != B_TAB) && (key != B_BACKSPACE) && (key != B_ENTER) &&
			(key != B_LEFT_ARROW) && (key != B_RIGHT_ARROW) &&
			(key != B_UP_ARROW) && (key != B_DOWN_ARROW))
			return B_SKIP_MESSAGE;
	}
	if (key > '9')
		return B_SKIP_MESSAGE;
	return B_DISPATCH_MESSAGE;
}


//====================================================================

TPrintView::TPrintView(BRect frame, BMessage *msg)
		   :BView(frame, "", B_FOLLOW_ALL, B_WILL_DRAW)
{
	rgb_color	c;

	if (msg->HasInt32("quality"))
		fPrintQuality = msg->FindInt32("quality" + QUALITY_DRAFT);
	else
		fPrintQuality = QUALITY_BEST;

	c.red = c.green = c.blue = VIEW_COLOR;
	SetViewColor(c);
	SetDrawingMode(B_OP_OVER);
	//SetFontName("Emily");
	SetFontSize(12);
}

//--------------------------------------------------------------------

void TPrintView::AttachedToWindow(void)
{
	bool		marked = FALSE;
	BButton		*button;
	BMenuItem	*item;
	BPopUpMenu	*menu;
	BRect		r;

	r.Set(COPIES_H, COPIES_V,
		  COPIES_H + COPIES_WIDTH, COPIES_V + COPIES_HEIGHT);
	fCopies = new BTextControl(r, "", COPIES_TEXT, "1",
								new BMessage(PRINT_COPIES));
	AddChild(fCopies);
	fCopies->SetAlignment(B_ALIGN_LEFT, B_ALIGN_RIGHT);
	fCopies->SetDivider(StringWidth(COPIES_TEXT) + 7);

	menu = new BPopUpMenu("");
	menu->SetRadioMode(TRUE);
	menu->AddItem(item = new BMenuItem("Draft", new BMessage(QUALITY_DRAFT)));
	if (fPrintQuality == QUALITY_DRAFT)
		item->SetMarked(TRUE);
	menu->AddItem(item = new BMenuItem("Good", new BMessage(QUALITY_GOOD)));
	if (fPrintQuality == QUALITY_GOOD)
		item->SetMarked(TRUE);
	menu->AddItem(item = new BMenuItem("Best", new BMessage(QUALITY_BEST)));
	if (fPrintQuality == QUALITY_BEST)
		item->SetMarked(TRUE);

	r.Set(ALL_H, ALL_V, ALL_H + ALL_WIDTH, ALL_V + ALL_HEIGHT);
	fAllButton = new BRadioButton(r, "", ALL_TEXT, new BMessage(RANGE_ALL));
	AddChild(fAllButton);
	Window()->Lock();
	fAllButton->SetValue(1);
	Window()->Unlock();

	r.Set(SELECTION_H, SELECTION_V,
		  SELECTION_H + SELECTION_WIDTH, SELECTION_V + SELECTION_HEIGHT);
	fFromButton = new BRadioButton(r, "", SELECTION_TEXT,
								   new BMessage(RANGE_SELECTION));
	AddChild(fFromButton);

	r.Set(FROM_H, FROM_V, FROM_H + FROM_WIDTH, FROM_V + FROM_HEIGHT);
	fFrom = new BTextControl(r, "", FROM_TEXT, "", new BMessage(RANGE_FROM));
	AddChild(fFrom);
	fFrom->SetAlignment(B_ALIGN_LEFT, B_ALIGN_RIGHT);
	fFrom->SetDivider(StringWidth(FROM_TEXT) + 7);

	r.Set(TO_H, TO_V, TO_H + TO_WIDTH, TO_V + TO_HEIGHT);
	fTo = new BTextControl(r, "", TO_TEXT, "", new BMessage(RANGE_TO));
	AddChild(fTo);
	fTo->SetAlignment(B_ALIGN_LEFT, B_ALIGN_RIGHT);
	fTo->SetDivider(StringWidth(TO_TEXT) + 7);

	r.Set(QUALITY_H, QUALITY_V, QUALITY_H + 150, QUALITY_V + MENU_HEIGHT);
	fQuality = new BMenuField(r, "", QUALITY_TEXT, menu);
	fQuality->SetDivider(StringWidth(QUALITY_TEXT) + 7);
	AddChild(fQuality);

	r.Set(PRINT_CANCEL_BUTTON_H, PRINT_CANCEL_BUTTON_V,
		  PRINT_CANCEL_BUTTON_H + PRINT_BUTTON_WIDTH,
		  PRINT_CANCEL_BUTTON_V + PRINT_BUTTON_HEIGHT);
	button = new BButton(r, "", PRINT_CANCEL_BUTTON_TEXT,
						new BMessage(PRINT_CANCEL));
	AddChild(button);

	r.Set(PRINT_OK_BUTTON_H, PRINT_OK_BUTTON_V,
		  PRINT_OK_BUTTON_H + PRINT_BUTTON_WIDTH,
		  PRINT_OK_BUTTON_V + PRINT_BUTTON_HEIGHT);
	button = new BButton(r, "", PRINT_OK_BUTTON_TEXT, new BMessage(PRINT_OK));
	AddChild(button);
	button->MakeDefault(TRUE);
}

//--------------------------------------------------------------------

void TPrintView::Draw(BRect rect)
{
	BRect	r;

	r = Bounds();

	SetHighColor(255, 255, 255);
	StrokeLine(BPoint(r.left, r.top), BPoint(r.right, r.top));
	StrokeLine(BPoint(r.left, r.top + 1), BPoint(r.left, r.bottom - 1));
	StrokeLine(BPoint(r.left + 3, PRINT_LINE_V + 4),
			   BPoint(r.right - 3, PRINT_LINE_V + 4));

	SetHighColor(120, 120, 120);
	StrokeLine(BPoint(r.right, r.top + 1), BPoint(r.right, r.bottom));
	StrokeLine(BPoint(r.right - 1, r.bottom), BPoint(r.left, r.bottom));
	StrokeLine(BPoint(r.left + 3, PRINT_LINE_V + 3),
			   BPoint(r.right - 3, PRINT_LINE_V + 3));

	SetHighColor(0, 0, 0);
	MovePenTo(RANGE_H, RANGE_V);
	DrawString(RANGE_TEXT);
}

//--------------------------------------------------------------------

void TPrintView::MessageReceived(BMessage *msg)
{
	switch (msg->what) {
		case PRINT_COPIES:
			break;

		case RANGE_ALL:
		case RANGE_SELECTION:
			break;

		case RANGE_FROM:
		case RANGE_TO:
			Window()->Lock();
			fFromButton->SetValue(1);
			Window()->Unlock();
			break;

		case QUALITY_DRAFT:
		case QUALITY_GOOD:
		case QUALITY_BEST:
			fPrintQuality = msg->what;
			break;
	}
}

//--------------------------------------------------------------------

void TPrintView::UpdateMessage(BMessage *msg)
{
	long	start;
	long	end;

	if (msg->HasInt32("copies"))
		msg->ReplaceInt32("copies", atoi(fCopies->Text()));
	else
		msg->AddInt32("copies", atoi(fCopies->Text()));

	if (fAllButton->Value()) {
		start = 1;
		end = -1;
	}
	else {
		start = atoi(fFrom->Text());
		end = atoi(fTo->Text());
	}
	if (msg->HasInt32("begin"))
		msg->ReplaceInt32("begin", start);
	else
		msg->AddInt32("begin", start);
	if (msg->HasInt32("end"))
		msg->ReplaceInt32("end", end);
	else
		msg->AddInt32("end", end);

	if (msg->HasInt32("quality"))
		msg->ReplaceInt32("quality", fPrintQuality - QUALITY_DRAFT);
	else
		msg->AddInt32("quality", fPrintQuality - QUALITY_DRAFT);
}
