/* ++++++++++

   FILE:  HP_LaserJet.cpp
   REVS:  $Revision: 1.27 $
   NAME:  Benoit

   Copyright (c) 1996 by Be Incorporated.  All Rights Reserved.

+++++ */
//------------------------------------------------------------------

#include <fcntl.h>  
#include <unistd.h>
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>

#ifndef _DEBUG_H
#include <Debug.h>
#endif
#ifndef _APPLICATION_H
#include <Application.h>
#endif

#ifndef _STORAGE_DEFS_H
#include <StorageDefs.h>
#endif
#ifndef _FILE_H
#include <File.h>
#endif
#ifndef _VOLUME_H
#include <Volume.h>
#endif
#ifndef _DIRECTORY_H
#include <Directory.h>
#endif
#ifndef _ALERT_H
#include <Alert.h>
#endif
#ifndef _VIEW_H
#include <View.h>
#endif
#ifndef	_TEXTCONTROL_H
#include <TextControl.h>
#endif
#ifndef	_BITMAP_H
#include <Bitmap.h>
#endif
#ifndef _REGION_H
#include <Region.h>
#endif

#include <InterfaceDefs.h>
#include <File.h>
#include <Path.h>
#include <FindDirectory.h>
#include <PrintJob.h>

//------------------------------------------------------------------
#include "HP_LaserJet.h"
#include "HP_Print.h"
#include "HP_Setup.h"
#include "Status.h"
//------------------------------------------------------------------

extern char** environ;

//------------------------------------------------------------------
// Here is the list of standard functions that you need to export
// to get a working print server addon.
// More functions might be added later but those 3 functions are the
// strict minimum, later functions will be optionals.
//------------------------------------------------------------------

#pragma export on

BMessage	*take_job(BFile *spool_file, BFile *printer_file, BMessage* msg);
BMessage	*config_page(BFile *printer_file, BMessage* msg);
BMessage	*config_job(BFile *printer_file, BMessage* msg);
char		*add_printer(char *driver_name);

#pragma export reset


//------------------------------------------------------------------
// This is the magic matrix currently used to do the gray scale to
// 1 bit dithering.
//------------------------------------------------------------------

#define	K	8
long	magic[4][4] = {
		{300/K,2100/K,3000/K,5100/K},
		{3600/K,4200/K,900/K,1500/K},
		{2400/K,600/K,4650/K,2700/K},
		{3900/K,3300/K,1800/K,1200/K}
};

//------------------------------------------------------------------
// The ImageView is the view that will be used to image the picture
// extracted from the SpoolFile into the offscreen bitmap. This is
// a most normal view.
//------------------------------------------------------------------

ImageView::ImageView(BRect rect, char *name)
	   	   : BView(rect, name, B_FOLLOW_ALL, B_WILL_DRAW)
{
}

//------------------------------------------------------------------
// Very impressive constructor for the HP... put your stuff here !
//------------------------------------------------------------------

	TPrintDrv::TPrintDrv()
{
}

//------------------------------------------------------------------
// Very impressive destructor too.
//------------------------------------------------------------------

	TPrintDrv::~TPrintDrv()
{
}

//------------------------------------------------------------------
// THe PrintDriver will use an offscreen bitmap which is 2400 pixel
// wide (this is 2400/(72*4) inches in this case) meaning 8.3 inches.
// The Height of the bitmap is the imaging band size we choose.
// Note that we use an 8 bit bitmap here which will work pretty well
// for gray scale and simple color stuff, a more advance printer might
// want to work with a real truecolor bitmap and change the dithering
// code.
//------------------------------------------------------------------

void	TPrintDrv::setup_bitmap()
{
	BRect			a_rect;
	
	a_rect.Set(0, 0, 2399, BAND_SIZE-1);
	a_bitmap = new BBitmap(a_rect, B_COLOR_8_BIT, TRUE);
	
// Add the temporary imaging view to the bitmap.

	a_view = new ImageView(a_rect, "Test");
	
	a_bitmap->AddChild(a_view);
}

//------------------------------------------------------------------
// do_line will take a line in the source bitmap and dither it to
// a 1 bit bitmap line. y is the vertical position in the band and
// is used to keep in sync with the dithering matrix.
// ptr is the source 8 bit bitmap, buffer is the destination 1 bit
// bitmap. 
//------------------------------------------------------------------

void	TPrintDrv::do_line(uchar *buffer, uchar *ptr, long y)
{
	long		i, k, rnd;
	long		intensity;	
	rgb_color	c;
	color_map 	*system_map;
	uchar		cur;	
	long		xx;

	system_map = system_colors();

	y &= 0x03;
	xx = 0;
	for (i = 0; i < 300; i++) {
		cur = 0;
		for (k = 0; k < 8; k++) {
			xx++;
			c = system_map->color_list[*ptr++];
			intensity = c.red + c.green + c.blue;
			//intensity /= 3;

// magic is scaled up 3x, so no need to divide here.
			
			if (intensity < magic[y][xx&0x03])
				cur |= (0x80 >> k);
		}
		buffer[i] = cur;
	}
}

//------------------------------------------------------------------
// external function declaration for the compress (packbits) function.

int	compress(char *outrow, char *inrow, long bytes);

//------------------------------------------------------------------
// stream_bitmap will take the current band in the bitmap and send
// it to the HP printer after dithering etc...
//------------------------------------------------------------------

long	TPrintDrv::stream_bitmap()
{
	long				j;
	long				k;
	unsigned	char	buffer[1024];
	unsigned	char	out[1024];
	unsigned	char	buffer1[32];
	unsigned	char	*ptr;
	long				cnt;
	long				cnt1;
	long				error;

// Printer command to set printer resolution.

	sprintf((char *)buffer1, "%c*b2M\0", 27);
	error = write(ffd, buffer1, strlen((char*)buffer1));
	if (error < 0) {
		return error;
	}	

// For each line in the band do ....

	for (j = 0; j < BAND_SIZE; j++) {
		ptr = (unsigned char *)a_bitmap->Bits();
		ptr += (j * 2400);
		
		
// dither the current line.

		do_line(buffer, ptr, j);
		cnt = 1;

// find the last non zero byte in the buffer... this will avoid doing
// useless work.

		for (k = 0; k < 300; k++) {
			if (buffer[k])
				cnt = k;
		}
		cnt++;

// compress that line using PackBits compression.


		cnt1 = compress((char *)out, (char *)buffer, cnt);
		sprintf((char *)buffer1, "%c*b%ldW\0", 27, cnt1);
		error = write(ffd, buffer1, strlen((char*)buffer1));
		if (error < 0) {
			return error;
		}	

// Send the line to the printer.

		error = write(ffd, out, cnt1);
		if (error < 0) {
			return error;
		}	
	}
	return 0;
}

//------------------------------------------------------------------
// This routine will do the multiple band imaging for a given page...
// It receives an array of picture which will have to be draw into
// each band each time... note that this is a pretty memory hungry
// way to do things since this array of picture could take a lot of
// space.

long	TPrintDrv::do_image(long picture_count, BPicture *a_picture[], BPoint where[], BRect clips[])
{
	char	buffer[32];
	long	i;
	long	p;
	long	error;
	BRect	tmp_clip;
	BRegion	*tmp_region;

	sprintf(buffer, "%c*t300R\0", 27);
	error = write(ffd, buffer, strlen(buffer));
	if (error < 0)
		return error;
	
	sprintf(buffer, "%c*r0A\0", 27);
	error = write(ffd, buffer, strlen(buffer));
	if (error < 0)
		return error;
	
// In this specific driver, we use 3 bands to do the whole vertical
// size of the page.

	for (i = 0; i < 3; i++) {

// First erase the page to white... nobody will do it for you !

		a_view->SetHighColor(255, 255, 255);
		a_view->FillRect(a_view->Bounds());

// Specify to the view the the resolution should be 4*72 DPI.

		a_view->SetScale(4.0);

// Got thru the list of pictures...

		for (p = 0; p < picture_count; p++) {

// Handle the rectangular clipping for a given picture.

			tmp_clip = clips[p];
			tmp_clip.top += where[p].y;
			tmp_clip.bottom += where[p].y;
			tmp_clip.left += where[p].x;
			tmp_clip.right += where[p].x;

// Since we work at (72*4) DPI, rescale the clipping accordingly.			

			tmp_clip.left *= 4.0;
			tmp_clip.top *= 4.0;
			tmp_clip.bottom *= 4.0;
			tmp_clip.right *= 4.0;

			tmp_region = new BRegion();
			tmp_region->Set(tmp_clip);

// Set the clip.

			a_view->ConstrainClippingRegion(tmp_region);
			delete tmp_region;

// And Draw the picture at the correct place.

			a_view->DrawPicture(a_picture[p], BPoint((4.0*where[p].x) + 0, (4.0*where[p].y) + -i*250));
			
// Get certain that all the drawing is completed for that given picture.

			a_view->Sync();
		}

// We can now send that content of the bitmap to the printer

		error = stream_bitmap();
		if (error < 0)
			return error;
	}

	sprintf(buffer, "%c*rB\0", 27);
	error = write(ffd, buffer, strlen(buffer));
	if (error < 0)
		return error;
	sprintf(buffer, "%cE\0", 27);
	error = write(ffd, buffer, strlen(buffer));
	return error;
}

//------------------------------------------------------------------
// do_page will print page <page_number>. BFile is the spool file.

long	TPrintDrv::do_page(BFile *the_file, long page_number)
{
	long		data_size;
	char		*data;
	BPicture	*a_picture[128];
	BRect		tmp;
	BPoint		where[128];
	BRect		clips[128];
	long		picture_count;
	long		p;
	long		i;
	long		*pp;
	long		error;
	char		status[32];

// Update the status field in the spool file database entry... this will
// impress your friends and give good feedback to the user.

/*
	sprintf(status, "pge.%ld", page_number+1);
	the_record->SetString("Status", status);
	the_record->Commit();
*/

// Open the parallel port.

	ffd = open("/dev/parallel2", O_RDWR);
	
// Find out how many pictures are used to describe that page.

	the_file->Read(&picture_count, 4);

// Read for all those picture the position at which they should
// go on the page,the clipping information etc... And the picture
// itself

	for (p = 0; p < picture_count; p++) {
		the_file->Read(&data_size, 4);
		the_file->Read(&where[p], sizeof(BPoint));
		the_file->Read(&clips[p], sizeof(BRect));
		if (data_size > 0 && data_size < MAX_SIZE) {
			data = (char *)malloc(data_size);
			if (data) {
				the_file->Read(data, data_size);
				pp= (long*)data;
				a_picture[p] = new BPicture(data, data_size);
				free((char *)data);
			}
		}
	}

// We need locking before drawing in the window/offscreen.

	a_view->Window()->Lock();
	error = do_image(picture_count, a_picture, where, clips);
	a_view->Window()->Unlock();

// cleanup.

	for (p = 0; p < picture_count; p++)
		delete a_picture[p];

// and close the port.

	close(ffd);
	return error;
}

//------------------------------------------------------------------
// This is the real start of handling a print_job... we will open
// the spool file, get the setup message , expand it into a real
// BMessage.
// Find out the number of pages and handle them. piece of cake.

long	TPrintDrv::handle_job(BFile *the_file)
{
	print_file_header	h;
	long				page;
	char				*setup_flat;
	long				setup_flat_size;
	long				error;

//Here we go, open the little spool file.

	//the_file->Open(B_READ_ONLY);
	the_file->Seek(0, SEEK_SET);

//Get the header.

	the_file->Read(&h, sizeof(h));
	
// Read the size of the setup message (the one you created during
// page setup).
// Then read the message itself.

	the_file->Read(&setup_flat_size, sizeof(setup_flat_size));
	setup_flat = (char *)malloc(setup_flat_size);
	the_file->Read(setup_flat, setup_flat_size);

// Re-Hydrate the message.

	setup = new BMessage();
	setup->Unflatten(setup_flat);

	free((char *)setup_flat);


// Now we can handle the pages... since this is a very simple driver,
// we do not use the setup message.

	for (page = 0; page < h.page_count; page++) {
		error = do_page(the_file, page);
		if (error < 0)
			goto out;
	}
out:;

// Clean up things now.

	delete a_bitmap;

	//the_file->Close();
	if (error >= 0) {
		//unlink(filename);
	}
	else {
		BAlert *alert = new BAlert("Error", "Printer not responding.", "OK");
		alert->Go();
	}

	delete setup;
	return(error);
}

//------------------------------------------------------------------

BMessage	*TPrintDrv::take_job(BFile *the_file, BMessage *msg)
{
	BMessage	*result;
	long		error;

	setup_bitmap();
	//the_record = the_file->Record();
	error = handle_job(the_file);
	if (error >= 0)
		result = new BMessage('okok');
	else
		result = new BMessage('baad');
	return(result);
}

//------------------------------------------------------------------
// Main entry point for job handling.
// We receive the spool_file, the printer configuration file, and the
// setup message.

BMessage	*take_job(BFile* spool_file, BFile *printer_file, BMessage* msg)
{
	BMessage	*result;
	TPrintDrv	*the_drv;

	the_drv = new TPrintDrv();

	result = the_drv->take_job(spool_file, msg);
	delete the_drv;

	return result;
}

//------------------------------------------------------------------

long	TPrintDrv::do_page_setup(BMessage *msg, char *pr_name)
{
	BSetup	*a_setup;

	a_setup = new BSetup(msg, pr_name);
	return(a_setup->Go());
}

//------------------------------------------------------------------

BMessage	*TPrintDrv::page_setup(BMessage *msg, char *pr_name)
{
	BMessage	*result;
	BRect		page_size;
	BAlert		*alert;
	long		val;

	msg = new BMessage(msg);			//clone it !
	val = do_page_setup(msg, pr_name);
	if (val < 0) {
		delete msg;
		return(0);
	}

	msg->what = 'okok';

	//msg->FindRect("paper_rect");
	//msg->FindFloat("scale");
	//msg->FindInt32("orientation"); // 0-Portrait, 1-Landscape

	page_size.top = 0;
	page_size.left = 0;
	page_size.bottom = 11 * 72;
	page_size.right = 8.5 * 72;

	if (!msg->HasRect("paper_rect"))
		msg->AddRect("paper_rect", page_size);	
	
	page_size.top = 0.5 * 72;
	page_size.left = 0.5 * 72;
	page_size.bottom = (11-0.5) * 72;
	page_size.right = (8.5-0.5) * 72;
	if (!msg->HasRect("printable_rect")) {
		msg->AddRect("printable_rect", page_size);	
	}

	return(msg);
}

//------------------------------------------------------------------

long	TPrintDrv::do_print_setup(BMessage *msg, char *pr_name)
{
	BPrint	*a_setup;

	a_setup = new BPrint(msg, pr_name);
	return(a_setup->Go());
}

//------------------------------------------------------------------

BMessage	*TPrintDrv::print_setup(BMessage *msg, char *pr_name)
{
	BMessage	*result;
	BRect		page_size;
	BAlert		*alert;
	long		val;

	msg = new BMessage(msg);			//clone it !
	val = do_print_setup(msg, pr_name);
	if (val < 0) {
		delete msg;
		return(0);
	}

	msg->what = 'okok';

	//msg->FindInt32("copies");
	//msg->FindInt32("begin");	// 1 to -1 = all
	//msg->FindInt32("end");
	//msg->FindInt32("quality"); // 0 - draft, 1 - good, 2 - best

	return(msg);
}

//------------------------------------------------------------------
// This is the main entry point used to handle dialogs/user interaction
// specifing the way printing should be done for the CURRENT job !
// printer_file is the printer configuration for the driver, the BMessage
// is either a blank message OR a previously saved page setup message
// from a previous job.
// This function will return a new BMessage containing the page setup
// fields + any optional field.
// This BMessage will be passed again to the driver when the job is
// started.
//------------------------------------------------------------------

BMessage	*config_page(BFile *printer_file,BMessage* msg)
{
	char			name[128];
	TPrintDrv		*the_drv;
	BMessage		*result;
	long			err;

// We will use the printer name as window name.

	err = printer_file->ReadAttr("Printer Name",
								 B_ASCII_TYPE,
					  			 0,
								 name,
								 128);
	

	the_drv = new TPrintDrv();
	result = the_drv->page_setup(msg, name);
	delete the_drv;
	return(result);
}

//------------------------------------------------------------------

BMessage	*config_job(BFile *printer_file,BMessage* msg)
{
	char			name[128];
	TPrintDrv		*the_drv;
	BMessage		*result;
	long			err;

// We will use the printer name as window name.

	err = printer_file->ReadAttr("Printer Name",
								 B_ASCII_TYPE,
					  			 0,
								 name,
								 128);

	the_drv = new TPrintDrv();
	result = the_drv->print_setup(msg, name);
	delete the_drv;
	return(result);
}

//------------------------------------------------------------------
// Nice to get functions in logical order... this is the foward
// declaration for save_setup.

void	save_setup(char *driver_name, char *printer_name);

//------------------------------------------------------------------
// Configure printer is called when the user wants to create a new
// printer accessed with the current driver.
// The driver name is passed to the function since for localisation
// issue, the name might be different from the initial one.
// add_printer will display any dialog/user interaction needed to
// identify/name a printer to be accessed using the driver.

char *add_printer(char *driver_name)
{
	char			*name;
	BRect			r;	
	long			sem;
	TStatusWindow	*t;

	name = (char *)malloc(128);

// We will create a nice simple little dialog that will request a printer
// name.

#define NAME_TOP		100
#define NAME_LEFT		150

	r.top = NAME_TOP;
	r.bottom = r.top + STATUS_HEIGHT;
	r.left 	= NAME_LEFT;
	r.right = r.left + STATUS_WIDTH;
	sem = create_sem(0, "print_setup");

// create the dialog

	t = new TStatusWindow(r, sem,
					      driver_name);

// wait for the dialog to complete.

	acquire_sem(sem);

// Get a copy of the name the user typed in.

	strcpy(name, t->tmp_string);

// close the dialog.

	t->Lock();
	t->Close();
	delete_sem(sem);

// if the name is null, the user pressed cancel, just clean up and exit.

	if (strlen(name) == 0) {
		free((char *)name);
		return NULL;
	}

// otherwise create a new setup.

	save_setup(driver_name, name);
	return name;
}

//------------------------------------------------------------------
// This function called by add_printer will create a new printer
// file which will define all the info later required by the driver
// to access that printer.

void	save_setup(char *driver_name, char *printer_name)
{
	long			foo;
	long			err;
	BPath			setting_printer_dir;
	BFile			*a_file;

// We want to create the printer file on the boot volume in the
// /boot/system/settings/printers directory.		

	find_directory(B_COMMON_SETTINGS_DIRECTORY, &setting_printer_dir, true);
	setting_printer_dir.Append("printers");
	setting_printer_dir.Append(printer_name);
	a_file = new BFile(setting_printer_dir.Path(), O_CREAT | O_RDWR);

//	sprintf(buf, "/boot/system/settings/printers/%s", printer_name);
//	a_file = new BFile(buf, O_CREAT | O_RDWR);
	
	err = a_file->Write(&foo, sizeof(foo));
	if (err >= 0) {

// You can use the data part of the file to write any private information
// you want that you will need to find that printer in the future.
// In this case we just write some fake data.

			//a_file->Write(&foo, sizeof(foo));

// We need to say which driver has to be used for that printer

			err = a_file->WriteAttr("Driver Name",
									B_ASCII_TYPE,
							  		0,
									driver_name,
									strlen(driver_name)+1);


// And putting a comment with that printer is always a nice touch.

			err = a_file->WriteAttr("Comments",
									B_ASCII_TYPE,
							  		0,
									"Parallel HP Printer",
									strlen("Parallel HP Printer")+1);

			err = a_file->WriteAttr("Printer Name",
									B_ASCII_TYPE,
							  		0,
									printer_name,
									strlen(printer_name)+1);

// and cleanup.

			delete a_file;
		}
}
