/*
 * StackHack.cpp
 * -------------
 * A couple of functions that you can use to trace the
 * current stack of function calls.
 *
 * Copyright 1999, Be Incorporated.   All Rights Reserved.
 * This file may be used under the terms of the Be Sample Code License.
 */

#include "StackHack.h"
#include <image.h>
#include <malloc.h>
#include <stdlib.h>
#include <stdio.h>
#include <String.h>

#define SYMBOL_SIZE 64
#define BLOCK_SIZE 1024;


typedef struct {
	char	symbol[SYMBOL_SIZE];
	uint32	address;
} sym_info;

#define SYM_INFO_SIZE sizeof(sym_info)


static uint32 num_symbols = 0;
static sym_info *sym_table = 0;

static uint32 get_caller_address(uint32 level);
static void lookup_symbol(uint32 addr, const char **symbol, uint32 *offset);

static int compare_symbols(const void *a, const void *b)
{
	return ((sym_info *)a)->address -
		   ((sym_info *)b)->address;
}


#if __INTEL__

uint32 get_caller_address(uint32 level)
{
	uint32 fp = 0, nfp, ret = 0;

	level += 2;
	
	fp = (uint32)get_stack_frame();
	if (fp < 0x80000000)
		return 0;
	nfp = *(uint32 *)fp;
	while (nfp && --level > 0) {
		if (fp < 0x80000000)
			return 0;
		nfp = *(uint32 *)fp;
		ret = *(uint32 *)(fp + 4);
		if (ret < 0x80000000)
			break;
		fp = nfp;
	}

	return ret;
}


void init_sym_table()
{
	image_info info;
	int32 cookie = 0;
	
	uint32 sym_table_size = BLOCK_SIZE;
	sym_table = (sym_info*)malloc(SYM_INFO_SIZE *
								  sym_table_size);
	sym_info *sym_ptr = sym_table;
	
	while (get_next_image_info(0, &cookie, &info) == B_OK) {
		int32 sym_index = 0;
		
		int32 size = SYMBOL_SIZE;
		int32 type;
		
		while (get_nth_image_symbol(info.id, sym_index++,
						sym_ptr->symbol, &size,	&type,
						(void**)&sym_ptr->address) == B_OK) {
			size = SYMBOL_SIZE;
			if (type == B_SYMBOL_TYPE_TEXT) {
				if (++num_symbols == sym_table_size) {
					sym_table_size += BLOCK_SIZE;
					sym_table = (sym_info*)realloc(sym_table,
							SYM_INFO_SIZE * sym_table_size);
				}
				sym_ptr = &sym_table[num_symbols];
			}
		}
	}
	sym_table = (sym_info*)realloc(sym_table,
							SYM_INFO_SIZE * num_symbols);
	
	qsort(sym_table, num_symbols, SYM_INFO_SIZE,
			  compare_symbols);
}

#else 

static __asm unsigned long *get_caller_frame()
{
	lwz     r3, 0 (r1)
	blr
}


uint32 get_caller_address(uint32 level)
{
	uint32 *cf = get_caller_frame();
	uint32 ret = 0;
	
	level += 2;
	
	while (cf && --level > 0) {
		ret = cf[2];
		if (ret < 0x80000000 || ret > 0xfc000000)
			break;
		cf = (uint32 *)*cf;
	}

	return ret;
}


void init_sym_table()
{
	image_info info;
	int32 cookie = 0;
	
	uint32 sym_table_size = BLOCK_SIZE;
	sym_table = (sym_info*)malloc(SYM_INFO_SIZE *
								  sym_table_size);
	sym_info *sym_ptr = sym_table;
	
	while (get_next_image_info(0, &cookie, &info) == B_OK) {
		BString path = info.name;
		path += ".xMAP";
		FILE* file = fopen(path.String(), "r");
		if (file == 0)
			continue;
		char line[512];
		
		while (fgets(line, sizeof(line), file) != NULL) {
			if (strncmp(line, "Code", 4) == 0)
				break;
		}

		while (fgets(line, sizeof(line), file) != NULL) {
			if (strncmp(line, "Data", 4) == 0)
				break;
			char addr[32];
			char type[32];
			char symbol[512];
			
			if (sscanf(line, "%s %s %s\n",
						addr, type, symbol) == 3 &&
						strcmp(type, "PR") == 0 &&
						*symbol != '@') {
				
					symbol[63] = 0;
					strcpy(sym_table[num_symbols].symbol,
						symbol + 1);
					sym_table[num_symbols].address = strtol(addr, 0, 16) +
						(uint32)info.text;
					if (++num_symbols == sym_table_size) {
						sym_table_size += BLOCK_SIZE;
						sym_table = (sym_info*)realloc(
							sym_table, SYM_INFO_SIZE *
							sym_table_size);
				}
			}
		}
		fclose(file);
	}
	sym_table = (sym_info*)realloc(sym_table,
							SYM_INFO_SIZE * num_symbols);
	
	qsort(sym_table, num_symbols, SYM_INFO_SIZE,
			  compare_symbols);
}

#endif


void lookup_symbol(uint32 addr, const char **symbol,
				   uint32 *offset)
{
	uint32 lower = 0;
	uint32 upper = num_symbols - 1;
	uint32 index = 0;
	
	while (upper - lower > 1) {
		index = (upper + lower) / 2;
		if (sym_table[index].address > addr)
			upper = index;
		else
			lower = index;
	}
	
	if (sym_table[upper].address <= addr)
		index = upper;
	else
		index = lower;
		
	*symbol = sym_table[index].symbol;
	*offset = addr - sym_table[index].address;
}


void dump_call_stack()
{
	uint32 addr;
	
	for(uint32 level = 0; level < 3; level++) {
		addr = get_caller_address(level);
		
		const char *symbol;
		uint32 offset;
		lookup_symbol(addr, &symbol, &offset);
		printf("0x%08lx: %s + 0x%08lx\n", addr, symbol, offset);
	}
}

void cleanup_sym_table()
{
	if (sym_table)
		free(sym_table);	
}
