
/* Non-lagged Drag and Drop */


#include <interface/View.h>
#include <app/Application.h>
#include <interface/Window.h>
#include <stdio.h>

class QuickDrag : public BView
{
	public:
						QuickDrag(BRect frame);
						~QuickDrag();
						
		void			MouseDown(BPoint where);
		void			MouseMoved(BPoint where, uint32 code, const BMessage *msg);
		void			MouseUp(BPoint where);

		void			Pulse();
		
	private:
		bool			fClicked;
		BPoint			fClickPt;
		bool			fDragging;
};

#include <interface/Bitmap.h>
#include <interface/Screen.h>

QuickDrag::QuickDrag(BRect frame) :
	BView(frame, "QuickDrag", B_FOLLOW_ALL_SIDES, B_WILL_DRAW | B_PULSE_NEEDED),
	fClicked(false),
	fClickPt(B_ORIGIN),
	fDragging(false)
{
}


QuickDrag::~QuickDrag()
{
}

void 
QuickDrag::MouseDown(BPoint where)
{
	// We could be starting a drag session
	// mark that the mouse click originated in our view
	// and cache the click point 
	fClicked = true;
	fClickPt = where;
	// set the mouse mask to get mouse moved messages from outside
	// the view
	SetMouseEventMask(B_POINTER_EVENTS, 0);
}

void 
QuickDrag::MouseMoved(BPoint where, uint32 code, const BMessage *msg)
{
	// ignore if the click did not originate in our view
	// or if already dragging
	if (!fClicked || fDragging)
		return;

	// only start the drag process if the mouse has moved more than
	// ten pixels vertically or horizontally
	BPoint diff = where - fClickPt;
	if (diff.x > 10 || diff.x < -10 || diff.y > 10 || diff.y < -10)
	{
		fDragging = true;
		
		// create a new bitmap to be dragged
		BBitmap *bitmap = new BBitmap(Bounds(), B_RGB32);
		
		// rgb_colors are in rgba order in memory
		// RGB32 bitmaps are in bgra order
		// swap red and blue components
		rgb_color color = ViewColor();
		uint8 r = color.red;
		color.red = color.blue;
		color.blue = r;

		// fill in the bitmap
		int32 * bits = (int32 *)bitmap->Bits();
		int32 * end = bits + (bitmap->BitsLength()/sizeof(int32));
		while (bits < end)
		{
			*bits++ = *((int32 *)&color); 
		}
		// start the drag session with a bogus message
		BMessage dumb('dumb');
		DragMessage(&dumb, bitmap, B_OP_BLEND, where);
	}	
}

void 
QuickDrag::MouseUp(BPoint where)
{
	// we are definitely done, so unset everything
	fClicked = false;
	fDragging = false;
	SetMouseEventMask(0, 0);
}

void 
QuickDrag::Pulse()
{
	// don't cycle colors if we are dragging
	if (fDragging)
		return;
		
	// change the view color
	rgb_color color = ViewColor();
	color.red -= 1;
	color.green -= 2;
	color.blue -= 3;
	SetViewColor(color);
	// invalidate so it actually draws
	Invalidate();
}

class SimpleWin : public BWindow
{
	public:
		SimpleWin(BRect rect) : BWindow(rect, "QuickDrag", B_TITLED_WINDOW, 0){};
		bool	QuitRequested() {
			be_app->PostMessage(B_QUIT_REQUESTED);
			return true;
		};
};

int main()
{
	BApplication app("application/x-vnd.BeDTS-QuickDrag");
	BRect rect(50,50,250,250);
	SimpleWin *win = new SimpleWin(rect);
	
	QuickDrag *view = new QuickDrag(win->Bounds());
	win->AddChild(view);
	win->Show();
	app.Run();
	return 0;
}