/*
	Copyright 1999, Be Incorporated.   All Rights Reserved.
	This file may be used under the terms of the Be Sample Code License.
*/
#include "Layout.h"
#include "FlipPage.h"

#include <Window.h>
#include <stdio.h>

Layout::Layout()
{
	fPrintSettings = NULL;
	fNumFlipsOnCurPage = 0;

	// views cannot be printed unless they are attached to a window,
	// so we create a small "dummy win" far offscreen, which is only
	// exploited for its connection to the app server
	fDummyWin = new BWindow(BRect(31000,31000, 31005,31005), "dummy", B_TITLED_WINDOW, 0);
	fDummyWin->Show();

	Reposition();
}

Layout::~Layout()
{
	delete fPrintSettings;
}

void
Layout::SetFrameSize(BRect frameSize)
{
	fFlipPageRect.Set(0,0,frameSize.Width()+50+5,
						frameSize.Height()+5+5);
	Reposition();
}

void
Layout::AddFrame(BBitmap *image)
{
	int32 item = fNumFlipsOnCurPage;
	float top = (item % fVerticalPages) * fFlipPageRect.Height();
	float left = floor(item / fVerticalPages) * fFlipPageRect.Width();

	BRect frame(left, top, left + fFlipPageRect.Width(),
				top + fFlipPageRect.Height());
	frame.OffsetBy(fOffset); // center the flip pages on printed page
	
	fDummyWin->Lock();
	FlipPage *page = new FlipPage(frame, image);
	fDummyWin->AddChild(page);
	fPrintJob->DrawView(page, page->Bounds(), frame.LeftTop());
	fNumFlipsOnCurPage++;
	fDummyWin->RemoveChild(page);
	fDummyWin->Unlock();
	delete page;
	
	if(fNumFlipsOnCurPage == FlipPagesPerPage()){
		fPrintJob->SpoolPage();
		fNumFlipsOnCurPage = 0;
	}

}

int32
Layout::FlipPagesPerPage()
{
	return fHorizontalPages * fVerticalPages;
}

status_t
Layout::JobSetup()
{
	BPrintJob job("jobsetup");
	status_t result;
	
	if(fPrintSettings == NULL){
		result = PageSetup();
		if(result != B_OK){
			return result;
		}		
	}
	
	job.SetSettings(new BMessage(fPrintSettings));
	
	result = job.ConfigJob();
	if(result == B_OK){
		delete fPrintSettings;
		fPrintSettings = job.Settings();
	}

	return result;
}

status_t
Layout::Begin()
{
	status_t result = JobSetup();
	if(result == B_OK){
		fPrintJob = new BPrintJob("Flipbook");
		fPrintJob->SetSettings(new BMessage(fPrintSettings));
		fPrintJob->BeginJob();
	}
	return result;
}

void
Layout::FinishLastPage()
{
	fPrintJob->SpoolPage();
	fPrintJob->CommitJob();
	delete fPrintJob;
	FlipPage::ResetPageNums();
	fNumFlipsOnCurPage = 0;
}


status_t
Layout::PageSetup()
{
	BPrintJob job("pagesetup");
	if(fPrintSettings){
		job.SetSettings(new BMessage(fPrintSettings));
	}
	
	status_t result;
	result = job.ConfigPage();
	if(result == B_OK){
		delete fPrintSettings;
		fPrintSettings = job.Settings();
	}

	Reposition();
	return result;
}

void
Layout::Reposition()
{
	BPrintJob job("reposition");
	if(fPrintSettings){
		job.SetSettings(new BMessage(fPrintSettings));
	}

	BRect printableRect = job.PrintableRect();
	
	fHorizontalPages = (int32)floor(printableRect.Width() / fFlipPageRect.Width());
	fVerticalPages = (int32)floor(printableRect.Height() / fFlipPageRect.Height());

	// fOffset is for centering the flippages on the printed page
	float off_x = (printableRect.Width() - fHorizontalPages*fFlipPageRect.Width())/2;
	float off_y = (printableRect.Height() - fVerticalPages*fFlipPageRect.Height())/2;
	fOffset.Set(off_x, off_y);
}

void
Layout::PrinterChanged()
{
	/* called when a B_PRINTER_CHANGED message is received */
	/* this forces us to re-run PageSetup */
	delete fPrintSettings;
	fPrintSettings = NULL;
	Reposition();
}