/*
	Copyright 1999, Be Incorporated.   All Rights Reserved.
	This file may be used under the terms of the Be Sample Code License.
*/
#include "FlipView.h"
#include "FlipWin.h"
#include "FlipMsg.h"
#include "FlipPage.h"

#include <MediaFile.h>
#include <Alert.h>
#include <stdio.h>

const int32 MOVIE_BORDER = 10;
const int32 MIN_MOVIE_WIDTH = 300;
const int32 MIN_MOVIE_HEIGHT = 200;

float max(float a, float b)
{
	if(a > b)	return a;
	else		return b;
}

FlipView::FlipView(BRect R)
	: BView(R, "flipview", B_WILL_DRAW, B_FOLLOW_ALL_SIDES)
{
	fOpenPanel = NULL;
	fOpenPanelTarget = NULL;
	fCurFrame = NULL;
	fVideoTrack = NULL;
	fVideoFile = NULL;

	fLayout = new Layout;

	SetViewColor(216,216,216);
	
	// create the movie view
	BRect rect(0,0,R.Width(),MIN_MOVIE_HEIGHT);
	fMovieView = new MovieView(rect, this);
	AddChild(fMovieView);

	// create the control view	
	rect.Set(0,MIN_MOVIE_HEIGHT,R.right-R.left,R.bottom-R.top);
	fControlView = new ControlView(rect, this);
	AddChild(fControlView);
}

FlipView::~FlipView()
{
	delete fLayout;
	delete fOpenPanel;
	delete fOpenPanelTarget;	
}

void
FlipView::MessageReceived(BMessage* msg)
{
	switch(msg->what)
	{
	 case OPEN_MOVIE:
		ShowOpenPanel();		
		break;
	 case PAGE_SETUP:
		fLayout->PageSetup();
		fControlView->UpdatePrintStats();
		break;
	 case B_REFS_RECEIVED:
		msg->FindRef("refs", &fMovieRef);
		if(PrepareMovie() == B_OK){
			DoResize();
			fControlView->EnableGenerateButton(true);
			fControlView->UpdatePrintStats();
		} else {
			fControlView->EnableGenerateButton(false);
		}
		break;
	 case GENERATE:
	 {
		thread_id tid;
		tid = spawn_thread(FlipView::GenerateHook, "async", B_NORMAL_PRIORITY, this);
		if (tid >= 0) {
			resume_thread(tid);
		} else {
			fprintf(stderr,"Flipbook: Couldn't spawn a new thread!\n");
		}
		break;
	 }
 	 case LEFT_HANDED:
		FlipPage::IsRightHanded(false);
		break;
	 case RIGHT_HANDED:
		FlipPage::IsRightHanded(true);
		break;
	 case B_PRINTER_CHANGED:
		fLayout->PrinterChanged();
		break;
	 default:
		BView::MessageReceived(msg);
		break;
	}
}

void
FlipView::DoResize()
{
	// current size
	float width = fMovieView->Bounds().Width();
	float height = fMovieView->Bounds().Height();

	// new size	
 	float newWidth =  max(MIN_MOVIE_WIDTH, fCurFrame->Bounds().Width() +2 * MOVIE_BORDER);
	float newHeight = max(MIN_MOVIE_HEIGHT, fCurFrame->Bounds().Height()+2 * MOVIE_BORDER);

	float diffx = newWidth - width;
	float diffy = newHeight - height;

	// resize the views
	fMovieView->ResizeTo(newWidth,newHeight);
	fControlView->MoveBy(diffx/2, diffy);
	
	// resize the window
	Window()->ResizeBy(diffx, diffy);
	
	fMovieView->Invalidate();
}

void
FlipView::ShowOpenPanel()
{
	if(fOpenPanel == NULL){
		fOpenPanelTarget = new BMessenger(this);	
		fOpenPanel = new BFilePanel(B_OPEN_PANEL, fOpenPanelTarget,
						NULL, 0, false);
	}	
	fOpenPanel->Show();					
}

bool
FlipView::HasMovie() const
{
	return (fVideoFile != NULL);
}

status_t
FlipView::PrepareMovie()
{
	delete fVideoFile; fVideoFile = NULL;
	delete fCurFrame; fCurFrame = NULL;
	fVideoTrack = NULL;	// BMediaFile owns the track, just set it to NULL
	fMovieView->NewFrame(NULL);		// clears the current frame
		
	fVideoFile = new BMediaFile(&fMovieRef);
	if(fVideoFile->InitCheck() != B_OK){
		(new BAlert("", "Couldn't properly open your file..."
			"Are you sure this is a MOVIE file?", "uhh, well...no"))->Go();
		return B_ERROR;
	}

	media_format encodedFormat;
	int32 numTracks = fVideoFile->CountTracks();

	for(int32 i=0; i < numTracks; i++){

		BMediaTrack *tryTrack = fVideoFile->TrackAt(i);
		if (tryTrack == NULL || tryTrack->InitCheck() != B_OK) {
			(new BAlert("", "Media file looks bad. Can't find any tracks.", "eek!"))->Go();
			return B_ERROR;
		}
		
		tryTrack->EncodedFormat(&encodedFormat);
		if (encodedFormat.type == B_MEDIA_ENCODED_VIDEO) {
			BRect frameRect(0, 0, 
				encodedFormat.u.encoded_video.output.display.line_width - 1,
				encodedFormat.u.encoded_video.output.display.line_count - 1);
			fCurFrame = new BBitmap(frameRect, B_RGB_32_BIT, true);
			encodedFormat.u.raw_video.display.format = B_RGB_32_BIT;
			encodedFormat.u.raw_video.display.bytes_per_row = fCurFrame->BytesPerRow();

			media_format decodedFormat;
			decodedFormat.type = B_MEDIA_RAW_VIDEO;
			decodedFormat.u.raw_video.last_active = (uint32) (frameRect.Height() - 1);
			decodedFormat.u.raw_video.orientation = B_VIDEO_TOP_LEFT_RIGHT;
			decodedFormat.u.raw_video.pixel_width_aspect = 1;
			decodedFormat.u.raw_video.pixel_height_aspect = 3;
			decodedFormat.u.raw_video.display.format = fCurFrame->ColorSpace();
			decodedFormat.u.raw_video.display.line_width = (int32) frameRect.Width();
			decodedFormat.u.raw_video.display.line_count = (int32) frameRect.Height();
			decodedFormat.u.raw_video.display.bytes_per_row = fCurFrame->BytesPerRow();
			tryTrack->DecodedFormat(&decodedFormat);
			fVideoTrack = tryTrack;
			break;		// found the video track
		}
	}

	if(fVideoTrack == NULL){
		(new BAlert("", "Hey, your file has no video track!", "Naughty!"))->Go();
		return B_ERROR;
	}
	
	int64 val;
	fVideoTrack->ReadFrames((char*)fCurFrame->Bits(), &val);	
	fLayout->SetFrameSize(fCurFrame->Bounds());

	fMovieView->PositionFrame(fCurFrame->Bounds());
	fMovieView->NewFrame(fCurFrame);
	
	return B_OK;
}

int32
FlipView::GenerateHook(void *data)
{
	int32 result = B_ERROR;
	
	FlipView *fv = static_cast<FlipView*>(data);
	FlipWin *fw = dynamic_cast<FlipWin*>(fv->Window());

	// acquire the lock so the user can't quit while we're
	// generating a flipbook...
	if(acquire_sem(fw->generate_sem) == B_OK){

		fv->fControlView->EnableGenerateButton(false);
		result = fv->GenerateFlipbook();

		fv->fControlView->EnableGenerateButton(true);	
		release_sem(fw->generate_sem);
	}
	return result;
}

int32
FlipView::GenerateFlipbook()
{
	if(fVideoTrack == NULL){
		// shouldn't happen, since the Generate button isn't enabled until
		// we have a video track
		(new BAlert("", "You need to open a movie before you make a flipbook!\n", "Ugh"))->Go();
		return B_ERROR;
	}
	
	// start the print job
	if(fLayout->Begin() != B_OK){
		return B_ERROR;
	}	

	// get back to the first frame of the movie
	int64 seekFrame = 0;
	fVideoTrack->SeekToFrame(&seekFrame);
	
	int64 val;
	int64 numFrames = fVideoTrack->CountFrames();
	int32 markCount = fControlView->MarkCount();

	// step through the frames one at a time, draw each one on the screen,
	// and print the "marked" ones	
	for(int64 i=0; i < numFrames; i++){
		fVideoTrack->ReadFrames((char*)fCurFrame->Bits(), &val);
		fMovieView->NewFrame(fCurFrame);		
		if(i % markCount == 0){ 
			fLayout->AddFrame(fCurFrame);
		}
	}
	
	// finish the print job
	fLayout->FinishLastPage();
	return B_OK;
}


