/*
	Copyright 1999, Be Incorporated.   All Rights Reserved.
	This file may be used under the terms of the Be Sample Code License.
*/
#include "ControlView.h"
#include "FlipView.h"
#include <String.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>

ControlView::ControlView(BRect R, FlipView *parent)
	: BView(R, "controlview", B_WILL_DRAW, 0)
{
	fParent = parent;
	fMarkCount = 1;
	SetViewColor(216,216,216);	
}

ControlView::~ControlView()
{
	// empty
}

void
ControlView::AttachedToWindow()
{
	const char *MarkMessage = "Print one frame per ";
	// draw status box
	fStatusBox = new StatusBox(STAT_BOX);
	AddChild(fStatusBox);

	// draw frame-marking control
	fMarkControl = new BTextControl(MARK_BOX, "markcontrol",
					MarkMessage, "1", new BMessage(MARK_COUNT));	
	fMarkControl->SetTarget(this);
	float swidth = StringWidth(MarkMessage);
	fMarkControl->SetDivider(swidth+3);
	AddChild(fMarkControl);

	// draw Generate button	
	float top = Bounds().bottom-10-GEN_BTN_HEIGHT;
	float left = Bounds().right-10-GEN_BTN_LEN;
	BRect R(left, top, left+GEN_BTN_LEN, top+GEN_BTN_HEIGHT);
	fGenerate = new BButton(R, "gen", "Generate", new BMessage(GENERATE));
	fGenerate->SetEnabled(false);
	AddChild(fGenerate);
	
}

void
ControlView::MessageReceived(BMessage *msg)
{
	switch(msg->what)
	{
	 case MARK_COUNT:
		SetMarkCount(fMarkControl->Text());
		UpdatePrintStats();
	 default:
		BView::MessageReceived(msg);
		break;
	}
}

void
ControlView::SetMarkCount(const char *num)
{
	/* be a little bit careful before calling atoi */
	bool nonDigit = false;
	int32 len = strlen(num);
	for(int32 i=0; i<len; i++){
		if(isdigit(num[i]) == false){
			nonDigit = true;
		}
	}
	
	if(nonDigit || len == 0){
		goto error;
	}

	fMarkCount = atoi(num);
	if(fMarkCount == 0){
		goto error;
	}
	return;

error:
	fMarkCount = 1;
	fMarkControl->SetText("1");
	fMarkControl->Invalidate();
}

void
ControlView::UpdatePrintStats()
{
	if(fParent->HasMovie() == false){
		return;
	}

	int64 marked = fParent->fVideoTrack->CountFrames()/fMarkCount;
	fStatusBox->FramesMarked(marked);
	
	int64 perPage = fParent->fLayout->FlipPagesPerPage();
	fStatusBox->FramesPerPage(perPage);

	int64 total = (marked / perPage) + 1;
	fStatusBox->TotalPages(total);

	fStatusBox->Invalidate();
}

void
ControlView::EnableGenerateButton(bool val)
{
	Window()->Lock();
	fGenerate->SetEnabled(val);
	Window()->Unlock();
}

int32
ControlView::MarkCount() const
{
	return fMarkCount;
}

/*************************************************************/
StatusBox::StatusBox(BRect R)
	: BBox(R, "status")
{
	SetLabel("Print Information");
	fMarked = 0;
	fFramesPerPage = 0;
	fPages = 0;
}

StatusBox::~StatusBox()
{
	// empty
}

void
StatusBox::FramesMarked(int64 val)
{
	fMarked = val;
}

void
StatusBox::FramesPerPage(int64 val)
{
	fFramesPerPage = val;
}

void
StatusBox::TotalPages(int64 val)
{
	fPages = val;
}

void
StatusBox::Draw(BRect R)
{
	SetFont(be_plain_font);
	
	BString str;
	SetFontSize(12);
	BPoint pt(10,30);

	MovePenTo(pt);	
	str = "Number of flip pages: ";
	str << fMarked;
	DrawString(str.String());
	
	pt.Set(10,50);
	MovePenTo(pt);	
	str = "Number of flip pages per printed page: ";
	str << fFramesPerPage;
	DrawString(str.String());
	
	pt.Set(10,70);
	MovePenTo(pt);	
	str = "Number of printed pages: ";
	str << fPages;
	DrawString(str.String());

	SetFont(be_bold_font);
	
	BBox::Draw(R);
}

/*************************************************************/

