// gameplay.cpp
// ------------
// A simple application that demonstrates the new GameSound
// interface.
//
// Copyright 1999, Be Incorporated.   All Rights Reserved.
// This file may be used under the terms of the Be Sample Code License.

#include <SimpleGameSound.h>
#include <FileGameSound.h>
#include <Window.h>
#include <View.h>
#include <Application.h>
#include <MenuField.h>
#include <PopUpMenu.h>
#include <Button.h>
#include <Bitmap.h>
#include <MenuItem.h>
#include <NodeInfo.h>
#include <Bitmap.h>
#include <Alert.h>
#include <Menu.h>
#include <Slider.h>

#include <string>
#include <string.h>
#include <stdio.h>

#include "TrackReader.h"



//	You can change this to make changes take effect faster (or slower)
#define FADE_DELAY 200000LL
#define RATE_DELAY 500000LL

class AView;


//#pragma mark --- Declarations ---

class AWindow : public BWindow {
public:
		AWindow();
		~AWindow();
		bool QuitRequested();
		void MessageReceived(BMessage * message);
		enum {
			cmdStartPlaying = 'awi2',
			cmdStopPlaying,
			cmdSetSoundKind,
			cmdSetFile,
			cmdSetVolume,
			cmdSetRate
		};
		enum {
			kindSimple,
			kindFile,
			kindPush
		};
private:
		BGameSound * m_sound;
		AView * m_view;
		int32 m_kind;
		float m_samplingRate;

		void StartPlaying();
		void StopPlaying();
		void SetSoundKind(int32 kind);
		void SetFile(const entry_ref * ref);
		void SetVolume(float v);
		void SetRate(float r);
};

class AView : public BView {
public:
		AView(BRect area);
		~AView();
		void Draw(BRect area);
		void MouseMoved(BPoint where, uint32 code, const BMessage *a_message);
		void MessageReceived(BMessage * message);
private:
		BBitmap * m_bitmap;
		string m_name;
		bool m_highlight;
};


//#pragma mark --- AWindow ---

AWindow::AWindow() :
	BWindow(BRect(100,300,400,450), "GamePlay", B_TITLED_WINDOW,
			B_NOT_RESIZABLE | B_NOT_ZOOMABLE | B_ASYNCHRONOUS_CONTROLS)
{
	m_sound = 0;
	m_kind = kindFile;
	//	create background
	BView * bg = new BView(Bounds(), "bg", B_FOLLOW_ALL, B_WILL_DRAW);
	bg->SetViewColor(216, 216, 216);
	//	create drop target area
	BRect r(bg->Bounds());
	r.top = r.bottom-52;
	m_view = new AView(r);
	bg->AddChild(m_view);
	//	create type selector
	BPopUpMenu * pum = new BPopUpMenu("Kind");
	BMenuItem * mi;
	BMessage msg(cmdSetSoundKind);
	msg.AddInt32("kind", kindSimple);
	mi = new BMenuItem("BSimpleGameSound", new BMessage(msg));
	pum->AddItem(mi);
	msg.ReplaceInt32("kind", kindFile);
	mi = new BMenuItem("BFileGameSound", new BMessage(msg));
	pum->AddItem(mi);
	BMenuField * mf = new BMenuField(BRect(10,10,150,30), "Kind", "Kind", pum, true);
	mf->SetDivider(mf->StringWidth("Kind")+6);
	bg->AddChild(mf);
	//	stop button
	BButton * stop = new BButton(BRect(160,10,220,33), "Stop", "Stop", new BMessage(cmdStopPlaying));
	bg->AddChild(stop);
	//	start button
	BButton * start = new BButton(BRect(230,10,290,33), "Start", "Start", new BMessage(cmdStartPlaying));
	bg->AddChild(start);

	r = Bounds();
	r.top = start->Frame().bottom+10;
	r.bottom = r.top+30;
	r.left = 10;
	r.right = 145;
	BSlider * sr = new BSlider(r, "Volume", "Volume", new BMessage(cmdSetVolume), 0, 100);
	sr->SetValue(100);
	sr->SetModificationMessage(new BMessage(cmdSetVolume));
	bg->AddChild(sr);
	r.OffsetBy(r.Width()+10, 0);
	sr = new BSlider(r, "Rate", "Rate", new BMessage(cmdSetRate), 0, 200);
	sr->SetValue(100);
	sr->SetModificationMessage(new BMessage(cmdSetRate));
	bg->AddChild(sr);
	//	display it all
	AddChild(bg);
	//	set the title
	mf->Menu()->Superitem()->SetLabel("BFileGameSound");
}


AWindow::~AWindow()
{
	delete m_sound;
}

bool 
AWindow::QuitRequested()
{
	StopPlaying();
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

void 
AWindow::MessageReceived(BMessage *message)
{
	switch (message->what) {
	case cmdStartPlaying:
		StartPlaying();
		break;
	case cmdStopPlaying:
		StopPlaying();
		break;
	case cmdSetSoundKind: {
			int32 kind;
			if (!message->FindInt32("kind", &kind)) {
				SetSoundKind(kind);
			}
		}
		break;
	case cmdSetFile: {
			entry_ref ref;
			if (!message->FindRef("refs", &ref)) {
				SetFile(&ref);
			}
		}
		break;
	case cmdSetVolume: {
			int32 value;
			if (!message->FindInt32("be:value", &value)) {
				SetVolume(value/100.0);
			}
		}
		break;
	case cmdSetRate: {
			int32 value;
			if (!message->FindInt32("be:value", &value)) {
				SetRate(value/100.0);
			}
		}
		break;
	default:
		BWindow::MessageReceived(message);
	}
}

void
AWindow::StartPlaying()
{
	if (m_sound == 0) return;
	m_sound->StartPlaying();
}

void
AWindow::StopPlaying()
{
	if (m_sound == 0) return;
	m_sound->StopPlaying();
}

void
AWindow::SetSoundKind(int32 kind)
{
	m_kind = kind;
}

void
AWindow::SetFile(const entry_ref *ref)
{
	//	Stop current sound (if any)
	StopPlaying();
	delete m_sound;
	//	Create new sound of indicated type
	switch (m_kind) {
	case kindSimple: {
	//	We could just as easily create the SimpleGameSound with the file in question.
	//	However, I want to introduce the ATrackReader class, so we do it the hard way.
	//		m_sound = new BSimpleGameSound(ref);
			ATrackReader * trk = open_sound_file(ref);
			if (trk != 0) {
				void * d = malloc(trk->CountFrames() * trk->FrameSize());
				trk->ReadFrames(d, trk->CountFrames());
				media_raw_audio_format raf(trk->Format().u.raw_audio);
				gs_audio_format fmt(reinterpret_cast<gs_audio_format &>(raf));
				m_sound = new BSimpleGameSound(d, trk->CountFrames(), &fmt);
				free(d);
				delete trk;
			}
		}
		break;
	case kindFile:
		m_sound = new BFileGameSound(ref);
		break;
	default:
		m_sound = 0;
	}
	if (m_sound && (m_sound->InitCheck() != B_OK)) {
		char msg[300];
		sprintf(msg, "Cannot open file '%s': %s", ref->name, strerror(m_sound->InitCheck()));
		(new BAlert("", msg, "Stop"))->Go();
		delete m_sound;
		m_sound = 0;
	}
	else if (!m_sound) {
		char msg[300];
		sprintf(msg, "Cannot open file '%s'.", ref->name);
		(new BAlert("", msg, "Stop"))->Go();
	}
	else {
		m_samplingRate = m_sound->Format().frame_rate;
		BControl * v = dynamic_cast<BControl *>(FindView("Volume"));
		if (v != 0) {
			SetVolume(v->Value()/100.0);
		}
		v = dynamic_cast<BControl *>(FindView("Rate"));
		if (v != 0) {
			SetRate(v->Value()/100.0);
		}
	}
}

void
AWindow::SetVolume(
	float v)
{
	if (m_sound == 0) return;
	m_sound->SetGain(v, FADE_DELAY);
}

void
AWindow::SetRate(
	float r)
{
	if (m_sound == 0) return;
	gs_attribute sr;
	sr.attribute = B_GS_SAMPLING_RATE;
	sr.duration = RATE_DELAY;
	if (r < 0.1) r = 0.1;
	sr.value = r * m_samplingRate;
	sr.flags = 0;
	status_t err = m_sound->SetAttributes(&sr, 1);
	if (err != B_OK) {
		fprintf(stderr, "Set sampling rate failed: %s\n", strerror(err));
	}
}

//#pragma mark --- AView ---

AView::AView(BRect area) :
	BView(area, "AView", B_FOLLOW_ALL, B_WILL_DRAW),
	m_name("<Drop sound file here>")
{
	m_bitmap = 0;
	m_highlight = false;
	SetLowColor(216, 216, 216);
	SetViewColor(LowColor());
}


AView::~AView()
{
	delete m_bitmap;
}

void 
AView::Draw(BRect area)
{
	//	Draw the icon and file name.
	SetDrawingMode(B_OP_COPY);
	DrawString(m_name.c_str(), BPoint(52,34));
	if (m_highlight) {
		SetHighColor(255, 0, 0);
		SetPenSize(2.0);
		StrokeRect(Bounds());
		SetHighColor(0, 0, 0);
	}
	if (m_bitmap) {
		SetDrawingMode(B_OP_OVER);
		DrawBitmap(m_bitmap, BPoint(10,10));
	}
}

void 
AView::MouseMoved(BPoint where, uint32 code, const BMessage *a_message)
{
	SetDrawingMode(B_OP_COPY);
	if (code == B_ENTERED_VIEW) {
		if ((a_message != 0) && a_message->HasRef("refs")) {
			m_highlight = true;
			FillRect(Bounds(), B_SOLID_LOW);
			Draw(Bounds());
			Flush();
		}
	}
	else if (code == B_EXITED_VIEW) {
		m_highlight = false;
		FillRect(Bounds(), B_SOLID_LOW);
		Draw(Bounds());
		Flush();
	}
}

void 
AView::MessageReceived(BMessage *message)
{
	m_highlight = false;
	SetDrawingMode(B_OP_COPY);
	//	The user dropped a file on us. Tell the window to use it.
	entry_ref ref;
	if (message->FindRef("refs", &ref)) {
		FillRect(Bounds(), B_SOLID_LOW);
		Draw(Bounds());
		Flush();
		return;		//	no file we could use
	}
	BMessage msg(AWindow::cmdSetFile);
	msg.AddRef("refs", &ref);
	Window()->PostMessage(&msg);
	//	and get the icon thereof
	delete m_bitmap;
	m_bitmap = new BBitmap(BRect(0,0,31,31), B_CMAP8);
	BNodeInfo::GetTrackerIcon(&ref, m_bitmap);
	m_name = ref.name;
	bool playNow = false;
	if ((message->FindBool("play_now", &playNow) == B_OK) && playNow) {
		Window()->PostMessage(AWindow::cmdStartPlaying);
	}
	//	force re-draw
	FillRect(Bounds(), B_SOLID_LOW);
	Draw(Bounds());
	Flush();
}


//#pragma mark --- main ---

int
main(
	int argc,
	char * argv[])
{
	//	Check arguments and create app object
	if ((argc > 1) && (argv[1][0] == '-')) {
		fprintf(stderr, "usage: gameplay [ file ]\n");
		fprintf(stderr, "Plays indicated sound file (defaults to raw CD-audio for unknown files)\n");
		return 1;
	}
	BApplication app("application/x-vnd.be.newsletter-gameplay");

	//	Create a BSimpleGameSound with the format we want for the connection
	//	to the GameSound system (the format is taken from the first sound that's
	//	created or loaded).
	gs_audio_format fmt = { 44100.0, 2, gs_audio_format::B_GS_S16,
		B_HOST_IS_BENDIAN ? B_MEDIA_BIG_ENDIAN : B_MEDIA_LITTLE_ENDIAN,
		0 };
	BSimpleGameSound * sgs = new BSimpleGameSound(0, 10, &fmt);
	delete sgs;

	//	Let's get this show on the road.
	AWindow * w;
	(w = new AWindow)->Show();
	entry_ref ref;
	if ((argv[1] != 0) && (get_ref_for_path(argv[1], &ref) == B_OK)) {
		BMessage msg(B_REFS_RECEIVED);
		msg.AddRef("refs", &ref);
		msg.AddBool("play_now", true);
		w->PostMessage(&msg, w->FindView("AView"));
	}

	app.Run();

	return 0;
}
