/*******************************************************************************
/
/	File:			tester.c
/
/	Description:	A sample application that tests the atomizer module.
/
/	Copyright 1999, Be Incorporated, All Rights Reserved.
/	This file may be used under the terms of the Be Sample Code License.
/
*******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

/* atomizer driver API */
#include "atomizer_driver.h"
/* the atomizer_info structure */
#include "atomizer.h"

typedef void (iterator_func)(int fd, const void *);

void list_atomizers(int fd, iterator_func *ifunc) {
	atomizer_info ai;
	atomizer_get_next_atomizer_info gnai;
	void *cookie = 0;
	status_t result;
	gnai.magic = ATOMIZER_PRIVATE_DATA_MAGIC;
	gnai.cookie = &cookie;
	gnai.info = &ai;
	while ((result = ioctl(fd, T_ATOMIZER_GET_NEXT_ATOMIZER_INFO, &gnai, sizeof(gnai))) == B_OK) {
		if (ifunc) ifunc(fd, ai.atomizer);
		else printf("Atomizer Info:\n\ttoken: %p\n\tname: %s\n\tcount: %lu\n\n", ai.atomizer, ai.name, ai.atom_count);
	}
}

const void *make_atomizer(int fd, const char *name) {
	atomizer_find_or_make_atomizer foma;
	status_t result;
	foma.magic = ATOMIZER_PRIVATE_DATA_MAGIC;
	foma.name = name;
	foma.atomizer = 0;
	result = ioctl(fd, T_ATOMIZER_FIND_OR_MAKE_ATOMIZER, &foma, sizeof(foma));
	printf("make_atomizer() returns %p\n", foma.atomizer);
	return (result == B_OK) ? foma.atomizer : 0;
}

void string_for_token(int fd, const void *atomizer, const void *atom) {
	atomizer_string_for_token sft;
	char buffer[256];
	status_t result;

	sft.magic = ATOMIZER_PRIVATE_DATA_MAGIC;
	sft.atomizer = atomizer;
	sft.atom = atom;
	sft.string = buffer;
	sft.max_size = sizeof(buffer);
	sft.length = 0;
	result = ioctl(fd, T_ATOMIZER_STRING_FOR_TOKEN, &sft, sizeof(sft));
	if (result == B_OK) printf("token: %p string: %s\n\n", atom, buffer);
}

void list_atoms(int fd, const void *atomizer) {
	atomizer_get_next_token gnt;
	status_t result;
	uint32 cookie = 0;

	gnt.magic = ATOMIZER_PRIVATE_DATA_MAGIC;
	gnt.atomizer = atomizer;
	gnt.cookie = &cookie;
	while ((result = ioctl(fd, T_ATOMIZER_GET_NEXT_ATOM, &gnt, sizeof(gnt))) == B_OK) {
		string_for_token(fd, atomizer, gnt.atom);
	}
}

const void *atomize(int fd, const void *atomizer, const char *name) {
	atomizer_atomize	a;
	status_t result;

	a.magic = ATOMIZER_PRIVATE_DATA_MAGIC;
	a.atomizer = atomizer;
	a.string = name;
	a.create = 1;
	a.atom = 0;

	result = ioctl(fd, T_ATOMIZER_ATOMIZE, &a, sizeof(a));
	return (result == B_OK) ? a.atom : 0;
}

int main(int argc, char **argv) {

	int fd;
	/* open the atomizer device */
	if ((fd = open("/dev/misc/atomizer", O_RDWR)) >= 0) {
		const void *my_atomizer;
		const void *sys_atomizer;
		const void *foo, *bar, *bar_sys;

		list_atomizers(fd, 0);
		my_atomizer = make_atomizer(fd, "my atomizer");
		printf("After making my_atomizer:\n");
		list_atomizers(fd, 0);
		sys_atomizer = make_atomizer(fd, 0);
		printf("After making the system atomizer:\n");
		list_atomizers(fd, 0);
		foo = atomize(fd, my_atomizer, "foo");
		bar = atomize(fd, my_atomizer, "bar");
		bar_sys = atomize(fd, sys_atomizer, "bar");
		printf("tokens for\n\tfoo: %p\n\tbar: %p\n\tbar (in system atomizer): %p\n", foo, bar, bar_sys);
		list_atomizers(fd, 0);
		foo = atomize(fd, my_atomizer, "foo");
		bar = atomize(fd, my_atomizer, "bar");
		bar_sys = atomize(fd, sys_atomizer, "bar");
		printf("again, tokens for\n\tfoo: %p\n\tbar: %p\n\tbar (in system atomizer): %p\n", foo, bar, bar_sys);
		list_atomizers(fd, 0);
		printf("dumping all tokens/strings:\n");
		list_atomizers(fd, list_atoms);
		close(fd);
	} else {
		printf("Yikes!  Couldn't open device:\n\t%s\n\t%s\n", strerror(fd), strerror(errno));
	}
	return 0;
}
