//  SettingsPanel.cpp
//
// 	This panel will get user settings. 
// 
//	russ 5/21/98
// 
//	Copyright 1997-1999, Be Incorporated.   All Rights Reserved.
//	This file may be used under the terms of the Be Sample Code License.

#include <string.h>
#include <stdlib.h>

#include <Alert.h>
#include <Box.h>
#include <Button.h>
#include <Screen.h>
#include <TextControl.h>
#include <Window.h>
#include <StringView.h>
#include <PopUpMenu.h>
#include <MenuField.h>
#include <MenuItem.h>
#include <CheckBox.h>

#include "Resource.h"
#include "SettingsPanel.h"
#include "IO.h"

const char* kSettingsBtnDone = "Done";
const char* kSettingsBtnCancel = "Cancel";
const char* kSettingsMsgCancel = "Cancel";
const char* kSettingsMsgOK = "OK";

static float
FontHeight(BView* target, bool full)
{
	font_height finfo;		
	target->GetFontHeight(&finfo);
	float h = finfo.ascent + finfo.descent;

	if (full)
		h += finfo.leading;
	
	return h;
}

static void CenterWindowOnScreen(BWindow* w)
{
	BRect	screenFrame = (BScreen(B_MAIN_SCREEN_ID).Frame());
	BPoint 	pt;
	pt.x = screenFrame.Width()/2 - w->Bounds().Width()/2;
	pt.y = screenFrame.Height()/2 - w->Bounds().Height()/2;

	if (screenFrame.Contains(pt))
		w->MoveTo(pt);
}

const int32 kPanelWidth = 300; 
const int32 kPanelHeight = 260;

SettingsPanel::SettingsPanel(BCallbackHandler* cb)
	:BWindow(BRect(0,0,kPanelWidth, kPanelHeight), "E-Drive Settings", B_MODAL_WINDOW,  
		B_NOT_RESIZABLE | B_NOT_ZOOMABLE | B_NOT_CLOSABLE)
{	
	m_CallBack = cb;	
	CreatePanelUI();
	
	Update(kFillControls);
	
	CenterWindowOnScreen(this);
	Show();      
} 

SettingsPanel::~SettingsPanel()
{
}

void SettingsPanel::MessageReceived(BMessage *msg) 
{
 		switch (msg->what){
			case MSG_DONE:
				m_CallBack->Done(B_NO_ERROR);
				PostMessage(B_QUIT_REQUESTED);
			break;
			
			case MSG_CANCEL:
				m_CallBack->Done(B_ERROR);
				PostMessage(B_QUIT_REQUESTED);
			break;
					
		default:
			BWindow::MessageReceived(msg);
			break;
		}
}

void SettingsPanel::Update(int16 direction)
{		
    // Load up the controls
    if (direction == kFillControls){ 
   		m_srcIPAddressCtl->SetText(m_srcIPAddress);
		m_dstIPAddressCtl->SetText(m_dstIPAddress);
		m_dstMacAddressCtl->SetText(m_dstMacAddress);
		m_showPacketsCtl->SetValue(m_showPackets);
		m_pingReplyCtl->SetValue(m_pingReply);
		m_showBroadcastsCtl->SetValue(m_showBroadcasts);
	}
    else{
        // Fill members from controls
		strcpy(m_srcIPAddress, m_srcIPAddressCtl->Text());
		strcpy(m_dstIPAddress, m_dstIPAddressCtl->Text());
		strcpy(m_dstMacAddress, m_dstMacAddressCtl->Text());
		m_showPackets = m_showPacketsCtl->Value();
		m_pingReply = m_pingReplyCtl->Value();	
		m_showBroadcasts = m_showBroadcastsCtl->Value();			   
    }
}

void SettingsPanel::CreatePanelUI() 
{
	BTextView* tv;
	
	BRect rect(Bounds());
	rect.InsetBy(-1.0, -1.0);
	m_MainBG = new BBox(rect);
	m_MainBG->SetFont(be_plain_font);
	AddChild(m_MainBG);
	
	rect.left = 11; rect.right = Bounds().Width() - 11;
	rect.top = 13; rect.bottom = rect.top + FontHeight(m_MainBG,true)+5;
	m_Label = new BStringView(rect, "", "");
	m_Label->SetFont(be_bold_font);
	m_Label->SetText("Settings for testing ethernet drivers.");
	m_MainBG->AddChild(m_Label);
	
	// Reset this here so that stringwidths can be calculated for plain.
	m_MainBG->SetFont(be_plain_font);
	
	rect.top = m_Label->Frame().bottom + 10;
	rect.bottom = rect.top + 1;
	m_srcIPAddressCtl = new BTextControl(rect, "", "Src IP address:", "", NULL);
	m_srcIPAddressCtl->SetDivider(m_MainBG->StringWidth("Src IP address:") + 10);
	m_srcIPAddressCtl->SetAlignment(B_ALIGN_RIGHT, B_ALIGN_LEFT);
	m_srcIPAddressCtl->MakeFocus(true);
	tv = m_srcIPAddressCtl->TextView();
	if (tv)
		tv->SetMaxBytes(128);
	m_MainBG->AddChild(m_srcIPAddressCtl);
	
	rect.top = m_srcIPAddressCtl->Frame().bottom + 10;
	rect.bottom = rect.top + 1;
	m_dstIPAddressCtl = new BTextControl(rect, "", "Dst IP address:", "", NULL);
	m_dstIPAddressCtl->SetDivider(m_MainBG->StringWidth("Dst IP address:") + 10);
	m_dstIPAddressCtl->SetAlignment(B_ALIGN_RIGHT, B_ALIGN_LEFT);
	tv = m_dstIPAddressCtl->TextView();
	if (tv)
		tv->SetMaxBytes(128);
	m_MainBG->AddChild(m_dstIPAddressCtl);
	
	rect.top = m_dstIPAddressCtl->Frame().bottom + 10;
	rect.bottom = rect.top + 1;
	m_dstMacAddressCtl = new BTextControl(rect, "", "Dst Ethernet Address:", "", NULL);
	m_dstMacAddressCtl->SetDivider(m_MainBG->StringWidth("Dst Ethernet Address:") + 10);
	m_dstMacAddressCtl->SetAlignment(B_ALIGN_RIGHT, B_ALIGN_LEFT);
	m_dstMacAddressCtl->MakeFocus(true);
	tv = m_dstMacAddressCtl->TextView();
	if (tv)
		tv->SetMaxBytes(128);
	m_MainBG->AddChild(m_dstMacAddressCtl);
	
	rect.top = m_dstMacAddressCtl->Frame().bottom + 10;
	rect.bottom = rect.top + 1;
	rect.left = 11;
	rect.right = Bounds().Width() - 10;	
	m_showPacketsCtl = new BCheckBox(rect, "", "Display packets", 
									new BMessage(999));
	m_MainBG->AddChild(m_showPacketsCtl);
	
	rect.top = m_showPacketsCtl->Frame().bottom + 10;
	rect.bottom = rect.top + 1;
	rect.left = 11;
	rect.right = Bounds().Width() - 10;	
	m_pingReplyCtl = new BCheckBox(rect, "", "Do ping replies", 
									new BMessage(999));
	m_MainBG->AddChild(m_pingReplyCtl);
	
	rect.top = m_pingReplyCtl->Frame().bottom + 10;
	rect.bottom = rect.top + 1;
	rect.left = 11;
	rect.right = Bounds().Width() - 10;	
	m_showBroadcastsCtl = new BCheckBox(rect, "", "Display broadcast packet", 
									new BMessage(999));
	m_MainBG->AddChild(m_showBroadcastsCtl);
			
	/////////////// Buttons ////////////////	
	rect.right = Bounds().Width() - 10;
	rect.left = rect.right - 75;
	rect.top = m_showBroadcastsCtl->Frame().bottom + 15;
	rect.bottom = rect.top + 1;
	m_DoneBtn = new BButton(rect, "", kSettingsBtnDone, new BMessage(MSG_DONE),
			B_FOLLOW_BOTTOM | B_FOLLOW_RIGHT);
	
	rect.right = rect.left - 10;
	rect.left = rect.right - 75;
	m_CancelBtn = new BButton(rect, "", kSettingsBtnCancel, new BMessage(MSG_CANCEL),
			B_FOLLOW_BOTTOM | B_FOLLOW_RIGHT);
	m_MainBG->AddChild(m_CancelBtn);	

	m_MainBG->AddChild(m_DoneBtn);
	SetDefaultButton(m_DoneBtn);
}

void SettingsPanel::SetSrcIPAddress(const char* srcIPAddress)
{	
	if (Lock()){
		strcpy(m_srcIPAddress, srcIPAddress);		
		Update(kFillControls);
		m_srcIPAddressCtl->MakeFocus(true);		    		  
    	Unlock();
    }
}

void SettingsPanel::SetDstIPAddress(const char* dstIPAddress)
{	
	if (Lock()){
		strcpy(m_dstIPAddress, dstIPAddress);		
		Update(kFillControls);	    		  
    	Unlock();
    }
}

void SettingsPanel::SetDstMacAddress(const char* dstMacAddress)
{	
	if (Lock()){
		strcpy(m_dstMacAddress, dstMacAddress);		
		Update(kFillControls);	    		  
    	Unlock();
    }
}

void SettingsPanel::SetShowPackets(const bool showPackets)
{	
	if (Lock()){
		m_showPackets = showPackets;		
		Update(kFillControls);	    		  
    	Unlock();
    }
}

void SettingsPanel::SetPingReply(const bool pingReply)
{	
	if (Lock()){
		m_pingReply = pingReply;		
		Update(kFillControls);	    		  
    	Unlock();
    }
}

void SettingsPanel::SetShowBroadcasts(const bool showBroadcasts)
{	
	if (Lock()){
		m_showBroadcasts = showBroadcasts;		
		Update(kFillControls);	    		  
    	Unlock();
    }
}

void SettingsPanel::GetSrcIPAddress(char* srcIPAddress)
{		
	Update(kEmptyControls);
	strcpy(srcIPAddress, m_srcIPAddress);
}

void SettingsPanel::GetDstIPAddress(char* dstIPAddress)
{		
	Update(kEmptyControls);
	strcpy(dstIPAddress, m_dstIPAddress);
}

void SettingsPanel::GetDstMacAddress(char* dstMacAddress)
{		
	Update(kEmptyControls);
	strcpy(dstMacAddress, m_dstMacAddress);
}

void SettingsPanel::GetShowPackets(bool* showPackets)
{		
	Update(kEmptyControls);
	*(bool*)showPackets = m_showPackets;
}

void SettingsPanel::GetPingReply(bool* pingReply)
{		
	Update(kEmptyControls);
	*(bool*)pingReply = m_pingReply;
}

void SettingsPanel::GetShowBroadcasts(bool* showBroadcasts)
{		
	Update(kEmptyControls);
	*(bool*)showBroadcasts = m_showBroadcasts;
}
