// 	IO.cpp
//
//	This class is a wrapper around some basic ethernet driver calls.
//	It handles all packet receives and transmits. There is also a simple
//	ping reply method built in to make driver development easier. User 
//  settings settings are also handled by this class.  
//
//  russ 4/28/99
// 
//	Copyright 1997-1999, Be Incorporated.   All Rights Reserved.
//	This file may be used under the terms of the Be Sample Code License.

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/uio.h>
#include <Alert.h>
#include <Path.h>
#include <FindDirectory.h>
#include <Directory.h>
#include <dirent.h>
#include <File.h>
#include <ByteOrder.h>
#include <netdb.h>

#include "IO.h"
#include "DriverNamePanel.h"
#include "Wnd.h"
#include "SettingsPanel.h"
#include "ByteArray.h"

// Thread that receives all packets. Normally these are broadcasts and
// our hardware address packets.
int32 Reader(void* data);
// Thread to send out ping requests.
int32 Transmitter(void* data);

#define PAYLOADLENGTH 1400 // The ICMP data length.
#define TIMEINTERVAL 1500000 // The transmit interval.

// Open is called after selecting a driver name from the name
// panel. It does an open on the driver and makes a couple
// io control calls to get the driver ready to be used by
// this app.
void IO::Open()
{
   	int		err;
	char    errbuf[512];
	BAlert*	alert;
	char tmp[5];
	char infobuf[512];
	int i;
		
	sprintf(infobuf, "Opening ethernet driver %s ...", m_DriverName); 
	m_parent->Print(infobuf);
	
	m_id = open(m_DriverName, O_RDWR);
	if (m_id < 0) {
		sprintf(errbuf, "Unable to open driver- %s", m_DriverName);   	
	    alert = new BAlert("Open", errbuf, "OK");
	    alert->Go();
		return;
	}	
	else{	    
	    err = ioctl(m_id, ETHER_INIT, &tmp[0], 0);
		if (err < 0) {
			m_parent->Print("Unable to initializing driver.");
			return;
		}
				
		err = ioctl(m_id, ETHER_GETADDR, &m_srcMacAddress[0], sizeof(m_srcMacAddress));
		if (err >= 0) {
			strcpy(infobuf, "Ethernet address ");
			for (i = 0; i < 5; i++){
				sprintf(tmp, "%2.2x:", m_srcMacAddress[i]);
				strcat(infobuf, tmp);
			}
			sprintf(tmp, "%2.2x", m_srcMacAddress[i]);
			strcat(infobuf, tmp);	
			m_parent->Print(infobuf);
			if (IOCtrl(MSG_RECEIVE_PACKETS))  
					m_parent->Print("Ok to choose transmit from file menu.");							
		}
	}	
}

void IO::Close()
{
	close(m_id);
}

bool IO::IOCtrl(uint32 msg)
{	
	if (msg == MSG_RECEIVE_PACKETS){		
		m_parent->Print("Spawning receiver thread...");
		
		m_rxThread = spawn_thread(Reader, "Reader_Thread", 
									B_NORMAL_PRIORITY, this);
									
		resume_thread(m_rxThread);	
	}
	
	if (msg == MSG_TRANSMIT_PACKETS){
		m_parent->Print("Spawning transmitter thread...");
		
		m_txThread = spawn_thread(Transmitter, "Transmiter_Thread", 
									B_NORMAL_PRIORITY, this);
									
		resume_thread(m_txThread);					
	}
	
	return true;
}

int32 Reader(void* data)
{
	IO* parent = (IO*)data;
	uchar RxPacket[kfixedPacketLen];
	uchar TxPacket[kfixedPacketLen];
	uint16 packetLength;	

	for (;;){
		if (read(parent->m_id, RxPacket, kfixedPacketLen) < 0){
			parent->m_parent->Print("No data received from read.");
			return 0;
		}			
		
		// Display received packets
		bool okToShow = false;
		if (parent->m_showPackets){
		    if (parent->m_showBroadcasts)
				parent->DumpPacket("Rx Packet:", RxPacket);
			else{ // Do not show broadcasts packets.
				for (int i=0; i<6; i++)
	 				if (((uint8)RxPacket[i]^0xff) != 0){ // Filter out broadcasts.
	                	okToShow = true;
						break;
					}
		        if (okToShow)
					parent->DumpPacket("Rx Packet:", RxPacket);
			}
		}			
							
		if (parent->PrepareArpReply(RxPacket, TxPacket, &packetLength)){
			if (parent->m_showPackets) 							
				parent->DumpPacket("Tx Arp Reply Packet:", TxPacket);
			
			write(parent->m_id, TxPacket, packetLength); // Arp reply
	 	}		
		
		if (parent->m_pingReply){
	 		if (parent->PreparePingReply(RxPacket, TxPacket, &packetLength)){
				if (parent->m_showPackets) 							
					parent->DumpPacket("Tx Ping Reply Packet:", TxPacket);
			
				write(parent->m_id, TxPacket, packetLength); // Ping reply
	 		}
		}
		
		// Check if this is a ping reply for us (response to our ping request),
		// if so then increment ICMP sequence number and do time profile. 	
		parent->ProcessPingReply(RxPacket, &packetLength);			
	}
	
	return 1;
}

int32 Transmitter(void* data)
{
	IO* parent = (IO*)data;
	uchar TxPacket[kfixedPacketLen];
	uint16 packetLength;
		
	for (;;){	
		parent->PreparePingRequest(TxPacket, &packetLength);
		
		if (parent->m_showPackets) 	
			parent->DumpPacket("Tx Ping Request Packet:", TxPacket);
		
		parent->m_timevalout = system_time();	
		write(parent->m_id, TxPacket, packetLength); // Ping request
		
		snooze(parent->m_timeInterval);
	}
	
	return 1;
}

bool IO::PrepareArpReply(uchar* pRxPacket, uchar* pTxPacket, uint16* packetLength)
{
	uint16 etherType;
	uint16 arpType;
	uint8 etherLength = 13; // 14 bytes to data, but since 0 offset adjust to 13.
	in_addr addr;
	char addr_buf[16];
	bool preparedArpReply = false;
	
	etherType = ntohs(readw((char*)pRxPacket+12));		
	arpType = ntohs(readw((char*)pRxPacket+etherLength+7));
	*packetLength =  42;
		
	if (etherType == 0x0806) // Arp	
		if (arpType == 1){  // Arp request
			
			addr.s_addr = readl((char*)pRxPacket+etherLength+25);
			strcpy(addr_buf, inet_ntoa(addr));
			if (strcmp(m_srcIPAddress, addr_buf) != 0)
				return (preparedArpReply = false);				
		
			for (int i=0; i < *packetLength; i++)
				*(pTxPacket+i) = *(pRxPacket+i);					
					
			// Fill in the destination and source mac address.
			for(int i=0; i<6; i++){
				*(pTxPacket+i) = *(pRxPacket+6+i);				        
				*(pTxPacket+6+i) = m_srcMacAddress[i];
				*(pTxPacket+etherLength+9+i) = m_srcMacAddress[i]; // Arp sender address.
				*(pTxPacket+etherLength+19+i) = *(pRxPacket+6+i); // Arp target address.
			}
					
			int32 ipSrcOffset = etherLength+15;
			int32 ipDstOffset = etherLength+25;
			// Exchange the destination and source ip address.
			for(int i=0; i<4; i++){
				*(pTxPacket+ipSrcOffset+i) = *(pRxPacket+ipDstOffset+i);				        
				*(pTxPacket+ipDstOffset+i) = *(pRxPacket+ipSrcOffset+i);
			}
				
			// Set the arp reply.
			writew(htons(2), (char*)pTxPacket+etherLength+7);			
						
			preparedArpReply = true;	
		}		
		
	return preparedArpReply;
}

bool IO::PreparePingReply(uchar* pRxPacket, uchar* pTxPacket, uint16* packetLength)
{
	uint16 etherType;
	uint8 ipProtocol;
	uint8 ipHeaderLength;
	uint8 etherLength = 13; // 14 bytes to data, but since 0 offset adjust to 13.
	uint8 icmpType;
	uint16 ipTotalLength;	
	bool preparedPingReply = false;
			
	etherType = ntohs(readw((char*)pRxPacket+12));
	// Get ip header length in 32 bit chunks so mult by 4.	
	ipHeaderLength = (readb((char*)pRxPacket+etherLength+1) & 0x0F) * 4; 
	ipProtocol = readb((char*)pRxPacket+etherLength+10);
	icmpType = readb((char*)pRxPacket+etherLength+ipHeaderLength+1);
	ipTotalLength = ntohs(readw((char*)pRxPacket+etherLength+3));
	*packetLength =  ipTotalLength + etherLength + 1;	
		
	if (etherType == 0x0800) // ip
	
		if (*packetLength == 1514) 
			m_parent->Print("Warning: No ip fragmentation support, may not reply.");
			
		if (ipProtocol == 1)   // icmp		
			if (icmpType == 8){   // Echo request (ping)
				for (int i=0; i < *packetLength; i++)
					*(pTxPacket+i) = *(pRxPacket+i);					
					
				// Exchange the destination and source mac address.
				for(int i=0; i<6; i++){
					*(pTxPacket+i) = *(pRxPacket+6+i);				        
					*(pTxPacket+6+i) = *(pRxPacket+i);
				}
					
				int32 ipSrcOffset = etherLength+13;
				int32 ipDstOffset = etherLength+17;
				// Exchange the destination and source ip address.
				for(int i=0; i<4; i++){
					*(pTxPacket+ipSrcOffset+i) = *(pRxPacket+ipDstOffset+i);				        
					*(pTxPacket+ipDstOffset+i) = *(pRxPacket+ipSrcOffset+i);
				}
				
				// Since not changing anything in the IP header there
				// is no need to do an IP checksum. Re-orders of address will not
				// effect the checksum.
									
				// Set the icmp echo reply.
				writeb(0, (char*)pTxPacket+etherLength+ipHeaderLength+1);
				
				// Changed packet so do icmp checksum.
				writew(0, (char*)pTxPacket+etherLength+ipHeaderLength+3);
				uint16 icmpSum = cksum((uchar*)pTxPacket+etherLength+ipHeaderLength+1, ipTotalLength-
																						ipHeaderLength);				
				
				// The writew will flip the byte order. For INTEL only. TBD
				writew(icmpSum, (char*)pTxPacket+etherLength+ipHeaderLength+3);				
							
				preparedPingReply = true;	
			}		
		
	return preparedPingReply;
}

bool IO::PreparePingRequest(uchar* pTxPacket, uint16* packetLength)
{	
	uint8 ipHeaderLength = 20; // Standard header no options.
	uint8 etherLength = 13; // 14 bytes to data, but since 0 offset adjust to 13.	
	uint8 dstMacAddress[5];
	uint16 ipTotalLength;	
	char* delimiter;
	char tmp[256];
		
	*packetLength = 42; // A standard packet 14-ether, 20-ip, 8-icmp
	*packetLength += m_payloadLength;	
	memset(pTxPacket, 0, *packetLength);
	ipTotalLength = *packetLength - (etherLength + 1);
	
	// Clean up hardware address.
	strcpy(tmp, m_dstMacAddress);
	char* next = tmp;
	int j = 0;	
	while ((delimiter = strchr(tmp, ':')) != NULL){		
		*delimiter = ' ';
		dstMacAddress[j] = (uint8)strtoul(next, NULL, 16);
		next = delimiter++;
		j++;
	}	
	dstMacAddress[5] = (uint8)strtoul(next, NULL, 16); // Last value
		
	for (int i=0; i<6; i++){		
		writeb(dstMacAddress[i], (char*)pTxPacket+i); // Hardware destination address.
		writeb(m_srcMacAddress[i], (char*)pTxPacket+6+i); // Hardware source address.
	}
		
	writew(htons(0x0800), (char*)pTxPacket+12); // Ethernet type.
	writeb(0x045, (char*)pTxPacket+etherLength+1); // Version and header length.
	writeb(0, (char*)pTxPacket+etherLength+2); // Type of service.
	writew(htons(ipTotalLength), (char*)pTxPacket+etherLength+3); // IP total length.
	writew(htons(999), (char*)pTxPacket+etherLength+5); // IP packet identifier. 
	writew(0, (char*)pTxPacket+etherLength+7); // Fragmentation offsets.
	writeb(0x0FF, (char*)pTxPacket+etherLength+9); // Time to live. (TTL)
	writeb(1, (char*)pTxPacket+etherLength+10); // Protocol (ICMP)
	
	// Fill in source and destination ip address.		
	int32 ipSrcOffset = etherLength+13;
	int32 ipDstOffset = etherLength+17;												
	writel(inet_addr(m_srcIPAddress), (char*)pTxPacket+ipSrcOffset);											
	writel(inet_addr(m_dstIPAddress), (char*)pTxPacket+ipDstOffset);
			
	writeb(8, (char*)pTxPacket+etherLength+ipHeaderLength+1); // ICMP type (ping request).
	writeb(0, (char*)pTxPacket+etherLength+ipHeaderLength+2); // ICMP code.
	writew(htons(m_txThread), (char*)pTxPacket+etherLength+ipHeaderLength+5); // Indentifier.
	writew(m_seqnum, (char*)pTxPacket+etherLength+ipHeaderLength+7); // Sequence number.
	
	// Do IP checksum.
	writew(0, (char*)pTxPacket+etherLength+11);
	uint16 ipSum = cksum((uchar*)pTxPacket+etherLength+1, ipHeaderLength);																													
	writew(ipSum, (char*)pTxPacket+etherLength+11);	
	
	// Do ICMP checksum.
	writew(0, (char*)pTxPacket+etherLength+ipHeaderLength+3);
	uint16 icmpSum = cksum((uchar*)pTxPacket+etherLength+ipHeaderLength+1, ipTotalLength -
																			ipHeaderLength);								
	writew(icmpSum, (char*)pTxPacket+etherLength+ipHeaderLength+3);			
			
	return true;
}

void IO::ProcessPingReply(uchar* pRxPacket, uint16* packetLength)
{ 	
	uint8 ipHeaderLength = 20; // Standard header no options.
	uint8 etherLength = 13; // 14 bytes to data, but since 0 offset adjust to 13.
	char buf[256];
	float bits, xfertime;
	
	*packetLength = 42; // A standard packet 14-ether, 20-ip, 8-icmp
	*packetLength += m_payloadLength;
	
	uint16 id = ntohs(readw((char*)pRxPacket+etherLength+ipHeaderLength+5)); // Indentifier.
	
	if (id == m_txThread){
	    m_timevalin = system_time();
		xfertime = (m_timevalin-m_timevalout);
		bits = (*packetLength)*2*8;
		sprintf(buf, "Ping round-trip %6.0f bits in %12.3f ms = %4.2f Kbits/s.", bits, xfertime/1000, bits/xfertime);
		m_parent->Print(buf);
		m_seqnum += 1;	
	}
	
	return;
}

// Dump the first 48 bytes of a packet. The output goes to the
// application's text window. This is a slow operation and effects
// performance so should be used when debugging slow network transactions.
void IO::DumpPacket(const char* msg, uchar* buf)
{
	char LineBuf[512];

	sprintf(LineBuf, "%s\n%2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x", msg,
	buf[0], buf[1], buf[2], buf[3], buf[4], buf[5], buf[6], buf[7],
	buf[8], buf[9], buf[10], buf[11], buf[12], buf[13], buf[14], buf[15]);
	
	m_parent->Print(LineBuf);

	sprintf(LineBuf, "%2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x",
	buf[16], buf[17], buf[18], buf[19], buf[20], buf[21], buf[22], buf[23],
	buf[24], buf[25], buf[26], buf[27], buf[28], buf[29], buf[30], buf[31]);

	m_parent->Print(LineBuf);
	
	sprintf(LineBuf, "%2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x %2.2x",
	buf[32], buf[33], buf[34], buf[35], buf[36], buf[37], buf[38], buf[39],
	buf[40], buf[41], buf[42], buf[43], buf[44], buf[45], buf[46], buf[47]);
	
	m_parent->Print(LineBuf);
}

//////////////////////////////////////////////////////////////////////
// Constructor and methods to save and get settings data from
// user and disk file.
IO::IO(Wnd* parent) 
{
    char driverPath[512];

	GetPersistantSettings();

    m_id = -1;
	m_parent = parent;
	m_SettingsPanelUp = false;
	m_NamePanelUp = true;
	m_seqnum = 0; // ICMP sequence number.
	m_payloadLength = PAYLOADLENGTH; 
	m_timeInterval = TIMEINTERVAL;
	m_txThread = -9;	 
	m_DriverNamePanel = new DriverNamePanel(this);
	sprintf(driverPath, "%s", m_DriverName);			
	m_DriverNamePanel->SetDriverName(driverPath);
}

void IO::GetPersistantSettings()
{
	BDirectory	dir;
	BEntry		entry;
	BPath		path;
    BFile* 		file;
    BMessage 	msg;
    char*		str;
    char* 		ver;
    uint32 		type;
    
   	find_directory(B_USER_SETTINGS_DIRECTORY, &path, true);
	path.Append("EDrive");	
	dir.SetTo(path.Path()); 
	if (dir.FindEntry("EDrive_data", &entry) == B_NO_ERROR) {
		file = new BFile(&entry, B_READ_WRITE);
		if (file->InitCheck() == B_NO_ERROR) {
			msg.Unflatten(file);
			
			// Make sure this is a valid data file
			if (msg.GetInfo("Version_info", &type) != B_OK)
				return;		
		
			// Make sure this is a valid format	
			msg.FindString("Version_info", (const char**) &ver);
			if (strcmp(ver, "V00.00") != 0)
				return;
				
			msg.FindString("driver_name", (const char**) &str);
			strcpy(m_DriverName, str);
			msg.FindString("src_ip", (const char**) &str);
			strcpy(m_srcIPAddress, str);
			msg.FindString("dst_ip", (const char**) &str);
			strcpy(m_dstIPAddress, str);
			msg.FindString("dst_mac", (const char**) &str);
			strcpy(m_dstMacAddress, str);
			msg.FindBool("display_packets", &m_showPackets);
			msg.FindBool("do_ping_reply", &m_pingReply);
			msg.FindBool("display_broadcasts", &m_showBroadcasts);
		} 
		else{
			strcpy(m_DriverName, "dev/net/Your driver name here.");
			strcpy(m_srcIPAddress, "10.192.99.2");
			strcpy(m_dstIPAddress, "10.192.99.3");
			strcpy(m_dstMacAddress, "00:00:c5:4f:00:34");
			m_showPackets = true;
			m_pingReply = true;
			m_showBroadcasts = true;
		}
	} 
	else{
		strcpy(m_DriverName, "/dev/net/Your driver name here.");
		strcpy(m_srcIPAddress, "10.192.99.2");
		strcpy(m_dstIPAddress, "10.192.99.3");
		strcpy(m_dstMacAddress, "00:00:c5:4f:00:34");
		m_showPackets = true;
		m_pingReply = true;
		m_showBroadcasts = true;
	}
}

void IO::SetPersistantSettings()
{
	BDirectory	dir;
	BEntry		entry;
	BPath		path;
    BFile* 		file;
    BMessage 	msg;
          			
   	// Window position and current profile 
   	find_directory(B_USER_SETTINGS_DIRECTORY, &path, true);
	path.Append("EDrive");	
	if (dir.SetTo(path.Path()) == B_ENTRY_NOT_FOUND)
		create_directory(path.Path(), 0777);
   	dir.SetTo(path.Path());
   	if (dir.FindEntry("EDrive_data", &entry) == B_NO_ERROR) {
		file = new BFile(&entry, B_READ_WRITE);
		if (file->InitCheck() == B_NO_ERROR) {
			msg.AddString("Version_info", "V00.00");
			msg.AddString("driver_name", m_DriverName);
			msg.AddString("src_ip", m_srcIPAddress);
			msg.AddString("dst_ip", m_dstIPAddress);
			msg.AddString("dst_mac", m_dstMacAddress);
			msg.AddBool("display_packets", m_showPackets);
			msg.AddBool("do_ping_reply", m_pingReply);
			msg.AddBool("display_broadcasts", m_showBroadcasts);
			
		
			msg.Flatten(file);
		}		
	}
	else {
		file = new BFile();
		if (dir.CreateFile("EDrive_data", file) == B_NO_ERROR) {
			msg.AddString("Version_info", "V00.00");
			msg.AddString("driver_name", m_DriverName);
			msg.AddString("src_ip", m_srcIPAddress);
			msg.AddString("dst_ip", m_dstIPAddress);
			msg.AddString("dst_mac", m_dstMacAddress);
			msg.AddBool("display_packets", m_showPackets);
			msg.AddBool("do_ping_reply", m_pingReply);
			msg.AddBool("display_broadcasts", m_showBroadcasts);
		
			msg.Flatten(file);		
		}
	}
}

// Put up settings panel to allow user to change default
// values.
void IO::ShowIOSettings()
{
    if (!m_SettingsPanelUp){
    	m_SettingsPanelUp = true;
		m_SettingsPanel = new SettingsPanel(this);
		m_SettingsPanel->SetSrcIPAddress(m_srcIPAddress);
		m_SettingsPanel->SetDstIPAddress(m_dstIPAddress);
		m_SettingsPanel->SetDstMacAddress(m_dstMacAddress);
		m_SettingsPanel->SetShowPackets(m_showPackets);
		m_SettingsPanel->SetPingReply(m_pingReply);
		m_SettingsPanel->SetShowBroadcasts(m_showBroadcasts);
	}		
}

// Abstract virtual from CallbackHandler, called from
// settings and name panel when cancel or done. 
void	
IO::Done(status_t status) 
{
	if (status >= B_NO_ERROR){
		if (m_SettingsPanelUp){
			m_SettingsPanel->GetSrcIPAddress(m_srcIPAddress);
			m_SettingsPanel->GetDstIPAddress(m_dstIPAddress);
			m_SettingsPanel->GetDstMacAddress(m_dstMacAddress);
			m_SettingsPanel->GetShowPackets(&m_showPackets);
			m_SettingsPanel->GetPingReply(&m_pingReply);
			m_SettingsPanel->GetShowBroadcasts(&m_showBroadcasts);
			
			m_SettingsPanelUp = false;	
		}
		
		if (m_NamePanelUp){		
			m_DriverNamePanel->GetDriverName(m_DriverName);
			Open();
			m_NamePanelUp = false;
		}
	}
	else{
		if (m_SettingsPanelUp)
			m_SettingsPanelUp = false;
			
		if (m_NamePanelUp)
			m_NamePanelUp = false;		
	}			
}










