/*
	bone_util.h
	
	networking utilities module
	
	Copyright 1999-2000, Be Incorporated, All Rights Reserved.
*/

#ifndef H_BONE_UTIL
#define H_BONE_UTIL

#include <BeBuild.h>
#include <SupportDefs.h>
#include <KernelExport.h>
#include <module.h>
#include <bone_data.h>
#include <ByteOrder.h>
#include <profiler.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
 * benaphore
 */
typedef struct bone_benaphore
{
	int32	atom;
	sem_id	sem;
} bone_benaphore_t;


/*
 * blocking fifo
 */


typedef struct bone_fifo
{
	char 				name[64];
	ssize_t 			max_bytes;
	size_t				current_bytes;
	bone_benaphore_t    read_guard;
	bone_benaphore_t    lock;
	sem_id				readsem;
	bone_data_t 		*head;
	bone_data_t 		*tail;
} bone_fifo_t;

struct bone_proto_info;


typedef struct bone_util_info
{
	struct module_info info;
	
	/*
	 * bone_data utilities
	 */
	bone_data_t	*(*new_data)(const void *copyfrom, uint32 len, uint32 addrlen);
	bone_data_t *(*clone_data)(bone_data_t *dat);
	void 		(*delete_data)(bone_data_t *bdt);
	void		*(*data_malloc)(bone_data_t *bdt, size_t size);
	int 		(*coalesce_data)(bone_data_t *to, bone_data_t *from, uint32 offset, uint32 bytes);
	int			(*prepend_data)(bone_data_t *bdt, const void *data, uint32 length, freefunc freethis);			 
	int 		(*append_data)(bone_data_t *bdt, const void *data, uint32 length, freefunc freethis);
	int 		(*insert_data)(bone_data_t *bdt, uint32 offset, const void *data, uint32 length, freefunc freethis);
	int 		(*copy_to_data)(bone_data_t *bdt, uint32 offset, const void *data, uint32 length);			 
	uint32 		(*get_data_iovecs)(const bone_data_t *bdt, struct iovec *iovs, int32 numiov);
	uint32 		(*count_data_iovecs)(const bone_data_t *bdt);
	uint32 		(*copy_from_data)(const bone_data_t *bdt, uint32 offset, void *copyinto, uint32 bytes);
	uint32 		(*remove_from_data)(bone_data_t *bdt, uint32 offset, void *copyinto, uint32 bytes);
	int 		(*map_to_data)(bone_data_t *bdt, uint32 offset, void **mystruct, uint32 mystructlen);
	int 		(*pop_data)(bone_data_t *bdt, uint32 offset, void **datap, uint32 datalen);
	uint16 		(*get_data_checksum)(const bone_data_t *bdt);
	uint32 		(*get_data_size)(const bone_data_t *bdt);
	int 		(*trim_data)(bone_data_t *bdt, uint32 offset, uint32 bytes);
	uint16      (*get_checksum)(void *buf, size_t len, uint16 prev_sum);
	uint16      (*incremental_checksum)(uint16 oldsum, uint16 oldval, uint16 newval);
	void        (*default_freefunc)(void *mem);

	/*
	 *	benaphore utilities
	 */
	int 			(*create_benaphore)(bone_benaphore_t *bbt, const char *name);
	void			(*delete_benaphore)(bone_benaphore_t *bbt);
	
	status_t		(*lock_benaphore)(bone_benaphore_t *bbt);
	void			(*unlock_benaphore)(bone_benaphore_t *bbt);

	
	/*
	 * fifo utilities.
	 */
	
	bone_fifo_t	*(*new_fifo)(const char *name, ssize_t max_bytes);
	void		(*delete_fifo)(bone_fifo_t *bft);
	status_t	(*enqueue_fifo_data)(bone_fifo_t *bft, bone_data_t *datum);
	ssize_t    	(*dequeue_fifo_data)(bone_fifo_t *bft, iovec *into, int numiov, bigtime_t timeout, struct sockaddr *addr, int addrlen);
	ssize_t    	(*peek_fifo_data)(bone_fifo_t *bft, iovec *into, int numiov, struct sockaddr *addr, int addrlen);
	bone_data_t *(*pop_fifo_data)(bone_fifo_t *bft, bigtime_t timeout);
	void		(*clear_fifo)(bone_fifo_t *bft);
	
	/*
	 *	masked compare and copy
	 */
	int 		(*masked_cmp)(const char *buf1, const char *buf2, const char *mask, int32 len);
	void		(*masked_copy)(char *to, const char *from, const char *mask, int32 len);
	
	/*
	 * config file functions
	 */
	
	status_t	(*get_outbound_protos)(int family, int type, int proto, struct bone_proto_info *stack[], int *numInStack);
	status_t	(*get_inbound_proto)(void **vcache, int32 family, int32 proto, struct bone_proto_info **protomod);
	int32		(*get_config_int32)(const char *key, int32 def);
	
	/*
	 * Allocator
	 */
	
	void		*(*falloc)(size_t size); /* fast mempool allocater; expected life < 100 us */
	void		*(*zalloc)(size_t size); /* zero'd mempool allocater */
	void		*(*malloc)(size_t size); /* zero'd general malloc() */
	void		(*free)(void *mem);
	
	/*
	 * profiler
	 */
	void 		(*enter_prof_func)(const char *file, const char *func, int flags);
	void 		(*exit_prof_func)(const char *file, const char *func, int flags);
	void 		(*log_data_path)(bone_data_t *data, const char *what);
	void		(*print_memuse)(void);

} bone_util_info_t;

#define BONE_UTIL_MOD_NAME "network/bone_util"
#define BONE_PEEPER_MOD_NAME "network/bone_peeper"


#ifdef __cplusplus
}
#endif


#endif /* H_BONE_UTIL */       
