// Copyright 1992-1999, Be Incorporated, All Rights Reserved.
/*
	This file may be used under the terms of the Be Sample Code License.
*/

#include <Bitmap.h>
#include <Button.h>
#include <Debug.h>
#include <Dragger.h>
#include <MenuItem.h>
#include <PopUpMenu.h>
#include <Window.h>

#include "MiniPulse.h"
//#include "iconfile.h"

extern const char *app_signature;
//const char *app_signature = "application/x-vnd.Be-MiniPulse";
// the application signature used by the replicant to find the supporting
// code

MiniPulse::MiniPulse(BRect frame, const char *name,
	rgb_color n_framecolor,
	rgb_color n_activecolor,
	rgb_color n_idlecolor,
	uint32 resizeMask, uint32 flags)
		: BView(frame, name, resizeMask, flags)
		, framecolor(n_framecolor)
		, activecolor(n_activecolor)
		, idlecolor(n_idlecolor)
{
	Update();
	Update();
	SetViewColor(200, 200, 200);
	BRect rect(Bounds());
	rect.OffsetTo(B_ORIGIN);
	rect.top = rect.bottom - 7;
	rect.left = rect.right - 7;
	AddChild(new BDragger(rect, this, B_FOLLOW_RIGHT | B_FOLLOW_BOTTOM));
}

MiniPulse::MiniPulse(BMessage *message)
	: BView(message)
{
	rgb_color *rgb;
	ssize_t size;
	status_t status;
	status = message->FindData("framecolor", B_RGB_COLOR_TYPE, (const void **)&rgb, &size);
	framecolor = status == B_NO_ERROR ? *rgb : default_framecolor;
	status = message->FindData("activecolor", B_RGB_COLOR_TYPE, (const void **)&rgb, &size);
	activecolor = status == B_NO_ERROR ? *rgb : default_activecolor;
	status = message->FindData("idlecolor", B_RGB_COLOR_TYPE, (const void **)&rgb, &size);
	idlecolor = status == B_NO_ERROR ? *rgb : default_idlecolor;
}


MiniPulse::~MiniPulse()
{
}

// archiving overrides
MiniPulse *
MiniPulse::Instantiate(BMessage *data)
{
	if (!validate_instantiation(data, "MiniPulse"))
		return NULL;
	return new MiniPulse(data);
}

status_t 
MiniPulse::Archive(BMessage *data, bool deep) const
{
	uint32 winresizingmode = ResizingMode();
	((BView*)this)->BView::SetResizingMode(B_FOLLOW_LEFT|B_FOLLOW_TOP);
	BView::Archive(data, deep);
	((BView*)this)->SetResizingMode(winresizingmode);

	data->AddString("add_on", app_signature);
	data->AddData("framecolor", B_RGB_COLOR_TYPE, &framecolor, sizeof(rgb_color));
	data->AddData("activecolor", B_RGB_COLOR_TYPE, &activecolor, sizeof(rgb_color));
	data->AddData("idlecolor", B_RGB_COLOR_TYPE, &idlecolor, sizeof(rgb_color));
	return B_NO_ERROR;
}


void 
MiniPulse::Draw(BRect rect)
{
	BView::Draw(rect);
	BRect bounds(Bounds());
		
	if(cpu_count > B_MAX_CPU_COUNT) {
		return;
	}
	if(cpu_count < 0) return;
	SetDrawingMode(B_OP_COPY);

	int h = (int)bounds.Height()-2;
	float top = 1;
	float bottom = top+h;
	float left = 1;
	float barwidth = (bounds.Width())/cpu_count-2;
	float right = barwidth+left;
	for(int i=0; i<cpu_count; i++) {
		int barsize = (int)(cputime[i] * (h+1));
		if (barsize > h)
			barsize = h;
		double rem = cputime[i] * (h+1) - barsize;
		//const rgb_color fractioncolor = {32, 64+128*rem, 32, 0};
		rgb_color fractioncolor;
#define FRACTIONCALC(comp) fractioncolor.comp = (uint8)(idlecolor.comp + rem * (activecolor.comp - idlecolor.comp))
		FRACTIONCALC(red);
		FRACTIONCALC(blue);
		FRACTIONCALC(green);
		FRACTIONCALC(alpha);
#undef FRACTIONCALC
		int idlesize = (int)(h-barsize);
		SetHighColor(framecolor);
		StrokeRect(BRect(left-1, top-1, right+1, bottom+1));
		if(idlesize > 0) {
			SetHighColor(idlecolor);
			FillRect(BRect(left, top, right, top+idlesize-1));
		}
		SetHighColor(fractioncolor);
		FillRect(BRect(left, bottom-barsize, right, bottom-barsize));
		if(barsize > 0) {
			SetHighColor(activecolor);
			FillRect(BRect(left, bottom-barsize+1, right, bottom));
		}
		left += barwidth+2;
		right += barwidth+2;
	}
}

void
MiniPulse::MessageReceived(BMessage *message)
{
	BView::MessageReceived(message);
}

void 
MiniPulse::AttachedToWindow()
{
	Window()->SetPulseRate(200000);
}

void 
MiniPulse::Update()
{
	system_info fSysInfo;
	get_system_info(&fSysInfo);
	
	bigtime_t now = system_time();
	//
	// Compute time spend in idle threads
	//
	cpu_count = fSysInfo.cpu_count;
	for(int i=0; i<cpu_count; i++ ) {
		double CPUTime = (double)(fSysInfo.cpu_infos[i].active_time -
					  prevactive[i]) / (now - prevtime);
		prevactive[i] = fSysInfo.cpu_infos[i].active_time;
		if(CPUTime < 0) CPUTime = 0;
		if(CPUTime > 1) CPUTime = 1;
		cputime[i] = CPUTime;
	}
	prevtime = now;
}

void 
MiniPulse::Pulse()
{
	Update();
	Draw(Bounds());
}

void
MiniPulse::MouseDown(BPoint /*point*/)
{
#if 0
	for (int32 count = 0; count < 20; count++) {
		uint32 mouseButtons;
		BPoint where;
		GetMouse(&where, &mouseButtons, true);
		if (!mouseButtons) {
			// button just clicked, play, pause or eject
			return;
		}
		snooze(10000);
	}
	ConvertToScreen(&point);
#endif
//	do not call inherited here, we do not wan't to block by tracking the button
}

