/*
 * ColorMap - approximate inverse colour mapping object
 *
 *  1996, Chris Herborth (chrish@qnx.com)
 *
 * $Id$
 *
 * Provide an approximate inverse colour mapping into the set of system
 * colours.
 *
 * Creating a ColorMap object gets the system colours and sets up the
 * approximate inverse colour map for those colours.
 *
 * To find the closest match in the system colour, call getColor() with
 * an rgb_color or set of RGB values.  To find the index into the system
 * colour map of the closest match, call getIndex().
 *
 * If the BeOS ever adds the ability to change the system colour map,
 * you'll have to call Refresh() when the map changes.
 *
 * This code is based on a technique described by Tim Kientzle in the
 * August 1996 C/C++ User's Journal ("Approximate Inverse Color Mapping").
 *
 */

#pragma once

#include <interface/InterfaceDefs.h>

struct ColorMapping {
	unsigned long map;
	unsigned int  index;
};

class ColorMap {
public:
	ColorMap();
	~ColorMap();

	// Refresh the internal colour map; not needed as long as BeOS doesn't
	// let anyone switch the system colour map.
	//
	// Returns B_NO_ERROR if successful, or B_NO_MEMORY if it was unable
	// to allocate memory for the mapping array.
	int Refresh( void );

	unsigned int getIndex( const rgb_color colour ) {
		return search( colour );
	}

	unsigned int getIndex( const unsigned int r,
						   const unsigned int g,
						   const unsigned int b ) {
		return search( r, g, b );
	}

	rgb_color getColor( const rgb_color colour ) {
		return sysColours[search( colour )];
	}

	rgb_color getColor( const unsigned int r,
					    const unsigned int g,
					    const unsigned int b ) {
		return sysColours[search( r, g, b )];
	}

private:
	unsigned long interleave( const rgb_color colour ) {
		return interleave( colour.red, colour.green, colour.blue );
	}

	unsigned long interleave( const unsigned int r,
							  const unsigned int g,
							  const unsigned int b ) {
		return ( (expand[r] << 2) | (expand[g] << 1) | expand[b] );
	}

	unsigned int search( const rgb_color colour ) {
		return search( colour.red, colour.green, colour.blue );
	}

	unsigned int search( const unsigned int r,
						 const unsigned int g,
						 const unsigned int b );

	unsigned long expand[256];	// Used by interleave().

	rgb_color    *sysColours;
	ColorMapping *mapping;
};
