// ColorMap - see colormap.h for more information
//
//  1996, Chris Herborth (chrish@qnx.com)

#include "ColorMap.h"

#include <support/Errors.h>
#include <stdlib.h>

// Prototype for a compare function to use with qsort() on the ColorMapping
// array.
static int compare_it( const void *thing1, const void *thing2 );

static int compare_it( const void *thing1, const void *thing2 )
{
	ColorMapping *t1 = (ColorMapping *)thing1;
	ColorMapping *t2 = (ColorMapping *)thing2;

	if( t1->map < t2->map ) {
		return -1;
	}
	if( t1->map > t2->map ) {
		return 1;
	}
	return 0;
}

ColorMap::ColorMap()
{
	// Create a table to help us expand the colours into digits.
	unsigned int i, j;
	unsigned long e;
	for( i = 0; i < 256; i++ ) {
		e = 0L;

		for( j = 0; j < 8; j++ ) {
			e |= ( i & ( 1 << j ) ) << ( 2 * j );
		}

		expand[i] = e;
	}

	Refresh();
}

ColorMap::~ColorMap()
{
	if( mapping ) {
		delete [] mapping;
	}
}

int ColorMap::Refresh( void )
{
	color_map *sysMap = system_colors();

	sysColours = sysMap->color_list;

	if( mapping ) {
		delete [] mapping;
	}
	mapping = new ColorMapping[256];
	if( mapping == NULL ) {
		return B_NO_MEMORY;
	}

	// Now we expand the colour tuples into single 24-bit integers.
	for( unsigned int i = 0; i < 256; i++ ) {
		mapping[i].map   = interleave( sysColours[i]  );
		mapping[i].index = i;
	}

	// Sort the colours by their expanded 24-bit representation.
	qsort( mapping, 256, sizeof( ColorMapping ), compare_it );

	return B_NO_ERROR;
}

unsigned int ColorMap::search( const unsigned int r,
							   const unsigned int g,
							   const unsigned int b )
{
	// Modified binary search to find the closest match to r, g, b in
	// the mapping array.

	unsigned long target = interleave( r, g, b );

	unsigned int start_pt = 0;		// Start of array.
	unsigned int end_pt   = 255;	// End of array.

	while( start_pt <= end_pt ) {
		unsigned int mid = ( start_pt + end_pt ) / 2;

		if( mapping[mid].map == target ) {
			return mapping[mid].index;
		}

		if( mid == start_pt || mid == end_pt ) {
			return mapping[mid].index;
		}

		if ( mapping[mid].map > target ) {
			end_pt = mid - 1;
		}

		if ( mapping[mid].map < target ) {
			start_pt = mid + 1;
		}
	}

	return mapping[start_pt].index;
}
