/*
 * Copyright (c) 1992, Brian Berliner and Jeff Polk
 * Copyright (c) 1989-1992, Brian Berliner
 * 
 * You may distribute under the terms of the GNU General Public License as
 * specified in the README file that comes with the CVS source distribution.
 * 
 * Created from log.c by Fred Fish (fnf@cygnus.com).
 * 
 * Generates patch kits from cvs log information for the specified files.
 * With no argument, generates patch kits for all the files in the current
 * directory, recursively by default.
 *
 * FIXME:
 *
 *   Currently generates a shell script that must be run after-the-fact.
 *   Modify to generate patchkits directly.
 *
 *   The code was created basically by hacking log.c to death, so parts
 *   of the remaining code may be irrelevant to what we are trying to do
 *   go back later and strip this out.
 *
 *   A '$' in the generated script, such as from a log message, will
 *   confuse the shell.  Fix that.
 *
 */

#include "cvs.h"
#include <ctype.h>

#ifndef isascii
# define isascii(c) (((c) & ~0x7f) == 0)	/* If C is a 7 bit value.  */
#endif

/* This structure holds the information about each revision that we need to collect while scanning the logs of the repository
   files, and save for use after the scan to create the patch kits.  The filename that this revision belongs to, and the
   associated log message, are both used as keys to locate this info, so we don't need to save copies of them in the struct. */

struct patchinfo
{
  char *version;
  char *date;
  char *author;
  char *state;
  int dead;
};

typedef struct patchinfo PInfo;

/* This structure holds information parsed from the -r option.  */

struct option_revlist
{
  /* The next -r option.  */

  struct option_revlist *next;

  /* The first revision to consider.  This is NULL if the range is :rev, or if no revision is given.  */

  char *first;

  /* The last revision to consider.  This is NULL if the range is rev:, or if no revision is given.  If there is no colon, first
     and last are the same.  */

  char *last;

  /* Nonzero if there was a trailing `.', which means to consider only the head revision of a branch.  */

  int branchhead;
};

/* This structure holds information derived from option_revlist given a particular RCS file.  */

struct revlist
{
  /* The next pair.  */

  struct revlist *next;

  /* The first numeric revision to consider.  */

  char *first;

  /* The last numeric revision to consider.  */

  char *last;

  /* The number of fields in these revisions (one more than numdots).  */

  int fields;
};

/* This structure holds information parsed from the -d option.  */

struct datelist
{
  /* The next date.  */

  struct datelist *next;

  /* The starting date.  */

  char *start;

  /* The ending date.  */

  char *end;

  /* Nonzero if the range is inclusive rather than exclusive.  */

  int inclusive;
};

/* This structure is used to pass information through start_recursion.  */
struct log_data
{
  /* Nonzero if the -b option was seen, meaning that only revisions on the default branch should be printed.  */

  int default_branch;

  /* If not NULL, the value given for the -r option, which lists sets of revisions to be printed.  */

  struct option_revlist *revlist;

  /* If not NULL, the date pairs given for the -d option, which select date ranges to print.  */

  struct datelist *datelist;

  /* If not NULL, the single dates given for the -d option, which select specific revisions to print based on a date.  */

  struct datelist *singledatelist;

  /* If not NULL, the list of states given for the -s option, which only prints revisions of given states.  */

  List *statelist;

  /* If not NULL, the list of login names given for the -w option, which only prints revisions checked in by given users.  */

  List *authorlist;
};

/* This structure is used to pass information through walklist.  */

struct log_data_and_rcs
{
  struct log_data *log_data;
  struct revlist *revlist;
  RCSNode *rcs;
};

/* The list of log messages, keyed by the message.  Each member of the
   list is another list, keyed by the filename, that holds the version
   info for each file that has the same log message.  There must be
   only one member in the secondary filename list for any given
   filename or else we have botched things. */

static List *logmsglist;

static Dtype log_dirproc PROTO ((void *callerdat, char *dir, char *repository, char *update_dir, List *entries));
static int pk_fileproc PROTO ((void *callerdat, struct file_info *finfo));
static struct option_revlist *log_parse_revlist PROTO ((const char *));
static void log_parse_date PROTO ((struct log_data *, const char *));
static void log_parse_list PROTO ((List **, const char *));
static struct revlist *pk_expand_revlist PROTO ((RCSNode *, struct option_revlist *, int));
static void log_free_revlist PROTO ((struct revlist *));
static int log_version_requested PROTO ((struct log_data *, struct revlist *, RCSNode *, RCSVers *));
static int log_fix_singledate PROTO ((Node *, void *));
static void log_tree PROTO ((struct file_info *finfo, struct log_data *, struct revlist *, RCSNode *, const char *));
static void log_abranch PROTO ((struct file_info *, struct log_data *, struct revlist *, RCSNode *, const char *));
static void log_version PROTO ((struct file_info *, struct log_data *, struct revlist *, RCSNode *, RCSVers *, int));
static int version_compare PROTO ((const char *, const char *, int));
static void save PROTO ((struct file_info *, RCSNode *, RCSVers *, char *key));
static void duplog PROTO ((RCSNode *, char *));
static void emitcmds PROTO ((void));

static const char *const patchkit_usage[] =
{
  "Usage: %s %s [-lb] [-r[revisions]] [-d dates] [-s states] [-w[logins]] [files...]\n",
  "\t-l\tLocal directory only, no recursion.\n",
  "\t-b\tOnly consider revisions on the default branch.\n",
  "\t-r[revisions]\tSpecify revision(s)s to consider.\n",
  "\t-d dates\tSpecify dates (D1<D2 for range, D for latest before).\n",
  "\t-s states\tOnly consider revisions with specified states.\n",
  "\t-w[logins]\tOnly consider revisions checked in by specified logins.\n",
  "(Specify the --help global option for a list of other help options)\n",
  NULL
};

int
patchkit (argc, argv)
     int argc;
     char **argv;
{
  int c;
  int err = 0;
  int local = 0;
  struct log_data log_data;
  struct option_revlist *rl, **prl;

  if (argc == -1)
    {
      usage (patchkit_usage);
    }

  memset (&log_data, 0, sizeof log_data);

  optind = 0;
  while ((c = getopt (argc, argv, "+bd:lr::s:w::")) != -1)
    {
      switch (c)
	{
	case 'b':
	  log_data.default_branch = 1;
	  break;
	case 'd':
	  log_parse_date (&log_data, optarg);
	  break;
	case 'l':
	  local = 1;
	  break;
	case 'r':
	  rl = log_parse_revlist (optarg);
	  for (prl = &log_data.revlist; *prl != NULL; prl = &(*prl)->next) {;}
	  *prl = rl;
	  break;
	case 's':
	  log_parse_list (&log_data.statelist, optarg);
	  break;
	case 'w':
	  if (optarg != NULL)
	    {
	      log_parse_list (&log_data.authorlist, optarg);
	    }
	  else
	    {
	      log_parse_list (&log_data.authorlist, getcaller ());
	    }
	  break;
	case '?':
	default:
	  usage (patchkit_usage);
	  break;
	}
    }

  wrap_setup ();

#ifdef CLIENT_SUPPORT
  if (client_active)
    {
      int i;

      /* We're the local client.  Fire up the remote server.  */
      start_server ();
	
      ign_setup ();

      for (i = 1; i < argc && argv[i][0] == '-'; i++)
	{
	  send_arg (argv[i]);
	}

      send_files (argc - i, argv + i, local, 0, SEND_NO_CONTENTS);
      send_file_names (argc - i, argv + i, SEND_EXPAND_WILD);

      send_to_server ("patchkit\012", 0);
      err = get_responses_and_close ();
      return (err);
    }
#endif

  err = start_recursion (pk_fileproc, (FILESDONEPROC) NULL, log_dirproc, (DIRLEAVEPROC) NULL, (void *) &log_data,
			 argc - optind, argv + optind, local, W_LOCAL | W_REPOS | W_ATTIC, 0, 1, (char *) NULL, 1);
  emitcmds ();
  return (err);
}

/*
 *  Parse a revision list specification and return pointer to the head
 *  of a linked list of revision specifications.  Each revision spec 
 *  contains pointers to strings that represent the first and last
 *  revision numbers in the range to consider, and a flag indicating
 *  if the last revision is the branch head revision.
 */

static struct option_revlist *
log_parse_revlist (argstring)
     const char *argstring;
{
  char *copy;
  struct option_revlist *ret, **pr;

  ret = NULL;
  pr = &ret;

  /* Copy the argument into memory so that we can change it.  We don't want to change the argument because, at least as of this
     writing, we will use it if we send the arguments to the server.  FIXME: We never bother to free up our copy.  */

  copy = xstrdup (argstring);
  while (copy != NULL)
    {
      char *comma;
      char *cp;
      char *first, *last;
      struct option_revlist *r;

      comma = strchr (copy, ',');
      if (comma != NULL)
	{
	  *comma++ = '\0';
	}

      first = copy;
      cp = strchr (copy, ':');
      if (cp == NULL)
	{
	  last = copy;
	}
      else
	{
	  *cp++ = '\0';
	  last = cp;
	}

      if (*first == '\0')
	{
	  first = NULL;
	}
      if (*last == '\0')
	{
	  last = NULL;
	}

      r = (struct option_revlist *) xmalloc (sizeof *r);
      r->next = NULL;
      r->first = first;
      r->last = last;
      if (first != last || first[strlen (first) - 1] != '.')
	{
	  r->branchhead = 0;
	}
      else
	{
	  r->branchhead = 1;
	  first[strlen (first) - 1] = '\0';
	}

      *pr = r;
      pr = &r->next;

      copy = comma;
    }

  return (ret);
}

/*
 * Parse a date specification.
 */

static void
log_parse_date (log_data, argstring)
     struct log_data *log_data;
     const char *argstring;
{
  char *orig_copy;
  char *copy;

  /* Copy the argument into memory so that we can change it.  We don't want to change the argument because, at least as of this
     writing, we will use it if we send the arguments to the server.  */

  copy = xstrdup (argstring);
  orig_copy = copy;
  while (copy != NULL)
    {
      struct datelist *nd, **pd;
      char *cpend, *cp, *ds, *de;

      nd = (struct datelist *) xmalloc (sizeof *nd);

      cpend = strchr (copy, ';');
      if (cpend != NULL)
	{
	  *cpend++ = '\0';
	}

      pd = &log_data->datelist;
      nd->inclusive = 0;

      if ((cp = strchr (copy, '>')) != NULL)
	{
	  *cp++ = '\0';
	  if (*cp == '=')
	    {
	      ++cp;
	      nd->inclusive = 1;
	    }
	  ds = cp;
	  de = copy;
	}
      else if ((cp = strchr (copy, '<')) != NULL)
	{
	  *cp++ = '\0';
	  if (*cp == '=')
	    {
	      ++cp;
	      nd->inclusive = 1;
	    }
	  ds = copy;
	  de = cp;
	}
      else
	{
	  ds = NULL;
	  de = copy;
	  pd = &log_data->singledatelist;
	}

      if (ds == NULL)
	{
	  nd->start = NULL;
	}
      else if (*ds != '\0')
	{
	  nd->start = Make_Date (ds);
	}
      else
	{
	  /* 1970 was the beginning of time, as far as get_date and
	     Make_Date are concerned.  FIXME: That is true only if time_t
	     is a POSIX-style time and there is nothing in ANSI that
	     mandates that.  It would be cleaner to set a flag saying
	     whether or not there is a start date.  */
	  nd->start = Make_Date ("1/1/1970 UTC");
	}

      if (*de != '\0')
	{
	  nd->end = Make_Date (de);
	}
      else
	{
	  /* We want to set the end date to some time sufficiently far
	     in the future to pick up all revisions that have been
	     created since the specified date and the time `cvs log'
	     completes.  FIXME: The date in question only makes sense
	     if time_t is a POSIX-style time and it is 32 bits
	     and signed.  We should instead be setting a flag saying
	     whether or not there is an end date.  Note that using
	     something like "next week" would break the testsuite (and,
	     perhaps less importantly, loses if the clock is set grossly
	     wrong).  */
	  nd->end = Make_Date ("2038-01-01");
	}

      nd->next = *pd;
      *pd = nd;

      copy = cpend;
    }

  free (orig_copy);
  return;
}

/*
 * Parse a comma separated list of items, and add each one to *PLIST.
 */

static void
log_parse_list (plist, argstring)
     List **plist;
     const char *argstring;
{
  while (1)
    {
      Node *p;
      char *cp;

      p = getnode ();
      cp = strchr (argstring, ',');
      if (cp == NULL)
	{
	  p->key = xstrdup (argstring);
	}
      else
	{
	  size_t len;

	  len = cp - argstring;
	  p->key = xmalloc (len + 1);
	  strncpy (p->key, argstring, len);
	  p->key[len + 1] = '\0';
	}
      if (*plist == NULL)
	{
	  *plist = getlist ();
	}
      if (addnode (*plist, p) != 0)
	{
	  freenode (p);
	}
      if (cp == NULL)
	{
	  break;
	}
      argstring = cp + 1;
    }
  return;
}

/*
 * Process the log information for one file.
 */

static int
pk_fileproc (callerdat, finfo)
     void *callerdat;
     struct file_info *finfo;
{
  struct log_data *log_data = (struct log_data *) callerdat;
  Node *p;
  RCSNode *rcsfile;
  char buf[256];
  struct revlist *revlist;
  struct log_data_and_rcs log_data_and_rcs;

  if ((rcsfile = finfo->rcs) == NULL)
    {
      /* no rcs file.  What *do* we know about this file? */
      p = findnode (finfo->entries, finfo->file);
      if (p != NULL)
	{
	  Entnode *e;
	    
	  e = (Entnode *) p->data;
	  if (e->version[0] == '0' && e->version[1] == '\0')
	    {
	      if (!really_quiet)
		{
		  error (0, 0, "%s has been added, but not committed",
			 finfo->file);
		}
	      return (0);
	    }
	}
	
      if (!really_quiet)
	{
	  error (0, 0, "nothing known about %s", finfo->file);
	}
	
      return (1);
    }

  /* We will need all the information in the RCS file.  */
  RCS_fully_parse (rcsfile);

  /* Turn any symbolic revisions in the revision list into numeric revisions.  */
  revlist = pk_expand_revlist (rcsfile, log_data->revlist, log_data->default_branch);

  /* If a symbolic revision was not found in this file, then ignore this file.
     This is different from what the cvs log command does. */
  if (revlist == NULL)
    {
      return (0);
    }


  if (rcsfile->access != NULL)
    {
      const char *cp;

      cp = rcsfile->access;
      while (*cp != '\0')
	{
	  const char *cp2;

	  cp2 = cp;
	  while (! isspace ((unsigned char) *cp2) && *cp2 != '\0')
	    ++cp2;
	  cp = cp2;
	  while (isspace ((unsigned char) *cp) && *cp != '\0')
	    ++cp;
	}
    }

  log_data_and_rcs.log_data = log_data;
  log_data_and_rcs.revlist = revlist;
  log_data_and_rcs.rcs = rcsfile;

  /* If any single dates were specified, we need to identify the revisions they select.  Each one selects the single revision,
     which is otherwise selected, of that date or earlier.  The log_fix_singledate routine will fill in the start date for each
     specific revision.  */

  if (log_data->singledatelist != NULL)
    {
      walklist (rcsfile->versions, log_fix_singledate, (void *) &log_data_and_rcs);
    }

  if (rcsfile->head != NULL)
    {
      p = findnode (rcsfile->versions, rcsfile->head);
      if (p == NULL)
	{
	  error (1, 0, "can not find head revision in `%s'", finfo->fullname);
	}
      while (p != NULL)
	{
	  RCSVers *vers;

	  vers = (RCSVers *) p->data;
	  log_version (finfo, log_data, revlist, rcsfile, vers, 1);
	  if (vers->next == NULL)
	    {
	      p = NULL;
	    }
	  else
	    {
	      p = findnode (rcsfile->versions, vers->next);
	      if (p == NULL)
		{
		  error (1, 0, "can not find next revision `%s' in `%s'", vers->next, finfo->fullname);
		}
	    }
	}
      log_tree (finfo, log_data, revlist, rcsfile, rcsfile->head);
    }

  /* Free up the new revlist and restore the old one.  */
  log_free_revlist (revlist);

  /* If singledatelist is not NULL, free up the start dates we added to it.  */
  if (log_data->singledatelist != NULL)
    {
      struct datelist *d;

      for (d = log_data->singledatelist; d != NULL; d = d->next)
	{
	  if (d->start != NULL)
	    {
	      free (d->start);
	    }
	  d->start = NULL;
	}
    }

  return (0);
}

/*
 * Fix up a revision list in order to compare it against versions.
 * Expand any symbolic revisions.
 */

static struct revlist *
pk_expand_revlist (rcs, revlist, default_branch)
     RCSNode *rcs;
     struct option_revlist *revlist;
     int default_branch;
{
  struct option_revlist *r;
  struct revlist *ret, **pr;

  ret = NULL;
  pr = &ret;
  for (r = revlist; r != NULL; r = r->next)
    {
      struct revlist *nr;

      nr = (struct revlist *) xmalloc (sizeof *nr);
      if (r->first == NULL && r->last == NULL)
	{
	  /* If both first and last are NULL, it means that we want
	     just the head of the default branch, which is RCS_head.  */
	  nr->first = RCS_head (rcs);
	  nr->last = xstrdup (nr->first);
	  nr->fields = numdots (nr->first) + 1;
	}
      else if (r->branchhead)
	{
	  char *branch;

	  /* Print just the head of the branch.  */
	  if (isdigit ((unsigned char) r->first[0]))
	    nr->first = RCS_getbranch (rcs, r->first, 1);
	  else
	    {
	      branch = RCS_whatbranch (rcs, r->first);
	      if (branch == NULL)
		{
		  error (0, 0, "warning: `%s' is not a branch in `%s'", r->first, rcs->path);
		  free (nr);
		  continue;
		}
	      nr->first = RCS_getbranch (rcs, branch, 1);
	      free (branch);
	    }
	  if (nr->first == NULL)
	    {
	      error (0, 0, "warning: no revision `%s' in `%s'", r->first, rcs->path);
	      free (nr);
	      return (NULL);
	    }
	  nr->last = xstrdup (nr->first);
	  nr->fields = numdots (nr->first) + 1;
	}
      else
	{
	  if (r->first == NULL || isdigit ((unsigned char) r->first[0]))
	    {
	      nr->first = xstrdup (r->first);
	    }
	  else
	    {
	      if (RCS_nodeisbranch (rcs, r->first))
		{
		  nr->first = RCS_whatbranch (rcs, r->first);
		}
	      else
		{
		  nr->first = RCS_gettag (rcs, r->first, 1, (int *) NULL);
		}
	      if (nr->first == NULL)
		{
		  error (0, 0, "warning: no revision `%s' in `%s'", r->first, rcs->path);
		  free (nr);
		  return (NULL);
		}
	    }

	  if (r->last == r->first)
	    {
	      nr->last = xstrdup (nr->first);
	    }
	  else if (r->last == NULL || isdigit ((unsigned char) r->last[0]))
	    {
	      nr->last = xstrdup (r->last);
	    }
	  else
	    {
	      if (RCS_nodeisbranch (rcs, r->last))
		{
		  nr->last = RCS_whatbranch (rcs, r->last);
		}
	      else
		{
		  nr->last = RCS_gettag (rcs, r->last, 1, (int *) NULL);
		}
	      if (nr->last == NULL)
		{
		  error (0, 0, "warning: no revision `%s' in `%s'", r->last, rcs->path);
		  if (nr->first != NULL)
		    {
		      free (nr->first);
		    }
		  free (nr);
		  return (NULL);
		}
	    }

	  /* Process the revision numbers the same way that rlog does.  This code is a bit cryptic for my tastes, but keeping
	     the same implementation as rlog ensures a certain degree of compatibility.  */

	  if (r->first == NULL)
	    {
	      nr->fields = numdots (nr->last) + 1;
	      if (nr->fields < 2)
		{
		  nr->first = xstrdup (".0");
		}
	      else
		{
		  char *cp;

		  nr->first = xstrdup (nr->last);
		  cp = strrchr (nr->first, '.');
		  strcpy (cp, ".0");
		}
	    }
	  else if (r->last == NULL)
	    {
	      nr->fields = numdots (nr->first) + 1;
	      nr->last = xstrdup (nr->first);
	      if (nr->fields < 2)
		{
		  nr->last[0] = '\0';
		}
	      else
		{
		  char *cp;

		  cp = strrchr (nr->last, '.');
		  *cp = '\0';
		}
	    }
	  else
	    {
	      nr->fields = numdots (nr->first) + 1;
	      if (nr->fields != numdots (nr->last) + 1
		  || (nr->fields > 2
		      && version_compare (nr->first, nr->last, nr->fields - 1) != 0))
		{
		  error (0, 0, "invalid branch or revision pair %s:%s in `%s'", r->first, r->last, rcs->path);
		  free (nr->first);
		  free (nr->last);
		  free (nr);
		  continue;
		}
	      if (version_compare (nr->first, nr->last, nr->fields) > 0)
		{
		  char *tmp;

		  tmp = nr->first;
		  nr->first = nr->last;
		  nr->last = tmp;
		}
	    }
	}

      nr->next = NULL;
      *pr = nr;
      pr = &nr->next;
    }

  /* If the default branch was requested, add a revlist entry for
     it.  This is how rlog handles this option.  */
  if (default_branch && (rcs->head != NULL || rcs->branch != NULL))
    {
      struct revlist *nr;

      nr = (struct revlist *) xmalloc (sizeof *nr);
      if (rcs->branch != NULL)
	{
	  nr->first = xstrdup (rcs->branch);
	}
      else
	{
	  char *cp;

	  nr->first = xstrdup (rcs->head);
	  cp = strrchr (nr->first, '.');
	  *cp = '\0';
	}
      nr->last = xstrdup (nr->first);
      nr->fields = numdots (nr->first) + 1;

      nr->next = NULL;
      *pr = nr;
    }

  return (ret);
}

/*
 * Free a revlist created by pk_expand_revlist.
 */

static void
log_free_revlist (revlist)
     struct revlist *revlist;
{
  struct revlist *r;

  r = revlist;
  while (r != NULL)
    {
      struct revlist *next;

      if (r->first != NULL)
	{
	  free (r->first);
	}
      if (r->last != NULL)
	{
	  free (r->last);
	}
      next = r->next;
      free (r);
      r = next;
    }
  return;
}

/*
 * Return nonzero if a given revision should appear in the patchkit, based on the options provided.
 */

static int
log_version_requested (log_data, revlist, rcs, vnode)
     struct log_data *log_data;
     struct revlist *revlist;
     RCSNode *rcs;
     RCSVers *vnode;
{
  /* Handle the list of states from the -s option.  */
  if (log_data->statelist != NULL && findnode (log_data->statelist, vnode->state) == NULL)
    {
      return (0);
    }

  /* Handle the list of authors from the -w option.  */
  if (log_data->authorlist != NULL)
    {
      if (vnode->author != NULL && findnode (log_data->authorlist, vnode->author) == NULL)
	{
	  return (0);
	}
    }

  /* rlog considers all the -d options together when it decides whether to print a revision, so we must be compatible.  */

  if (log_data->datelist != NULL || log_data->singledatelist != NULL)
    {
      struct datelist *d;

      for (d = log_data->datelist; d != NULL; d = d->next)
	{
	  int cmp;

	  cmp = RCS_datecmp (vnode->date, d->start);
	  if (cmp > 0 || (cmp == 0 && d->inclusive))
	    {
	      cmp = RCS_datecmp (vnode->date, d->end);
	      if (cmp < 0 || (cmp == 0 && d->inclusive))
		{
		  break;
		}
	    }
	}

      if (d == NULL)
	{
	  /* Look through the list of specific dates.  We want to select the revision with the exact date found in the start
	     field.  The commit code ensures that it is impossible to check in multiple revisions of a single file in a single
	     second, so checking the date this way should never select more than one revision.  */
	  for (d = log_data->singledatelist; d != NULL; d = d->next)
	    {
	      if (d->start != NULL && RCS_datecmp (vnode->date, d->start) == 0)
		{
		  break;
		}
	    }

	  if (d == NULL)
	    {
	      return (0);
	    }
	}
    }

  /* If the -r or -b options were used, REVLIST will be non NULL, and we print the union of the specified revisions.  */

  if (revlist != NULL)
    {
      char *v;
      int vfields;
      struct revlist *r;

      /* This code is taken from rlog.  */
      v = vnode->version;
      vfields = numdots (v) + 1;
      for (r = revlist; r != NULL; r = r->next)
	{
	  if (vfields == r->fields + (r->fields & 1)
	      && version_compare (v, r->first, r->fields) >= 0
	      && version_compare (v, r->last, r->fields) <= 0)
	    {
	      return (1);
	    }
	}

      /* If we get here, then the -b and/or the -r option was used, but did not match this revision, so we reject it.  */

      return (0);
    }

  /* By default, we accept all revisions.  */
  return (1);
}

/*
 * Sort out a single date specification by narrowing down the date
 * until we find the specific selected revision.
 */

static int
log_fix_singledate (p, closure)
     Node *p;
     void *closure;
{
  struct log_data_and_rcs *data = (struct log_data_and_rcs *) closure;
  Node *pv;
  RCSVers *vnode;
  struct datelist *holdsingle, *holddate;
  int requested;

  pv = findnode (data->rcs->versions, p->key);
  if (pv == NULL)
    {
      error (1, 0, "missing version `%s' in RCS file `%s'", p->key, data->rcs->path);
    }
  vnode = (RCSVers *) pv->data;

  /* We are only interested if this revision passes any other tests.  Temporarily clear log_data->singledatelist to avoid
     confusing log_version_requested.  We also clear log_data->datelist, because rlog considers all the -d options together.  We
     don't want to reject a revision because it does not match a date pair if we are going to select it on the basis of the
     singledate.  */

  holdsingle = data->log_data->singledatelist;
  data->log_data->singledatelist = NULL;
  holddate = data->log_data->datelist;
  data->log_data->datelist = NULL;
  requested = log_version_requested (data->log_data, data->revlist, data->rcs, vnode);
  data->log_data->singledatelist = holdsingle;
  data->log_data->datelist = holddate;

  if (requested)
    {
      struct datelist *d;

      /* For each single date, if this revision is before the
	 specified date, but is closer than the previously selected
	 revision, select it instead.  */
      for (d = data->log_data->singledatelist; d != NULL; d = d->next)
	{
	  if (RCS_datecmp (vnode->date, d->end) <= 0 && (d->start == NULL || RCS_datecmp (vnode->date, d->start) > 0))
	    {
	      if (d->start != NULL)
		{
		  free (d->start);
		}
	      d->start = xstrdup (vnode->date);
	    }
	}
    }

  return (0);
}

/*
 * Print the list of changes, not including the trunk, in reverse
 * order for each branch.
 */

static void
log_tree (finfo, log_data, revlist, rcs, ver)
     struct file_info *finfo;
     struct log_data *log_data;
     struct revlist *revlist;
     RCSNode *rcs;
     const char *ver;
{
  Node *p;
  RCSVers *vnode;

  p = findnode (rcs->versions, ver);
  if (p == NULL)
    {
      error (1, 0, "missing version `%s' in RCS file `%s'", ver, rcs->path);
    }
  vnode = (RCSVers *) p->data;
  if (vnode->next != NULL)
    {
      log_tree (finfo, log_data, revlist, rcs, vnode->next);
    }
  if (vnode->branches != NULL)
    {
      Node *head, *branch;

      /* We need to do the branches in reverse order.  This breaks the List abstraction, but so does most of the branch
	 manipulation in rcs.c.  */

      head = vnode->branches->list;
      for (branch = head->prev; branch != head; branch = branch->prev)
	{
	  log_abranch (finfo, log_data, revlist, rcs, branch->key);
	  log_tree (finfo, log_data, revlist, rcs, branch->key);
	}
    }
  return;
}

/*
 * Log the changes for a branch, in reverse order.
 */

static void
log_abranch (finfo, log_data, revlist, rcs, ver)
     struct file_info *finfo;
     struct log_data *log_data;
     struct revlist *revlist;
     RCSNode *rcs;
     const char *ver;
{
  Node *p;
  RCSVers *vnode;

  p = findnode (rcs->versions, ver);
  if (p == NULL)
    {
      error (1, 0, "missing version `%s' in RCS file `%s'", ver, rcs->path);
    }
  vnode = (RCSVers *) p->data;
  if (vnode->next != NULL)
    {
      log_abranch (finfo, log_data, revlist, rcs, vnode->next);
    }
  log_version (finfo, log_data, revlist, rcs, vnode, 0);
  return;
}

/*
 *  Collect the log information for a single version.
 */

static void
log_version (finfo, log_data, revlist, rcs, ver, trunk)
     struct file_info *finfo;
     struct log_data *log_data;
     struct revlist *revlist;
     RCSNode *rcs;
     RCSVers *ver;
     int trunk;
{
  Node *p;
  int year, mon, mday, hour, min, sec;
  char buf[16 * 1024];			/* FIXME: Possible overrun */
  char buf2[16];
  Node *padd, *pdel;

  if (! log_version_requested (log_data, revlist, rcs, ver))
    {
      return;
    }

  p = findnode (RCS_getlocks (rcs), ver->version);
  (void) sscanf (ver->date, SDATEFORM, &year, &mon, &mday, &hour, &min, &sec);
  if (year < 1900)
    {
      year += 1900;
    }
  sprintf (buf, "%04d/%02d/%02d %02d:%02d:%02d", year, mon, mday, hour, min, sec);

  if (! trunk)
    {
      padd = findnode (ver->other, ";add");
      pdel = findnode (ver->other, ";delete");
    }
  else if (ver->next == NULL)
    {
      padd = NULL;
      pdel = NULL;
    }
  else
    {
      Node *nextp;
      RCSVers *nextver;

      nextp = findnode (rcs->versions, ver->next);
      if (nextp == NULL)
	{
	  error (1, 0, "missing version `%s' in `%s'", ver->next, rcs->path);
	}
      nextver = (RCSVers *) nextp->data;
      pdel = findnode (nextver->other, ";add");
      padd = findnode (nextver->other, ";delete");
    }

  p = findnode (ver->other, "log");

  /* The p->data == NULL case is the normal one for an empty log message (rcs-14 in sanity.sh).  I don't think the case where
     p->data is "" can happen (getrcskey in rcs.c checks for an empty string and set the value to NULL in that case).  My guess
     would be the p == NULL case would mean an RCS file which was missing the "log" keyword (which is illegal according to
     rcsfile.5).  */

  if (p == NULL || p->data == NULL || p->data[0] == '\0')
    {
      p->data = xstrdup ("*** empty log message ***");
    }
  strncpy (buf2, ver->date, 8);
  buf2[8] = '\000';
  if ((strlen (ver -> author) + strlen (p -> data)) > (sizeof (buf) - 128))
    {
      error (0, 0, "Internal error, buf too small at %s:%d", __FILE__, __LINE__);
      return;
    }
  sprintf (buf, "Date: %s\nAuthor: %s\n%s", buf2, ver->author, p->data);
  save (finfo, rcs, ver, buf);
  return;
}

/*
 * Print a warm fuzzy message
 */

/* ARGSUSED */
static Dtype
log_dirproc (callerdat, dir, repository, update_dir, entries)
     void *callerdat;
     char *dir;
     char *repository;
     char *update_dir;
     List *entries;
{
  if (!isdir (dir))
    {
      return (R_SKIP_ALL);
    }

  if (!quiet)
    {
      error (0, 0, "Examining %s", update_dir);
    }
  return (R_PROCESS);
}

/*
 * Compare versions.  This is taken from RCS compartial.
 */

static int
version_compare (v1, v2, len)
     const char *v1;
     const char *v2;
     int len;
{
  while (1)
    {
      int d1, d2, r;

      if (*v1 == '\0')
	{
	  return (1);
	}
      if (*v2 == '\0')
	{
	  return (-1);
	}

      while (*v1 == '0')
	++v1;
      for (d1 = 0; isdigit ((unsigned char) v1[d1]); ++d1)
	{
	  ;
	}

      while (*v2 == '0')
	{
	  ++v2;
	}
      for (d2 = 0; isdigit ((unsigned char) v2[d2]); ++d2)
	{
	  ;
	}

      if (d1 != d2)
	{
	  return (d1 < d2 ? -1 : 1);
	}

      r = memcmp (v1, v2, d1);
      if (r != 0)
	{
	  return (r);
	}

      --len;
      if (len == 0)
	{
	  return (0);
	}

      v1 += d1;
      v2 += d1;

      if (*v1 == '.')
	{
	  ++v1;
	}
      if (*v2 == '.')
	{
	  ++v2;
	}
    }
}

static char *
shortmsg (key)
     char *key;
{
  char *msg;
  static char buf[128];

  for (msg = key ; *msg && !isascii (*msg); msg++) {;}
  strncpy (buf, msg, sizeof (buf));
  msg = &buf[sizeof(buf)-5];
  *msg++='.';
  *msg++='.';
  *msg++='.';
  *msg++='\000';
  for (msg = buf; *msg; msg++)
    {
      if (*msg == '\n')
	{
	  if (*(msg + 1) == '\000')
	    {
	      *msg = '\000';
	    }
	  else
	    {
	      *msg = '-';
	    }
	}
      if (*msg == '\t')
	{
	  *msg = ' ';
	}
    }
  return (buf);
}

static void duplog (rcs, key)
     RCSNode *rcs;
     char *key;
{
  error (0, 0, "ignoring duplicate log message `%s' in RCS file `%s'", shortmsg (key), rcs->path);
  return;
}

static void
save (finfo, rcs, ver, key)
     struct file_info *finfo;
     RCSNode *rcs;
     RCSVers *ver;
     char *key;
{
  Node *p;
  List *flist;
  PInfo *pinfo;

  if (logmsglist == NULL)
    {
      logmsglist = getlist ();
    }
  p = findnode (logmsglist, key);
  if (p == NULL)
    {
      p = getnode ();
      p->type = RCSVERS;	/* FIXME */
      p->key = xstrdup (key);	/* Is xstrdup really needed? */
      p->data = (char *) getlist();
      (void) addnode (logmsglist, p);
    }
  flist = (List *) p->data;
  p = findnode (flist, finfo->fullname);
  if (p != NULL)
    {
      duplog (rcs, key);
    }
  else
    {
      /* First create and populate the patch info struct */
      pinfo = (PInfo *) xmalloc (sizeof (PInfo));
      memset (pinfo, 0, sizeof (PInfo));
      pinfo -> version = xstrdup (ver -> version);
      pinfo -> date = xstrdup (ver -> date);
      pinfo -> author = xstrdup (ver -> author);
      pinfo -> state = xstrdup (ver -> state);
      pinfo -> dead = ver -> dead;
      /* Now add it to our save data set */
      p = getnode ();
      p->type = RCSVERS;			/* FIXME */
      p->key = xstrdup (finfo->fullname);
      p->data = (char *) pinfo;
      (void) addnode (flist, p);
    }
  return;
}

/* Don't overload the server with rapid requests.  It can
   hang the inet daemon, at least under linux. */
#define DELAY if(server_active)cvs_output("sleep 2\n", 0);

static void
emitcmds ()
{
  List *mlist;
  Node *mhead;
  Node *mnode;
  List *flist;
  Node *fhead;
  Node *fnode;
  PInfo *pinfo;
  int lastrev;
  char *sp;
  char *patchdate;
  char baserev[256];	/* FIXME: possible overrun */
  char buf[256];	/* FIXME: possible overrun */
  
  if (logmsglist != NULL)
    {
      mhead = logmsglist -> list;
      for (mnode = mhead -> next; mnode != mhead; mnode = mnode -> next)
	{
	  flist = (List *) mnode -> data;
	  fhead = flist -> list;
	  pinfo = (PInfo *) fhead -> next -> data;
	  patchdate = pinfo -> date;
	  sprintf (buf, "cat >patch-%s <<\\EOF\n", patchdate);
	  cvs_output (buf, 0);
	  cvs_output (mnode -> key, 0);
	  cvs_output ("\nEOF\n", 0);
	  for (fnode = fhead -> next; fnode != fhead; fnode = fnode -> next)
	    {
	      pinfo = (PInfo *) fnode -> data;
	      if (pinfo -> dead)
		{
		  sprintf (buf, "cvs diff -cN -r %s %s >>patch-%s\n", pinfo -> version, fnode -> key, patchdate);
		  cvs_output (buf, 0);
		  DELAY;
		}
	      else
		{
		  sp = strrchr (pinfo->version, '.');
		  lastrev = atoi (++sp);
		  strncpy (baserev, pinfo->version, sp - pinfo->version);
		  baserev[sp - pinfo->version] = '\000';
		  if (lastrev == 1)
		    {
		      /* Lowest rev on a branch, diff against the branchpoint. */
		      sp = strrchr (baserev, '.');
		      *sp = '\000';
		      sp = strrchr (baserev, '.');
		      if (sp == NULL)
			{
			  /* must be 1.1, fake it */
			  strcpy (baserev, "0");
			}
		      else
			{
			  /* Truncate another revision to get to branchpoint revision. */
			  *sp = '\000';
			}
		    }
		  else
		    {
		      /* Generate a revision number on this branch that is one less than this one. */
		      sprintf (&baserev[sp - pinfo->version], "%d", lastrev - 1);
		    }
		  if (baserev[0] == '0' && baserev[1] == '\000')
		    {
		      sprintf (buf, "cvs -q update -p -r %s %s >/tmp/cvs-$$-temp\n", pinfo -> version, fnode -> key);
		      cvs_output (buf, 0);
		      DELAY;
		      sprintf (buf, "echo 'Index: %s' >>patch-%s\n", fnode -> key, patchdate);
		      cvs_output (buf, 0);
		      sprintf (buf, "echo '===================================================================' >>patch-%s\n", patchdate);
		      cvs_output (buf, 0);
		      sprintf (buf, "echo 'diff -N %s' >>patch-%s\n", fnode -> key, patchdate);
		      cvs_output (buf, 0);
		      sprintf (buf, "diff -cp /dev/null /tmp/cvs-$$-temp >>patch-%s\n", patchdate);
		      cvs_output (buf, 0);
		    }
		  else
		    {
		      sprintf (buf, "cvs diff -cpN -r %s -r %s %s >>patch-%s\n", baserev, pinfo->version, fnode -> key, patchdate);
		      cvs_output (buf, 0);
		      DELAY;
		    }
		}
	    }
	}
    }
  return;
}
