/*****************************************************************************
//
//	File:		3dFaceBody.h
//
//	Description:	The base class representing a fully defined polygonal object
//
//	Copyright 1997, Be Incorporated
//
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
//***************************************************************************/

#ifndef _3D_FACE_H
#define _3D_FACE_H

#include <unistd.h>

#include "3dBody.h"
#include "3dFaceModel.h"
#include "3dFaceLook.h"
#include "3dMaterial.h"
#include "3dPointLook.h"

/*	This describes a plane with a normal to the plane and any arbitrary point in the plane */
struct B3dPlane {
	B3dVector normal;
	B3dVector pointInPlane;
};

/*	Completely describes a single independent face.  Note that all the contents of
	this should be considered READ-ONLY.  The exception is when using ProcessFaces;
	the material may be changed by filters called from an invocation of this
	method. */
struct B3dFaceFilterDesc {
	int16		numPts;				/* 	Number of points in the face */
	B3dVector		*faceNormal;			/*	The face's normal vector */
	B3dVector		*points;				/*	The point locations */
	B3dVector		*normals;				/*	The vertex normals */
	bool			hasPointLooks;		/*	true if point looks are defined */
	B3dPointLook	*pointLooks;			/*	Point look structures per point
										NOTE: this is only valid if hasPointLooks
										is true */
	B3dMaterial	*material;			/*	The face's material */
};

enum {
	FACE_IS_IN		= 0x00000001,
	STOP_RECURSION	= 0x00000002,
	
	MUST_SHARE_1		= 1,
	MUST_SHARE_2		= 2
};

typedef uint32 (*surface_filter)(void *userInfo, B3dFaceFilterDesc *in, 
		B3dFaceFilterDesc* candidate, int32 neighborsIn);
/*	Function type which can be passed to GatherSurface.  It takes the description
	of a face which is known to be part of the surface and a description of a candidate
	face.  If the candidate face should be included in the surface extent, the returned
	bitmask should include FACE_IS_IN.  If the returned bitmask contains STOP_RECURSION,
	the neighbors of this face which are not already part of the surface are not
	considered as candidates.  "userInfo" is specified by the call to GatherSurface. 
	neighborsIn is the number of neighbors this face has (so far) that are part of the
	surface.  Said another way, it's the number of times this face has been tested,
	including this one.
	
	IMPORTANT: The B3dFaceFilterDesc should be considered READ-ONLY for this call. */

typedef uint32 (*point_func)(void *randomInfo, B3dVector *thePoint);
/* The point can be modified by this call */

typedef uint32 (*normal_func)(void *randomInfo, B3dVector *theNormal, B3dVector *thePoint);
/*	The normal and point can both be modified by this call */

typedef uint32 (*pointlook_func)(void *randomInfo, B3dPointLook *thePointLook, 
	B3dVector *theNormal, B3dVector *thePoint);
/*	The point look, normal, and point can all be modified by this call.
	IMPORTANT: if ProcessPointLooks is called with a selection containing undefined point looks
	(i.e. any point look used by a material which does not have per-point information, which
	right now if anything which is not texture mapped), thePointLook WILL BE NULL.  This is
	because there is no information stored for point looks that are undefined.  Therefore, 
	be sure the material is set for texture mapped surfaces BEFORE setting point looks. */

typedef uint32 (*face_func)(void *randomInfo, B3dFaceFilterDesc *desc);

/*	A triangle within a clump.  "normals" is ignored for clumps without normal
	lists, but must be defined otherwise.  "pointLooks" must only be defined
	for triangles which specify a material which requires point looks.  */
struct B3dClumpFace {
	int32		points[3];
	int32		normals[3];
	
	int32		pointLooks[3];
	int32		material;

	bool			backVisible;
	int32		backPointLooks[3];
	int32		backMaterial;
};

/*	A B3dClump is a way of describing a set of interconnected
	faces, points, normals, point looks, and materials.  It
	follows no specific rules about what can share what.  It
	can be used to fully describe a polygonal object to the
	B3dFace constructor.
	
	It is legal for the "normals" list, the "pointLooks" list, or
	both to be empty.  Normals will be generated if needed.
	However, if a material is specified which needs point looks
	and no point looks exist for that face, the point looks
	generated will be undefined in value.  */
class B3dClump {
	
	public:

						B3dClump();
						~B3dClump();

	BArray<B3dVector>		points;
	BArray<B3dVector>		normals;
	BArray<B3dPointLook>	pointLooks;
	BArray<B3dMaterial>	materials;
	BArray<B3dClumpFace>	faces;
};

struct B3dHandle;
class HandleList;
class ElementList;
class B3dSelection;
class B3dImporter;
class Bitmap;
class B3dPackedPointLook;
class B3dPackedMaterial;

/*	
	A selection is a collection of elements.  Each one of those elements
	is considered to be "grabbed" by one or more "handles".  There is only a single handle
	for each face, but points, normals and point looks can have an arbitrary number of handles,
	one for each face that references that element.  Thus, a point shared between three faces
	can be grabbed by up to three handles.  Which handles an element is grabbed by affects the
	operation of many calls, with the notable exception of the Read... and Write... calls.
	
	Selections can have 
*/

class B3dFaceBody : public B3dBody
{

	public:
	
						B3dFaceBody(			char *name,
											B3dWorld *world,
											B3dClump *clump=NULL,
									        uint32 status = NULL);
						B3dFaceBody(			char *name,
											B3dWorld *world,
											B3dImporter *importer,
									        uint32 status = NULL);
						B3dFaceBody(			B3dFaceBody *face,
											char *name,
											ulong clone_flag);
virtual 					~B3dFaceBody();
virtual 	B3dThing			*Clone(				char *name = 0L,
											ulong clone_flag = 0);

virtual	void 			Draw(				B3dLighter *	lighter,
											B3dLens *		projector,
											B3dRenderer *	renderer,
											B3dVector *	offset0,
											B3dMatrix *	rot,
											B3dSharedState *buf);

virtual void				TouchDown(			B3dTouchDesc *	touch,
											B3dVector *		touchOnScreen,
											uint32			buttons);
	
virtual void				TouchMoved(			B3dTouchDesc *	touch,
											B3dVector *		touchOnScreen,
											uint32			buttons);
		/*	TouchDown and TouchMoved override the default B3dThing behavior in the
			case of special materials.  If a touch impacts on a face on which is
			textured a B3dOffView, the B3dOffView is informed of the touch on it's
			screen, and the touch "falls through" into the camera space of that
			B3dOffView (this may mean it falls through a wormhole into a different
			universe).  TouchDown does the actual fallthrough, and TouchMoved
			keeps the B3dOffView updated about the current intersection of the
			touch with it's screen. */

virtual	bool				GetTouchInfo(			B3dVector *	origin,
											B3dVector *	axis,
											B3dVector *	touch,
											void **		personalInfo);
virtual	void				FreeTouchInfo(		void *		personalInfo);
		/*	These are the two methods needed to participate in the B3dView's default touch behavior.
			GetTouchInfo calls SurfaceIntersection to find the touch point, and allocates a new
			selection containing the touched face and returns it all the "personalInfo" pointer.
			"personalInfo" gets put in the filed of the same name in the B3dTouchDesc, so
			TouchDown and TouchMoved can then refer to it.  FreeTouchInfo is called when the
			buttons is released, and it frees the selection. */

virtual	void				AssumeDefaultLook(		B3dFaceModel *model, bool isSmooth=true);
		/*	This creates a boring default look to use with the given model, and starts using
			that look and model. */

		/* Here is the public manipulation API. */

		/* Selection manipulations */
		selection_id		NewSelection();
		status_t			FreeSelection(		selection_id selection);
		status_t			ClearSelection(		selection_id selection);
		status_t			CopySelection(		selection_id copyInto,
											selection_id copyFrom=B_OBJECT_SELECTION_ID,
											uint32 selectionTypes=B_3DSELECT_ALL,
											int32 elementNumber=B_ALL_ELEMENTS);
		status_t			CombineSelection(		selection_id source1Selection,
											selection_id source2Selection,
											selection_id destinationSelection,
											int32 operation);

		int32			CountElements(		selection_id selection,
											int32 selectionTypes=B_3DSELECT_ALL);
		status_t			RemoveElement(		selection_id selection,
											int32 elementNumber,
											int32 selectionTypes=B_3DSELECT_ALL);
		/*	Generic element couting and removal (from selection).  Only the specified types
			are counted, and in removing elements the nth element which fits the type mask
			is removed. */

		status_t			GrabByAllHandles(		selection_id sourceSelection,
											selection_id destinationSelection);
		/*	This selects into destinationSelection the exact same elements as are in sourceSelection,
			except that the new selection's elements are all grabbed by all possible handles. */

		status_t			Split(				selection_id selection);
		/*	This ensures that for each element in the given selection, the handles I have grabbed
			that element by are the only ones which exist to that element.  If any other handles to
			the element exist, they are given a separate copy of the element.  Note that this may cause
			other selections to gain elements: if a selection has grabbed an element by two handles
			and those handles end up on opposite sides of a split, that selection now contains
			two elements grabbed by one handle each. */

		status_t			Merge(				selection_id selection,
											int32 types=B_3DSELECT_ALL,
											int32 mergeRules=B_MERGE_AVERAGED);
		/*	This is the opposite of Split.  It takes all of the elements in the given selection
			and welds them together.  They must all by the same type, or an error results.  Note
			that after this operation, the selection will contain one element only, grabbed by
			the union of all the handles that all the individual elements were grabbed by before
			the call.  Got that?  Other selections which contain elements involved in the merge
			will be updated properly. */

		status_t			SelectAtPoint(		B3dVector *v,
											selection_id selectInto,
											int32 selectionTypes=B_3DSELECT_POINTS,
											selection_id selectFrom=B_OBJECT_SELECTION_ID,
											int32 maxNumElements=1,
											float maxDistance=-1.0);
		status_t			SelectInPlanes(		B3dPlane *planes,
											int32 numPlanes,
											selection_id selectInto,
											int32 selectionTypes=B_3DSELECT_POINTS,
											selection_id selectFrom=B_OBJECT_SELECTION_ID,
											int32 faceCriteria=3);
		status_t			SelectInBox(			B3dVector *origin,
											B3dVector *x, B3dVector *y, B3dVector *z,
											selection_id selectInto,
											int32 selectionTypes=B_3DSELECT_POINTS,
											selection_id selectFrom=B_OBJECT_SELECTION_ID,
											int32 faceCriteria=3);
		status_t			SelectInPyramid(		B3dVector *origin,
											B3dVector *ray1, B3dVector *ray2,
											B3dVector *ray3, B3dVector *ray4,
											selection_id selectInto,
											int32 selectionTypes=B_3DSELECT_POINTS,
											selection_id selectFrom=B_OBJECT_SELECTION_ID,
											int32 faceCriteria=3);
		/*	These calls allow location-oriented selection.  All coordinates are in object space,
			and point looks and normals for the purposes of these calls are considered to have
			the location of their associated point.  For SelectInPlanes, SelectInBox and
			SelectInPyramid, 'faceCriteria' is the number of points of a face which must be
			within the given region for the face to be considered in it.  For SelectInPlanes,
			a point/face is in the region if it falls on the positive (normal facing) side of
			the plane for each plane in the list of planes given (the lenght of which is given
			by 'numPlanes'. */

		status_t			SurfaceIntersection(	B3dVector *origin, B3dVector *ray,
											selection_id selectInto,
											B3dVector *touchPoint,
											int32 nthIntersection=0);
		/*	This finds the point of intersection (in object space) of the
			given ray with the object's faces.  nthIntersection specifies which
			intersection along the ray to return if there are more than one, and
			the face which the ray passes through is selected into selectInto */
		
		int32			CountPoints(			selection_id selection);
		status_t			RemovePoint(			selection_id selection,
											int32 pointNumber);
		/*	Count the number of points in the selection, or remove the Nth point (from the selection) */

		status_t			ProcessPoints(		selection_id pointSelection,
											point_func pointProcessFunc,
											void *userData=NULL,
											selection_id destinationSelection=B_NO_SELECTION_ID);
		/*	This calls pointProcessFunc once for each point in the selection.  If destinationSelection
			is not B_NO_SELECTION_ID, it selects any point which pointProcessFunc returns non-zero
			for into destinationSelection.  This method can thus be used as a selection-to-selection
			filter, or as a batch transform of the points, etc. */

		status_t			ReadPoints(			selection_id selection, 
											B3dVector *point,
											int32 whichPoint=B_ALL_POINTS);
		status_t			WritePoints(			selection_id selection,
											B3dVector *point,
											int32 whichPoint=B_ALL_POINTS);
		/*	Gets or sets the location of one or all points selected in selection.  For WritePoints,
			if whichPoint is B_ONE_TO_ALL then "point" should point to a single B3dVector location which all
			points in the selection should be set to.  Otherwise, the mapping is one-to-one. */

		status_t			GrabPointNormals(		selection_id destinationSelection,
											selection_id pointSelection=B_OBJECT_SELECTION_ID,
											int32 whichPoint=B_ALL_POINTS);
		status_t			GrabPointPointLooks(	selection_id destinationSelection,
											selection_id pointSelection=B_OBJECT_SELECTION_ID,
											int32 whichPoint=B_ALL_POINTS);
		status_t			GrabPointFaces(		selection_id destinationSelection,
											selection_id pointSelection=B_OBJECT_SELECTION_ID,
											int32 whichPoint=B_ALL_POINTS);
		/*	Selects into destinationSelection all the normals, point looks, or faces associated with
			a point. Point looks and normals are grabbed only by the cooresponding handle that the given
			point(s) is/are grabbed by. */

		int32			CountNormals(			selection_id selection);
		status_t			RemoveNormal(			selection_id selection,
											int32 normalNumber);
		/*	Count the number of normals in the selection, or remove the Nth normal (from the selection) */

		status_t			ProcessNormals(		selection_id normalSelection,
											normal_func normalProcessFunc,
											void *userData,
											selection_id destinationSelection=B_NO_SELECTION_ID);
		/*	This works just as ProcessPoints */

		status_t			ReadNormals(			selection_id selection,
											B3dVector *normal,
											int32 whichNormal=B_ALL_NORMALS);
		status_t			WriteNormals(			selection_id selection,
											B3dVector *normal,
											int32 whichNormal=B_ALL_NORMALS);
		/*	Gets or sets the vector of one or all vertex normals selected in selection.  For WriteNormals,
			if whichNormal is B_ONE_TO_ALL then "normal" should point to a single B3dVector vector which all
			normals in the selection should be set to.  Otherwise, the mapping is one-to-one. */

		status_t			GenerateNormals(		selection_id selection,
											int32 whichNormal=B_ALL_NORMALS);
		/*	This is a handy routine to automatically generate surface normal values.  It generates
			a guess for each normal based on which handles it is grabbed by.  It assumes that the
			surface is smooth over the faces it is grabbed with relation to, and ignores any other
			faces, regardless of whether or not other faces share it. */

		status_t			GrabNormalPoints(		selection_id destinationSelection,
											selection_id pointSelection=B_OBJECT_SELECTION_ID,
											int32 whichNormal=B_ALL_NORMALS);
		status_t			GrabNormalPointLooks(	selection_id destinationSelection,
											selection_id pointSelection=B_OBJECT_SELECTION_ID,
											int32 whichNormal=B_ALL_NORMALS);
		status_t			GrabNormalFaces(		selection_id destinationSelection,
											selection_id pointSelection=B_OBJECT_SELECTION_ID,
											int32 whichNormal=B_ALL_NORMALS);
		/*	Selects into destinationSelection all the points, point looks, or faces associated with a normal.
			Points and point looks are grabbed only by the cooresponding handle that the given normal(s)
			is/are grabbed by. */

		int32			CountPointLooks(		selection_id selection);
		status_t			RemovePointLook(		selection_id selection,
											int32 pointLookNumber);
		/*	Count the number of point looks in the selection, or remove the Nth point look (from the selection) */

		status_t			ProcessPointLooks(		selection_id pointLookSelection,
											pointlook_func pointLookProcessFunc,
											void *userData=NULL,
											selection_id destinationSelection=B_NO_SELECTION_ID);
		/*	This works just as ProcessPoints, however, note that comment at the pointlook_func definition
			that undefined point looks will be reported as NULL. */

		status_t			ReadPointLooks(		selection_id selection,
											B3dPointLook *pointLook,
											int32 whichPointLook=B_ALL_POINTLOOKS);
		status_t			WritePointLooks(		selection_id selection,
											B3dPointLook *pointLook,
											int32 whichPointLook=B_ALL_POINTLOOKS);
		/*	Gets or sets the point look of one or all vertex normals selected in selection.  For WritePointLooks,
			if whichPointLook is B_ONE_TO_ALL then "pointLook" should point to a single B3dPointLook which all
			point looks in the selection should be set to.  Otherwise, the mapping is one-to-one. */

		status_t			GrabPointLookPoints(	selection_id destinationSelection,
											selection_id pointLookSelection=B_OBJECT_SELECTION_ID,
											int32 whichPointLook=B_ALL_POINTLOOKS);
		status_t			GrabPointLookNormals(	selection_id destinationSelection,
											selection_id pointLookSelection=B_OBJECT_SELECTION_ID,
											int32 whichPointLook=B_ALL_POINTLOOKS);
		status_t			GrabPointLookFaces(	selection_id destinationSelection,
											selection_id pointLookSelection=B_OBJECT_SELECTION_ID,
											int32 whichPointLook=B_ALL_POINTLOOKS);
		/*	Selects into destinationSelection all the points, normals, or faces associated with a point look.
			Points and normals are grabbed only by the the cooresponding handle that the given point look(s)
			is/are grabbed by. */

		int32			CountFaces(			selection_id selection);
		status_t			RemoveFace(			selection_id selection,
											int32 faceNumber);
		/*	Count the number of faces in the selection, or remove the Nth face (from the selection) */

		status_t			ProcessFaces(			selection_id faceSelection,
											face_func faceProcessFunc,
											void *userData=NULL,
											selection_id destinationSelection=B_NO_SELECTION_ID);
		/*	This works just as ProcessPoints */

		status_t			ReadFaceNormals(		selection_id faceSelection,
											B3dVector *faceNormals,
											int32 whichFace=B_ALL_FACES);
		/*	This gets the normal of the face.  The face normal is not directly writable, but rather
			is a characteristic determined by the locations of the face's points. */
		
		status_t			ReadFaceMaterials(		selection_id selection,
											B3dMaterial *material,
											int32 whichFace=B_ALL_FACES);
		status_t			WriteFaceMaterials(	selection_id selection,
											B3dMaterial *material,
											int32 whichFace=B_ALL_FACES);
		/*	Gets or sets the material of one or all faces selected in selection.  For WriteFaceMaterials,
			if whichMaterial is B_ONE_TO_ALL then material should point to a single material which all
			faces in the selection should be set to.  Otherwise, the mapping of material to face is one-to-one. */

		status_t			GrabFacePoints(		selection_id destinationSelection,
											selection_id faceSelection=B_OBJECT_SELECTION_ID,
											int32 whichFace=B_ALL_FACES);
		status_t			GrabFaceNormals(		selection_id destinationSelection,
											selection_id faceSelection=B_OBJECT_SELECTION_ID,
											int32 whichFace=B_ALL_FACES);
		status_t			GrabFacePointLooks(	selection_id destinationSelection,
											selection_id faceSelection=B_OBJECT_SELECTION_ID,
											int32 whichFace=B_ALL_FACES);
		/*	Selects into destinationSelection all the points, normals, or point looks used by a face.
			These points, normals, or point looks are grabbed only by the handle associated with this face */

		status_t			GrabFaceBack(			selection_id destinationSelection,
											selection_id faceSelection=B_OBJECT_SELECTION_ID,
											int32 whichFace=B_ALL_FACES);
		/*	Grabs a handle to the back side of the specified face(s).  Remember that B_OBJECT_SELECTION_ID
			only contains handles to the fronts of faces. */
			
		status_t			GatherSurface(		selection_id startingFace,
											selection_id filterInto,
											int32 maxRecursions=-1,
											surface_filter filterFunction=NULL,
											void *userData=NULL,
											int32 neighborRule=MUST_SHARE_2);
		/*	This is a general-purpose surface selection routine.  It takes an initial face (if there is
			more than one face in startingFace it uses the first) and calls filterFunction on all
			of that face's neighbors.  Any of the neighbors that pass (see the description of
			surface_filter, above) are selected into filterInto. This is then applied recursively to
			the neighbors.  maxRecursions specifies the maximum number of times to recurse.  
			randomData is passed directly to the face_filter routine, and should hold any
			state data you want passed to it.  neighborRule is either MUST_SHARE_1 or MUST_SHARE_2.
			MUST_SHARE_1 means that only one point must be shared between faces for them to
			be considered neighbors, while MUST_SHARE_2 means that they must share an actual
			edge.  If faceFilter is NULL, the test always passes, and the call simply selects
			all faces within maxResursions hops from the original face. (Clarifications:
			"startingFace" is always a member of the resulting selection; the default value of
			maxRecursions==-1 will recurse until all connected faces are visited) */

		status_t			AddClump(				B3dClump *clump);
		/*	The creation API.  Short, huh? */

		status_t			ExtractClump(			B3dClump *clump,
											selection_id clumpSelection=B_OBJECT_SELECTION_ID);
		/*	This will take clumpSelection and generate a clump which describes it.  No optimization
			of the clump is done, if you want an optimal clump then call Optimize before calling this. */

		status_t			Optimize(				int32 level);
		/* Optimize the object for memory and speed.  See 3dDefs.h for a description of optimization levels. */

		status_t 			AlignColinearEdges();
		/*	This is a fast routine to fix cases in which a large triangle shares a single edge
			with two smaller triangles.  This can cause rendering errors because the exact
			projection of the midpoint of the edge (when rendering the two smaller triangles)
			can be different from the interpolated edge as rendered by the larger triangle.
			This routine splits the larger triangle in two and shares the midpoint between
			them.  This is probably only useful when you're importing files from non-picky
			rendering formats, but it _is_ useful then. */

/*****************/
/* LOW-LEVEL API */
/*****************/

		/*	This is stuff to interface selections with the process of creating an object.
			For instance, I could create a model and have very specific surfaces or points
			on it that I wish to be able to use later as selections.  Because I created the
			object, I know the exact indices of the faces or points, and don't want to worry
			about the high level process of selecting the faces by location, or criteria,
			or whatever, I just want a selection I can refer to later.  These are methods to
			allow that.  These should not, in general, be used at some later time after object
			creation (once the selection API might have been used on it), because indices
			may have changed from when the object was created.  Therefore, you'll probably
			want to use this only at object creation time.  See the Book demo for an
			example of how to use this API. */

		status_t			SelectFaceIndices(		selection_id faceSelection,
											int32 *faceIndices,
											int32 numFaceIndices);
		/*	Select the given face indices into faceSelection.  The indices should be
			into the face lists of the model (indices into B3dFaceModel::m_faces and 
			B3dFaceModel::m_advancedFaces (if applicable)). */

		status_t			SelectPointIndices(	selection_id pointSelection,
											int32 *pointIndices,
											int32 numPointIndices);
		/*	Select the given face indices into pointSelection. The indices should be
			into the shared point list of the model (indices into m_points). All handles
			for each index are grabbed and placed into pointSelection. */

		status_t			SelectNormalIndices(	selection_id normalSelection,
											int32 *normalIndices,
											int32 numNormalIndices);
		/*	Select the given normal indices into normalSelection. The indices should be
			into the shared normal list of the model (indices into m_norms).  All handles
			for each index are grabbed and placed into normalSelection.  Note that
			if the model level is B_FACE_MODEL (i.e. the model has no vertex normals)
			this call is invalid, and will return an error. */

		B3dVector *		PointList();
		B3dVector *		NormalList();
		/*	These give direct access to the shared point and normal lists.  If the
			model doesn't have vertex normals, NormalList will return NULL. */

		status_t			RecalcFaceNormals(		int32 whichFace=B_ALL_FACES);		
		/*	This will recalculate the face normals of one or all faces.  This should be
			used after mucking with the point list directly. */

	private:

		B3dSelection *	m_freeSelections;
		B3dSelection *	m_activeSelections;
		int32			m_ownership;
		int32			m_state;
		int32			m_optimizationState;

		void				Construct(B3dClump *clump);
		
		void				FixupHandles(HandleList &handles, int32 flags=1);
		int32			ElementLength(HandleList &handles, int32 start);
		void				SplitHandle(B3dHandle h);
		int32			Handle2Index(B3dHandle h);
		bool				HandlesShare(B3dHandle h1, B3dHandle h2);
		void				BuildElementList(	HandleList &handles, 
										ElementList &elements,
										int32 types=B_3DSELECT_ALL,
										struct ElementDesc **pointers=NULL);
		void				CollectHandles(int32 type, int32 index, int32 isBack, HandleList *handles);
		bool				UpgradeModel(int32 upgradeTo);
		bool				UpgradeLook(int32 upgradeTo);
		int32			GetHandleCount(selection_id selection, int32 types);
		B3dHandle			GetNthHandle(selection_id selection, int32 n, int32 types);
		int32			GetHasForPointLook(B3dHandle h);
		status_t			AnalyzeClump(	B3dClump *	clump,
									int32 *		lookFlagsP,
									int32 *		modelLevelP,
									int32 **		point2NormalMap,
									int32 **		point2PointLookMap,
									bool *		mustReorderNormals);
		status_t			UnoptimizeFaceNormals();
		status_t			UnoptimizeNormals();
		status_t			UnoptimizePointLooks();
		status_t			AutoCalcVertexNormals(int32 whichNormals);
		status_t			BuildBitmaps(	Bitmap *points,
									Bitmap *normals,
									Bitmap *pointLooks,
									Bitmap *materials);
		bool				FillFaceIndices(int32 faceIndex, bool back, struct FaceIndices *fi);
		status_t			ProcessFacesI(selection_id faceSelection,
									face_func faceProcessFunc,
									void *userData,
									selection_id destinationSelection,
									int32 whichFace);
		status_t			ProcessThings(	selection_id selection,
										void *processFunc,
										void *userData,
										selection_id destinationSelection,
										int32 whichThing,
										int32 type);
		status_t			Component2Component(	selection_id destinationSelection,
											selection_id componentSelection,
											int32 whichComponent,
											int32 fromType,
											int32 toType);
		status_t			Face2Components(	selection_id destinationSelection,
										selection_id faceSelection,
										int32 whichFace,
										int32 type);
		status_t			SplitInternal(	HandleList &toSplit, int32 toSplitStart, int32 toSplitEnd,
										HandleList &theRest, int32 theRestStart, int32 theRestEnd,
										int32 type, int32 index, int32 isBack, bool smallSet,
										bool pointsAreSame);
		status_t			AllHandles(		HandleList &allHandles, int32 types);
		status_t			CollectRelevantHandles(HandleList &globals, ElementList &el, bool sort=true);
inline B3dPackedPointLook*	PLI2PPL(int32 index);
inline B3dPackedMaterial*	MI2PM(int32 index);
};

/* These are convenience inlines */

inline int32 B3dFaceBody::CountPoints(selection_id selection)
{
	return CountElements(selection,B_3DSELECT_POINTS);
};

inline status_t B3dFaceBody::RemovePoint(selection_id selection, int32 n)
{
	return RemoveElement(selection,n,B_3DSELECT_POINTS);
};

inline int32 B3dFaceBody::CountNormals(selection_id selection)
{
	return CountElements(selection,B_3DSELECT_NORMALS);
};

inline status_t B3dFaceBody::RemoveNormal(selection_id selection, int32 n)
{
	return RemoveElement(selection,n,B_3DSELECT_NORMALS);
};

inline int32 B3dFaceBody::CountFaces(selection_id selection)
{
	return CountElements(selection,B_3DSELECT_FACES);
};

inline status_t B3dFaceBody::RemoveFace(selection_id selection, int32 n)
{
	return RemoveElement(selection,n,B_3DSELECT_FACES);
};

inline int32 B3dFaceBody::CountPointLooks(selection_id selection)
{
	return CountElements(selection,B_3DSELECT_POINTLOOKS);
};

inline status_t B3dFaceBody::RemovePointLook(selection_id selection, int32 n)
{
	return RemoveElement(selection,n,B_3DSELECT_POINTLOOKS);
};

#endif
