/*****************************************************************************
//
//	File:		3dSpin.h
//
//	Description:	Class representing a volume of rotation
//
//	Copyright 1997, Be Incorporated
//
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
//***************************************************************************/

#ifndef _3D_SPIN_H
#define _3D_SPIN_H

#include <unistd.h>

#ifndef _3D_FACE_H 
#include "3dFace.h"
#endif
#ifndef _3D_SPIN_MODEL_H 
#include "3dSpinModel.h"
#endif

/*
	This structure defines a "spin" object.  A spin object is
	simply defined as a volume of rotation.  "shape" is the
	two-dimensional curve (defined by "shapeCount" B2dVectors) which
	is to be rotated around it's Y axis.  The X coordinate of each
	2d vector is the distance from the axis, and the Y coordinate is
	the distance along the axis.  Thus, the first and last X
	coordinate should always be 0.
	
	NOTE: Certain nice optimizations can be made only if the Y
	coordinate increases from one point to the next.  If this
	does not hold, the object will be created but will not be as
	fast to draw.  For example, consecutive values of Y resulting in
	a fast-to-draw object might be : -0.6, -0.4, -0.2, -0.2, 0.1,
	0.3, 0.5.
	
	The resolution of the spin object is partially determined by
	"radiusCount" which specifies the number of times to duplicate
	"shape" around it's Y axis.

	If "texture" is non-null, then texture coordinates are generated
	for the object and the given texture is mapped over it's surface.
	"textureRepeatX" is the number of times the texture repeats in one
	revolution around the object, and "textureRepeatY" is the number of
	times the texture repeats from the "top" of the object to the
	"bottom" (i.e. first Y to last Y).  If "textureDirectY", the texture
	Y coordinates are determined by the equation:
	
		texY = textureRepeatY * (((maxY-curveY) + minY) / (maxY-minY))
	
	where curveY is the current point on the curve, and minY and maxY
	are the minimum and maximum Y coordinates that the curve assumes.
	Otherwise, the Y texture coordinate is calculated by:
	
		texY = textureRepeatY * t
		
	where t is the normalized distance along the curve from the first
	point to the last.
*/

struct B3dSpinDesc {
	int32		shapeCount;
	B2dVector 	*shape;
	int32		radiusCount;
	B3dChannel	*texture;
	bool			textureDirectY;
	int32		textureRepeatX;
	float		textureRepeatY;
};

/**************************************
// B3dSpin.
/*************************************/

class B3dSpin : public B3dFaceBody {

/*************************************
// HIGH-LEVEL API                   */

 public:
	B3dSpin(	char			*name,
			B3dWorld    *world,
			B3dSpinDesc	*desc,
			int32		creationFlags);
	/*	Creates a spin object named 'name' and placed in
		the universe 'uni'.  The object is constructed as
		specified by "desc" (to format of which is detailed
		above).
		
		if (creationFlags & B_OWN_MODEL_DESC) then the
		constructor will free the "shape" element of
		"desc" and will also free "desc" itself. */

	virtual ~B3dSpin();
};

#endif
