//*****************************************************************************
//
//	File:		 3dmovView.cpp
//
//	Description: 3d demo constructor using the 3d Kit.
//
//	Copyright 1996, Be Incorporated
//
//*****************************************************************************

#ifndef _3D_MOV_VIEW_H
#include "3dmovView.h"
#endif
#ifndef _3D_WORLD_H
#include "3dWorld.h"
#endif
#ifndef _3D_SPIN_H
#include "3dSpin.h"
#endif
#ifndef _3D_CUBE_H
#include "3dCube.h"
#endif
#ifndef _3D_STD_LINKS_H
#include "3dStdLinks.h"
#endif
#ifndef _3D_RADIAL_LENS_H
#include "3dRadialLens.h"
#endif
#ifndef _3D_MOV_APP_H
#include "3dmovApp.h"
#endif
#ifndef _3D_MATH_LIB_H
#include <3dMathLib.h>
#endif

#include <math.h>
#include <Debug.h>



/**********************************************
/*  Cube.
/* This is a special class overloading 3dCube
/* to define a few special properties.
/*********************************************/

class Z3dCube : public B3dCube {
public:	
	Z3dCube(char *name, B3dWorld *world, B3dVector *size);
	~Z3dCube();
	virtual void MessageReceived(BMessage *message, B3dTouchDesc *touch);

private:
	selection_id    tmp;
	selection_id	faces[6];
};

static B3dPointLook looksA[3] = {
	{ 1.0, 0.0, 0.0 },
	{ 0.0, 1.0, 0.0 },
	{ 1.0, 1.0, 0.0 }
};

static B3dPointLook looksB[3] = {
	{ 1.0, 0.0, 0.0 },
	{ 0.0, 0.0, 0.0 },
	{ 0.0, 1.0, 0.0 }
};

Z3dCube::Z3dCube(char *name, B3dWorld *world, B3dVector *size)
: B3dCube(name, world, size) {
	int           i;
	int32         listIndice[2];
	B3dMaterial   material;

	// allocate a working selection
	tmp = NewSelection();
	// set the default texture on all faces
	material.SetTexture(default_texture);
	material.SetProperties(material.Properties() & ~B_MATERIAL_IS_SMOOTH);
	WriteFaceMaterials(B_OBJECT_SELECTION_ID, &material, B_ONE_TO_ALL_FACES);
	// set the point look for standard mapping.
	for (i=0; i<12; i++) {
		GrabFacePointLooks(tmp, B_OBJECT_SELECTION_ID, i);
		WritePointLooks(tmp, (i&1)?looksB:looksA);
	}
	// create 6 selections representing the 6 'faces' of the cube.
	for (i=0; i<6; i++) {
		listIndice[0] = i*2;
		listIndice[1] = i*2+1;
		faces[i] = NewSelection();
		SelectFaceIndices(faces[i], listIndice, 2);
	}
}

Z3dCube::~Z3dCube() {
	int       i;
	
	for (i=0; i<6; i++)
		FreeSelection(faces[i]);
	FreeSelection(tmp);
}

void Z3dCube::MessageReceived(BMessage *msg, B3dTouchDesc *touch) {
	int          i;
	entry_ref	 *tref;
	
	// Check if the drag&drop touches oen of the selection. If it's the
	// case, call the application giving a description of the file dragged,
	// the object and the selection affected by the drag&drop.
	if (msg->HasRef("refs")) {
		tref = new entry_ref();
		msg->FindRef("refs", tref);
		for (i=0; i<6; i++) {
			SelectAtPoint(&touch->touchInObject, tmp, B_3DSELECT_FACES, faces[i], 1, 0.001);
			if (CountElements(tmp) > 0) {
				((Z3dApplication*)be_app)->SelectChannel(tref, this, faces[i], i);
				break;
			}
		}
	}
}

/**********************************************
/*  Sphere
/* This is a special class overloading 3dSpin
/* to define a nicer mapping on the poles and
/* add special properties.
/*********************************************/

class Z3dSphere : public B3dSpin {
public:	
	Z3dSphere(char *name, B3dWorld *world, int32 bands, int32 rotations);
	~Z3dSphere();
	virtual void MessageReceived(BMessage *message, B3dTouchDesc *touch);

private:
	selection_id	all_material;
		
	static	B3dSpinDesc *	CreateSpinDesc(int32 bands, int32 rotations);
};

Z3dSphere::Z3dSphere(char *name, B3dWorld *world, int32 bands, int32 rotations)
: B3dSpin(name, world, CreateSpinDesc(bands,rotations), B_OWN_DESC) {
	all_material = NewSelection();
	// create a selection for the whole sphere.
	CopySelection(all_material, B_OBJECT_SELECTION_ID, B_3DSELECT_FACES);
}

Z3dSphere::~Z3dSphere() {
	FreeSelection(all_material);
}

void Z3dSphere::MessageReceived(BMessage *msg, B3dTouchDesc *touch) {
	entry_ref	 *tref;
	
	// Call the application giving a description of the file dragged,
	// the object and the selection of the whole sphere.
	if (msg->HasRef("refs")) {
		tref = new entry_ref();
		msg->FindRef("refs", tref);
		((Z3dApplication*)be_app)->SelectChannel(tref, this, all_material, 6);
	}
}

B3dSpinDesc *Z3dSphere::CreateSpinDesc(int32 bands, int32 rotations) {
	B3dSpinDesc *desc = new B3dSpinDesc;
	int32 sc = bands + 1;
	int32 rc = rotations;
	B2dVector *shape = new B2dVector[sc];
	float angle,cs,sn;
	
	// create the description of a slice of the sphere, and build a 
	// spin descriptor based on that.
	shape[0].x = 0;
	shape[0].y = -1.0;
	shape[sc-1].x = 0;
	shape[sc-1].y = 1.0;

	for (int32 i=1;i<sc-1;i++) {
		angle = (PI*i)/(sc-1);
		cs = b_cos(angle);
		sn = b_sin(angle);
		shape[i].x = sn;
		shape[i].y = -cs;
	};
	
	// used extended options to have the 3dSpin constructor build 
	// both the model and the default look (with a typical spherical
	// mapping using teh defautl texture.
	desc->shapeCount = sc;
	desc->shape = shape;
	desc->radiusCount = rc;
	desc->texture = default_texture;
	desc->textureRepeatX = 2;
	desc->textureRepeatY = 1;

	return desc;
}

/**********************************************
/*  Pulse
/* This is a special class defining the Pulse
/* object.
/*********************************************/

enum { RES_MAX = 30 };

class Z3dPulse : public B3dFaceBody {
	friend void pulse_hook(void *data, B3dTimeBase *base);	
public:
	Z3dPulse(char *name, B3dWorld *world, float size, long step);
	virtual ~Z3dPulse();
	virtual void MessageReceived(BMessage *message, B3dTouchDesc *touch);
	void    SetShape(float time);

private:
	long          offset[RES_MAX], count[RES_MAX];
	long          radius_count;
	float         time;
	float         *radius, *radius2z;
	int32         token, point_count;
	ushort        *index_radius;
	selection_id  tmp;
	selection_id  faces[2];
	
	static B3dFaceModelDesc *BuildModel(float size, long step);
};

Z3dPulse::Z3dPulse(char *name, B3dWorld *world, float size, long step)
: B3dFaceBody(name, world) {
	int             i, i0, j, cf;
	int32           *indices;
	int32           mi;
	float           r, coeff;	
	B3dVector       *pts;
	B3dMaterial     material;
	B3dFaceModel    *myModel;
	B3dFaceLookDesc desc;

	// create the model of the object.
	myModel = new B3dFaceModel(BuildModel(size, step), B_OWN_POINTS|B_OWN_FACES|B_OWN_DESC);

	// create the look of the object.
	desc.flags = B_3D_LOCAL_POINT_LOOK|B_3D_BACKS_VISIBLE;

	// define the material tables.
	mi = 0;
	material.SetTexture(default_texture);
	desc.numMaterials = 1;
	desc.materials = &material;
	desc.materialIndices = &mi;

	// define the point looks tables
	point_count = myModel->m_points.CountItems();
	cf = myModel->m_faces.CountItems();
	desc.numPointLooks = 2*point_count;
	desc.pointLooks = (B3dPointLook*)malloc(sizeof(B3dPointLook)*desc.numPointLooks);
	desc.pointLookIndices = (int32*)malloc(3*sizeof(int32)*cf*2);

	// point looks coordinates
	i0 = desc.numPointLooks/2;
	for (i=0; i<i0; i++) {
		desc.pointLooks[i].x = 0.5-myModel->m_points[i].x;
		desc.pointLooks[i].y = myModel->m_points[i].y+0.5;
		desc.pointLooks[i].alpha = 0.0;
		desc.pointLooks[i+i0].x = 1.0-desc.pointLooks[i].x;
		desc.pointLooks[i+i0].y = desc.pointLooks[i].y;
		desc.pointLooks[i+i0].alpha = 0.0;
	}	
	
	// point looks indices
	indices = desc.pointLookIndices;
	for (i=0; i<cf; i++) {
		*indices++ = myModel->m_faces[i].points[0];
		*indices++ = myModel->m_faces[i].points[1];
		*indices++ = myModel->m_faces[i].points[2];
	}
	for (i=0; i<cf; i++) {
		*indices++ = myModel->m_faces[i].points[0]+i0;
		*indices++ = myModel->m_faces[i].points[1]+i0;
		*indices++ = myModel->m_faces[i].points[2]+i0;
	}

	// set the look
	SetLook(new B3dFaceLook(name, myModel ,&desc));
		
	// The user is responsible for freeing the memory he allocated for
	// the descriptor.
	free(desc.pointLooks);
	free(desc.pointLookIndices);
	
	// radius effect management. Preprocess the radial distance between each
	// point of the pulse object and the central axis. Group them by similar
	// value (with a 1e-5 tolerance) to keep only a list with each radius
	// once, and an index table going from points to radius.
	radius = (float*)malloc(sizeof(float)*(step*step+40));
	index_radius = (ushort*)malloc(sizeof(float)*myModel->m_points.CountItems());
	radius_count = 0;
	coeff = 2.0/size;
	pts = &myModel->m_points[0];
	for (i=0; i<myModel->m_points.CountItems(); i++) {
		r = b_sqrt(pts[i].x*pts[i].x + pts[i].y*pts[i].y)*coeff;
		for (j=0; j<radius_count; j++)
			if ((radius[j] < (r+1e-5)) && (radius[j] > (r-1e-5)))
				break;
		if (j == radius_count)
			radius[radius_count++] = r;
		index_radius[i] = j;
	}
	radius2z = (float*)malloc(sizeof(float)*radius_count);

	// This register a FrameHook to the universe. This hook will be called
	// for each time changed (in universe time) and will be used to chnage
	// the shape of the pulse object. As it's considered to be a "long"
	// operation, the HEAVY_FUTY flag asks the universe to spawn a new
	// thread just to do that (all heavy-duty operations can be process in
	// parallel as they should affect only local contexts.
	time = 0;
	token = Universe()->AddFrameHook(pulse_hook, (void*)this, B_HEAVY_DUTY);

	// create selections for both sides, and a temp selection
	tmp = NewSelection();
	faces[0] = NewSelection();
	CopySelection(faces[0], B_OBJECT_SELECTION_ID, B_3DSELECT_FACES);
	faces[1] = NewSelection();
	GrabFaceBack(faces[1]);
}

Z3dPulse::~Z3dPulse() {
	Universe()->RemoveFrameHook(token);

	free(radius);
	free(radius2z);
	free(index_radius);
	
	FreeSelection(faces[0]);
	FreeSelection(faces[1]);
	FreeSelection(tmp);
}

B3dFaceModelDesc *Z3dPulse::BuildModel(float size, long step) {
	long             i, j, k, index, ipt, ipt2, nb_pt, nb_face;
	long             begin[50];
	float            dx, dy, sx, sy;
	B3dVector        *listPt, *curPt;
	B3dFaceDesc      *listFace, *curFace;
	B3dFaceModelDesc *desc;

// calculate pt and face count.
	nb_pt = 3*step*(step+1)+1;
	nb_face = 12*step*step;
// buffer allocation
	listPt = (B3dVector*)malloc(nb_pt*sizeof(B3dVector));
	listFace = (B3dFaceDesc*)malloc(nb_face*sizeof(B3dFaceDesc));
// create points.
	curPt = listPt;
	dy = size/(2.0*(float)step);
	dx = dy*0.866;
	index = 0;
	begin[0] = 0;
	for (i=-step; i<=step; i++) {
		if (i < 0) k = 2*step+i;
		else k = 2*step-i;
		index += k+1;
		begin[i+step+1] = index;
		sx = dx * (float)i;
		sy = -0.5 * dy * (float)k;
		for (j=0; j<=k; j++) {
			curPt->x = sx;
			curPt->y = sy;
			curPt->z = 0.0;
			curPt++;
			sy += dy; 
		}
	}
// create face definition
	curFace = listFace;
	index = 0;
	for (i=-step; i<0; i++) {
		ipt = begin[i+step];
		ipt2 = begin[i+step+1];
		k = begin[i+step+2] - ipt - 2;
		for (j=0; j<k; j++) {
			if (j & 1) {
				curFace->points[0] = ipt; 
				curFace->points[1] = ipt+1;
				curFace->points[2] = ipt2+1;
				ipt++;
				ipt2++;
			}
			else {
				curFace->points[0] = ipt; 
				curFace->points[1] = ipt2+1;
				curFace->points[2] = ipt2;
			}
			curFace++;
			index++;
		}
	}
	for (i=0; i<step; i++) {
		ipt = begin[i+step];
		ipt2 = begin[i+step+1];
		k = begin[i+step+2] - ipt - 2;
		for (j=0; j<k; j++) {
			if (j & 1) {
				curFace->points[0] = ipt+1; 
				curFace->points[1] = ipt2+1;
				curFace->points[2] = ipt2;
				ipt++;
				ipt2++;
			}
			else {
				curFace->points[0] = ipt; 
				curFace->points[1] = ipt+1;
				curFace->points[2] = ipt2;
			}
			curFace++;
			index++;
		}
	}
	
// create, fill and return constructor.
	desc = (B3dFaceModelDesc*)malloc(sizeof(B3dFaceModelDesc));
	desc->pointCount = nb_pt;
	desc->points = listPt;
	desc->faceCount = nb_face;
	desc->faces = listFace;
	desc->modelType = B_SMOOTH_MODEL;
	desc->normalCount = 0;
	desc->normals = NULL;
	return desc;
}

void Z3dPulse::MessageReceived(BMessage *msg, B3dTouchDesc *touch) {
	int          i;
	entry_ref	 *tref;
	
	// Check if the drag&drop touches oen of the selection. If it's the
	// case, call the application giving a description of the file dragged,
	// the object and the selection affected by the drag&drop.
	if (msg->HasRef("refs")) {
		tref = new entry_ref();
		msg->FindRef("refs", tref);
		for (i=0; i<2; i++) {
			CombineSelection((selection_id)touch->personalInfo, faces[i],
							 tmp, B_SELECTION_INTERSECTION);
			if (CountElements(tmp) > 0) {
				((Z3dApplication*)be_app)->SelectChannel(tref, this, faces[i], 7+i);
				break;
			}
		}
	}
}

void pulse_hook(void *data, B3dTimeBase *base) {
	Z3dPulse     *z;

	// this hook is called by the universe for each change of time (each new
	// frame when Pause is not selected.
	z = (Z3dPulse*)data;
	z->time += base->universe_new-base->universe_old;
	z->SetShape(z->time);
}

void Z3dPulse::SetShape(float time) {
	int         i;
	B3dVector   *points;
	
	// reset the z coordinates of every points, depending of their radial
	// distance to the center axis and the current time. Preprocess all
	// the new z first.
	for (i=0; i<radius_count; i++)
		radius2z[i] = 0.16*b_cos(radius[i]*10.0-time*3e-6)/(1.0+3.0*radius[i]);
	points = PointList();
	for (i=0; i<point_count; i++)
		points[i].z = radius2z[index_radius[i]];
	// update the normals of each face
	RecalcFaceNormals();
	// update the curve normal on each vertex.
	GenerateNormals(B_OBJECT_SELECTION_ID);
}

/**********************************************
/*  Book
/* the book animation (not commented. have fun :-)
/* It's the same architecture as the pulse,
/* just a bit more complicated.
/*********************************************/

class Z3dPage : public B3dFaceBody {
	friend void page_hook(void *data, B3dTimeBase *base);	
 public:
	Z3dPage(char *name, B3dWorld *world, float size, long step,
			float alpha0, bool moving, int32 index0, int32 index1);
	virtual       ~Z3dPage();
	virtual void  MessageReceived(BMessage *message, B3dTouchDesc *touch);
	virtual void  TouchDown(B3dTouchDesc *touch, B3dVector *touchOnScreen, uint32 buttons);
	virtual void  TouchMoved(B3dTouchDesc *touch, B3dVector *touchOnScreen, uint32 buttons);
	
 private:
	int32         index_channel[2];
	int32         mode;
	int32         step, point_count, token;
	float         inertia, size;
	float         elevation, direction, alpha, traction;
	float         elevation0, direction0, alpha0, traction0;
	B3dVector     point0;
	selection_id  tmp;
	selection_id  pages[2];
	
	void          SetShape(float alpha, float inertia, float traction);
	static B3dFaceModelDesc *BuildModel(float size, long step);
};

static float angles_max[24] = {
	0.8, 1.12, 1.34, 1.5,
	1.6, 1.67, 1.72, 1.76,
	1.8, 1.8, 1.8, 1.8,
	1.8, 1.8, 1.8, 1.8,
	1.8, 1.8, 1.8, 1.8,
	1.8, 1.8, 1.8, 1.8
};

Z3dPage::Z3dPage(char *name, B3dWorld *world, float size0, long step0,
				 float alpha0, bool moving, int32 index0, int32 index1)
:B3dFaceBody(name, world) {
	int             i, j, k, cf;
	int32           *indices;
	int32           *mi, *m, *face_list, *f;
	float           coeff, x, y;	
	B3dMaterial     material[2];
	B3dFaceModel    *myModel;
	B3dPointLook    *pt_look;
	B3dFaceLookDesc desc;

	index_channel[0] = index0;
	index_channel[1] = index1;
	step = step0;
	size = size0;
	myModel = new B3dFaceModel(BuildModel(size, step), B_OWN_POINTS|B_OWN_FACES|B_OWN_DESC);

	desc.flags = B_3D_LOCAL_FACE_LOOK|B_3D_LOCAL_POINT_LOOK|B_3D_BACKS_VISIBLE;

	point_count = myModel->m_points.CountItems();
	cf = myModel->m_faces.CountItems();

	mi = (int32*)malloc(sizeof(int32)*cf*2);
	m = mi;
	for (k=0; k<2; k++) {
		for (j=0; j<2*step; j++) m[j] = 0;
		m += 2*step;
		for (i=1; i<step-1; i++) {
			m[0] = m[1] = 0;
			for (j=2; j<2*step-2; j++) m[j] = 1;
			m[2*step-2] = m[2*step-1] = 0;
			m += 2*step;
		}
		for (j=0; j<2*step; j++) m[j] = 0;
		m += 2*step;
	}
	
	material[0].SetTexture(default_texture);
	material[1].SetTexture(default_texture2);
	desc.numMaterials = 2;
	desc.materials = material;
	desc.materialIndices = mi;

	desc.numPointLooks = 2*point_count;
	desc.pointLooks = (B3dPointLook*)malloc(sizeof(B3dPointLook)*desc.numPointLooks);
	desc.pointLookIndices = (int32*)malloc(3*sizeof(int32)*cf*2);

	coeff = 1.0/(float)(step-2);
	pt_look = desc.pointLooks;

	x = 1.0+coeff;
	for (i=0; i<=step; i++) {
		y = 1.0+coeff; 
		for (j=0; j<=step; j++) {
			pt_look->x = x;
			pt_look->y = y;
			pt_look->alpha = 0.0;
			pt_look++;
			y -= coeff;
		}
		x -= coeff;
	}

	x = -coeff;
	for (i=0; i<=step; i++) {
		y = 1.0+coeff; 
		for (j=0; j<=step; j++) {
			pt_look->x = x;
			pt_look->y = y;
			pt_look->alpha = 0.0;
			pt_look++;
			y -= coeff;
		}
		x += coeff;
	}

	indices = desc.pointLookIndices;
	for (i=0; i<cf; i++) {
		*indices++ = myModel->m_faces[i].points[0];
		*indices++ = myModel->m_faces[i].points[1];
		*indices++ = myModel->m_faces[i].points[2];
	}
	for (i=0; i<cf; i++) {
		*indices++ = myModel->m_faces[i].points[0]+point_count;
		*indices++ = myModel->m_faces[i].points[1]+point_count;
		*indices++ = myModel->m_faces[i].points[2]+point_count;
	}

	SetLook(new B3dFaceLook(name, myModel ,&desc));
		
	free(mi);
	free(desc.pointLooks);
	free(desc.pointLookIndices);

	// create the two side selection, and the tmp selection
	tmp = NewSelection();
	pages[0] = NewSelection();
	pages[1] = NewSelection();
	cf = 2*(step-2)*(step-2);
	f = face_list = (int32*)malloc(sizeof(int32)*cf);
	for (i=1; i<step-1; i++)
		for (j=1; j<step-1; j++) {
			f[0] = 2*(j+step*i);
			f[1] = f[0]+1;
			f+=2;
		}
	SelectFaceIndices(pages[0], face_list, cf);
	free(face_list);
	GrabFaceBack(pages[1], pages[0]);	

	if (!moving) {
		token = -1;	
		if (alpha0 == 1.0)
			ClearSelection(pages[1]);
		else
			ClearSelection(pages[0]);
	}
	else
		token = Universe()->AddFrameHook(page_hook, (void*)this, B_HEAVY_DUTY);
	
	alpha = alpha0;
	inertia = 0.0;
	traction = 0.0;
	SetShape(alpha, inertia, traction);
}

Z3dPage::~Z3dPage() {
	if (token >= 0)
		Universe()->RemoveFrameHook(token);
}

void Z3dPage::TouchDown(B3dTouchDesc *touch, B3dVector *touchOnScreen, uint32 buttons) {
	float          x_ref, yz_ref;
	B3dMatrix      rot;

	x_ref = touch->touchInObject.x/size;
	yz_ref = b_sqrt(touch->touchInObject.y*touch->touchInObject.y +
					touch->touchInObject.z*touch->touchInObject.z)/size;
	if (yz_ref < 0.45) {
		mode = 0;
		rot = *Rotation();
		elevation0 = asin(rot.X().z);
		direction0 = atan2(rot.X().y, rot.X().x);
		if (direction0 < 0.0)
			direction0 += 3.1416;
		else
			direction0 -= 3.1416;
	}
	else if (yz_ref > 0.65) {
		if (token == -1)
			mode = 2;
		else {
			alpha0 = alpha;
			traction0 = 2.5*(x_ref-0.5);
			mode = 1;
		}
	}
	else mode = 2;
	point0 = touch->touchOnScreen;
}

void Z3dPage::TouchMoved(B3dTouchDesc *touch, B3dVector *touchOnScreen, uint32 buttons) {
	float          move;
	B3dVector      point;
	B3dMatrix      mat;
	
	point = *touchOnScreen;
	if (mode == 0) {
		direction = direction0+(point0.x-point.x)*2.0;
		elevation = elevation0+(point0.y-point.y)*2.5;
		if (direction > 0.3)
			direction = 0.3;
		else if (direction < -0.3)
			direction = -0.3;
		if (elevation > 1.2)
			elevation = 1.2;
		else if (elevation < 0.3)
			elevation = 0.3;
		if (GroupMaster() != 0L) {
			mat.Set(3.1416+direction, -elevation, 0.0);
			GroupMaster()->SetRotation(&mat);
		}
	}
	else if (mode == 1) {
		move = (point0.x-point.x)*b_cos(alpha*1.5708)+
			   (point.y-point0.y)*b_sin(alpha*1.5708);
		alpha = alpha0 + move*1.6;
		if (alpha > 1.0)
			alpha = 1.0;
		else if (alpha < -1.0)
			alpha = -1.0;
		inertia += (alpha0-alpha);
		if (inertia > 1.0)
			inertia = 1.0;
		else if (inertia < -1.0)
			inertia = -1.0;
		traction = traction0;
		alpha0 = alpha;
		point0 = point;
	}
}

B3dFaceModelDesc *Z3dPage::BuildModel(float size, long step) {
	long             i, j, ipt, nb_pt, nb_face;
	float            dx, dy, sx, sy;
	B3dVector        *listPt, *curPt;
	B3dFaceDesc      *listFace, *curFace;
	B3dFaceModelDesc *desc;

	if (step > 24) step = 24;
	step &= 0x1e;
// calculate count of point and faces
	nb_pt = (step+1) * (step+1);
	nb_face = 2*step*step;
// buffer allocation
	listPt = (B3dVector*)malloc(nb_pt*sizeof(B3dVector));
	listFace = (B3dFaceDesc*)malloc(nb_face*sizeof(B3dFaceDesc));
// create fake points
	curPt = listPt;
	dy = dx = size/(float)step;
	sy = 0.0;
	for (i=0; i<=step; i++) {
		sx = 0.0;
		for (j=0; j<=step; j++) {
			curPt->x = sx;
			curPt->y = sy;
			curPt->z = 0.0;
			sx += dx;
			curPt++;
		}
		sy += dy;
	}
// create faces (real ones).
	curFace = listFace;
	ipt = step-1;
	for (i=0; i<step; i++) {
		for (j=step-1; j>=0; j--) {
			curFace[0].points[0] = ipt; 
			curFace[0].points[1] = ipt+step+1;
			curFace[0].points[2] = ipt+1;
			curFace[1].points[0] = ipt+step+1;
			curFace[1].points[1] = ipt+step+2;
			curFace[1].points[2] = ipt+1;
			curFace += 2;
			ipt--;
		}
		ipt+=2*step+1;
	}
// create, fill and return constructor.
	desc = (B3dFaceModelDesc*)malloc(sizeof(B3dFaceModelDesc));
	desc->pointCount = nb_pt;
	desc->faceCount = nb_face;
	desc->faces = listFace;
	desc->points = listPt;
	desc->modelType = B_SMOOTH_MODEL;
	desc->normalCount = 0;
	desc->normals = NULL;
	return desc;
}

void Z3dPage::MessageReceived(BMessage *msg, B3dTouchDesc *touch) {
	int          i;
	entry_ref	 *tref;

	if (msg->HasRef("refs")) {
		tref = new entry_ref();
		msg->FindRef("refs", tref);
		for (i=0; i<2; i++) {
			CombineSelection((selection_id)touch->personalInfo, pages[i],
							 tmp, B_SELECTION_INTERSECTION);
			if (CountElements(tmp) > 0) {
				((Z3dApplication*)be_app)->SelectChannel(tref, this, pages[i],
														 index_channel[i]);
				break;
			}
		}
	}
}

void page_hook(void *data, B3dTimeBase *base) {
	float       alpha0;
	Z3dPage     *z;

	z = (Z3dPage*)data;
	z->SetShape(z->alpha, z->inertia, z->traction);

	if (z->inertia > 0.04)
		z->inertia -= 0.04+(z->inertia-0.04)*0.2;
	else if (z->inertia < -0.04)
		z->inertia += 0.04+(0.04-z->inertia)*0.2;
	else
		z->inertia = 0.0;

	alpha0 = z->alpha;
    z->alpha += z->alpha*0.05;
	if (z->alpha > 1.0)
		z->alpha = 1.0;
	else if (z->alpha < -1.0)
		z->alpha = -1.0;
	z->inertia += (alpha0-z->alpha)*1.0;
	if (z->inertia > 1.0)
		z->inertia = 1.0;
	else if (z->inertia < -1.0)
		z->inertia = -1.0;
	
	if (z->traction > 0.04)
		z->traction -= 0.04+(z->traction-0.04)*0.2;
	else if (z->traction < -0.04)
		z->traction += 0.04+(0.04-z->traction)*0.2;
	else
		z->traction = 0.0;
}

void Z3dPage::SetShape(float alpha, float inertia, float traction) {
	long        i, j, k, hstep, step1;
	float       angle, coeff, c_iner, pas, prev_angle, a0, d_move, move;
	B3dVector   centre[25];
	B3dVector   norm[25];
	B3dVector   *curPt, *curNorm, *Pt, *Norm;
	B3dVector   *points, *norms;
	B3dFaceDesc *curFace;

	points = PointList();
	norms = NormalList();
	coeff = 1.0/(float)step;
	pas = size*coeff;
	hstep = step/2;
	step1 = step+1;
// build the central curve
	// first point
	centre[0].x = size*0.5;
	centre[0].y = 0.0;
	centre[0].z = 0.0;
	prev_angle = 0.0;
	for (i=0; i<step; i++) {
		c_iner = (float)(step-i)*coeff*inertia;
		if (c_iner > 0.0)
			angle = angles_max[i] * (c_iner + alpha*(1.0-c_iner)); 
		else
			angle = angles_max[i] * (c_iner + alpha*(1.0+c_iner));
	// next point.
		centre[i+1].x = centre[i].x;
		centre[i+1].y = centre[i].y + pas * b_sin(angle);
		centre[i+1].z = centre[i].z + pas * b_cos(angle);
	// previous norm.
		a0 = (angle+prev_angle) * 0.5 - 1.5708;
		norm[i].x = 0.0;
		norm[i].y = b_sin(a0);
		norm[i].z = b_cos(a0);
		prev_angle = angle;
	}
    // last norm
	a0 = angle - 1.5708;
	norm[step].x = 0.0;
	norm[step].y = b_sin(a0);
	norm[step].z = b_cos(a0);
// rebuild points and point's norms.
	curPt = points;
	curNorm = norms;
// first half, without traction
	for (i=0; i<=hstep; i++) {
		curPt->x = centre[i].x - pas * (float)hstep;
		curPt->y = centre[i].y;
		curPt->z = centre[i].z;
		curNorm[0] = norm[i];
		for (j=1; j<=step; j++) {
			curNorm[j] = curNorm[0];
			curPt[j].x = curPt[j-1].x + pas;
			curPt[j].y = curPt[0].y;
			curPt[j].z = curPt[0].z;
		}
		curPt += step1;
		curNorm += step1;
	}
// the axis is normal
	Pt = curPt+hstep;
	Norm = curNorm+hstep;
	for (i=1; i<= hstep; i++) {
		*Pt = centre[i+hstep];
		*Norm = norm[i+hstep];
		Pt += step1;
		Norm += step1;
	}
// second half, with traction.
	if (traction >= 0.0) {
		for (i=0; i<hstep*step1; i+=step1)
			for (j=hstep-1; j>=0; j--) {
				Pt = curPt+j+i;
				Norm = curNorm+j+i;
				Pt->x = Pt[1].x-pas;
				Pt->y = Pt[1].y;
				Pt->z = Pt[1].z;
				*Norm = Norm[1];
			}
	}
	else {
		d_move = -3.0 * (traction * inertia * pas)/(float)(hstep*hstep);
		move = 0.0;
		for (k=1; k<step; k++) {
			if (k>=hstep)
				if ((move < 0.12) && (move > -0.12))
					move += d_move;
			i = k-hstep;
			if (i<0) i = 0;
			for (;i<hstep ;i++) {
				j = hstep-k+i;
				if (j>=hstep) break;
				Pt = curPt+j+i*step1;
				Norm = curNorm+j+i*step1;
				Pt->x = Pt[1].x-pas;
				Pt->y = Pt[1].y + move*(Norm[1].y + Norm[-step1].y);
				Pt->z = Pt[1].z + move*(Norm[1].z + Norm[-step1].z);
				Norm->x = Norm[1].x + (Pt[1].x + Pt[-step1].x - 2*Pt->x) * move;
				Norm->y = Norm[1].y + (Pt[1].y + Pt[-step1].y - 2*Pt->y) * move;
				Norm->z = Norm[1].z + (Pt[1].z + Pt[-step1].z - 2*Pt->z) * move;
				Norm->Norm();
			}
		}
	}

	if (traction <= 0.0) {
		for (i=0; i<hstep*step1; i+=step1)
			for (j=hstep+1; j<=step; j++) {
				Pt = curPt+j+i;
				Norm = curNorm+j+i;
				Pt->x = Pt[-1].x+pas;
				Pt->y = Pt[-1].y;
				Pt->z = Pt[-1].z;
				*Norm = Norm[-1];
			}
	}
	else {
		d_move = 3.0 * (traction * inertia * pas)/(float)(hstep*hstep);
		move = 0.0;
		for (k=1+hstep; k<step+hstep; k++) {
			if (k>=step)
				move += d_move;
			i = k-step;
			if (i<0) i = 0;
			for (;i<hstep ;i++) {
				j = k-i;
				if (j<=hstep) break;
				Pt = curPt+j+i*step1;
				Norm = curNorm+j+i*step1;
				Pt->x = Pt[-1].x+pas;
				Pt->y = Pt[-1].y + move*(Norm[-1].y + Norm[-step1].y);
				Pt->z = Pt[-1].z + move*(Norm[-1].z + Norm[-step1].z);
				Norm->x = Norm[-1].x + (Pt[-1].x + Pt[-step1].x - 2*Pt->x) * move;
				Norm->y = Norm[-1].y + (Pt[-1].y + Pt[-step1].y - 2*Pt->y) * move;
				Norm->z = Norm[-1].z + (Pt[-1].z + Pt[-step1].z - 2*Pt->z) * move;
				Norm->Norm();
			}
		}
	}
// calculate norms of all faces.
	curNorm = &((B3dFaceModel*)(look->model))->m_faceNorms[0];
	curFace = &((B3dFaceModel*)(look->model))->m_faces[0];
	i = ((B3dFaceModel*)(look->model))->m_faces.CountItems();
	for (; i>0; i--) {
		curNorm[0] = (points[curFace->points[2]]-points[curFace->points[1]])^
			         (points[curFace->points[1]]-points[curFace->points[0]]);
		curNorm[0].Norm();
		curNorm++;
		curFace++;
	}
}

/**********************************************
/*  View
/*********************************************/

Z3dView::Z3dView(BRect rect, char *name, long index, B3dUniverse *uni)
: B3dView(name, rect, uni) {
	myIndex = index;
}

void Z3dView::AttachedToWindow() {
	B3dLight          *light;
	B3dMatrix         mat;
	B3dVector         v, v2;
	RGBAColor         color;

	// need to call the inherited method.
	inherited::AttachedToWindow();

	// need to lock the universe before modifying it
	Universe()->Lock();

	switch (myIndex) {
	case 0:
		// create a cube.
		v.Set(1.0, 1.0, 1.0);
		mat.Set(0.0, -0.55, 0.0);
		myBody = new Z3dCube("Test cube", World(), &v);
		v2.Set(2.3, 0.0, 0.0);
		myBody->SetOrigin(&v2);
		myBody->SetRotation(&mat);
		v.Set(0.0, 0.0, 1.0);
		myBody->LinkTo(new B3dAxisLink(0.25, &v, &v2));
		// create lights
		color.Set(1.0, 1.0, 1.0, 0.0);
		new B3dAmbientLight("Test Light2", World(), 0.5, &color);		
		v.Set(0.42, 0.43, 0.8);
		new B3dParallelLight("Test Light1", World(), 0.7, &color, &color, &v);
		break;
	case 1:
		// create a sphere.
		mat.Set(1.57, 0.2, 0.0);
		myBody = new Z3dSphere("Test sphere", World(), 14, 28);
		myBody->SetOrigin(2.8, 0.0, 0.0);
		myBody->SetRotation(&mat);
		// create lights
		color.Set(1.0, 1.0, 1.0, 0.0);
		new B3dAmbientLight("Test Light2", World(), 0.4, &color);		
		v.Set(0.2, -0.9, 0.4);
		light = new B3dParallelLight("Test Light1", World(), 0.8, &color, &color, &v);
		light->LinkTo(new B3dJigLink(0.4, 0.3, 1.0));
		break;
	case 2:
		// create a pulse.
		myBody = new Z3dPulse("Test pulse", World(), 1.0, 12);
		myBody->SetOrigin(0.0, -1.04, -0.73);
		// create lights
		color.Set(1.0, 1.0, 1.0, 0.0);
		new B3dAmbientLight("Test Light2", World(), 0.3, &color);		
		v.Set(0.42, 0.43, 0.8);
		light = new B3dParallelLight("Test Light1", World(), 0.8, &color, &color, &v);
		break;
	case 3:
		// create a book.
		pages[0] = new Z3dPage("Left page", World(), 1.0, 12, 1.0, FALSE, 9, 9);
		pages[1] = new Z3dPage("Right page", World(), 1.0, 12, -1.0, FALSE, 10, 10);
		pages[2] = new Z3dPage("Moving page", World(), 1.0, 12, -1.0, TRUE, 11, 12);
		pages[2]->SetOrigin(-1e-4, 0.0, 1e-4);
		v.Set(0.0, 0.0, 0.0);
		group_things(3, pages, &myBody, "Book", &v);
		mat.Set(3.1416, -0.7, 0.0);
		myBody->SetRotation(&mat);
		myBody->SetOrigin(-1.7, 0.0, 0.0);
		// create lights
		color.Set(1.0, 1.0, 1.0, 0.0);
		new B3dAmbientLight("Test Light2", World(), 0.4, &color);		
		v.Set(0.43, 0.8, 0.42);
		light = new B3dParallelLight("Test Light1", World(), 0.7, &color, &color, &v);
		break;
	}
		
	// turn the camera to look at the box.
	Camera()->ViewPoint()->LookAt(myBody);

	if (myIndex == 3) {
		mat.Set(0.0, -0.31, 0.0);
		mat = *Camera()->ViewPoint()->Rotation()*mat;
		Camera()->ViewPoint()->SetRotation(&mat);
	}
		
	// end of universe modification
	Universe()->Unlock();	
}












