/*****************************************************************************
//
//	File:			3dMaterial.h
//
//	Description:	Class describing the material appearance of things.
//
//	Copyright 1997, Be Incorporated
//
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
//***************************************************************************/

#ifndef _3D_MATERIAL_H
#define _3D_MATERIAL_H

#include "3dChannel.h"

enum {
	B_MATERIAL_IS_VISIBLE		= 0x00000001,
	/*	If this isn't set, anything with this material isn't rendered */

	B_MATERIAL_IS_FILLED		= 0x00000002,
	/*	The material is solid; otherwise, wireframe rendering is needed */

	B_MATERIAL_IS_COLORED		= 0x00000004,
	/* 	The material is a particular color.  This should always be
		enabled unless the material is textured or mirrored.  If the
		material is textured or mirrored, this being enabled means that
		the texture should be blended with the material color (this
		can be quite expensive in software) */

	B_MATERIAL_IS_LIT			= 0x00000010,
	/*	The material is affected by lighting.  This should usually be
		enabled, although it can be turned off for mirrored materials,
		for instance, or for materials that are meant to be emissive. */

	B_MATERIAL_IS_SMOOTH		= 0x00000020,
	/*	Material lighting is per-vertex; otherwise it is per-face 
		Per-vertex lighting gives the impression of a smooth surface,
		while per-face lighting gives a flat or faceted appearance.
		This only has any effect if B_MATERIAL_IS_LIT is enabled. */

	B_MATERIAL_IS_SHINY		= 0x00000008,
	/*	The material needs specular reflection.  This should be used for
		materials that should look shiny (like metal, plastic, etc.) */

	B_MATERIAL_IS_TEXTURED		= 0x00000040,
	/*	The material is texture-mapped */

	B_MATERIAL_IS_MIRRORED		= 0x00000080,
	/*	The material is pseudo-environment-mapped */

	B_MATERIAL_IS_TRANSPARENT	= 0x00000200,
	/*	Alpha blending is enabled for this material.  The material
		will look transparent or translucent.  The degree of
		transparency is determined by the alpha component of
		the face's color (1.0 == opaque, 0.0 == transparent) */

	B_MATERIAL_IS_GRIMY		= 0
	/*	Whoever instantiated this object should be ashamed of
		themselves... this material badly needs to be cleaned. */
};

struct B3dPackedMaterial;
class B3dFaceBody;
class B3dFaceLook;

class B3dMaterial
{

	public:

					B3dMaterial();

inline	int32		Properties() const;
		status_t		SetProperties(int32 props);
		/*	Get and set material's properties (see previous enums) */

inline	RGBAColor		Color() const;
		void			SetColor(RGBAColor *color);
		/*	Get and set the material color */

inline	RGBAColor		ShineColor() const;
inline	float		Shininess() const;
		void			SetShine(RGBAColor *color, float specularCoeff);
		/*	Get and set the specular color and the specular reflection coefficient */

inline	B3dChannel *	Texture() const;
		void			SetTexture(B3dChannel *texture);
		/*	Texture mapping uses a B3dChannel as the texture source.  B3dChannel
			is the common encapsulated format for a RGB_32_BIT graphic
			buffer or stream of buffers (picture, movie, live video or 3d) */

inline	float		MirrorZoom() const;
		/*	This is a tricky coefficient controling the zoom of the texture used
			for mirror mapping. A typical value is around 0.8.  This coefficient
			allow you to adjust the zoom on the environment texture if you have
			problems on the borders.  Such is the price of pseudo-environment
			mapping. */
		void			SetMirror(B3dChannel *texture, float zoom);
		/*	Specifies both the environment map and zoom constant to use for
			mirror mapping. */

inline	float		Blending() const;
		void			SetBlending(float blendFactor);
		/*	Get and set the ratio of texture to color
			1.0 == all texture, 0.0 == all color */

inline	bool			operator==(const B3dMaterial &other) const;
inline	bool			operator!=(const B3dMaterial &other) const;

/* This is public for fast access by low-level code, but you
   should not modify them directly; use the above methods. */

        RGBAColor		m_ambientColor;
		RGBAColor		m_diffuseColor;
		RGBAColor		m_specularColor;
		float		m_specularCoeff;
		B3dChannel *	m_texture;
		float		m_zoomFactor;
		float		m_blendFactor;

		int32		m_properties;

		bool			Equal(const B3dMaterial &other) const;
	
					B3dMaterial(int32 rid, B3dPackedMaterial *pm);
};


float		B3dMaterial::Blending() const
{
	return m_blendFactor;
};

int32 		B3dMaterial::Properties() const
{
	return m_properties;
};

RGBAColor 	B3dMaterial::Color() const
{
	return m_diffuseColor;
};

RGBAColor 	B3dMaterial::ShineColor() const
{
	return m_specularColor;
};

float 		B3dMaterial::Shininess() const
{
	return m_specularCoeff;
};

B3dChannel *	B3dMaterial::Texture() const
{
	return m_texture;
};

float		B3dMaterial::MirrorZoom() const
{
	return m_zoomFactor;
};

bool			B3dMaterial::operator==(const B3dMaterial &other) const
{
	if (Properties() != other.Properties()) return false;
	return Equal(other);
};

bool			B3dMaterial::operator!=(const B3dMaterial &other) const
{
	return !((*this) == other);
};

#endif
