/*****************************************************************************
//
//	File:			3dLens.h
//
//	Description:	Class describing the projection (optical properties) used.
//                  to look at the world.
//
//	Copyright 1997, Be Incorporated
//
// ******** 3dKit ARCHITECTURE NOTE :
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
//***************************************************************************/

#ifndef _3D_LENS_H
#define _3D_LENS_H

#ifndef _3D_DEFS_H 
#include "3dDefs.h"
#endif
#ifndef _3D_THING_H
#include "3dThing.h"
#endif
#ifndef _3D_LOOK_H
#include "3dLook.h"
#endif
#ifndef _3D_MODEL_H
#include "3dModel.h"
#endif

/**************************************
// LOW-LEVEL API
/*************************************/

enum {
	B_SEE_POINT =   0x00000001,
	B_SEE_NORM =    0x00000002,
	B_SEE_CLIP =    0x00000004
};

enum {
	B_CLIPPED_OUT = 0x00000001,
	B_CLIPPED_IN =  0x00000002,
	B_UNCLIPPED =   0x00000003
};

class B3dLensOptions {
public:	
	B3dVector Origin;
	B3dMatrix Rotation;
	float     buf_hMin;
	float     buf_hMax;
	float     buf_vMin;
	float     buf_vMax;
	float     hMin;
	float     hMax;
	float     vMin;
	float     vMax;
};

class B3dLensPoint : public B3dVector {
 public:
	int32	h;
	int32	v;
	float	invZ;
};

class B3dLensImage {

	public:
	
	void *		lensPoints;
	int32		lensPointSize;
	B3dVector *	lensVectors;

	B3dVector		lensOrigin;
	B3dVector		lensDirection;
	B3dVector		lensUp;
	
	inline void      *lensPointAt(int32 index);
	inline B3dVector *lensVectorAt(int32 index);
};

void *B3dLensImage::lensPointAt(int32 index)
{
	return (void*)((int32)lensPoints+index*lensPointSize);
}

B3dVector *B3dLensImage::lensVectorAt(int32 index)
{
	return lensVectors+index;
}

class B3dCamera;
struct render_desc;

class B3dLens {

	public:
	
					B3dLens();
virtual 				~B3dLens();

virtual	B3dVector		Screen2Camera(B3dVector vectorInScreenSpace)=0;
inline	B3dVector		Pixel2Camera(BPoint pixel, float z);

virtual	B3dVector		Camera2Screen(B3dVector vectorInCameraSpace)=0;
virtual	B3dVector		Pixel2Screen(BPoint pixel, float z)=0;

virtual	BPoint		Screen2Pixel(B3dVector vectorInScreenSpace)=0;
inline	BPoint		Camera2Pixel(B3dVector vectorInCameraSpace);
	
virtual	void			GetOptions(void **options);
virtual	void			SetOptions();
virtual	bool			See(	B3dModel     *model,
						B3dMatrix    *rotation,
						B3dVector    *translation,
						long         status,
						B3dLensImage *lensImage);
virtual	void			See(	B3dCamera    *camera,
						B3dMatrix    *rotation,
						float        trans_z);
virtual	void			CalcAxis(B3dVector *vectorOnScreen, B3dVector *axis);
virtual	long			CheckSteric(float radius, B3dMatrix *rotation, B3dVector *translation);
virtual	void			Clip(render_desc *desc, PNLD *array);
virtual	void			GetBuffers(B3dLensPoint *array[8]);

		void			Lock();
		void			Unlock();

	private:
	
		int32		lens_lock;
		sem_id		lens_sem;
};

B3dVector B3dLens::Pixel2Camera(BPoint pixel, float z)
{
	return Screen2Camera(Pixel2Screen(pixel,z));
};

BPoint B3dLens::Camera2Pixel(B3dVector vectorInCameraSpace)
{
	return Screen2Pixel(Camera2Screen(vectorInCameraSpace));
};

#endif

