/*****************************************************************************
//
//	File:			3dCamera.h
//
//	Description:	Class use to create view of a universe.
//
//	Copyright 1997, Be Incorporated
//
// ******** 3dKit ARCHITECTURE NOTE :
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
//***************************************************************************/

#ifndef _3D_CAMERA_H
#define _3D_CAMERA_H

#ifndef _3D_DEFS_H 
#include "3dDefs.h"
#endif
#ifndef _3D_MODEL_H 
#include "3dModel.h"
#endif
#ifndef _3D_UNIVERSE_H
#include "3dUniverse.h"
#endif
#ifndef _3D_LENS_H
#include "3dLens.h"
#endif

/**************************************
// LOW-LEVEL API
/*************************************/

/* Struct defining the row buffer description for rendering */
typedef struct camera_buffer {
	int32			optionFlags;

	void				*pixels;
	int32			pixelBytesPerRow;

	void				*zbuffer;
	int32			zBytesPerRow;

	BRect			frame;
	BRect			clip;
	uchar			*indexMap;
	rgb_color			*colorList;
	color_space		mode;
	float           zoom_factor;
} camera_buffer;

/* Hook used for buffer and buffer flow control */
typedef void switch_buffer_hook(void *, camera_buffer *);


/* Hook used to erase part or all of a buffer */
typedef void erase_buffer_hook(void *, camera_buffer *);

class B3dLens;
class B3dRadialLens;
class B3dRenderer;
class B3dLighter;

class B3dSharedState {
 public:
	B3dSharedState();
	~B3dSharedState();
	void    *GetBuffer(int32 size);
 private:
	int32   size;
	void    *buffer;
};

/* 

There are five coordinate spaces:

	Object coordinates are in the reference frame of a Thing in the world
	World coordinates are in the reference frame of the world itself
	Camera coordinates are in the reference frame of the camera doing the rendering
	Screen coordinates are camera coordinates projected onto the camera's screen,
		plus the Z coordinates of the point in camera space which was projected.
		Note that screen coordinates are normalized left-handed coordinates, with
		their origin at the screen's center, and are as such independent of the
		resolution of the screen.
	Pixel coordinates are screen coordinates scaled to the resolution of the screen
*/

/*	Class used to encapsulate information from hit detection.  Note that this structure
	does _not_ change as the touch is tracked.  All the information here is information
	gleaned from the touchdown. */
struct B3dTouchDesc {
	B3dCamera		*camera;
		/*	A pointer to the actual camera that was used to
			project the object onto the screen it is being
			manipulated through */

	/*	The touch point in several reference frames: */
	B3dVector		touchOnScreen;
		/*	The position on the screen through which the touch
			originally traveled on it's way to the object, in
			screen coordinates */

	B3dVector		touchInWorld;
		/*	The position of the original touch in world space */

	B3dVector		touchInObject;
		/*	The position of the touch in object space */

	B3dThing *	object;
		/*	A pointer to the thing that recieved the touch */
	void *		personalInfo;
		/*	This is for information the object needs for the duration of
			the touch.  B3dFace, for instance, stores a selection_id here for
			the face that was touched.  This is filled in by B3dBody::GetTouchInfo()
			and may be freed by B3dBody::FreeTouchInfo().*/

	B3dMatrix		initialObjectRotation;
	B3dVector		initialObjectOrigin;
		/*	The initial origin and rotation of the object at the time of the
			touch.  They are using internally, so don't modify them, but they
			may come in handy. */

	float		side;
		/*	Used internally */
};

class B3dWorldSort;

/**************************************
// B3dCamera.
/*************************************/

/* Camera class, the object controling the rendering of views of an universe.
   */
class B3dCamera {

/*************************************
// HIGH-LEVEL API                   */

	public:	
	
inline	const char	*Name();
		void			SetName(char *name);
		/* Return and set the name of the camera */

inline	B3dLens		*Lens();
		/*	Returns the current lens used by the camera. The default one is a
			B3dRadialLens.  This is useful if you want to change the opening
			of your view */

inline	B3dLighter	*Lighter();
		/*	Returns the current lighter used by the camera. The default one is
			a B3dLighter.  This is useful if you want to change the settings
			of the Lighter */

		B3dThing		*ViewPoint();
		/* Return the object used as origin of the camera view */

		int32		SetViewPoint(B3dThing *view_point);
		void			SetViewPoint(B3dVector originInWorldSpace, B3dWorld *world);
		/*	Set the new ViewPoint of the camera as being a thing or a static
			point described by its coordinates, and its world */

		B3dVector		Camera2World(B3dVector vectorInCameraSpace);
inline	B3dVector		Screen2World(B3dVector vectorInScreenSpace);
inline	B3dVector		Pixel2World(BPoint pixel, float z);

		B3dVector		World2Camera(B3dVector vectorInWorldSpace);
inline	B3dVector		Screen2Camera(B3dVector vectorInScreenSpace);
inline	B3dVector		Pixel2Camera(BPoint pixel, float z);

inline	B3dVector		World2Screen(B3dVector vectorInWorldSpace);
inline	B3dVector		Camera2Screen(B3dVector vectorInCameraSpace);
inline	B3dVector		Pixel2Screen(BPoint pixel, float z);

inline	BPoint		World2Pixel(B3dVector vectorInWorldSpace);
inline	BPoint		Camera2Pixel(B3dVector vectorInCameraSpace);
inline	BPoint		Screen2Pixel(B3dVector vectorInScreenSpace);
		/*	These are some handy routines to convert between coordinate
			spaces the camera knows about.  The camera doesn't know about
			object spaces, so to use object coordinates, you'll have to
			convert to or from world coordinates (with B3dThing::Object2World
			and B3dThing::World2Object).  These are only for convenience,
			for anything in which performance is important, use GetViewOperator
			(below) and do your own matrix manipulations.*/
	
/*************************************
// LOW-LEVEL API                    */

	B3dCamera(char               *name,
			  B3dWorld           *world,
			  switch_buffer_hook *switch_hook,
			  erase_buffer_hook  *erase_hook,
			  void               *client,
			  B3dLens            *lens = 0L,
			  B3dLighter         *lighter = 0L);
	virtual             ~B3dCamera();

	inline B3dUniverse  *Universe();
	
	void                SetLens(B3dLens *lens);
	void                SetLighter(B3dLighter *lighter);
	
	void                GetViewOperator(B3dVector **translation, B3dMatrix **rotation);

	float               Threshold();
	void                SetThreshold(float threshold);
	
// low level public calls
	inline B3dWorldSort *WorldSort();
	inline B3dSharedState *SharedState();
	
	virtual void        InitFrame();
	virtual void        RenderFrame();
	virtual void        Update();
	
	bool          GetTouchDesc(		B3dVector *touchOnScreen,
								B3dTouchDesc *touch);
	void          CalcTouchDesc(	B3dVector *touchOnScreen,
								B3dVector *origin0, B3dVector *axis0,
								B3dTouchDesc *touch, B3dMatrix *mat);
	
/*************************************
// PRIVATE STUFF                    */

 private:
	char               *name;
	void               *client;
	bool               own_view_point;
	long               camera_lock;
	float              threshold;
	BRect              up_to_date;
	sem_id             camera_sem;
	int32              optionFlags;
	B3dLens            *lens;
	B3dThing           *view_point;
	B3dVector          translation;
	B3dMatrix          rotation;
	B3dLighter         *lighter;
	B3dRenderer        *renderer;
	B3dUniverse        *uni;
	B3dWorldSort       *world_sort;
	camera_buffer      buffer;
	B3dSharedState     *shared_state;
	erase_buffer_hook  *erase_hook;
	switch_buffer_hook *switch_hook;
};

/**************************************
// INLINE FUNCTION DEFINITIONS.
/*************************************/

B3dLens *B3dCamera::Lens() {
	return lens;
}

B3dLighter *B3dCamera::Lighter() {
	return lighter;
}

B3dUniverse *B3dCamera::Universe() {
	return uni;
}

B3dWorldSort *B3dCamera::WorldSort() {
	return world_sort;
}

B3dSharedState *B3dCamera::SharedState() {
	return shared_state;
}

B3dVector B3dCamera::Screen2Camera(B3dVector vectorInScreenSpace)
{
	return lens->Screen2Camera(vectorInScreenSpace);
};

B3dVector B3dCamera::Pixel2Camera(BPoint pixel, float z)
{
	return lens->Pixel2Camera(pixel,z);
};

B3dVector B3dCamera::Camera2Screen(B3dVector vectorInCameraSpace)
{
	return lens->Camera2Screen(vectorInCameraSpace);
};

B3dVector B3dCamera::Pixel2Screen(BPoint pixel, float z)
{
	return lens->Pixel2Screen(pixel,z);	
};

BPoint B3dCamera::Camera2Pixel(B3dVector vectorInCameraSpace)
{
	return lens->Camera2Pixel(vectorInCameraSpace);
};

BPoint B3dCamera::Screen2Pixel(B3dVector vectorInScreenSpace)
{
	return lens->Screen2Pixel(vectorInScreenSpace);
};

B3dVector B3dCamera::Screen2World(B3dVector vectorInScreenSpace)
{
	return Camera2World(lens->Screen2Camera(vectorInScreenSpace));
};

B3dVector B3dCamera::Pixel2World(BPoint pixel, float z)
{
	return Camera2World(lens->Pixel2Camera(pixel,z));
};

B3dVector B3dCamera::World2Screen(B3dVector vectorInWorldSpace)
{
	return lens->Camera2Screen(World2Camera(vectorInWorldSpace));
};

BPoint B3dCamera::World2Pixel(B3dVector vectorInWorldSpace)
{
	return lens->Camera2Pixel(World2Camera(vectorInWorldSpace));
};

#endif
