/*****************************************************************************
//
//	File:			2dVector.h
//
//	Description:	Generic 2d vector.
//
//	Copyright 1997, Be Incorporated
//
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
//***************************************************************************/

#ifndef _2D_VECTOR_H
#define _2D_VECTOR_H

#include <SupportDefs.h>
#include <stdio.h>

/**************************************
// B2dVector.
/*************************************/

class B2dVector {

/*************************************
// HIGH-LEVEL API                   */

 public:
	float   x;
	float   y;
	/* Coordinates in 2 dimensional space.
	   */
	inline void Set(float new_x, float new_y);
	/* set them.
	   */	

/*************************************
// LOW-LEVEL API                    */

	float Length() const;  
	/* return the length of the vector, without modifying it.
	   */
	float Norm(bool return_prev = false);
	/* change the length of the vector to make it equal to 1,
	   but don't change its direction and orientation. This
	   function can also return the previous length is needed.
	   */  
	inline float Square();
	/* returns Length()*Length().
	   */
	inline float operator* (const B2dVector& v2) const;
	/* return the scalar product of two vectors.
	   */
	inline B2dVector& operator* (const float k) const;
	/* scale a vector by a factor k.
	   */
	inline B2dVector& operator- (const B2dVector& v2) const;
	inline void operator-= (const B2dVector v2);
	inline B2dVector& operator+ (const B2dVector& v2) const;
	inline void operator+= (const B2dVector v2);
	/* Add or substract vectors.
	   */
	inline float operator^ (const B2dVector& v2) const;
	/* Calculate the determinant of two vectors.
	   */
	inline void operator<<= (const B2dVector v2);
	/* Get the minimal component of the 2 vectors, component per
	   component.
	   */
	inline void operator>>= (const B2dVector v2);
	/* Get the maximal component of the 2 vectors, component per
	   component.
	   */
	   
/*************************************
// PRIVATE STUFF                    */

	void Debug();
};

/**************************************
// HIGH-LEVEL API
/*************************************/

extern B2dVector B_2D_VECTOR_NULL;
/* shared NULL 2dVector.
   */

/**************************************
// INLINE FUNCTION DEFINITIONS
/*************************************/

float B2dVector::operator* (const B2dVector& v2) const {
	return x*v2.x + y*v2.y;
}

B2dVector& B2dVector::operator* (const float k) const {
	B2dVector w;

	w.x = x*k;
	w.y = y*k;
	return w;
}

B2dVector& B2dVector::operator- (const B2dVector& v2) const {
	B2dVector w;

	w.x = x-v2.x;
	w.y = y-v2.y;
	return w;
}

void B2dVector::operator-= (const B2dVector v2) {
	x -= v2.x;
	y -= v2.y;
}

B2dVector& B2dVector::operator+ (const B2dVector& v2) const {
	B2dVector w;

	w.x = x+v2.x;
	w.y = y+v2.y;
	return w;
}

void B2dVector::operator+= (const B2dVector v2) {
	x += v2.x;
	y += v2.y;
}

float B2dVector::operator^ (const B2dVector& v2) const {
	return x*v2.y-y*v2.x;
}

void B2dVector::operator<<= (const B2dVector v2) {
	if (v2.x < x) x = v2.x;
	if (v2.y < y) y = v2.y;
}

void B2dVector::operator>>= (const B2dVector v2) {
	if (v2.x > x) x = v2.x;
	if (v2.y > y) y = v2.y;
}

void B2dVector::Set(float new_x, float new_y) {
	x = new_x;
	y = new_y;
}

float B2dVector::Square() {
	return (x*x + y*y);
}

#endif







