/*ldir/lread
   Program to read Linux extended 2 filesystems under DOS

   Main Program  main.c

   Copyright information and copying policy see file README.TXT

   V1.0: Written by Jason Hunter and Dave Lutz.
   V2.0: Rewritten by Werner Zimmermann, FHTE Esslingen, Germany, Oct. 1996
   -new user interface, new command line options
   -new drive detection code
   -debugged buffering, memory allocation to handle big directories
   -error handling
   V2.0a:Integrated floppy disk support. Full support for (short) symlinks.
   W.Zimmermann, Nov. 9, 96
   V2.0b:Completely rewrote memory allocation stuff in dir.c,
   improved harddisk partition detection.
   W.Zimmermann, Nov. 16,96
   V2.0c:Now also single files can be ldir'ed.
   (Limited) support for wildcards, see README.TXT.
   W.Zimmermann, Nov. 21,96
   V2.0d:lread can now copy multiple files.
   W.Zimmermann, Dec. 6,96
   V2.0e: Initialized inode_num in search_dir to fix bug when doing
   multiple read of files.
   W.Zimmermann, Dec. 11,96
   V2.1: Dynamic buffer allocation in function readdisk in readdisk.c,
   removed READ_BUF_SIZE.
   W.Zimmermann, Jan. 24,97
   V2.2: SCSI support has now also been tested. Some comments modified.
   Now under Windows95 also long filenames for lread are supported
   (unfortunately this does not seem to work with DJPP's gcc)
   W.Zimmermann, May 10, 97
   V2.3: Fixed partition table read. Could not handle multiple linux
   partitions in DOS extended partitions. Bug found by Jakob
   Saternus. Thanks, Jakob!
   Included better Windows 95 detection, now can distinguish between
   pure DOS7 and Win95. Thanks to Santiago Garcia Mantinan!
   W.Zimmermann, June 6, 97
   V2.4: Now works under Windows NT (must be compiled for NT, see makefile.dos).
   Some minor bugfixes. This release was inspired by and contains code
   supplied by Christophe Grenier. Thanks, Christophe!
   W.Zimmermann, Sept. 1, 98
   V2.5: Bugfixes in argument parsing in main.c and in readdisk.c. Again
   in cooperation with Christophe Grenier.
   W.Zimmermann, Sept. 1, 98
   V2.6: Minor modification in comp_name(). Again in cooperation with Christophe Grenier.
   W.Zimmermann, Oct. 12, 98
   V2.7: Bugfixes in path handling code
   comp_name():lread now allow to read path with . and .. 
   eatpath():lread won't anymore display / as a file
   main():correct the overflow of 'source' by eatpath()
   Christophe GRENIER, Nov. 8, 98
   Old bug not corrected: symlinks pointing to directory won't work.
   V2.70:Command line argument parsing changed, experimental write support,
   changing UID,GID,file mode works, deleting works too, copying not
   yet implemented
   New environment variable LDRIVE=/dev/... to set default drive.
   New environment variable LDIR=<LinuxPath> to set a base directory.
   W.Zimmermann with preworks by F.Miller, Jan. 12, 1999
   V2.8: Now copying Dos-Files to Linux works, but yet is not completely
   tested. The code still contains a lot of debugging stuff.
   'Support' for compiling lread/ldir under Linux discontinued.
   W.Zimmermann, Jan. 15, 1999
   V2.9: Write buffer/cache implemented to speed up lwrite.
   W.Zimmermann, Jan. 19, 1999
   V3.0: Some beautifying, no real modifications.
   W.Zimmermann, Feb. 1, 1999
   V3.1: Implemented read buffer/cache to speed up lread.
   Compilation with Gnu's gcc is no longer supported (did not work
   under V3.0 anyhow).
   ldir now tries to find out, if it's running under Windows NT and
   calls ldirNT automatically.
   W.Zimmermann, Feb. 15, 1999
   V3.2: Identical with V3.1, but LREADhttp, lread's http based graphical
   user interface included in the package.
   W.Zimmermann, Mar. 15, 1999
   V3.3: Integrated support for UNIX and QDOS operating systems (LINUX x86, 
   68K, Sun Solaris, QDOS). This support was hacked by Robert Zidlicky. 
   If you have problems with running LDREAD under UNIX systems, please 
   directly contact Richard under <rdzidlic@geocities.com> or 
   <rdzidlic@cip.informatik.uni-erlangen.de>
   readdisk.c now is a wrapper for all operating system specific func-
   tions. The original 'readdisk' is renamed to DOSdisk.h and inclu-
   ded by readdisk.c. UNIX/QDOS's harddisk access is in UNIXdisk.h, which
   alternatively is included by readdisk.c. Nevertheless there still
   is some operating system specific stuff in the other files, mainly
   to convert data structures to and from big endian, as some UNIX systems
   runs on big endian machines. dos_long_create and lock_drive have been 
   moved to DOSdisk.h.
   Richard also fixed a bug in inode.c.
   W.Zimmermann, Mar. 27, 1999      
 */

#define VERSION  "V3.3"

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <ctype.h>
#ifndef UNIX
#include <sys\stat.h>
#include <time.h>
#include <string.h>
#include <math.h>
#include <dir.h>
#endif

#include "ext2.h"
#include "proto.h"
#include "ldir.h"

#ifndef UNIX
#include <dos.h>
#include <conio.h>
#include <io.h>
#include "windows.h"
#else
#include <unistd.h>
#endif

/* !!!!!!!!!!!!!!!!!!!!!!!!!! Miller !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
extern int BLOCK_SIZE;
extern void *MALLOC(size_t size);
extern void FREE(void *block);
extern super_block sb;
extern group_desc *gt;
/* !!!!!!!!!!!!!!!!!!!!!!!!!! Ende Miller !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */

unsigned int disk_no;
unsigned int part_no;
char *disk_name;
#ifdef UNIX
int disk_fd;
#endif

char ext2_disk[32] = LINUX_DISK;
enum
{
    LDIR, LREAD, LWRITE, LTEST
}
modus = LDIR;
enum
{
    CHANGE, DEL, COPY
}
u_modus = CHANGE;
/*CHANGE        =change file access rights, GID or UID
   DEL           =delete file
   WRITE         =write file
   COPY          =copy file from DOS to Linux
 */
enum
{
    NO, YES
}
quiet = NO;

int main(int argc, char **argv)
{
    inode *in, newIn;
    int fd;
    long bytes;
    unsigned long inode_num;
    unsigned long block_num, x, y, totalBlocks, blockList[NBLOCK], blocksInBuf, *sind = NULL, *dind = NULL, *tind = NULL,
     sindBlock = 0, dindBlock = 0, tindBlock = 0, indBlocks = 0;
    char Fmode[16] = "700", Uid[16] = "0", Gid[16] = "0", inode_name[256], sourcebak[256], source[256] = "/", destin[256] = "STDOUT",
     destin_mode = NORM, *temp, ans;
    FILE *file;
    char sourceNameFound = 0;
    struct dir newDirEntry;
#ifdef UNIX
    static int blocks_per_block;
#endif

#ifndef UNIX
#if !defined(__WIN32__)
    union REGS regs;
    struct SREGS sregs;

/*try to find out if we are under Windows NT */
    temp = getenv("PROCESSOR_ARCHITECTURE");
    if (!strcmp(temp, "x86"))					/*this is Windows NT, this variable is not defined under DOS or Windows 9x */
    {
	fnsplit(argv[0], sourcebak, inode_name, NULL, NULL);	/*change ldir to ldirNT */
	fnmerge(sourcebak, sourcebak, inode_name, "ldirNT", NULL);
	for (x = 1; x < argc; x++)
	{
	    strcat(sourcebak, " ");
	    strcat(sourcebak, argv[x]);
	}
#ifdef DEBUG_MAIN
	printf("Windows NT detected  OSmajor:%d  OSminor:%d\n", _osmajor, _osminor);
#endif
	system(sourcebak);					/*under Windows NT  ldirNT.exe is called instead of ldir.exe */
	exit(0);
    }
#endif
#else
    test_endianness();
#endif

    disk_name = getenv("LDRIVE");				/*Get default drive from environment variable LDRIVE, if set */
    if (disk_name != NULL)
	strncpy(ext2_disk, disk_name, 32);			/*option overrides variable LINUX_DISK in ldir.h */
    /*and is overridden by command line param '-s' */
    temp = getenv("LDIR");					/*Get default base directory from environment variable LDIR, if set */
    if (temp != NULL)
    {
	if (temp[0] == '/')
	{
	    strncpy(source, temp, 256);
	    if (source[strlen(source) - 1] != '/')
		strcat(source, "/");
	} else
	{
	    fprintf(STDERR, "The LinuxPath base directory, which you specified with 'lcd LinuxPathBaseDirectory', must begin with '/'\n");
	    return (-1);
	}
    }
    if ((argc > 1) && strstr(argv[1], "-READ"))			/* See if its ldir, lread or lwrite */
    {
	modus = LREAD;
	x = 2;
    } else if ((argc > 1) && strstr(argv[1], "-WRITE"))
    {
	modus = LWRITE;
	x = 2;
    } else
    {
	modus = LDIR;
	x = 1;
    }

    for (; x < argc; x++)					/* Parse command line */
    {
	/* See if help is requested */
	if ((strncmp(argv[x], "-h", 2) == 0) || (strncmp(argv[1], "-v", 2) == 0) || (strncmp(argv[1], "-?", 2) == 0))
	{
	    fprintf(STDERR, "\n******************************************************************************\n");
	    fprintf(STDERR, "LDIR/LREAD/LWRITE version %s (C) 1997-99 Werner.Zimmermann@fht-esslingen.de\n", VERSION);
#ifdef UNIX
	    fprintf(STDERR, "UNIX/QDOS port by Richard Zidlicky\n");
#endif
	    fprintf(STDERR, "******************************************************************************\n");
	    fprintf(STDERR, "LDIR lists, LREAD reads and LWRITE writes files on Linux Extended 2 filesystems\n\n");
	    fprintf(STDERR, "usage: ldir    [-s/dev/hd..] /LinuxPath/LinuxDir\n");
	    fprintf(STDERR, "  or   lread   [-s/dev/hd..] /LinuxPath/LinuxFile [DosFile]\n");
	    fprintf(STDERR, "  or   lwrite  [-s/dev/hd..] [-f.. -u.. -g..] DosFile /LinuxPath/LinuxFile\n");
	    fprintf(STDERR, "  or   lchange [-s/dev/hd..] [-fMODE] [-uUID] [-gGID] /LinuxPath/LinuxFile\n");
	    fprintf(STDERR, "  or   ldel    [-s/dev/hd..] /LinuxPath/LinuxFile\n");
	    fprintf(STDERR, "  or   lcd     /LinuxBaseDirectory\n\n");
	    fprintf(STDERR, "  or   ldir -h\n\n");
	    fprintf(STDERR, "You can set your default Linux drive via 'set LDRIVE=dev/hd..'\n");
#ifndef UNIX
	    fprintf(STDERR, "To find out which partitions you have, use\n       ldir -part\n");
	    fprintf(STDERR, "If you have problems (especially SCSI drives!), read the README file\n");
	    fprintf(STDERR, "Please note: In a Windows 9x DOS box LWRITE may spin off your floppy disk drive.\n");
#endif
	    return (-1);
	}
	/* See if disk drive is specified */
	else if (strncmp(argv[x], "-s", 2) == 0)
	{
	    strncpy(ext2_disk, &argv[x][2], 32);		/* strip -s when copying */
	    continue;
	}
	/* See if it is '-q', i.e. quiet operation without warnings for WRITE operation */
	else if (strncmp(argv[x], "-q", 2) == 0)
	{
	    quiet = YES;
	    continue;
	}
	/* See if we want to display the partition table of all drives (undocumented) */
	else if (strncmp(argv[x], "-part", 5) == 0)
	{
	    modus = LTEST;
	    break;
	}
	/* See if it is -f, -u or -g */
	else if ((strncmp(argv[x], "-f", 2) == 0) || (strncmp(argv[x], "-u", 2) == 0) || (strncmp(argv[x], "-g", 2) == 0))
	{
	    if (modus == LWRITE)
	    {
		u_modus = CHANGE;				/* read and check UID, GID, FMODE is done later */
		continue;
	    } else
	    {
		fprintf(STDERR, "\n'-f','-u' or '-g' not allowed here, please try 'ldir -h'\n");
		return (-1);
	    }
	}
	/* See if it is -del */
	else if (strncmp(argv[x], "-del", 4) == 0)
	{
	    if (modus == LWRITE)
	    {
		u_modus = DEL;
		continue;
	    } else
	    {
		fprintf(STDERR, "\n'-del'not allowed here, please try 'ldir -h'\n");
		return (-1);
	    }
	}
	/* See if it is -copy */
	else if (strncmp(argv[x], "-copy", 4) == 0)
	{
	    if (modus == LWRITE)
	    {
		u_modus = COPY;
		if (argc <= x + 2)
		{
		    fprintf(STDERR, "\nYou must specify source and destination filenames\n");
		    return (-1);
		}
		strncpy(destin, source, 256);
		strncpy(source, argv[x + 1], 256);

		if (argv[x + 2][0] == '/')
		    strncpy(destin, argv[x + 2], 256);
		else
		    strncat(destin, argv[x + 2], 256 - strlen(source));
		sourceNameFound = 1;
		break;
	    } else
	    {
		fprintf(STDERR, "\n'-copy' not allowed here, please try 'ldir -h'\n");
		return (-1);
	    }
	}
	/* if it is none of these it must be a file/directory name */
	else
	{
	    if (sourceNameFound)
	    {							/* must be destination file name */
		strncpy(destin, argv[x], 256);
		if (!strcmp(destin, "."))			/*copy destin to same filename as source? */
		{
		    strcpy(destin, source);
		    destin_mode = SELF;
		}
	    } else
	    {							/* must be source file name */
		if (argv[x][0] == '/')
		    strncpy(source, argv[x], 256);
		else
		    strncat(source, argv[x], 256 - strlen(source));
		sourceNameFound = 1;
	    }
	    continue;
	}
    }

    if ((modus == LWRITE) && ((u_modus == DEL) || (u_modus == COPY)) && (quiet == NO))
    {
	fprintf(STDERR, "\nLWRITE is experimental. It may destroy your files or even your disk!\n");
	fprintf(STDERR, "USE AT YOUR OWN RISK! LWRITE will set your filesystem to 'not clear'\n");
	fprintf(STDERR, "On most installations, this will cause an automatic run of fsck when\n");
	fprintf(STDERR, "you boot up Linux. If not, run fsck (e2fsck) manually!\n\n");
	fprintf(STDERR, "Do you really want to continue [Y/N] ?  ");
	scanf("%c", &ans);
	if ((char) toupper(ans) != 'Y')
	    return (-1);
	fprintf(STDERR, "\n");
    }
    /* With LREAD or LWRITE at least a source file name must be given */
    if (((modus == LREAD) || (modus == LWRITE)) && (!sourceNameFound))
    {
	fprintf(STDERR, "You must specify a source filename (help: lread -h  or lwrite -h)\n");
	return (-1);
    }
#ifdef UNIX
    disk_name = ext2_disk;
#else
    if (strstr(ext2_disk, "/dev/hda"))				/* figure out disk number for DOS */
	disk_no = 128;
    else if (strstr(ext2_disk, "/dev/hdb"))
	disk_no = 129;
    else if (strstr(ext2_disk, "/dev/hdc"))
	disk_no = 130;						/*correct number??? */
    else if (strstr(ext2_disk, "/dev/hdd"))
	disk_no = 131;						/*correct number??? */
    else if (strstr(ext2_disk, "/dev/sda"))
	disk_no = 128;						/*correct number??? */
    else if (strstr(ext2_disk, "/dev/sdb"))
	disk_no = 129;						/*correct number??? */
    else if (strstr(ext2_disk, "/dev/fd0"))
	disk_no = 0;
    else if (strstr(ext2_disk, "/dev/fd1"))
	disk_no = 1;
    else
    {
	printf("Unknown Drive Specification: %s   %i\n", ext2_disk, disk_no);
	return (-1);
    }
    if (disk_no >= 128)						/*do it only for harddisks ??? */
    {
	strcpy(sourcebak, ext2_disk);				/*figure out partition number */
	if (strstr(ext2_disk, "dev/hd"))
	{
	    temp = strtok(sourcebak, "/dev/hd");
	} else
	{
	    temp = strtok(sourcebak, "dev/sd");
	    fprintf(STDERR, "SCSI drive support is experimental, see the README file\n");
	}
	part_no = temp[1] - 0x30;				/*convert char to int */
	if ((part_no < 1) || (part_no > 9))
	{
	    part_no = 0;
	}
    } else
	part_no = 1;						/*on floppy disks we only have one partition */
    _fmode = O_BINARY;						/* we want all files binary for this */
#endif

    if (modus != LTEST)
    {
	if (examine_drive())
	    exit(-1);
    } else
    {
	for (disk_no = 128; disk_no <= 131; disk_no++)
	{
	    part_no = 0;
	    examine_drive();
	}
	return (0);
    }

    if (load_super() == -1)					/* Couldn't get superblock */
    {
	printf("Error Loading Superblock\n");
	return (-1);
    }
#ifdef UNIX
    blocks_per_block = BLOCK_SIZE / 4;
#endif
    if (load_groups() == -1)					/* Couldn't read groups */
    {
	printf("Error Reading Groups\n");
	return (-1);
    }
#ifdef DEBUG_MAIN
    printf("Printing Group--------------------------\n");
    print_groups();
#endif

    if ((modus == LWRITE) && (u_modus == COPY))
    {
	strcpy(sourcebak, destin);
	if (eatpath(destin, inode_name, INIT))			/*Does the file exist? */
	{
	    fprintf(STDERR, "Linux-file '%s' exists - can't overwrite - please delete first\n", sourcebak);
	    return (-1);
	}
	strcpy(destin, sourcebak);
	for (x = strlen(destin) - 1; x > 0; x--)		/*Extract the path */
	{
	    if (destin[x] == '/')
		break;
	}
	destin[x + 1] = '\0';

	modus = LDIR;
	if ((inode_num = eatpath(destin, inode_name, INIT)) == 0)	/*Reload the associated directory Inode */
	{
	    fprintf(STDERR, "Invalid Linux path! LWRITE can only copy files to existing directories\n");
	    return (-1);
	}
	modus = LWRITE;
	strcpy(destin, sourcebak);				/*Extract the filename */
	temp = (char *) strrchr(destin, '/');
	strcpy(destin, (char *) &temp[1]);
    } else
    {
	strcpy(sourcebak, source);
	inode_num = eatpath(source, inode_name, INIT);
    }
    if ((inode_num == 0) && !((modus == LWRITE) && (u_modus == COPY)))	/* File or Directory not found */
    {
	printf("Linux Path Invalid\n");
	return (-1);
    }
#ifdef DEBUG_MAIN
    printf("1: inode_num=%ld\n", inode_num);
    printf("argc:%i---x:%li---Modus:%i---Source:%s---Destination:%s---\n", argc, x, modus, sourcebak, destin);
#endif

    switch (modus)						/* See if a dir is requested */
    {
	case LDIR:
	    {
		search_dir(NULL, NULL, inode_name, KILL_BUF);	/*close buffer */
		if ((in = load_inode(inode_num)) == NULL)	/*load directory inode */
		{
		    printf("Inode Error\n");
		    return (-1);
		}
		list_dir(in, sourcebak);			/*print out directory */
		return (0);
	    }

/* !!!!!!!!!!!!!!!!!!!!!!!!! Miller 18.11.98 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
	case LWRITE:
	    {
		switch (u_modus)				/*check LWRITE submode */
		{
		    case CHANGE:
			{
#ifndef UNIX
#if !defined(__WIN32__)
			    lockDrive(1);			/*lock the drive */
#endif
#endif
			    while (inode_num)
			    {
				in = load_inode(inode_num);
				if (in == NULL)
				{
				    printf("Inode Error\n");
				    return (-1);
				}
				for (x = 1; x < argc; x++)	/* Parse command line again */
				{
				    if (strncmp(argv[x], "-f", 2) == 0)		/* change file mode (access rights) */
				    {
					strncpy(Fmode, &argv[x][2], 16);	/* strip -f when copying */
					in->i_mode = in->i_mode & 0xF000;	/* set 12 rightmost bits to zero */
					if (strlen(Fmode) == 1)	/* convert octal values and set access rights */
					    in->i_mode = in->i_mode | (((int) Fmode[0] - '0') & 0x7);
					else if (strlen(Fmode) == 2)
					    in->i_mode = in->i_mode | (((int) Fmode[1] - '0') & 0x7) | (((int) Fmode[0] - '0') & 0x7) << 3;
					else if (strlen(Fmode) == 3)
					    in->i_mode = in->i_mode | (((int) Fmode[2] - '0') & 0x7) | (((int) Fmode[1] - '0') & 0x7) << 3 | (((int) Fmode[0] - '0') & 0x7) << 6;
					else if (strlen(Fmode) == 4)
					    in->i_mode = in->i_mode | (((int) Fmode[3] - '0') & 0x7) | (((int) Fmode[2] - '0') & 0x7) << 3 | (((int) Fmode[1] - '0') & 0x7) << 6 | (((int) Fmode[0] - '0') & 0x7) << 9;
					else
					    fprintf(STDERR, "\nAccess rights must be between 1 and three octal digits\n");
				    } else if (strncmp(argv[x], "-u", 2) == 0)	/*change UID */
				    {
					strncpy(Uid, &argv[x][2], 16);	/* strip -u when copying */
					in->i_uid = atoi(Uid);
				    } else if (strncmp(argv[x], "-g", 2) == 0)	/*change GID */
				    {
					strncpy(Gid, &argv[x][2], 16);	/* strip -g when copying */
					in->i_gid = atoi(Gid);
				    }
				}
				store_inode(inode_num, in);
				inode_num = search_dir(NULL, NULL, inode_name, KEEP_BUF);
			    }
			    search_dir(NULL, NULL, inode_name, KILL_BUF);
#ifndef UNIX
#if !defined(__WIN32__)
			    lockDrive(0);			/*unlock the drive */
#endif
#endif
			    return (0);
			}
/* !!!!!!!!!!!!!!!!!!!!!!!!! Ende Miller 18.11.98 !!!!!!!!!!!!!!!!!!!!!!!!!! */
		    case DEL:
			{
/* This is incomplete:
   -When the filename contains wildcards, this only works with the first file.
 */
#ifndef UNIX
#if !defined(__WIN32__)
			    lockDrive(1);			/*lock the drive */
#endif
#endif
			    search_dir(NULL, NULL, inode_name, KILL_BUF);	/*Free buffer */
			    if ((in = load_inode(inode_num)) == NULL)	/*Load Directory inode */
			    {
				fprintf(STDERR, "Could not load directory inode\n");
				return (-1);
			    }
			    strcpy(newDirEntry.name, "");
			    newDirEntry.inode_num = 0;
			    if ((inode_num = modify_dir(in, inode_name, &newDirEntry, 0)) == 0)		/*Delete entry in directory table */
			    {
				fprintf(STDERR, "Could not delete file entry in directory\n");
				return (-1);
			    }
			    if ((in = load_inode(inode_num)) == NULL)	/*Load file inode */
			    {
				fprintf(STDERR, "Could not load file inode\n");
				return (-1);
			    }
			    block_num = block_list(in);		/*get first block number */
			    while (block_num > 0)
			    {
				modifyBlockBitmap(block_num, 0);	/*clear blocks in block bitmap */
#ifdef DEBUG_MAIN
				printf("Deleting %4ld.block: blocknum=%ld\n", x++, block_num);
#endif
				sb.s_free_blocks_count++;	/*adapt super block's free block count */
				gt[(block_num - 1) / sb.s_blocks_per_group].bg_free_blocks_count++;
				block_num = block_list(NULL);	/*get next block number */
			    }
			    modifyBlockBitmap((int) NULL, 4);	/*write bitmap to disk */

			    modifyInodeBitmap(inode_num, 0);	/*clear inode in inode bitmap */
			    modifyInodeBitmap((int) NULL, 4);	/*write bitmap to disk */
			    sb.s_free_inodes_count++;		/*adapt superblock's free inode count */
			    gt[(inode_num - 1) / sb.s_inodes_per_group].bg_free_inodes_count++;

			    store_groups();

			    memset(in, 0, sizeof(inode));	/*clear file inode */
			    store_inode(inode_num, in);		/*write inode to disk */

			    /*for debugging: */ sb.s_state = 0;
			    /*mark file system 'not clean' */
			    if (write_super() == -1)		/*write superblock to disk */
			    {
				printf("Error Writing Superblock\n");
				return (-1);
			    }
#ifndef UNIX
#if !defined(__WIN32__)
			    lockDrive(0);			/*unlock the drive */
#endif
#endif
			    return (0);
			}
		    case COPY:
			{
			    search_dir(NULL, NULL, inode_name, KILL_BUF);	/*Free buffer */
			    if ((file = fopen(source, "rb")) == NULL)
			    {
				fprintf(STDERR, "Can't open DOS-file '%s'", source);
				return (-1);
			    }
			    if (!((sb.s_free_inodes_count > 0) && (sb.s_free_blocks_count > 0)))
			    {
				fprintf(STDERR, "Disk full no free inodes or blocks\n");
				return (-1);
			    }
			    if ((temp = MALLOC(NBLOCK * BLOCK_SIZE)) == NULL)	/*allocate temp. buffer */
			    {
				fprintf(STDERR, "Memory allocation error in lwrite\n");
				return (-1);
			    }
#ifdef DEBUG_MAIN
			    printf("\ninode_num=%ld\n", inode_num);
#endif
			    if ((in = load_inode(inode_num)) == NULL)	/*Load Directory inode */
			    {
				fprintf(STDERR, "Could not load directory inode\n");
				return (-1);
			    }
			    /*Looking for a free inode */
			    for (inode_num = 1; inode_num <= sb.s_inodes_count; inode_num++)
			    {
				if (modifyInodeBitmap(inode_num, 2) == 0)
				    break;
			    }
#ifdef DEBUG_MAIN
			    printf("Found free inode=%ld\n", inode_num);
#endif
			    newDirEntry.inode_num = inode_num;
			    strcpy(newDirEntry.name, destin);
#ifndef UNIX
#if !defined(__WIN32__)
			    lockDrive(1);			/*lock the drive */
#endif
#endif
			    modify_dir(in, "", &newDirEntry, 1);	/*Insert entry in directory table */

			    modifyInodeBitmap(inode_num, 1);	/*set inode in inode bitmap */
			    modifyInodeBitmap((int) NULL, 4);	/*write bitmap to disk */

			    block_num = 1;
			    memset(&newIn, 0, sizeof(inode));	/*initialize new inode */
			    totalBlocks = bytes = blocksInBuf = 0;

			    do
			    {					/*read DOS file into temp buffer */
				bytes = fread(temp, 1, NBLOCK * BLOCK_SIZE, file);
#ifdef DEBUG_MAIN
				printf("reading %lu byte from DOS\n", bytes);
#endif
				if (bytes == 0)
				    break;			/*EOF */

				blocksInBuf = bytes / BLOCK_SIZE;
				if (bytes % BLOCK_SIZE)
				    blocksInBuf++;

				/*Allocate the blocks */
				for (x = 0; x < blocksInBuf; x++)
				{				/*triple indirection handling */
				    if ((totalBlocks - 12 - (BLOCK_SIZE / 4)) == ((BLOCK_SIZE / 4L) * (BLOCK_SIZE / 4L)))
				    {
					if ((tind = (unsigned long *) MALLOC(BLOCK_SIZE)) == NULL)
					{
					    fprintf(STDERR, "Memory allocation error during file copy\n");
					    blocksInBuf = x;
					    break;
					}
					/*Looking for a free block */
					for (; block_num <= sb.s_blocks_count; block_num++)	/*continue at the last block_num */
					{
					    if (modifyBlockBitmap(block_num, 2) == 0)
						break;
					}
					if (block_num > sb.s_blocks_count)
					{
					    fprintf(STDERR, "Disk full no more blocks free - could not completely copy file\n");
					    blocksInBuf = x;
					    break;
					}
					modifyBlockBitmap(block_num, 1);	/*set blocks in block bitmap */
					sb.s_free_blocks_count--;	/*adapt super block's free block count */
					gt[(block_num - 1) / sb.s_blocks_per_group].bg_free_blocks_count--;

					tindBlock = block_num;
#ifdef DEBUG_MAIN
					printf("totalBlocks=%ld  Triple indirection block=%ld\n", totalBlocks, tindBlock);
#endif
					newIn.i_block[14] = tindBlock;

					memset((byte *) tind, 0, BLOCK_SIZE);
					indBlocks++;
				    }
				    /*double indirection handling */
				    if ((totalBlocks - 12 - (BLOCK_SIZE / 4)) % ((BLOCK_SIZE / 4L) * (BLOCK_SIZE / 4L)) == 0)
				    {
					if (dind != NULL)
					{
#ifdef DEBUG_MAIN
					    printf("writing double indirection block\n");
#endif

#ifdef UNIX
					    {			/* fix byteorder, reverse done in inode.c */
						int i;
						_u32 *p = dind;
						for (i = 0; i < blocks_per_block; i++, p++)
						    *p = cpu_to_le32(*p);
					    }
#endif
					    writedisk((byte *) dind, dindBlock * BLOCK_SIZE, BLOCK_SIZE);
					} else
					{
					    if ((dind = (unsigned long *) MALLOC(BLOCK_SIZE)) == NULL)
					    {
						fprintf(STDERR, "Memory allocation error during file copy\n");
						blocksInBuf = x;
						break;
					    }
					}
					/*Looking for a free block */
					for (; block_num <= sb.s_blocks_count; block_num++)	/*continue at the last block_num */
					{
					    if (modifyBlockBitmap(block_num, 2) == 0)
						break;
					}
					if (block_num > sb.s_blocks_count)
					{
					    fprintf(STDERR, "Disk full no more blocks free - could not completely copy file\n");
					    blocksInBuf = x;
					    break;
					}
					modifyBlockBitmap(block_num, 1);	/*set blocks in block bitmap */
					sb.s_free_blocks_count--;	/*adapt super block's free block count */
					gt[(block_num - 1) / sb.s_blocks_per_group].bg_free_blocks_count--;

					dindBlock = block_num;
#ifdef DEBUG_MAIN
					printf("totalBlocks=%ld  Double indirection block=%ld\n", totalBlocks, dindBlock);
#endif
					if (totalBlocks == 12 + BLOCK_SIZE / 4)
					    newIn.i_block[13] = dindBlock;
					else
					    tind[((totalBlocks - 12) / (BLOCK_SIZE / 4) - 1) % (BLOCK_SIZE / 4)] = dindBlock;

					memset((byte *) dind, 0, BLOCK_SIZE);
					indBlocks++;
				    }
				    /*single indirection handling */
				    if ((totalBlocks - 12) % (BLOCK_SIZE / 4) == 0)
				    {
					if (sind != NULL)	/*write previous sind-block */
					{
#ifdef DEBUG_MAIN
					    printf("writing single indirection block\n");
#endif
#ifdef UNIX
					    {			/* fix byteorder, reverse done in inode.c */
						int i;
						_u32 *p = sind;
						for (i = 0; i < blocks_per_block; i++, p++)
						    *p = cpu_to_le32(*p);
					    }
#endif
					    writedisk((byte *) sind, sindBlock * BLOCK_SIZE, BLOCK_SIZE);
					} else
					{
					    if ((sind = (unsigned long *) MALLOC(BLOCK_SIZE)) == NULL)
					    {
						fprintf(STDERR, "Memory allocation error during file copy\n");
						blocksInBuf = x;
						break;
					    }
					}
					/*Looking for a free block */
					for (; block_num <= sb.s_blocks_count; block_num++)	/*continue at the last block_num */
					{
					    if (modifyBlockBitmap(block_num, 2) == 0)
						break;
					}
					if (block_num > sb.s_blocks_count)
					{
					    fprintf(STDERR, "Disk full no more blocks free - could not completely copy file\n");
					    blocksInBuf = x;
					    break;
					}
					modifyBlockBitmap(block_num, 1);	/*set blocks in block bitmap */
					sb.s_free_blocks_count--;	/*adapt super block's free block count */
					gt[(block_num - 1) / sb.s_blocks_per_group].bg_free_blocks_count--;

					sindBlock = block_num;
#ifdef DEBUG_MAIN
					printf("totalBlocks=%ld  Single Indirection block=%ld\n", totalBlocks, sindBlock);
#endif
					if (totalBlocks == 12)
					    newIn.i_block[12] = sindBlock;
					else
					    dind[((totalBlocks - 12) / (BLOCK_SIZE / 4) - 1) % (BLOCK_SIZE / 4)] = sindBlock;

					memset((byte *) sind, 0, BLOCK_SIZE);
					indBlocks++;
				    }
				    /*Looking for a free block */
				    for (; block_num <= sb.s_blocks_count; block_num++)		/*continue at the last block_num */
				    {
					if (modifyBlockBitmap(block_num, 2) == 0)
					    break;
				    }
				    if (block_num > sb.s_blocks_count)
				    {
					fprintf(STDERR, "Disk full no more blocks free - could not completely copy file\n");
					blocksInBuf = x;
					break;
				    }
				    modifyBlockBitmap(block_num, 1);	/*set blocks in block bitmap */
				    sb.s_free_blocks_count--;	/*adapt free block counts */
				    gt[(block_num - 1) / sb.s_blocks_per_group].bg_free_blocks_count--;

				    if (totalBlocks < 12)
				    {
					newIn.i_block[totalBlocks] = block_num;		/*pointers to blocks (no indirection) */
					totalBlocks++;
				    } else
					/*pointers to blocks (indirection) */
				    {
					sind[(totalBlocks - 12) % (BLOCK_SIZE / 4)] = block_num;
					totalBlocks++;
				    }
				    blockList[x] = block_num;
				}

/*##############################################################################
   Note: what if we find no more free blocks -> break statements in the above block allocation routines
   we have to make sure, that we do not write too many blocks
   ##############################################################################
 */

				/*now write the blocks to the Linux disk */
				for (x = 0; x < blocksInBuf; x++)
				{
				    if (disk_no < 128)		/*with diskettes we write block per block */
				    {
					writedisk((byte *) & temp[x * BLOCK_SIZE], blockList[x] * BLOCK_SIZE, BLOCK_SIZE);	/*write to Linux-disk */
				    } else
					/*with harddisk we do a multiblock write for blocks in sequence, this assumes a disk in LBA mode!!!
					   else we may get write errors #4 !!! */
				    {
					for (y = x + 1; y < blocksInBuf; y++)
					{
					    if (blockList[y] != blockList[y - 1] + 1)
						break;
					}
#ifdef DEBUG_MAIN
					printf("x=%ld Sequence of %ld blocks from block=%ld to %ld\n", x, y - x, blockList[x], blockList[y - 1]);
#endif
					writedisk((byte *) & temp[x * BLOCK_SIZE], blockList[x] * BLOCK_SIZE, (y - x) * BLOCK_SIZE);	/*write to Linux-disk */
					x = y - 1;
				    }
				}

				newIn.i_size = newIn.i_size + bytes;	/*file size */

			    }
			    while (bytes == NBLOCK * BLOCK_SIZE);
			    modifyBlockBitmap((int) NULL, 4);		/*write bitmap to disk */

			    if (sind != NULL)			/*write sind-block */
			    {
#ifdef DEBUG_MAIN
				printf("writing single indirection block\n");
#endif
#ifdef UNIX
				{				/* fix byteorder, reverse done in inode.c */
				    int i;
				    _u32 *p = sind;
				    for (i = 0; i < blocks_per_block; i++, p++)
					*p = cpu_to_le32(*p);
				}
#endif
				writedisk((byte *) sind, sindBlock * BLOCK_SIZE, BLOCK_SIZE);
				FREE(sind);
			    }
			    if (dind != NULL)			/*write dind-block */
			    {
#ifdef DEBUG_MAIN
				printf("writing double indirection block\n");
#endif
#ifdef UNIX
				{				/* fix byteorder, reverse done in inode.c */
				    int i;
				    _u32 *p = dind;
				    for (i = 0; i < blocks_per_block; i++, p++)
					*p = cpu_to_le32(*p);
				}
#endif
				writedisk((byte *) dind, dindBlock * BLOCK_SIZE, BLOCK_SIZE);
				FREE(dind);
			    }
			    if (tind != NULL)			/*write tind-block */
			    {
#ifdef DEBUG_MAIN
				printf("writing triple indirection block\n");
#endif
#ifdef UNIX
				{				/* fix byteorder, reverse done in inode.c */
				    int i;
				    _u32 *p = tind;
				    for (i = 0; i < blocks_per_block; i++, p++)
					*p = cpu_to_le32(*p);
				}
#endif
				writedisk((byte *) tind, tindBlock * BLOCK_SIZE, BLOCK_SIZE);
				FREE(tind);
			    }
			    sb.s_free_inodes_count--;		/*adapt superblock's free inode count */
			    gt[(inode_num - 1) / sb.s_inodes_per_group].bg_free_inodes_count--;

			    store_groups();

			    newIn.i_mode = 0x81FF;		/* File mode */
			    newIn.i_uid = 0;			/* Owner Uid */
			    newIn.i_gid = 0;			/* Group Id */
			    newIn.i_ctime = time(NULL);		/* File creation time */
			    newIn.i_links_count = 1;		/* Links count */
			    newIn.i_blocks = (totalBlocks + indBlocks) * 2;	/* Blocks count */

			    for (x = 1; x < argc; x++)		/* Parse command line again */
			    {
				if (strncmp(argv[x], "-f", 2) == 0)	/* change file mode (access rights) */
				{
				    strncpy(Fmode, &argv[x][2], 16);	/* strip -f when copying */
				    newIn.i_mode = 0x8000;	/* set 9 rightmost bits to zero */
				    if (strlen(Fmode) == 1)	/* convert octal values and set access rights */
					newIn.i_mode = newIn.i_mode | (((int) Fmode[0] - '0') & 0x7);
				    else if (strlen(Fmode) == 2)
					newIn.i_mode = newIn.i_mode | (((int) Fmode[1] - '0') & 0x7) | (((int) Fmode[0] - '0') & 0x7) << 3;
				    else if (strlen(Fmode) == 3)
					newIn.i_mode = newIn.i_mode | (((int) Fmode[2] - '0') & 0x7) | (((int) Fmode[1] - '0') & 0x7) << 3 | (((int) Fmode[0] - '0') & 0x7) << 6;
				    else if (strlen(Fmode) == 4)
					newIn.i_mode = newIn.i_mode | (((int) Fmode[3] - '0') & 0x7) | (((int) Fmode[2] - '0') & 0x7) << 3 | (((int) Fmode[1] - '0') & 0x7) << 6 | (((int) Fmode[0] - '0') & 0x7) << 9;
				    else
					fprintf(STDERR, "\nAccess rights must be between 1 and three octal digits\n");
				} else if (strncmp(argv[x], "-u", 2) == 0)	/*change UID */
				{
				    strncpy(Uid, &argv[x][2], 16);	/* strip -u when copying */
				    newIn.i_uid = atoi(Uid);
				} else if (strncmp(argv[x], "-g", 2) == 0)	/*change GID */
				{
				    strncpy(Gid, &argv[x][2], 16);	/* strip -g when copying */
				    newIn.i_gid = atoi(Gid);
				}
			    }
#ifdef DEBUG_MAIN
			    printf("writing inode\n");
#endif
			    store_inode(inode_num, &newIn);	/*write inode to disk */

			    /*for debugging: */ sb.s_state = 0;
			    /*mark file system 'not clean' */
#ifdef DEBUG_MAIN
			    printf("writing superblock\n");
#endif
			    if (write_super() == -1)		/*write superblock to disk */
			    {
				printf("Error Writing Superblock\n");
				exit(-1);
			    }
#ifndef UNIX
#if !defined(__WIN32__)
			    lockDrive(0);			/*unlock the drive */
#endif
#endif
			    FREE(temp);
			    return (0);
			}
		}
		break;
	    }

	case LREAD:
	    {
		if ((temp = MALLOC(NBLOCK * BLOCK_SIZE)) == NULL)
		{
		    fprintf(STDERR, "Memory allocation error in lread\n");
		    return (-1);
		}
		while (inode_num)
		{
		    if ((in = load_inode(inode_num)) == NULL)
		    {
			fprintf(STDERR, "Inode Error\n");
			return (-1);
		    }
		    if (destin_mode == SELF)
			strncpy(destin, inode_name, sizeof(destin));
		    if (strcmp(destin, "STDOUT"))		/*See if output to a file */
		    {
			if ((quiet == NO) && ((file = fopen(destin, "rb")) != NULL))
			{
			    fclose(file);
			    fprintf(STDERR, "File '%s' exists, overwrite? [Y/N]  ", destin);
			    scanf("%c", &ans);
			    if ((char) toupper(ans) != 'Y')
			    {
				fprintf(STDERR, "Could not copy file '%s' to '%s'\n", source, destin);
				return (-1);
			    }
			}
#ifndef UNIX
#if !defined(__WIN32__)
			fd = dos_long_creat(destin);
#else
			fd = creat(destin, S_IREAD | S_IWRITE | O_BINARY);
#endif
#else
			fd = creat(destin, O_RDWR);
#endif
			if (fd == -1)
			{
			    printf("Problem opening second argument file (DOS file) %s %s\n", sourcebak, destin);
			    return (-1);
			}
		    } else
			/*we want output to stdout */
		    {
			fd = 1;
		    }

		    if ((bytes = read_inode(in, temp, MBLOCK * BLOCK_SIZE, KEEP_BUF)) == -1)	/* read first blocks */
		    {
			fprintf(STDERR, "Error Reading Inode\n");
			return (0);
		    }
		    do
		    {
			if ((x = write(fd, temp, bytes)) != bytes)	/* write to disk */
			{
			    fprintf(STDERR, "Error Writing DOS File %s  written %lu Byte  should write %lu Byte - Probably disk full\n", destin, x, bytes);
			    return (-1);
			}
			bytes = read_inode(NULL, temp, MBLOCK * BLOCK_SIZE, KEEP_BUF);
			if (bytes == -1)			/* problem reading */
			{
			    printf("Error Reading 2\n");
			    return (-1);
			}
		    }
		    while (bytes != 0);
		    if (bytes)
			printf("*** Problem encountered reading file %s***\n", sourcebak);
		    if (fd > 2)
			close(fd);
		    read_inode(NULL, NULL, 0, KILL_BUF);	/*release the buffer */
		    strcpy(source, sourcebak);
		    inode_num = search_dir(NULL, NULL, inode_name, KEEP_BUF);
		}
		search_dir(NULL, NULL, inode_name, KILL_BUF);
		FREE(temp);
		break;
	    }
	case LTEST:
	    {							/*we never should come here */
	    }

    }
    return (0);
}
