/*
 * Decompiled with CFR 0.152.
 */
package org.apache.asterix.transaction.management.service.recovery;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.channels.ClosedByInterruptException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.apache.asterix.common.exceptions.ACIDException;
import org.apache.asterix.common.transactions.Checkpoint;
import org.apache.asterix.common.transactions.CheckpointProperties;
import org.apache.asterix.common.transactions.ICheckpointManager;
import org.apache.asterix.common.transactions.ILogManager;
import org.apache.asterix.common.transactions.ITransactionManager;
import org.apache.asterix.common.transactions.ITransactionSubsystem;
import org.apache.asterix.transaction.management.service.recovery.CheckpointThread;
import org.apache.hyracks.api.exceptions.HyracksDataException;
import org.apache.hyracks.api.io.IPersistedResourceRegistry;
import org.apache.hyracks.api.util.IoUtil;
import org.apache.hyracks.util.file.FileUtil;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.Supplier;

public abstract class AbstractCheckpointManager
implements ICheckpointManager {
    private static final Logger LOGGER = LogManager.getLogger();
    private static final String CHECKPOINT_FILENAME_PREFIX = "checkpoint_";
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    public static final long SHARP_CHECKPOINT_LSN = -1L;
    private static final FilenameFilter filter = (dir, name) -> name.startsWith(CHECKPOINT_FILENAME_PREFIX);
    private static final long FIRST_CHECKPOINT_ID = 0L;
    private final File checkpointDir;
    private final int historyToKeep;
    private final int lsnThreshold;
    private final int pollFrequency;
    private final IPersistedResourceRegistry persistedResourceRegistry;
    protected final ITransactionSubsystem txnSubsystem;
    private CheckpointThread checkpointer;

    public AbstractCheckpointManager(ITransactionSubsystem txnSubsystem, CheckpointProperties checkpointProperties) {
        this.txnSubsystem = txnSubsystem;
        String checkpointDirPath = checkpointProperties.getCheckpointDirPath();
        if (LOGGER.isInfoEnabled()) {
            LOGGER.log(Level.INFO, "Checkpoint directory = " + checkpointDirPath);
        }
        if (!checkpointDirPath.endsWith(File.separator)) {
            checkpointDirPath = checkpointDirPath + File.separator;
        }
        this.checkpointDir = new File(checkpointDirPath);
        if (!this.checkpointDir.exists()) {
            if (LOGGER.isInfoEnabled()) {
                LOGGER.log(Level.INFO, "Checkpoint directory " + checkpointDirPath + " didn't exist. Creating one");
            }
            this.checkpointDir.mkdirs();
        }
        this.lsnThreshold = checkpointProperties.getLsnThreshold();
        this.pollFrequency = checkpointProperties.getPollFrequency();
        this.historyToKeep = checkpointProperties.getHistoryToKeep() + 1;
        this.persistedResourceRegistry = txnSubsystem.getApplicationContext().getPersistedResourceRegistry();
    }

    public Checkpoint getLatest() {
        LOGGER.log(Level.INFO, "Getting latest checkpoint");
        List<File> checkpointFiles = this.getCheckpointFiles();
        if (checkpointFiles.isEmpty()) {
            return null;
        }
        List<Checkpoint> orderedCheckpoints = this.getOrderedValidCheckpoints(checkpointFiles, false);
        if (orderedCheckpoints.isEmpty()) {
            return this.forgeForceRecoveryCheckpoint();
        }
        return orderedCheckpoints.get(orderedCheckpoints.size() - 1);
    }

    public void start() {
        this.checkpointer = new CheckpointThread(this, this.txnSubsystem.getLogManager(), this.lsnThreshold, (long)this.pollFrequency);
        this.checkpointer.start();
    }

    public void stop(boolean dumpState, OutputStream ouputStream) throws IOException {
        this.checkpointer.shutdown();
        this.checkpointer.interrupt();
        try {
            this.checkpointer.join();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    public void dumpState(OutputStream os) throws IOException {
    }

    public Path getCheckpointPath(long checkpointId) {
        return Paths.get(this.checkpointDir.getAbsolutePath() + File.separator + CHECKPOINT_FILENAME_PREFIX + checkpointId, new String[0]);
    }

    protected void capture(long minMCTFirstLSN, boolean sharp) throws HyracksDataException {
        ILogManager logMgr = this.txnSubsystem.getLogManager();
        ITransactionManager txnMgr = this.txnSubsystem.getTransactionManager();
        long nextCheckpointId = this.getNextCheckpointId();
        Checkpoint checkpointObject = new Checkpoint(nextCheckpointId, logMgr.getAppendLSN(), minMCTFirstLSN, txnMgr.getMaxTxnId(), sharp, 12);
        this.persist(checkpointObject);
        this.cleanup();
    }

    private Checkpoint forgeForceRecoveryCheckpoint() {
        return new Checkpoint(Long.MIN_VALUE, Long.MIN_VALUE, Integer.MIN_VALUE, 0L, false, 12);
    }

    private void persist(Checkpoint checkpoint) throws HyracksDataException {
        File file;
        Path path = this.getCheckpointPath(checkpoint.getId());
        if (LOGGER.isInfoEnabled()) {
            file = path.toFile();
            LOGGER.log(Level.INFO, "Persisting checkpoint file to " + file + " which " + (file.exists() ? "already exists" : "doesn't exist yet"));
        }
        try {
            if (path.toFile().exists()) {
                IoUtil.delete((Path)path);
            }
            byte[] bytes = OBJECT_MAPPER.writeValueAsBytes((Object)checkpoint.toJson(this.persistedResourceRegistry));
            FileUtil.writeAndForce((Path)path, (byte[])bytes);
            this.readCheckpoint(path);
        }
        catch (IOException e) {
            LOGGER.log(Level.ERROR, "Failed to write checkpoint to disk", (Throwable)e);
            throw HyracksDataException.create((Throwable)e);
        }
        if (LOGGER.isInfoEnabled()) {
            file = path.toFile();
            LOGGER.log(Level.INFO, "Completed persisting checkpoint file to " + file + " which now " + (file.exists() ? "exists" : " still doesn't exist"));
        }
    }

    private List<File> getCheckpointFiles() {
        Object[] checkpoints = this.checkpointDir.listFiles(filter);
        if (checkpoints == null || checkpoints.length == 0) {
            if (LOGGER.isInfoEnabled()) {
                LOGGER.log(Level.INFO, "Listing of files in the checkpoint dir returned " + (checkpoints == null ? "null" : "empty"));
            }
            return Collections.emptyList();
        }
        if (LOGGER.isInfoEnabled()) {
            LOGGER.log(Level.INFO, "Listing of files in the checkpoint dir returned " + Arrays.toString(checkpoints));
        }
        return Arrays.asList(checkpoints);
    }

    private List<Checkpoint> getOrderedValidCheckpoints(List<File> checkpoints, boolean deleteCorrupted) {
        ArrayList<Checkpoint> checkpointObjectList = new ArrayList<Checkpoint>();
        for (File file : checkpoints) {
            try {
                if (LOGGER.isWarnEnabled()) {
                    LOGGER.log(Level.WARN, "Reading checkpoint file: " + file.getAbsolutePath());
                }
                Checkpoint cp = this.readCheckpoint(Paths.get(file.getAbsolutePath(), new String[0]));
                checkpointObjectList.add(cp);
            }
            catch (ClosedByInterruptException e) {
                Thread.currentThread().interrupt();
                if (LOGGER.isWarnEnabled()) {
                    LOGGER.log(Level.WARN, "Interrupted while reading checkpoint file: " + file.getAbsolutePath(), (Throwable)e);
                }
                throw new ACIDException((Throwable)e);
            }
            catch (IOException e) {
                if (LOGGER.isWarnEnabled()) {
                    LOGGER.log(Level.WARN, "Failed to read checkpoint file: " + file.getAbsolutePath(), (Throwable)e);
                }
                if (!deleteCorrupted || !file.delete()) continue;
                Supplier[] supplierArray = new Supplier[1];
                supplierArray[0] = file::getAbsolutePath;
                LOGGER.warn("Deleted corrupted checkpoint file: {}", supplierArray);
            }
        }
        Collections.sort(checkpointObjectList);
        return checkpointObjectList;
    }

    private void cleanup() {
        List<File> checkpointFiles = this.getCheckpointFiles();
        List<Checkpoint> orderedCheckpoints = this.getOrderedValidCheckpoints(checkpointFiles, true);
        int deleteCount = orderedCheckpoints.size() - this.historyToKeep;
        for (int i = 0; i < deleteCount; ++i) {
            Checkpoint checkpoint = orderedCheckpoints.get(i);
            Path checkpointPath = this.getCheckpointPath(checkpoint.getId());
            LOGGER.warn("Deleting checkpoint file at: {}", (Object)checkpointPath);
            if (checkpointPath.toFile().delete()) continue;
            LOGGER.warn("Could not delete checkpoint file at: {}", (Object)checkpointPath);
        }
    }

    private long getNextCheckpointId() {
        List<File> checkpointFiles = this.getCheckpointFiles();
        if (checkpointFiles.isEmpty()) {
            return 0L;
        }
        long maxOnDiskId = -1L;
        for (File checkpointFile : checkpointFiles) {
            long fileId = Long.parseLong(checkpointFile.getName().substring(CHECKPOINT_FILENAME_PREFIX.length()));
            maxOnDiskId = Math.max(maxOnDiskId, fileId);
        }
        return maxOnDiskId + 1L;
    }

    private Checkpoint readCheckpoint(Path checkpointPath) throws IOException {
        JsonNode jsonNode = (JsonNode)OBJECT_MAPPER.readValue(Files.readAllBytes(checkpointPath), JsonNode.class);
        return (Checkpoint)this.persistedResourceRegistry.deserialize(jsonNode);
    }
}

