/* fontreader.c - 19:53 GMT +10:00 Thu 29 July 1993 - modifier Geoffrey Tobin */

/* From input file "../include/globals.p" */

#include "config.h"
#include "globals.h"
#include "files.h"
#include "screenio.h"
#include "vdu.h"
#include "options.h"
#include "dvireader.h"
#include "fontreader.h"

#include "new.h"
#include "lstring.h"


typedef struct _REC_TFMinfo
{
  int wdindex, htindex, dpindex;
} _REC_TFMinfo;

typedef struct _REC_charmetrics
{
  int width[4], height[4], depth[4];
} _REC_charmetrics;


Static int PTfile;        /* PK/TFM file descriptor              */
Static int PToffset;      /* current byte offset in PTfile       */
Static int currPTbuff;    /* starting byte offset in buffer      */
Static buffer PTbuffer;   /* input buffer                        */
Static int gpower[33];    /* 0,1,11,111,1111,...                 */
Static boolean turnon;    /* is current run black?               */
Static int dynf;          /* dynamic packing variable            */
Static int repeatcount;   /* times to repeat the next row        */
Static int bitweight;     /* for bits or nybbles from inputbyte  */
Static int inputbyte;     /* the current input byte              */
Static int lf, lh, bc, ec, nw, nh;  /* TFM file data; lf unused  */
Static _REC_TFMinfo TFMinfo[maxTeXchar+1];
Static _REC_charmetrics charmetrics[maxTeXchar+1];


/******************************************************************************/

#ifdef __STDC__
boolean CheckFontPtr (fontinfo * fontptr)
#else
boolean CheckFontPtr (fontptr)
    fontinfo * fontptr;
#endif
{
  boolean check = false;
  if (fontptr == (fontinfo *) NULL)
  {
    MesgString ("NULL font info passed to CheckFontPtr!");
    MesgLine();
    check = false;
  }
  else if (fontptr->fontnamelen == 0)
  {
    MesgString ("Zero length font name in font info passed to CheckFontPtr!");
    MesgLine();
    check = false;
  }
  else if (!strcmp (fontptr->fontname, ""))
  {
    MesgString ("Empty font name in font info passed to CheckFontPtr!");
    MesgLine();
    check = false;
  }
  else
  {
    check = true;
  }
  return check;
}
/* CheckFontPtr */

/******************************************************************************/

#ifdef __STDC__
boolean BuildFileSpec (const char * areas, const char * name, string spec)
#else
boolean BuildFileSpec (areas, name, spec)
    char * areas, * name;
    string spec;
#endif
{
  /* Build a complete file specification in spec.
     Firstly, test for valid areas and name.
  */

  boolean found = false, more = false;
  string paths;
  char * s = (char *) NULL;
  LString * Lspec = NewLString (maxfontspec);

  if (areas == (char *) NULL)
  {
    MesgString ("NULL areas passed to BuildFileSpec!");
    MesgLine();
    return false;
  }
  else if (strlen (areas) == 0)
  {
    MesgString ("Zero length areas passed to BuildFileSpec!");
    MesgLine();
    return false;
  }

  if (name == (char *) NULL)
  {
    MesgString ("NULL file name passed to BuildFileSpec!");
    MesgLine();
    return false;
  }
  else if (strlen (name) == 0)
  {
    MesgString ("Zero length file name passed to BuildFileSpec!");
    MesgLine();
    return false;
  }

  /* copy areas to paths because strtok() overwrites */
  strncpy (paths, areas, maxfontspec);
  paths[maxfontspec] = '\0';  /* to be careful */

  found = false;
  more = true;

  /* loop for each directory listed in paths */
  for (s = paths; !found && more; s = (char *) NULL)
  {
    /* NOTE:  strtok returns NULL if all characters of s occur in DIRSEP */
    char * onepath = strtok (s, DIRSEP);  /* Hope strtok is common in C's */
    if (onepath == (char *) NULL)
    {
      more = false;
    }
    else
    {
      /* start file spec. with one directory */
      Scopys (Lspec, onepath);
      Scats (Lspec, "/");

      /* append name */
      Scats (Lspec, name);

      strncpy (spec, Lspec->s, maxfontspec);
      spec[maxfontspec] = '\0';  /* to be careful */

      /* check whether file exists where spec indicates */
      { /* start block */
        FILE * fp = fopen (spec, "r");
        if (fp != (FILE *) NULL)
        {
          fclose (fp);
          found = true;
        }
      } /* end block */
    } /* fi */
  } /* loop for each directory listed in paths */

  FreeLString (Lspec);

  return found;
}
/* BuildFileSpec */

/******************************************************************************/

#ifdef __STDC__
Static boolean BuildTFMSpec (fontinfo *tfmptr)
#else
Static boolean BuildTFMSpec (tfmptr)
    fontinfo *tfmptr;
#endif
{
  /* Build a complete TFM file specification in tfmptr->fontspec.
     This will be done at most once per font.
     Test for valid tfmptr->fontname.
     ASSUMPTIONS:
       tfmptr->fontname contains the basename of the font.
     SIDE EFFECTS on success:
       tfmptr->fontspec will be set.
       tfmptr->fontspeclen will no longer be 0.
       tfmptr->fontexists becomes TRUE if the TFM file can be opened,
                                  FALSE, if it cannot.
       tfmptr->pkfont becomes FALSE.
       (tfmptr->psfont is unaffected.)
     SIDE EFFECTS on failure:
         EITHER:
       tfmptr is NULL.
         OR:
       tfmptr->fontspec will be unset (holds empty string "").
       tfmptr->fontspeclen will be 0.
       tfmptr->fontexists becomes FALSE.
       tfmptr->pkfont becomes FALSE.
       tfmptr->psfont becomes FALSE.
  */

  boolean found = false, more = false;
  string tfmdirs;
  char * s = (char *) NULL;
  LString * Ltfmspec = NewLString (maxfontspec);

  if (!CheckFontPtr (tfmptr))
    return false;

  /* Initialise correctly */
  strncpy (tfmptr->fontspec, "", maxfontspec);
  tfmptr->fontspeclen = 0;
  tfmptr->fontexists = false;
  tfmptr->pkfont = false;

  /* tfmptr->fontarea has been read from DVI file. */
  /* See "dvireadr.c". */
  /* We normally expect fontarea to be empty, for portability. */

  if (tfmptr->fontarealen > 0)
  {
    /* start TFM spec. with fontarea as read from DVI file */
    strncpy (tfmdirs, tfmptr->fontarea, maxfontspec);
    tfmdirs[maxfontspec] = '\0';  /* to be careful */
  }
  else
  {
    /* start TFM spec. with each directory from tfmdir, */
    /* as set in "options.c" */
    strncpy (tfmdirs, tfmdir, maxfontspec);
    tfmdirs[maxfontspec] = '\0';  /* to be careful */
  }

  found = false;
  more = true;

  /* loop for each TFM directory listed in tfmdirs */
  for (s = tfmdirs; !found && more; s = (char *) NULL)
  {
    /* NOTE:  strtok returns NULL if all characters of s occur in DIRSEP */
    char * onetfmdir = strtok (s, DIRSEP);
    if (onetfmdir == (char *) NULL)
    {
      more = false;
    }
    else
    {
      /* start TFM spec. with one TFM directory */
      Scopys (Ltfmspec, onetfmdir);
      Scats (Ltfmspec, "/");

      /* append <fontname>.tfm */
      Scats (Ltfmspec, tfmptr->fontname);
      Scats (Ltfmspec, ".tfm");

      strncpy (tfmptr->fontspec, Ltfmspec->s, maxfontspec);
      tfmptr->fontspec[maxfontspec] = '\0';  /* to be careful */
      tfmptr->fontspeclen = strlen (tfmptr->fontspec);  /* quite important */

      /* check whether TFM file exists where tfmptr->fontspec indicates */
      { /* start block */
        FILE * fp = fopen (tfmptr->fontspec, "r");
        if (fp != (FILE *) NULL)
        {
          fclose (fp);
          tfmptr->fontexists = true;
          tfmptr->pkfont = false;
          found = true;
        }
      } /* end block */
    } /* fi */
  } /* loop for each TFM directory listed in tfmdirs */

  FreeLString (Ltfmspec);

  /* Finalise correctly */
  if (!found)
  {
    strncpy (tfmptr->fontspec, "", maxfontspec);
    tfmptr->fontspeclen = 0;
    tfmptr->fontexists = false;
    tfmptr->pkfont = false;
    tfmptr->psfont = false;
  }

  return found;
}
/* BuildTFMSpec */

/******************************************************************************/

#ifdef __STDC__
Static boolean
FndFileInDir (Char *area, Char *name, int mag, fontinfo *pkfp)
#else
Static boolean
FndFileInDir (area, name, mag, pkfp)
    Char *area; Char *name; int mag; fontinfo *pkfp;
#endif
{
  /* Seek file <name>.<mag>pk in directory <area>/<name> or <area> .
     If it's found, then store that file spec. in pkfp->fontspec,
     set pkfp->fontspeclen to the length of that file spec,
     and return true.
     If the file spec. is truncated, or the file is not found,
     then return false.
  */

  boolean status = false;
  char  s[BUFSIZ];
  int len;

  /* Does file exist?  (BSD, POSIX have "#define F_OK 0".) */
  (Void) sprintf (s, "%s/%s/%s.%dpk", area, name, name, mag);
  if (access (s, 0))
    (Void) sprintf (s, "%s/%s.%dpk", area, name, mag);

  (Void) strncpy (pkfp->fontspec, s, maxfontspec);
  pkfp->fontspec[maxfontspec] = '\0';  /* to be careful */
  pkfp->fontspeclen = strlen (pkfp->fontspec);

  len = strlen (s);
  if (len > maxfontspec)
  {
    fprintf (logfile, "PK font's spec. length = %d\n", pkfp->fontspeclen);
    fprintf (logfile, "PK font's spec. = %s\n", pkfp->fontspec);
    fprintf (logfile, "That PK font's spec. was truncated!\n");
    fflush (logfile);

    status = false;
  }
  else
  {
    /* Can file be read?  (BSD, POSIX have "#define R_OK 4".) */
    if (!access (pkfp->fontspec, 4))
      status = true;
    else
      status = false;
  }
  return status;
}
/* FndFileInDir */

/******************************************************************************/

#define PK_MAGIC 4

#ifdef __STDC__
Static boolean BuildPKSpec (fontinfo * pkptr)
#else
Static boolean BuildPKSpec (pkptr)
fontinfo * pkptr;
#endif
{
  /* Build a complete PK file specification in pkptr->fontspec.
     This will be done at most once per font.
     Test for valid pkptr->fontname.
     Rounding error in resolution requires an off-by-one check.
     ASSUMPTIONS:
       The global variables  mag  and  xres  must be defined.
       pkptr->scaledsize  and  pkptr->designsize  must be defined.
       pkptr->fontname contains the basename of the font.
     SIDE EFFECTS on success:
       pkptr->fontspec will be set.
       pkptr->fontspeclen will no longer be 0.
       pkptr->fontexists becomes TRUE if the PK file can be opened,
                                 FALSE, if it cannot.
       pkptr->pkfont becomes TRUE if the PK file can be opened,
                             FALSE, if it cannot.
       pkptr->psfont becomes FALSE;
     SIDE EFFECTS on failure:
         EITHER:
       pkptr is NULL.
         OR:
       pkptr->fontspec will be unset (holds empty string "").
       pkptr->fontspeclen will be 0.
       pkptr->fontexists becomes FALSE.
       pkptr->pkfont becomes FALSE.
       pkptr->psfont becomes FALSE;
  */

  extern string pkdir;

  boolean found = false, more = true;
  string pkdir2;
  char * s = (char *) NULL;

  if (!CheckFontPtr (pkptr))
    return false;

  /* Initialise correctly */
  strncpy (pkptr->fontspec, "", maxfontspec);
  pkptr->fontspeclen = 0;
  pkptr->fontexists = false;
  pkptr->pkfont = false;
  pkptr->psfont = false;

  stringcopy (pkdir2, pkdir);

  found = false;
  more = true;

  /* loop for each directory in pkdir2 */
  for (s = pkdir2; !found && more; s = (char *) NULL)
  {
    char * onepkdir = (char *) NULL;
    int fontsize;
    int tempsize, tempsizelen;
    int i;

    /* NOTE:  strtok returns NULL if all characters of s occur in DIRSEP */
    onepkdir = strtok (s, DIRSEP);
    if (onepkdir == (char *) NULL)
    {
      more = false;
    }
    else
    {
      string fontname;
      int onepkdirlen = strlen (onepkdir);

      /* To be careful: */
      if (onepkdirlen >= maxfontspec)
      {
        onepkdirlen = maxfontspec - 1;

        fprintf (logfile, "The PK directory `%s' was just truncated!\n",
                          onepkdir);
        fflush (logfile);
      }

      strncpy (fontname, pkptr->fontname, maxfontspec);
      fontname[maxfontspec] = '\0';  /* to be careful */

      /* start PK spec. with one PK dir */
      strncpy (pkptr->fontspec, onepkdir, maxfontspec-1);
      pkptr->fontspec[maxfontspec] = '\0';  /* to be careful */
      pkptr->fontspeclen = strlen (pkptr->fontspec);

      pkptr->fontspec[onepkdirlen] = '/';
      pkptr->fontspec[onepkdirlen + 1] = '\0';

      /* gt - I hope "xres" is the right thing to use for the font size */
      fontsize = (int) (mag / 1000.0
          * (double) pkptr->scaledsize / pkptr->designsize
          * xres
        + 0.5);

      if (fontsize == 0)  /* allow for subtracting 1 */
        ++ fontsize;

      tempsize = fontsize;

      for (i=0; true; i++)  /* loop for adjacent font sizes */
      {
        /* Complete rest of fontspec,
           and return the position of first digit for fontsize.
           We have to try fontsize +/- 1 before giving up because
           rounding problems can occur in the above fontsize calculation.
         */

        { /* start block for tempsizelen */
          /* count decimal digits in tempsize */
          int j = tempsize;
          tempsizelen = 0;
          while (j > 0)
          {
            ++ tempsizelen;
            j /= 10;
          }
        } /* end block for tempsizelen */

        /* gt - WHAT and WHY is this MAGIC NUMBER (4) here? */
        if ((pkptr->fontnamelen + tempsizelen + PK_MAGIC)
            > sizeof (fontname))
        {
          break;
        }

        if (FndFileInDir (onepkdir, fontname, tempsize, pkptr))
        {
          pkptr->fontexists = true;  /* specified PK file exists */
          pkptr->pkfont = true;
          pkptr->psfont = false;
          found = true;
          break;
        }
        else if (i > 2)  /* original fontsize has been restored */
        {
          break;
        }
        else
        {
          if (i == 0)
            tempsize = fontsize - 1;  /* try fontsize-1 */ 
          else if (i == 1)
            tempsize = fontsize + 1;  /* try fontsize+1 */
          else /* i == 2 */
            tempsize = fontsize;      /* restore original fontsize */
        } /* inner if */
      } /* inner loop for adjacent font sizes */
    } /* outer if */
  } /* end loop for each directory in pkdir2 */

  /* Finalise correctly */
  if (!found)
  {
    strncpy (pkptr->fontspec, "", maxfontspec);
    pkptr->fontspeclen = 0;
    pkptr->fontexists = false;
    pkptr->pkfont = false;
    pkptr->psfont = false;
  }

  return found;
}
/* BuildPKSpec */

#undef PK_MAGIC

/******************************************************************************/

#ifdef __STDC__
boolean BuildFontSpec (fontinfo * fontptr)
#else
boolean BuildFontSpec (fontptr)
    fontinfo * fontptr;
#endif
{
  /* Build a complete PK or TFM font file specification in fontptr->fontspec.
     (May end up with a dummy PK font's or TFM metric's file spec.
      ShowStatistics() in "dvitovdu.c" will show any font substitutions.)
     This will be done at most once per font.
     Test for valid fontptr->fontname.
     ASSUMPTIONS:
       fontptr->fontname contains the basename of the font.
     SIDE EFFECTS on success:
       fontptr->fontspec will be set.
       fontptr->fontspeclen will no longer be 0.
       fontptr->fontexists becomes TRUE if the real or dummy PK or TFM
                                           file can be opened,
                                   FALSE, otherwise.
       fontptr->pkfont becomes TRUE if the real or dummy PK file
                                       can be opened,
                               FALSE, otherwise.
       fontptr->psfont becomes TRUE if the font's a PS or pure TFM font
                                       file that can be opened,
                               FALSE, otherwise.
       fontptr->honest becomes TRUE if the true font is used,
                               FALSE otherwise.
     SIDE EFFECTS on failure:
         EITHER:
       fontptr is NULL.
         OR:
       fontptr->fontspec will be unset (holds empty string "").
       fontptr->fontspeclen will be 0.
       fontptr->fontexists becomes FALSE.
       fontptr->pkfont becomes FALSE.
       fontptr->psfont becomes FALSE;
       fontptr->honest becomes FALSE.
  */

  boolean status = false;  /* success? */

  if (!CheckFontPtr (fontptr))
    return false;

  /**** Correct PK ? ****/

  if (fontptr->fontspeclen == 0)
  {
    /* Try PK bitmap font. */
    status = BuildPKSpec (fontptr);
  }
  
  if (fontptr->fontexists)
  {
    fontptr->pkfont = true;
    fontptr->psfont = false;
    fontptr->honest = true;

    fprintf (logfile, "Loading PK font file `%s'.\n", fontptr->fontspec);
    fflush (logfile);
  }
  else
  {
    /* PK font file doesn't exist. */

    fontptr->pkfont = false;
    fontptr->psfont = false;
    fontptr->honest = false;

#ifdef DEBUG
    fprintf (logfile, "Couldn't fopen PK font file `%s'!\n",
             (fontptr->fontspeclen == 0 ?
                  fontptr->fontname : fontptr->fontspec));
    fflush (logfile);
#endif /* DEBUG */

  } /* fi */

  /* POSTSCRIPT OR PURE TFM ? */

  /* Check for a PostScript font; if font is PS, then psfont becomes TRUE.
     NOTE that strtok skips sequences of separators, and does not make
     empty tokens.  Therefore, we cannot set an empty PS prefix in order
     to treat all missing PK fonts as PS fonts.
  */

  if (!fontptr->fontexists)
  {
    string prefixes;
    char * seps = DIRSEP;  /* use same separators as (PK and TFM) dir's! */
    char * ptr = (char *) NULL;
    boolean more;

    strncpy (prefixes, psprefix, maxstring);  /* !May TRUNCATE! */
    ptr = prefixes;

    do
    {
      char * prefix = strtok (ptr, seps);  /* Hope strtok is common in C's */
      more = (prefix != (char *) NULL);
      if (more)
      {
        ptr = (char *) NULL;  /* ready for next iteration */
        if (strncmp (fontptr->fontname, prefix, strlen (prefix)) == 0
            && (status = BuildTFMSpec (fontptr)) == true
            && fontptr->fontexists)
        {
          fontptr->psfont = true;
          fontptr->honest = true;
          fontptr->pkfont = false;
        }
        else
        {
          fontptr->psfont = false;
          fontptr->pkfont = false;
          fontptr->honest = false;
        } /* if */
      } /* if */
    }while (more && !fontptr->psfont);
    if (fontptr->psfont)
    {
      fprintf (logfile, "Loading PostScript font TFM `%s'.\n",
               fontptr->fontspec);
      fflush (logfile);
    }
    else
    {

#ifdef DEBUG
      fprintf (logfile, "`%s' is not a PostScript font.\n",
               fontptr->fontname);
      fflush (logfile);
#endif /* DEBUG */

    }
  } /* if */

  /**** Dummy PK? ****/

  if (!fontptr->fontexists)
  {
    /* Try dummy PK font. */

    string dummyspec;

    if (BuildFileSpec (pkdir, dummy_pk, dummyspec))
    {
      /* Fill pixeltable with dummy_pk values. */

      fontptr->fontexists = true;
      fontptr->pkfont = true;
      fontptr->psfont = false;
      fontptr->honest = false;

      strncpy (fontptr->fontspec, dummyspec, maxfontspec);
      fontptr->fontspeclen = strlen (fontptr->fontspec);
      status = true;

      fprintf (logfile, "Using dummy PK font `%s'\n", dummyspec);
      fflush (logfile);
    }
    else
    { 
      /* dummy PK font file doesn't exist. */

      fontptr->fontexists = false;

      fontptr->pkfont = false;
      fontptr->psfont = false;
      fontptr->honest = false;

#ifdef DEBUG
      fprintf (logfile, "Couldn't fopen dummy PK font file `%s'!\n",
               (fontptr->fontspeclen == 0 ?
                    fontptr->fontname : fontptr->fontspec));
      fflush (logfile);
#endif /* DEBUG */

    }
  } /* fi */

  /**** Correct TFM? ****/

  if (!fontptr->fontexists)
  {
    /* Try the correct font's TFM metric. */

    status = BuildTFMSpec (fontptr);

    if (fontptr->fontexists)
    {
      fontptr->pkfont = false;
      fontptr->psfont = false;
      fontptr->honest = true;

      fprintf (logfile, "Loading TFM metric file for `%s'.\n",
	       fontptr->fontname);
      fflush (logfile);
    }
    else
    {
      fontptr->pkfont = false;
      fontptr->psfont = false;
      fontptr->honest = false;

#ifdef DEBUG
      fprintf (logfile, "Couldn't fopen TFM metric file `%s'!\n",
               (fontptr->fontspeclen == 0 ?
                    fontptr->fontname : fontptr->fontspec));
      fflush (logfile);
#endif /* DEBUG */

    }
  } /* fi */

  /**** Dummy TFM? ****/

  if (!fontptr->fontexists)
  {
    /* Try dummy TFM metric. */

    string dummyspec;

    if (BuildFileSpec (pkdir, dummy_tfm, dummyspec))
    {
      status = true;
      fontptr->fontexists = true;

      fontptr->pkfont = false;
      fontptr->psfont = false;
      fontptr->honest = false;

      strncpy (fontptr->fontspec, dummyspec, maxfontspec);
      fontptr->fontspeclen = strlen (fontptr->fontspec);

      fprintf (logfile, "Using dummy TFM `%s'.\n", dummyspec);
      fflush (logfile);
    }
    else
    {
      status = false;
      fontptr->fontexists = false;

      fontptr->pkfont = false;
      fontptr->psfont = false;
      fontptr->honest = false;

#ifdef DEBUG
      fprintf (logfile, "Couldn't fopen dummy TFM metric file `%s'!\n",
               (fontptr->fontspeclen == 0 ?
                    fontptr->fontname : fontptr->fontspec));
      fflush (logfile);
#endif /* DEBUG */

    }
  } /* fi */

  /**** Resort to Terse display of this font, using DVI info. ****/

  if (!fontptr->fontexists)
  {
    /* Ran out of options. */

    fontptr->pkfont = false;
    fontptr->psfont = false;
    fontptr->honest = false;

    status = false;

#ifdef DEBUG
    fprintf (logfile, "No correct PK or TFM for `%s'\n", fontptr->fontname);
    fprintf (logfile, "  nor dummy PK `%s'\n", dummy_pk);
    fprintf (logfile, "  nor dummy TFM `%s'\n", dummy_tfm);
    fflush (logfile);
#endif /* DEBUG */

    fprintf (logfile, "Using DVI Terse display for `%s'\n", fontptr->fontname);
    fflush (logfile);

  } /* fi */

  return status;
}
/* BuildFontSpec */

/******************************************************************************/

#ifdef __STDC__
boolean OpenFontFile (Char *name)
#else
boolean OpenFontFile (name)
Char *name;
#endif
{
  /* Return TRUE if given file can be (unix-specifically) open-ed.
     Only one font file will be open at any given time.
  */

  extern int currPTbuff;
  extern int PTfile;

  currPTbuff = -1;   /* impossible value for first GetPTByte */
  PTfile = open (name, O_RDONLY, 0);  /* unix "open" returns -1 if fails */
  return (PTfile >= 0);
}
/* OpenFontFile */

/******************************************************************************/

Void CloseFontFile (VOID)
{
  /* If there is a currently open font file, then close it. */

  if (PTfile >= 0)
  {
    (Void) close (PTfile);
    /* Be cautious:  ensure PTfile is negative, after this function. */
    if (PTfile >= 0)
      PTfile = -1;
  }
}
/* CloseFontFile */

/******************************************************************************/

Static int GetPTByte (VOID)
{
  /* Returns the value (unsigned) of the byte at PToffset and
     advances PToffset for the next GetPTByte.
  */

  int Result, buffstart, result;

  buffstart = PToffset / bufflen * bufflen;   /* 0, bufflen, 2*bufflen... */
  if (buffstart != currPTbuff)
  {
    currPTbuff = buffstart;

    if (PTfile < 0)
    {
      StartText();
      ResetVDU();  /* let message stay */

      MesgString ("PTfile not open in GetPTByte!");
      MesgLine();

      RestoreTerminal();
      exit (1);
    }

    result = lseek (PTfile, buffstart, 0);
    /* DEBUG
       IF result <> buffstart THEN BEGIN
          writeln ('Lseek failed in GetPTByte!'); RestoreTerminal; EXIT (1);
       END;
     GUBED */
    if (result != buffstart)
    {
      StartText();
      ResetVDU();  /* let message stay */

      MesgString ("lseek failed in GetPTByte!");
      MesgLine();

      RestoreTerminal();
      exit (1);
    }
    result = read (PTfile, PTbuffer, bufflen);
    /* DEBUG
       IF result = -1 THEN BEGIN
          writeln ('Read failed in GetPTByte!'); RestoreTerminal; EXIT (1);
       END;
     GUBED */
    if (result == -1)
    {
      StartText();
      ResetVDU();  /* let message stay */

      MesgString ("Read failed in GetPTByte!");
      MesgLine();

      RestoreTerminal();
      exit (1);
    }
  }
  Result = PTbuffer[PToffset - buffstart];
  PToffset++;
  return Result;
}
/* GetPTByte */

/******************************************************************************/

Static int SignedPTByte (VOID)
{
  /* the next byte, signed */
  int b;

  b = GetPTByte();
  if (b < 128)
    return b;
  else
    return (b - 256);
}
/* SignedPTByte */

/******************************************************************************/

Static int GetTwoPTBytes (VOID)
{
  /* the next 2 bytes, unsigned */
  int a, b;

  a = GetPTByte();
  b = GetPTByte();
  return (a * 256 + b);
}
/* GetTwoPTBytes */

/******************************************************************************/

Static int SignedPTPair (VOID)
{
  /* the next 2 bytes, signed */
  int a, b;

  a = GetPTByte();
  b = GetPTByte();
  if (a < 128)
    return (a * 256 + b);
  else
    return ((a - 256) * 256 + b);
}
/* SignedPTPair */

/******************************************************************************/

/* UNUSED: */
#ifdef GET_THREE_PT_BYTES
Static int GetThreePTBytes (VOID)
{
  /* the next 3 bytes, unsigned */
  int a, b, c;

  a = GetPTByte();
  b = GetPTByte();
  c = GetPTByte();
  return ((a * 256 + b) * 256 + c);
}
/* GetThreePTBytes */
#endif /* GET_THREE_PT_BYTES */

/******************************************************************************/

typedef struct int_or_bytes
{
  boolean b;
  union
  {
    int int_;
    Char byt[4];
  } UU;
} int_or_bytes;

/******************************************************************************/

Static int SignedPTQuad (VOID)
{
  /* the next 4 bytes, signed */
  int w;

  byte3(w) = GetPTByte();
  byte2(w) = GetPTByte();
  byte1(w) = GetPTByte();
  byte0(w) = GetPTByte();
  return (w);
}
/* SignedPTQuad */

/******************************************************************************/

Static int GetNyb (VOID)
{
  /* Return next nybble in PK file. */

  if (bitweight == 0)
  {
    inputbyte = GetPTByte();
    bitweight = 16;   /* for next call of GetNyb */
    return (inputbyte / 16);   /* high nybble */
  }
  else
  {
    bitweight = 0;   /* for next call of GetNyb */
    return (inputbyte & 15);   /* low nybble */
  }
}
/* GetNyb */

/******************************************************************************/

Static int PackedNum (VOID)
{
  /* Return next run count using algorithm given in section 23 of PKtype.
     A possible side-effect is to set the global repeatcount value used
     to duplicate the current row.
  */

  int i, j;

  i = GetNyb();
  if (i == 0)
  {
    do
    {
      j = GetNyb();
      i++;
    } while (j == 0);
    while (i > 0)
    {
      j = j * 16 + GetNyb();
      i--;
    }
    return (j + (13 - dynf) * 16 + dynf - 15);
  }
  else if (i <= dynf)
    return i;
  else if (i < 14)
    return ((i - dynf - 1) * 16 + GetNyb() + dynf + 1);
  else
  {
    if (i == 14)   /* nybble = 15 */
      repeatcount = PackedNum();   /* recursive */
    else
      repeatcount = 1;
    return (PackedNum());   /* recursive */
  }
}
/* PackedNum */

/******************************************************************************/

#ifdef __STDC__
boolean GetBitmap (int ht, int wd, int mapadr, int_or_mptr *bitmap)
#else
boolean GetBitmap (ht, wd, mapadr, bitmap)
    int ht;  int wd;  int mapadr;  int_or_mptr * bitmap;
#endif
{
  /* Allocate space for bitmap and fill it in using information from
     character definition starting at mapadr in currently open PK file.
     Note that the memory used by a loaded bitmap is never deallocated.
     Each bitmap row uses an integral number of words (each 32 bits).
     Byte-aligned rows would use about 35% less memory but
     would increase the processing time needed to display each bitmap.
     It was felt that speed is more important than memory.
     Return true if the bitmap is obtained, false if function fails.
  */

  extern int bitweight;  /* local to "fontreader.c" */

  int_or_bptr wordptr, rowptr;
  int i, j, flagbyte, wordwidth, wordweight, rowsleft, hbit, count,
      bitmapwords;
  int word;
  Word * bitmapptr = (Word *) NULL;

  wordwidth = (wd + 31) / 32;   /* words in one row of bitmap */
  bitmapwords = ht * wordwidth;   /* memory required by bitmap */

  bitmapptr = cnew (bitmapwords, Word);
  if (bitmapptr == (Word *) NULL)
  {
    MesgString ("Character too big!  size=");
    MesgInt (bitmapwords);
    MesgLine();

    OutMem();

    return false;    /* Not Reached */
  }

  bitmap->UU.mptr = bitmapptr;   /* return start of bitmap */
  wordptr.UU.int_ = bitmap->UU.int_;
  PToffset = mapadr;   /* mapadr = flagbyte offset in PK file */

  flagbyte = GetPTByte();   /* assume < 240 */
  dynf = flagbyte / 16;   /* dynamic packing variable */
  turnon = ((flagbyte & 15) >= 8);   /* is 1st pixel black? */
  flagbyte &= 7;   /* value of bottom 3 bits */
  if (flagbyte < 4)   /* skip short char preamble */
    PToffset += 10;
  else if (flagbyte < 7)
    PToffset += 16;
  else
    PToffset += 36;

  bitweight = 0;   /* to get 1st inputbyte */
  if (dynf == 14)
  {
    /* raster info is a string of bits in the next (wd * ht + 7) DIV 8 bytes */

    for (i = 1; i <= ht; i++)
    {
      word = 0;   /* set all bits to 0 */
      wordweight = 31;   /* leftmost bit */
      for (j = 1; j <= wd; j++)
      {
        if (bitweight == 0)
        {
          inputbyte = GetPTByte();
          bitweight = 8;
        }
        bitweight--;   /* 7..0 */
        if ((unsigned)bitweight < 32 && ((1 << bitweight) & inputbyte))
              /* set bit */
                word |= 1 << wordweight;
        if (wordweight > 0)
          wordweight--;
        else
        {
          *wordptr.UU.bptr = word;
          wordptr.UU.int_ += 4;
          word = 0;
          wordweight = 31;
        }
      }
      if (wordweight < 31)
      {
        *wordptr.UU.bptr = word;
        wordptr.UU.int_ += 4;   /* start of next word */
      }
    }
  }
  else
  {
    /* raster info is encoded as run and repeat counts */

    rowsleft = ht;
    hbit = wd;
    repeatcount = 0;
    wordweight = 32;
    word = 0;
    rowptr = wordptr;   /* remember start of row */
    while (rowsleft > 0)
    {
      count = PackedNum();
      while (count > 0)
      {
        if (count < wordweight && count < hbit)
        {
          if (turnon)
            word = (word | gpower[wordweight]) & (~gpower[wordweight - count]);
          hbit -= count;
          wordweight -= count;
          count = 0;
          continue;
        }
        if (count >= hbit && hbit <= wordweight)
        {
          if (turnon)
            word = (word | gpower[wordweight]) & (~gpower[wordweight - hbit]);
          *wordptr.UU.bptr = word;
          /* end of current row, so duplicate repeatcount times */
          for (i = 1; i <= repeatcount; i++)
          {
            for (j = 1; j <= wordwidth; j++)
            {
              wordptr.UU.int_ += 4;
              *wordptr.UU.bptr = *rowptr.UU.bptr;
              rowptr.UU.int_ += 4;
            }
          }
          rowsleft += -repeatcount - 1;
          repeatcount = 0;
          word = 0;
          wordptr.UU.int_ += 4;
          rowptr = wordptr;   /* remember start of next row */
          wordweight = 32;
          count -= hbit;
          hbit = wd;
        }
        else
        {
          if (turnon)
            word |= gpower[wordweight];
          *wordptr.UU.bptr = word;
          wordptr.UU.int_ += 4;
          word = 0;
          count -= wordweight;
          hbit -= wordweight;
          wordweight = 32;
        }
      }
      turnon = !turnon;
    }
  }
  return true;
}
/* GetBitmap */

/******************************************************************************/

#ifdef __STDC__
Static int FixToDVI (int scale, int b0, int b1, int b2, int b3)
#else
Static int FixToDVI (scale, b0, b1, b2, b3)
    /* scale = currfont->scaledsize; */
    int scale;
    /* fix width: 4 bytes */
    int b0, b1, b2, b3;
#endif
{
  /* Convert the given fix width (made up of 4 bytes) into DVI units
     using the method recommended in DVITYPE.
     Added local scale to avoid changing scaledsize; thanks to Niel Kempson
     for reporting this bug.
  */

  int dviwidth = 0;
  int alpha, beta, temp;

  alpha = scale * 16;
  beta = 16;

  while (scale >= 0x800000)  /* 2^23sp = 128pt */
  {
    scale /= 2;
    beta /= 2;
  }

  if (beta == 0)
  {
    fprintf (logfile, "FixToDVI:  beta = 0, so divide by zero imminent!\n");
    fflush (logfile);
  }

  temp = ((b3 * scale / 0x100 + b2 * scale) / 0x100 + b1 * scale) / beta;

  if (b0 <= 0)
    dviwidth = temp;
  else if (b0 == 255)
    dviwidth = (temp - alpha);
  else
  {
    dviwidth = 0;  /* keep compiler happy, if necessary */

    StartText();
    ResetVDU();  /* let message stay */

    MesgString ("Bad TFM width! 1st byte=");
    MesgInt (b0);
    MesgLine();

    RestoreTerminal();
    exit (1);
  }

  return dviwidth;
}
/* FixToDVI */

/******************************************************************************/

#define pkid            89
#define pkpost          245
#define pknoop          246
#define pkpre           247

#ifdef __STDC__
Static Void PKFillPixelTable (fontinfo * currfont)
#else
Static Void PKFillPixelTable (currfont)
    fontinfo * currfont;
#endif
{
  /* Fill the pixeltable for currfont^ using the font directory info
     in the currently open PK file.
  */

  int i, j, flagbyte, flagpos;
  int chcode;   /* char. code, assumed to be <= maxTeXchar */
  int packetlen, endofpacket, b0, b1, b2, b3;   /* 4 bytes in TFM width */
  _REC_pixeltable *pix_tab;

  PToffset = 0;   /* move to first byte */

  if (GetPTByte() != pkpre)
  {
    StartText();
    ResetVDU();  /* let message stay */

    MesgString ("Bad PK pre command in `");
    MesgString (currfont->fontspec);
    MesgString ("'");
    MesgLine();

    RestoreTerminal();
    exit (1);
  }

  if (GetPTByte() != pkid)
  {
    StartText();
    ResetVDU();  /* let message stay */

    MesgString ("Bad PK id byte in `");
    MesgString (currfont->fontspec);
    MesgString ("'");
    MesgLine();

    RestoreTerminal();
    exit (1);
  }

  j = GetPTByte();   /* length of comment */
  PToffset += j + 16;   /* skip rest of preamble */

  for (i = 0; i <= maxTeXchar; i++)
  {
    pix_tab = &currfont->pixelptr[i];
    pix_tab->mapadr = 0;   /* all chars absent initially */
    pix_tab->bitmap.UU.mptr = (Word *) NULL;
  }

  while (true)
  {
    flagpos = PToffset;   /* remember position of flagbyte */
    flagbyte = GetPTByte();
    if (flagbyte < 240)
    {  /* read character definition */

      int scale = currfont->scaledsize;  /* For FixToDVI */

      flagbyte &= 7;   /* value of bottom 3 bits */
      if (flagbyte < 4)
      {  /* short char preamble */
        packetlen = flagbyte * 256 + GetPTByte();
        chcode = GetPTByte();
        endofpacket = packetlen + PToffset;
        pix_tab = &currfont->pixelptr[chcode];
        b1 = GetPTByte();
        b2 = GetPTByte();
        b3 = GetPTByte();
        pix_tab->dwidth = FixToDVI (scale, 0, b1, b2, b3);   /* b0 = 0 */
        pix_tab->pwidth = GetPTByte();
        pix_tab->wd = GetPTByte();
        pix_tab->ht = GetPTByte();
        pix_tab->xo = SignedPTByte();
        pix_tab->yo = SignedPTByte();
      }
      else if (flagbyte < 7)
      {
        packetlen = (flagbyte - 4) * 65536 + GetTwoPTBytes();
        chcode = GetPTByte();
        endofpacket = packetlen + PToffset;
        pix_tab = &currfont->pixelptr[chcode];
        b1 = GetPTByte();
        b2 = GetPTByte();
        b3 = GetPTByte();
        pix_tab->dwidth = FixToDVI (scale, 0, b1, b2, b3);   /* b0 = 0 */
        pix_tab->pwidth = GetTwoPTBytes();
        pix_tab->wd = GetTwoPTBytes();
        pix_tab->ht = GetTwoPTBytes();
        pix_tab->xo = SignedPTPair();
        pix_tab->yo = SignedPTPair();
      }
      else
      {
        packetlen = SignedPTQuad();
        chcode = SignedPTQuad();
        endofpacket = packetlen + PToffset;
        pix_tab = &currfont->pixelptr[chcode];
        b0 = GetPTByte();
        b1 = GetPTByte();
        b2 = GetPTByte();
        b3 = GetPTByte();
        pix_tab->dwidth = FixToDVI (scale, b0, b1, b2, b3);
        pix_tab->pwidth = SignedPTQuad() / 65536;   /* dx in pixels */
        PToffset += 4;   /* skip dy */
        pix_tab->wd = SignedPTQuad();
        pix_tab->ht = SignedPTQuad();
        pix_tab->xo = SignedPTQuad();
        pix_tab->yo = SignedPTQuad();
      }
      pix_tab = &currfont->pixelptr[chcode];   /* position of flagbyte */
      if (pix_tab->wd == 0 || pix_tab->ht == 0)
        pix_tab->mapadr = 0;   /* no bitmap */
      else
        pix_tab->mapadr = flagpos;
      PToffset = endofpacket;   /* skip raster info */
      continue;
    }

    switch (flagbyte)
    {
    case 240:
    case 241:
    case 242:
    case 243:
      i = 0;
      for (j = 240; j <= flagbyte; j++)
        i = i * 256 + GetPTByte();
      PToffset += i;   /* skip special parameter */
      break;

    case 244:   /* skip numspecial param */
      PToffset += 4;
      break;

    case pknoop:   /* do nothing */
      break;

    case pkpost:   /* no more char defs */
      goto _L888;
      break;

    default:
      StartText();
      ResetVDU();  /* let message stay */

      MesgString ("Bad PK flag byte in `");
      MesgString (currfont->fontspec);
      MesgString ("'");
      MesgLine();

      RestoreTerminal();
      exit (1);

      break;
    }
  }  /* of LOOP; flagbyte = pkpost */
_L888: ;

  /* extended short char preamble */
  /* long char preamble */
}
/* PKFillPixelTable */

#undef pkid
#undef pkpost
#undef pknoop
#undef pkpre

/******************************************************************************/

Static Void ReadTFMIntegers (VOID)
{
  /* Read the first 6 (six) 16-bit integers in the TFM file. */
  /* These give the dimensions of the arrays that constitute the rest */
  /* of the TFM file. */
  /* See TFtoPL section 8. */

  string out;  /* output string for messages */

  PToffset = 0;   /* start reading at 1st byte in TFM file */

  lf = GetTwoPTBytes();  /* length of TFM file, in words:  0 <= lf < 2^15. */
  lh = GetTwoPTBytes();  /* length of header, in words:  2 <= lh. */
  bc = GetTwoPTBytes();  /* smallest character code in font */
  ec = GetTwoPTBytes();  /* largest character code in font */
  nw = GetTwoPTBytes();  /* number of words in width table */
  nh = GetTwoPTBytes();  /* number of words in height table */

  /* sanity check of lf */
  if ( ! (0 <= lf  &&  lf < 1<<15))
  {
    StartText();
    ResetVDU();  /* let message stay */

    sprintf (out, "TFM file length given as:  lf = %d words.", lf);
    MesgString (out);
    MesgLine();
    sprintf (out, "  Should have 0 <= lf < %d words.", 1<<15);
    MesgString (out);
    MesgLine();

    RestoreTerminal();
    exit (1);
  }

  /* sanity check of lh */
  if (! (2 <= lh))
  {
    StartText();
    ResetVDU();  /* let message stay */

    sprintf (out, "TFM's Header length given as:  lh = %d words.", lh);
    MesgString (out);
    MesgLine();
    MesgString ("  Should have lh >= 2 (two) words.");
    MesgLine();

    RestoreTerminal();
    exit (1);
  }

  /* sanity check of bc and ec */
  if ( ! (0 <= bc  &&  bc-1 <= ec  &&  ec <= maxTeXchar))
  {
    StartText();
    ResetVDU();  /* let message stay */

    sprintf (out, "First & last char codes = (bc, ec) = (%d, %d).", bc, ec);
    MesgString (out);
    MesgLine();
    MesgString ("  Should have 0 <= bc <= ec <= maxTeXchar = 255.");
    MesgLine();

    RestoreTerminal();
    exit (1);
  }

  /* sanity check of nw */
  if ( ! (0 <= nw  &&  nw < 1<<15))
  {
    StartText();
    ResetVDU();  /* let message stay */

    sprintf (out, "Number of Width Table words given as:  nw = %d.", nw);
    MesgString (out);
    MesgLine();
    sprintf (out, "  Should have 0 <= nw < %d words.", 1<<15);
    MesgString (out);
    MesgLine();

    RestoreTerminal();
    exit (1);
  }

  /* sanity check of nh */
  if ( ! (0 <= nh  &&  nh < 1<<15))
  {
    StartText();
    ResetVDU();  /* let message stay */

    sprintf (out, "Number of Height Table words given as:  nh = %d", nh);
    MesgString (out);
    MesgLine();
    sprintf (out, "  Should have 0 <= nh < %d.", 1<<15);
    MesgString (out);
    MesgLine();

    RestoreTerminal();
    exit (1);
  }

}
/* ReadTFMIntegers */

/******************************************************************************/

Static Void ReadTFMCharInfo (VOID)
{
  /* Read the TFMinfo array.  See TFtoPL section 11. */

  int c, i;
  _REC_TFMinfo *tfm_inf;

  PToffset = (6 + lh) * 4;   /* offset of TFMinfo array (in bytes) */
                             /* 6 words of dimension info, plus header */
  for (c = bc; c <= ec; c++)
  {
    tfm_inf = &TFMinfo[c];
    tfm_inf->wdindex = GetPTByte() * 4;   /* offset from start of width array */
    i = GetPTByte();   /* 2nd byte contains htindex and dpindex */
    tfm_inf->htindex = i / 16 * 4;   /* offset from start of height array */
    tfm_inf->dpindex = (i & 15) * 4;   /* offset from start of depth array */
    PToffset += 2;   /* skip itindex and remainder bytes */
  }
}
/* ReadTFMCharInfo */

/******************************************************************************/

Static Void ReadTFMCharMetrics (VOID)
{
  /* Read the charmetrics array using the indices in TFMinfo. */

  int wdbase = lh * 4 + (ec - bc + 1) * 4 + 24;   /* offset of width array */
  int htbase = wdbase + nw * 4;   /* offset of height array */
  int dpbase = htbase + nh * 4;   /* offset of depth array */
  int c;

  for (c = bc; c <= ec; c++)
  {
    _REC_TFMinfo * tfm_inf = &TFMinfo[c];
    _REC_charmetrics * char_met = &charmetrics[c];
    int b;

    PToffset = wdbase + tfm_inf->wdindex;
    for (b = 0; b <= 3; b++)
      char_met->width[b] = GetPTByte();
    PToffset = htbase + tfm_inf->htindex;
    for (b = 0; b <= 3; b++)
      char_met->height[b] = GetPTByte();
    PToffset = dpbase + tfm_inf->dpindex;
    for (b = 0; b <= 3; b++)
      char_met->depth[b] = GetPTByte();
  }
}
/* ReadTFMCharMetrics */

/******************************************************************************/

#ifdef __STDC__
Static Void TFMFillPixelTable (fontinfo * currfont)
#else
Static Void TFMFillPixelTable (currfont)
    fontinfo * currfont;
#endif
{
  /* Fill the pixeltable for  * currfont
     (a PostScript or other font for which the PK file is missing),
     using information in the currently open TFM file.
  */

  int c;

  ReadTFMIntegers();   /* read lf..nh */
  ReadTFMCharInfo();   /* fill TFMinfo array */
  ReadTFMCharMetrics();   /* fill charmetrics array */

  for (c = 0; c < bc; c++)   /* chars < bc don't exist */
    currfont->pixelptr[c].mapadr = 0;

  for (c = ec + 1; c <= maxTeXchar; c++)   /* chars > ec don't exist */
    currfont->pixelptr[c].mapadr = 0;

  for (c = bc; c <= ec; c++)
  {
    _REC_pixeltable * pix_tab = &currfont->pixelptr[c];
    _REC_charmetrics * char_met = &charmetrics[c];
    int scale = currfont->scaledsize;  /* For FixToDVI */
    int dheight, pheight, ddepth, pdepth;

    pix_tab->dwidth = FixToDVI (scale,
                                char_met->width[0], char_met->width[1],
                                char_met->width[2], char_met->width[3]);

    dheight = FixToDVI (scale,
                        char_met->height[0], char_met->height[1],
                        char_met->height[2], char_met->height[3]);

    ddepth = FixToDVI (scale,
                       char_met->depth[0], char_met->depth[1],
                       char_met->depth[2], char_met->depth[3]);

    /* convert DVI units to pixels */

    pix_tab->pwidth = XPixelRound (pix_tab->dwidth);
    pheight = YPixelRound (dheight);
    pdepth = YPixelRound (ddepth);

    /* Since we don't have access to PK bitmap info,
       we will have to use the TFM width/height/depth info
       to approximate wd, ht, xo, yo.
    */

    pix_tab->wd = pix_tab->pwidth;
    pix_tab->wd -= pix_tab->wd / 8;   /* better approximation */
    pix_tab->ht = pheight + pdepth;
    pix_tab->xo = 0;
    pix_tab->yo = pheight - 1;
    if (pix_tab->wd == 0 || pix_tab->ht == 0)   /* anything but 0 */
      pix_tab->mapadr = 0;   /* char blank or not in font */
    else
      pix_tab->mapadr = 1;
    pix_tab->bitmap.UU.mptr = (Word *) NULL;
  }
}
/* TFMFillPixelTable */

/******************************************************************************/

/* number of characters in CM fonts */
#define CMSIZE  128

#ifndef OLD_CHARDIM
  /* Data taken from cmr10.pl = most common font (so usual dummy font) : */
  /* (width, height, depth, italic correction) of each cmr10 character */
  struct {double wd, ht, dp, ic;} cmr10[CMSIZE] =
  {
    {0.625002, 0.683332, 0, 0},  /* 00 */
    {0.833336, 0.683332, 0, 0},  /* 01 */
    {0.777781, 0.683332, 0, 0},  /* 02 */
    {0.694446, 0.683332, 0, 0},  /* 03 */
    {0.666669, 0.683332, 0, 0},  /* 04 */
    {0.750002, 0.683332, 0, 0},  /* 05 */
    {0.722224, 0.683332, 0, 0},  /* 06 */
    {0.777781, 0.683332, 0, 0},  /* 07 */
    {0.722224, 0.683332, 0, 0},  /* 08 */
    {0.777781, 0.683332, 0, 0},  /* 09 */
    {0.722224, 0.683332, 0, 0},  /* 0A */
    {0.583336, 0.694445, 0, 0.077779},  /* 0B */
    {0.555557, 0.694445, 0, 0},  /* 0C */
    {0.555557, 0.694445, 0, 0},  /* 0D */
    {0.833336, 0.694445, 0, 0},  /* 0E */
    {0.833336, 0.694445, 0, 0},  /* 0F */
    {0.277779, 0.430555, 0, 0},  /* 10 */
    {0.305557, 0.430555, 0.194445, 0},  /* 11 */
    {0.500002, 0.694445, 0, 0},  /* 12 */
    {0.500002, 0.694445, 0, 0},  /* 13 */
    {0.500002, 0.628473, 0, 0},  /* 14 */
    {0.500002, 0.694445, 0, 0},  /* 15 */
    {0.500002, 0.567777, 0, 0},  /* 16 */
    {0.750002, 0.694445, 0, 0},  /* 17 */
    {0.444446, 0, 0.170138, 0},  /* 18 */  /* sic : no height, only depth */
    {0.500003, 0.694445, 0, 0},  /* 19 */
    {0.722224, 0.430555, 0, 0},  /* 1A */
    {0.777781, 0.430555, 0, 0},  /* 1B */
    {0.500002, 0.527779, 0.097223, 0},  /* 1C */
    {0.902781, 0.683332, 0, 0},  /* 1D */
    {1.013891, 0.683332, 0, 0},  /* 1E */
    {0.777781, 0.731944, 0.048612, 0},  /* 1F */
    {0.277779, 0.430555, 0, 0},  /* 20 */
    {0.277779, 0.694445, 0, 0},  /* 21 */
    {0.500002, 0.694445, 0, 0},  /* 22 */
    {0.833336, 0.694445, 0.194443, 0},  /* 23 */
    {0.500002, 0.75, 0.055555, 0},  /* 24 */
    {0.833336, 0.75, 0.055555, 0},  /* 25 */
    {0.777781, 0.694445, 0, 0},  /* 26 */
    {0.277779, 0.694445, 0, 0},  /* 27 */
    {0.388889, 0.75, 0.25, 0},  /* 28 */
    {0.388889, 0.75, 0.25, 0},  /* 29 */
    {0.500002, 0.75, 0, 0},  /* 2A */
    {0.777781, 0.583334, 0.194443, 0},  /* 2B */
    {0.277779, 0.105556, 0.055555, 0},  /* 2C */
    {0.333334, 0.430555, 0, 0},  /* 2D */
    {0.277779, 0.105556, 0, 0},  /* 2E */
    {0.500002, 0.75, 0.25, 0},  /* 2F */
    {0.500002, 0.644444, 0, 0},  /* '0' */
    {0.500002, 0.644444, 0, 0},  /* '1' */
    {0.500002, 0.644444, 0, 0},  /* '2' */
    {0.500002, 0.644444, 0, 0},  /* '3' */
    {0.500002, 0.644444, 0, 0},  /* '4' */
    {0.500002, 0.644444, 0, 0},  /* '5' */
    {0.500002, 0.644444, 0, 0},  /* '6' */
    {0.500002, 0.644444, 0, 0},  /* '7' */
    {0.500002, 0.644444, 0, 0},  /* '8' */
    {0.500002, 0.644444, 0, 0},  /* '9' */
    {0.277779, 0.430555, 0, 0},  /* 3A */
    {0.277779, 0.430555, 0.194445, 0},  /* 3B */
    {0.277779, 0.5, 0.194445, 0},  /* 3C */
    {0.777781, 0.366875, -0.133125, 0},  /* 3D */
    {0.472224, 0.5, 0.194445, 0},  /* 3E */
    {0.472224, 0.694445, 0, 0},  /* 3F */
    {0.777781, 0.694445, 0, 0},  /* 40 */
    {0.750002, 0.683332, 0, 0},  /* 'A' */
    {0.708336, 0.683332, 0, 0},  /* 'B' */
    {0.722224, 0.683332, 0, 0},  /* 'C' */
    {0.763891, 0.683332, 0, 0},  /* 'D' */
    {0.680557, 0.683332, 0, 0},  /* 'E' */
    {0.652781, 0.683332, 0, 0},  /* 'F' */
    {0.784724, 0.683332, 0, 0},  /* 'G' */
    {0.750002, 0.683332, 0, 0},  /* 'H' */
    {0.361112, 0.683332, 0, 0},  /* 'I' */
    {0.51389,  0.683332, 0, 0},  /* 'J' */
    {0.777781, 0.683332, 0, 0},  /* 'K' */
    {0.625002, 0.683332, 0, 0},  /* 'L' */
    {0.916669, 0.683332, 0, 0},  /* 'M' */
    {0.750002, 0.683332, 0, 0},  /* 'N' */
    {0.777781, 0.683332, 0, 0},  /* 'O' */
    {0.680557, 0.683332, 0, 0},  /* 'P' */
    {0.777781, 0.683332, 0.194445, 0},  /* 'Q' */
    {0.736113, 0.683332, 0, 0},  /* 'R' */
    {0.555557, 0.683332, 0, 0},  /* 'S' */
    {0.722224, 0.683332, 0, 0},  /* 'T' */
    {0.750002, 0.683332, 0, 0},  /* 'U' */
    {0.750002, 0.683332, 0, 0.013888},  /* 'V' */
    {1.027781, 0.683332, 0, 0.013888},  /* 'W' */
    {0.750002, 0.683332, 0, 0},  /* 'X' */
    {0.750002, 0.683332, 0, 0.025},  /* 'Y' */
    {0.611113, 0.683332, 0, 0},  /* 'Z' */
    {0.277779, 0.75, 0.25, 0},  /* 5B */
    {0.500002, 0.694445, 0, 0},  /* 5C */
    {0.277779, 0.75, 0.25, 0},  /* 5D */
    {0.500002, 0.694445, 0, 0},  /* 5E */
    {0.277779, 0.667859, 0, 0},  /* 5F */
    {0.277779, 0.694445, 0, 0},  /* 60 */
    {0.500002, 0.430555, 0, 0},  /* 'a' */
    {0.555557, 0.694445, 0, 0},  /* 'b' */
    {0.444446, 0.430555, 0, 0},  /* 'c' */
    {0.555557, 0.694445, 0, 0},  /* 'd' */
    {0.444446, 0.430555, 0, 0},  /* 'e' */
    {0.305557, 0.694445, 0, 0.077779},  /* 'f' */
    {0.500002, 0.430555, 0.194445, 0},  /* 'g' */
    {0.555557, 0.694445, 0, 0},  /* 'h' */
    {0.277779, 0.667859, 0, 0},  /* 'i' */
    {0.305557, 0.667859, 0.194445, 0},  /* 'j' */
    {0.527781, 0.694445, 0, 0},  /* 'k' */
    {0.277779, 0.430555, 0, 0},  /* 'l' */
    {0.833336, 0.430555, 0, 0},  /* 'm' */
    {0.555557, 0.430555, 0, 0},  /* 'n' */
    {0.500002, 0.430555, 0, 0},  /* 'o' */
    {0.555557, 0.430555, 0.194445, 0},  /* 'p' */
    {0.527779, 0.430555, 0.194445, 0},  /* 'q' */
    {0.391668, 0.430555, 0, 0},  /* 'r' */
    {0.394445, 0.430555, 0, 0},  /* 's' */
    {0.38889,  0.61508,  0, 0},  /* 't' */
    {0.555557, 0.430555, 0, 0},  /* 'u' */
    {0.527781, 0.430555, 0, 0.013888},  /* 'v' */
    {0.722224, 0.430555, 0, 0.013888},  /* 'w' */
    {0.527781, 0.430555, 0, 0},  /* 'x' */
    {0.527781, 0.430555, 0.194445, 0.013888},  /* 'y' */
    {0.444446, 0.430555, 0, 0},  /* 'z' */
    {0.500002, 0.430555, 0, 0.027779},  /* 7B */
    {1.000003, 0.430555, 0, 0.027779},  /* 7C */
    {0.500002, 0.694445, 0, 0},  /* 7D */
    {0.500002, 0.667859, 0, 0},  /* 7E */
    {0.500002, 0.667859, 0, 0},  /* 7F */
  };
#endif /* Not OLD_CHARDIM */


#ifdef __STDC__
Static Void AbsentFillPixelTable (fontinfo * currfont)
#else
Static Void AbsentFillPixelTable (currfont)
    fontinfo * currfont;
#endif
{
  /* Fill the pixeltable for  * currfont
   * (a PostScript or other font that is missing and cannot be replaced),
   * using information in the DVI file, since we have nothing else.
   *
   * Updated (Wed 9 June 1993) to use cmr10 info., as that is the most
   * common font in most present-day documents.
   */

  int c;

  for (c = 0; c <= maxTeXchar; c++)
  {
    _REC_pixeltable * pix_tab = &currfont->pixelptr[c];

    /* scaled size of font in DVI units */
    int scale = currfont->scaledsize;  /* For character size estimate */
    int dheight, ddepth;  /* estimated height and depth, in DVI units */
    int pheight, pdepth;  /* same, in pixels */

#ifdef OLD_CHARDIM
    /* Let character be a square, scaledsize high and wide, zero depth. */
    /* For "example.tex", this width and height are not too bad. */
    pix_tab->dwidth = scale/2;  /* wild, hazardous guess */
    dheight = scale;
    ddepth = 0;
#else /* Not OLD_CHARDIM */
    if (c < CMSIZE)  /* 0 <= c < CMSIZE */
    {
      pix_tab->dwidth = scale * cmr10[c].wd;
      dheight = scale * cmr10[c].ht;
      ddepth = scale * cmr10[c].dp;
    }
    else  /* CMSIZE <= c <= maxTeXchar */
    {
      pix_tab->dwidth = 0;
      dheight = 0;
      ddepth = 0;
    }
#endif /* OLD_CHARDIM */

    /* convert DVI units to pixels */

    pix_tab->pwidth = XPixelRound (pix_tab->dwidth);
    pheight = YPixelRound (dheight);
    pdepth = YPixelRound (ddepth);

    /* Since we don't have access to PK bitmap or to TFM metric info,
       we will have to use the guessed (DVI) width/height/depth info
       to approximate wd, ht, xo, yo.
    */

    pix_tab->wd = pix_tab->pwidth;
    pix_tab->wd -= pix_tab->wd / 8;   /* better approximation */
    pix_tab->ht = pheight + pdepth;
    pix_tab->xo = 0;
    pix_tab->yo = pheight - 1;
    if (pix_tab->wd == 0 || pix_tab->ht == 0)   /* anything but 0 */
      pix_tab->mapadr = 0;   /* char blank or not in font */
    else
      pix_tab->mapadr = 1;
    pix_tab->bitmap.UU.mptr = (Word *) NULL;
  }
}
/* AbsentFillPixelTable */

#undef CMSIZE

/******************************************************************************/

#ifdef __STDC__
Void PixelTableRoutine (fontinfo * currfont)
#else
Void PixelTableRoutine (currfont)
  fontinfo * currfont;
#endif
{
  /* DVIReader has just allocated a new pixeltable for currfont^, and
     calls this routine from InterpretPage only ONCE per font
     (the first time the font is used).
     If this is the first time we've seen the font,
     then we build fontspec first.
     (Note that ShowStatistics, in the main program, may call BuildFontSpec
      first.)
     If we can't open the font file, we return dummy_pk or dummy_tfm values,
     but using the current font's scaledsize.
  */

#ifdef CURRFONT_TRACE
  fprintf (logfile, "PixelTableRoutine\n");
  fprintf (logfile, "currfont = %p\n", currfont);
  fprintf (logfile, "currfont->fontname = `%s'\n", currfont->fontname);
  fprintf (logfile, "currfont->fontspec = `%s'\n", currfont->fontspec);
  fprintf (logfile, "currfont->fontspeclen = %d\n", currfont->fontspeclen);
  fflush (logfile);
#endif /* CURRFONT_TRACE */

  if (!CheckFontPtr (currfont))
    return;

  /* Try to find and fopen (& fclose) PK file; if that fails, try TFM file. */
  if (currfont->fontspeclen == 0)
    (Void) BuildFontSpec (currfont);

  if (currfont->fontexists && !OpenFontFile (currfont->fontspec))
  {
    string mesg;

    StartText();
    ResetVDU();  /* let message stay */

    sprintf (mesg, "fopen-ed but couldn't open file `%s' for font `%s'",
             currfont->fontspec, currfont->fontname);
    MesgString (mesg);
    MesgLine();
    fprintf (logfile, "That shouldn't happen.\n");
    fflush (logfile);

    RestoreTerminal();
    exit (1);
  }

  /* gt - is this right when a dummy font is used? */

  if (currfont->fontexists)
  {
    if (currfont->pkfont)
      PKFillPixelTable (currfont);
    else
      TFMFillPixelTable (currfont);

    CloseFontFile();
  }
  else
  {
    AbsentFillPixelTable (currfont);
  }
}
/* PixelTableRoutine */

/******************************************************************************/

Void InitFontReader (VOID)
{
  /* This routine initializes some global variables. */

  int i;

  gpower[0] = 0;
  for (i = 1; i <= 32; i++)   /* used in GetBitmap */
    gpower[i] = gpower[i - 1] | (1 << (i - 1));
}
/* InitFontReader */

/******************************************************************************/

/* end fontreader.c */
