#include "RipData.h"
/*! \todo should replace all 252 with 255 */
#include "egapal.h"
#include <math.h>
#include <afxmt.h>
#include <vector>
#include <algorithm>


using namespace std;

#pragma warning (disable:4305)
#pragma warning (disable:4309)

/*
int standardPal[] =			{	0, 1, 2, 3, 4, 5, 6, 7, 
				 				56,57,58,59,60,61,62,63};
*/
/*
PALETTE Default_Palette = {      Define the palette for above EGA mode
  16, { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x14, 0x07,
        0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F  
		*/
int standardPal[] =		{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x14, 0x07,
						0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F };



char brushPatterns[][8]		= {	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},		//0
								{0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF},		//1
								{0xFF,0xFF,0x00,0x00,0xFF,0xFF,0x00,0x00},		//2
								{0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80},		//3								
								{0xE0,0xC1,0x83,0x07,0x0E,0x1C,0x38,0x70},		//4
								{0xF0,0x78,0x3C,0x1E,0x0F,0x87,0xC3,0xE1},		//5
								{0xA5,0xD2,0x69,0xB4,0x5A,0x2D,0x96,0x4B},		//6
								{0xFF,0x88,0x88,0x88,0xFF,0x88,0x88,0x88},		//7
								{0x81,0x42,0x24,0x18,0x18,0x24,0x42,0x81},		//8
								{0xCC,0x33,0xCC,0x33,0xCC,0x33,0xCC,0x33},		//9
								{0x80,0x00,0x08,0x00,0x80,0x00,0x08,0x00},		//10
								{0x88,0x00,0x22,0x00,0x88,0x00,0x22,0x00}};		//11

int lineStyles[] = {			~0xFFFFFFFF, ~0x33333333, ~0x1E3F1E3F, ~0x1F1F1F1F	};



CRipData::CRipData() :
	CBinaryData(),
	m_curX(0),
	m_curY(0),
	m_drawCol(0),
	m_writeMode(Normal),

	m_ownDC(0),
	m_pDC(0),
	m_pen(0),
	m_brush(0),
	m_fillBitmap(0),

	m_pClipboardDC(0),
	m_clipboardBitmap(0),
	m_clipboardWidth(0),
	m_clipboardHeight(0),
	m_savedDC(0)
{

	m_pen		 = new CPen();
	m_brush		 = new CBrush();
	m_fillBitmap = new CBitmap();	
	m_buffer.clear();
}


CRipData::~CRipData() {

		
	if (m_ownDC) {
		m_ownDC->DeleteDC();
		delete m_ownDC;
	}

	if (m_clipboardBitmap) {
		m_clipboardBitmap->DeleteObject();
		delete m_clipboardBitmap;
	}

	if (m_pClipboardDC) {
		m_pClipboardDC->DeleteDC();
		delete m_pClipboardDC;
	}
	
	if (m_pen) {
		m_pen->DeleteObject();
		delete m_pen;
	}

	if (m_brush) {
		m_brush->DeleteObject();
		delete m_brush;
	}

	if (m_fillBitmap) {
		m_fillBitmap->DeleteObject();
		delete m_fillBitmap;
	}

	m_buffer.clear();
}


void CRipData::init(CImageContext *context) {

	m_filePos = 0;

	int numBits		= 32;//context->m_pDC->GetDeviceCaps(BITSPIXEL);
	int numPlanes	= 1;
	
	if (context->m_animation!=ON) {

		if (!m_ownDC) {
			m_ownDC = new CDC();
		} else {
			m_ownDC->DeleteDC();
		}

		m_ownDC->CreateCompatibleDC(context->m_pDC);
		m_pDC = m_ownDC;

		if (!m_bitmap) {
			m_bitmap = new CBitmap();
			m_bitmap->CreateBitmap(m_width, m_height, numPlanes, numBits, NULL);
		}

		m_pDC->SelectObject(m_bitmap);
	} else {
			
		if (m_savedDC)
			context->m_pDC->RestoreDC(m_savedDC);

		m_pDC = context->m_pDC;
		context->m_savedDC = m_pDC->SaveDC();
	}


	m_pDC->SetBkMode(TRANSPARENT);
	m_pDC->PatBlt(0, 0, m_width, m_height, BLACKNESS);
	
	if (!m_pClipboardDC) {
		m_pClipboardDC = new CDC();
	} else {
		m_pClipboardDC->DeleteDC();
	}	
	m_pClipboardDC->CreateCompatibleDC(context->m_pDC);

	if (!m_clipboardBitmap) {
		m_clipboardBitmap = new CBitmap();
		m_clipboardBitmap->CreateBitmap(m_width, m_height, numPlanes, numBits, NULL);
	}
	m_pClipboardDC->SelectObject(m_clipboardBitmap);
	m_pClipboardDC->PatBlt(0, 0, m_width, m_height, BLACKNESS);
	m_pClipboardDC->SetBkMode(TRANSPARENT);
	m_clipboardWidth	= -1;
	m_clipboardHeight	= -1;

	m_fill.color		= 0;
	m_fill.pattern		= 0;
	
	m_line.thickness	= 1;
	m_line.pattern		= 0;
	m_drawCol			= 0;

	loadStandardPalette();
}


int CRipData::getNextCommand(CRipCommandData &command) {

	if (m_buffer.empty())
		return -1;

	CSingleLock singleLock(m_semaphore);
	singleLock.Lock();
	
	command = (CRipCommandData)m_buffer.front();
	m_buffer.pop_front();

	singleLock.Unlock();
	//Sleep(0);

	return m_buffer.size();
}


void CRipData::addCommand(CRipCommandData *command) {

	CSingleLock singleLock(m_semaphore);
	singleLock.Lock();

	m_buffer.push_back(*command);

	singleLock.Unlock();	
}


bool CRipData::hasCommands() {
	return (m_buffer.size()!=0);
}


void CRipData::executeCommand(CRipCommandData *command) {
	
	setBrush();
	setPen();
	switch(command->m_command) {
		case RIP_GET_IMAGE:				ripGetImage(command);		break;
		case RIP_PUT_IMAGE:				ripPutImage(command);		break;
		case RIP_TEXT:					ripText(command);			break;
		case RIP_TEXT_XY:				ripTextXY(command);			break;
		case RIP_FONT_STYLE:			ripFontStyle(command);		break;
		case RIP_COLOR:					ripColor(command);			break;
		case RIP_PIXEL:					ripPixel(command);			break;
		case RIP_LINE:					ripLine(command);			break;
		case RIP_RECTANGLE:				ripRectangle(command);		break;
		case RIP_ONE_PALETTE:			ripOnePalette(command);		break;
		case RIP_SET_PALETTE:			ripSetPalette(command);		break;
		case RIP_WRITE_MODE:			ripWriteMode(command);		break;
		case RIP_MOVE:					ripMove(command);			break;
		case RIP_BAR:					ripBar(command);			break;
		case RIP_CIRCLE:				ripCircle(command);			break;
		case RIP_OVAL_PIE_SLICE:		ripOvalPieSlice(command);	break;
		case RIP_OVAL:					ripOval(command);			break;
		case RIP_FILLED_OVAL:			ripFilledOval(command);		break;
		case RIP_POLYGON:				ripPolygon(command);		break;
		case RIP_FILL_POLYGON:			ripFillPolygon(command);	break;
		case RIP_POLYLINE:				ripPolyline(command);		break;
		case RIP_FILL:					ripFill(command);			break;
		case RIP_LINE_STYLE:			ripLineStyle(command);		break;
		case RIP_FILL_STYLE:			ripFillStyle(command);		break;
		case RIP_BEZIER:				ripBezier(command);			break;
		case RIP_ARC:					ripArc(command);			break;
		case RIP_OVAL_ARC:				ripOvalArc(command);		break;
		case RIP_PIE_SLICE:				ripPieSlice(command);		break;
		case RIP_FILL_PATTERN:			ripFillPattern(command);	break;
	}
}


/*! Help Functions */

#define PI	3.1415927
float degToRad(float deg) {
	return (deg*PI/180.f);
}


void CRipData::loadStandardPalette() {
	for (int i=0; i<16; i++) {
		m_pal[i] = standardPal[i];
	}
}


unsigned int CRipData::getCol(int index) {

	int r       = (char)egaPal[index][0]&0xFF;
	int g       = (char)egaPal[index][1]&0xFF;
	int b       = (char)egaPal[index][2]&0xFF;
	return (b<<16)|(g<<8)|(r);
}


void CRipData::setPen(int pattern, int userPattern) {
		
	int thickness = m_line.thickness;

	m_pen->DeleteObject();

	/*! \todo remove */
	if (pattern<100) {
		m_pen->CreatePen(PS_SOLID, thickness, getCol(m_pal[m_drawCol]));
	} else {
		/*! \todo implement line styles.. code is 6.0\riploader.cpp */
		/*
			 00      Normal, Solid Line    1111111111111111  FFFF             > v1.54
			 01      Dotted Line           0011001100110011  3333             > v1.54
			 02      Centered Line         0001111000111111  1E3F             > v1.54
			 03      Dashed Line           0001111100011111  1F1F             > v1.54
			 04      Custom Defined line (see about <user_pat> below)
		*/		
	}
}


void CRipData::setBrush() {

	m_brush->DeleteObject();
	if (m_fill.pattern==0) {

		m_brush->CreateSolidBrush(getCol(m_pal[0]));
	} else if (m_fill.pattern==1) {		

		m_brush->CreateSolidBrush(getCol(m_pal[m_fill.color]));
	} else {

		unsigned int ubits[64];

		m_fillBitmap->DeleteObject();
		m_fillBitmap->CreateBitmap(8,8,1,32,ubits);

		int col2 = getCol(m_pal[m_fill.color]);
		
		int r = col2&0xFF;
		int g = (col2>>8)&0xff;
		int b = (col2>>16)&0xff;

		col2 = (r<<16)|(g<<8)|b;

		int index = 0;
		for (int i=0; i<8; i++) {
			char line = brushPatterns[m_fill.pattern-2][i];
			for (int j=0; j<8; j++) {
				char bit = 1 << (8-j-1);
				ubits[index++] = (line&bit)? col2 : 0;
			}
		}
		
		/*! \note this should use ::SetDibBits */
		m_fillBitmap->SetBitmapBits(64*4, ubits);				
		m_brush->CreatePatternBrush(m_fillBitmap);
	}
}


/*! Command Implementations */

//RIP_GET_IMAGE
void CRipData::ripGetImage(CRipCommandData *command) { 

	int x0		= command->m_data[0];
	int y0		= command->m_data[1];
	int x1		= command->m_data[2];
	int y1		= command->m_data[3];
	char res	= command->m_data[4];

	m_clipboardWidth  = x1-x0;
	m_clipboardHeight = y1-y0;
	m_pClipboardDC->BitBlt(0, 0, m_clipboardWidth, m_clipboardHeight, m_pDC, x0, y0, SRCCOPY);
#ifdef DEBUG_RIP
	printf("RIP_GET_IMAGE: %d %d %d %d %d\n", x0, y0, x1, y1, res);
#endif
}

//RIP_PUT_IMAGE
void CRipData::ripPutImage(CRipCommandData *command) { 
	
	int x		= command->m_data[0];
	int y		= command->m_data[1];
	int mode	= command->m_data[2];
	char res	= command->m_data[3];

	int copyMode;
	switch(mode) {
		//Paste the image on-screen normally
		case 0: copyMode = SRCCOPY;		break;
		//Exclusive-OR  image with the one already on screen   
		case 1: copyMode = SRCINVERT;	break;
		//Logically OR  image with the one already on screen
		case 2: copyMode = SRCPAINT;	break;
		//Logically AND image with the one already on screen
		case 3: copyMode = SRCAND;		break;
		//Paste the inverse of the image on the screen
		case 4: copyMode = SRCINVERT;	break;
	}


	if (mode==4) {
		m_pClipboardDC->PatBlt(x, y, m_clipboardWidth, m_clipboardHeight, BLACKNESS);
	}

	m_pClipboardDC->BitBlt(x, y, m_clipboardWidth, m_clipboardHeight, m_pClipboardDC, x, y, copyMode);

#ifdef DEBUG_RIP
	printf("RIP_PUT_IMAGE: %d %d %d %d\n", x, y, mode, res);
#endif
}

//RIP_TEXT 
void CRipData::ripText(CRipCommandData *command) { 
	
	m_pDC->TextOut(m_curY, m_curX, command->m_text);
#ifdef DEBUG_RIP
	printf("RIP_TEXT: %s\n", command->text);
#endif
}
				
//RIP_TEXT_XY 
void CRipData::ripTextXY(CRipCommandData *command) { 
	
	int x				= command->m_data[0];
	int y				= command->m_data[1];
	m_pDC->TextOut(x, y, command->m_text);
#ifdef DEBUG_RIP
	printf("RIP_TEXT_XY: %d %d %s\n", x, y, command->m_text);
#endif
}
					
//RIP_FONT_STYLE
void CRipData::ripFontStyle(CRipCommandData *command) { 

	m_font.font		= command->m_data[0];
	m_font.direction= command->m_data[1];
	m_font.size		= command->m_data[2];
	m_font.res		= command->m_data[3];
#ifdef DEBUG_RIP
	printf("RIP_FONT_STYLE: %d, %d, %d, %d\n", m_font.font, m_font.direction, m_font.size, m_font.res);
#endif
}
					
//RIP_COLOR
void CRipData::ripColor(CRipCommandData *command) { 
	
	m_drawCol = command->m_data[0];
#ifdef DEBUG_RIP
	printf("RIP_COLOR: %d\n", m_drawCol);
#endif
}
			
//RIP_PIXEL
void CRipData::ripPixel(CRipCommandData *command) { 

	int x = command->m_data[0];
	int y = command->m_data[1];
	m_pDC->SetPixel(x, y, m_drawCol);
#ifdef DEBUG_RIP
	printf("RIP_PIXEL: %d, %d\n", x, y);
#endif
}

/*! This has to be implemented because GDI cannot do everything that BGI can do and is sometimes off by a pixel */
/*! Yes. A pixel. */
void drawLine(unsigned int col, int thickness, int pattern, int userPattern, POINT *pts, int numPts, CDC *pDC) {
	

	int bitpattern;
	unsigned int andAmt = 1<<31;
	if (pattern!=4) {
		bitpattern = ~lineStyles[pattern];
	} else {
	//	userPattern = ((userPattern&0xFF)<<8)|((userPattern&0xFF00)>>8);
		bitpattern = (userPattern)|((userPattern)<<16);
	}
	

	int cur = 0;
	while(cur<(numPts-1)) {
		
		POINT start = pts[cur];
		POINT end	= pts[cur+1];
		
		bool rev = (numPts==2);
		if (end.x<start.x) {
			int temp = end.x;
			end.x = start.x;
			start.x = temp;
			
			temp = end.y;
			end.y = start.y;
			start.y = temp;
			rev = true;
		}
		
		int dx = abs(end.x-start.x);	
		int dy = abs(end.y-start.y);
		
		int x = start.x;
		int y = start.y;
		
		int Xincr, Yincr;
		if (start.x > end.x) { Xincr=-1; } else { Xincr=1; }	
		if (start.y > end.y) { Yincr=-1; } else { Yincr=1; }
		
		if (dx >= dy)	
		{           
			y -= thickness/2;
			int dPr 	= dy<<1; 
			int dPru 	= dPr - (dx<<1);
			int P 		= dPr - dx;
			
			for (; dx>=0; dx--) {				

				if (bitpattern&andAmt) 
					for (int j=0; j<thickness; j++) 
						pDC->SetPixel(x, y+j, col);
				andAmt = _rotr(andAmt, 1);

				if ((rev && P>0) || (!rev && P>=0)) { 
					x+=Xincr;
					y+=Yincr;
					P+=dPru;
				} else {
					x+=Xincr;
					P+=dPr;
				}
			}		
		} else {
			
			x -= thickness/2;
			int dPr 	= dx<<1;
			int dPru 	= dPr - (dy<<1);
			int P 		= dPr - dy;
			
			for (; dy>=0; dy--)	 {
				
				if (bitpattern&andAmt) 
					for (int j=0; j<thickness; j++) 
						pDC->SetPixel(x+j, y, col);
				andAmt = _rotr(andAmt, 1);

				//if ((rev && P>0) || (!rev && P>=0)) { 
				if (P>=0) { 
					x+=Xincr;
					y+=Yincr;
					P+=dPru;
				} else {
					y+=Yincr;
					P+=dPr;
				}
			}		
		}	
		cur++;
	}
}


struct Edge {
	int x[2], xstart;
	int y[2];
	float m;
	bool xrev, yrev;

	int numer, denom, incr;
};


bool edgeCompare(const Edge &a, const Edge &b) {
	if (a.x[0]==b.x[0])
		return (a.y[0]<b.y[0]);		
	return (a.x[0]<b.x[0]);
}

enum JointType {
	JOINT_NONE=0,
	JOINT_SIDE,
	JOINT_TIP,
	JOINT_BOTTOM
};

JointType calcJoint(int cury, const Edge &a, const Edge &b) {	

	if (cury==a.y[0] && cury==b.y[0] && a.xstart==b.xstart)
		return JOINT_TIP;

	if (cury==a.y[1] && cury==b.y[0] && a.x[1]==b.x[0])
		return JOINT_SIDE;

	if (cury==a.y[1] && cury==b.y[1] && a.x[1]==b.x[1])
		return JOINT_BOTTOM;

	return JOINT_NONE;
}

#define FIXED_POINT (8)
/*! p.98 gamma et al */
void drawPoly(unsigned int col, POINT *pts, int numPts, CDC *pDC, RipPattern *pattern=0) {

	static vector<Edge>	edgeTable[480];
	static vector<Edge>	aet;


	//Clear edge table
	for (int i=0; i<480; i++) {
		edgeTable[i].clear();
	}

	//Initialize edge table
	int ymin = 480;
	int ymax = -1;
	for(i=0; i<numPts; i++) {

		POINT a = pts[i];
		POINT b = pts[(i+1)%numPts];

		Edge e;
		e.y[0] = __min(a.y, b.y);
		e.y[1] = __max(a.y, b.y);
		if (a.y>b.y) {
			
			e.yrev = false;
			if (a.x<b.x) {
				e.xrev = true;
			} 
			e.m    = (float)(a.x-b.x)/(float)(a.y-b.y);
			e.x[0] = b.x;
			e.x[1] = a.x;			
		} else if (b.y>a.y) {
			
			e.yrev = true;
			if (b.x<a.x) {
				e.xrev = true;
			} 
			e.m    = (float)(b.x-a.x)/(float)(b.y-a.y);
			e.x[0] = a.x;
			e.x[1] = b.x;
		} else {
			e.m    = 0;
			e.x[0] = __min(a.x, b.x);
			e.x[1] = __max(a.x, b.x);

			e.yrev = false;
			if (b.x<a.x)
				e.xrev = true;
		}
		
		e.numer = abs(e.x[1]-e.x[0]);
		e.denom = e.y[1]-e.y[0];
		e.incr  = e.denom;
		e.xstart = e.x[0];
		ymin  = __min(ymin, e.y[0]);
		ymax  = __max(ymax, e.y[1]);
		edgeTable[(int)e.y[0]].push_back(e);
	}

	//Clear active edge table
	aet.clear();
	
	do {
		vector<Edge>::iterator i;

		//Add new edges
		if (edgeTable[ymin].size()!=0) {
			for (i=edgeTable[ymin].begin(); i!=edgeTable[ymin].end(); i++) 
				aet.push_back(*i);
		}

		//If at the last scanline for the edge, set its x to be that last x value to avoid rounding errors
		for (i=aet.begin(); i!=aet.end(); i++)
			if ((*i).y[1]==ymin)
				(*i).x[0] = (*i).x[1];

		//Sort by x
		sort(aet.begin(), aet.end(), edgeCompare);

		//Pattern setup		
		char *pat;
		if (pattern) {
			if (pattern->pattern<10) {
				pat = &brushPatterns[pattern->pattern][0];
			} else {
				pat = &pattern->userPattern[0];
			}
		}


		//Draw the scanlines
		bool fill = true;
		int lastStart = -1;
		int andAmt = 1;
		for (i=aet.begin(); i!=(aet.end()-1); i++) {

			JointType joint = calcJoint(ymin, *i, *(i+1));
			if (joint==JOINT_SIDE)				
				continue;

/*			
			if (joint==JOINT_BOTTOM) {
				fill = !fill;
				continue;
			}
//*/
			if (fill) {

				int start = (*i).x[0];
				int end   = (*(i+1)).x[0];

				int scol=0, ecol=0xFFFFFF;

				if ((*i).m<0) {
					start = start + 1;// + (((*i).x[0]&andAmt)>>(FIXED_POINT-1));
				} else if ((*i).m>0 && ymin!=(*i).y[0])
					start--;
			//	else if ((*i).m==0)
			//		start=start-1;

				if (ymin!=(*(i+1)).y[0] && ymin!=(*(i+1)).y[1]) {
					if ((*(i+1)).m<0)
						end++;
					else if ((*(i+1)).m>0)
						end--;
				} else {
					end = end;
				}
				


/*	
				if ((*i).yrev) {
					scol=0xFF;
					if ((*i).m<0) {
						start = start+1;
						scol|=0xFF00;
					} else {
						start = start;
					}
				} else {
					scol=0xFF0000;
					if ((*i).m<0) {
						start = start+1;
						scol|=0xFF00;
					} else {
						start = start;
					}
				}
				
				if ((*(i+1)).yrev) {
					ecol=0xFF;
					if ((*(i+1)).m<0) {
						end = end+1;
						ecol|=0xFF00;
					} else {
						end = end;
					}
				} else {
					ecol=0xFF0000;
					if ((*(i+1)).m<0) {
						end = end+1;
						ecol|=0xFF00;
					} else {
						end = end;
					}
				}
*/

				if (joint==JOINT_TIP) {
					if ((*i).m<0)
						start--;
					end = start;
				}  
				
				/*
				if ((int)joint) {
					pDC->SetPixel(start, ymin, 0xFF);
					fill = false;
				} else {*/
				unsigned int bitpattern = (pattern)?(unsigned char)pat[(ymin%8)]:0xFF;
				bitpattern = bitpattern|(bitpattern<<8)|(bitpattern<<16)|(bitpattern<<24);
				unsigned int andAmt = 1<<((start+7)%8);
				for (int j=start; j<=end; j++) {
					if (bitpattern&andAmt) 
						pDC->SetPixel(j, ymin, col);
					andAmt = _rotr(andAmt, 1);
				}
				//}
				
				
				//pDC->SetPixel(start, ymin, scol);
				//pDC->SetPixel(end, ymin, ecol);

				lastStart = start;
			}
			fill = !fill;
		}

		//Remove edges
		i = i=aet.begin();
		while(i!=aet.end()) {
			if ((*i).y[1]==ymin) {
				aet.erase(i);
				i = aet.begin();
			} else {
				i++;
			}
		}
		
		//Update x values
		for (i=aet.begin(); i!=aet.end(); i++) {
			//(*i).x[0] += (*i).m; 
			(*i).incr += (*i).numer;
			while ((*i).incr>=(*i).denom) {
				(*i).incr -= (*i).denom;
				if ((*i).m<0)
					(*i).x[0]--;
				else if ((*i).m>0)
					(*i).x[0]++;
			}
		}
			
		ymin++;		
	} while(aet.size()>1 && ymin<ymax);
//	drawLine(col, 1, 0, 0, pts, numPts, pDC);
}




//RIP_LINE
void CRipData::ripLine(CRipCommandData *command) { 

	POINT pts[2];
	pts[0].x = command->m_data[0];
	pts[0].y = command->m_data[1];
	pts[1].x = command->m_data[2];
	pts[1].y = command->m_data[3];
/*	
	setPen(m_line.pattern, m_line.userPattern);
	m_pDC->SelectObject(m_pen);	
	m_pDC->Polyline(pts,2);
*/
	drawLine(getCol(m_pal[m_drawCol]), m_line.thickness, m_line.pattern, m_line.userPattern, pts, 2, m_pDC);
#ifdef DEBUG_RIP
	printf("RIP_LINE: %d %d %d %d\n", pts[0].x, pts[0].y, pts[1].x, pts[1].y);
#endif
}
							
//RIP_RECTANGLE 
void CRipData::ripRectangle(CRipCommandData *command) { 

	int x0 = command->m_data[0];
	int y0 = command->m_data[1];
	int x1 = command->m_data[2];
	int y1 = command->m_data[3];

	POINT pts[5];
	pts[0].x = x0;
	pts[0].y = y0;
	pts[1].x = x1;
	pts[1].y = y0;
	pts[2].x = x1;
	pts[2].y = y1;
	pts[3].x = x0;
	pts[3].y = y1;
	pts[4].x = x0;
	pts[4].y = y0;
	
	//setPen(m_line.pattern, m_line.userPattern);
	//m_pDC->SelectObject(m_pen);
	//m_pDC->Polyline(pts, 5);
	drawLine(getCol(m_pal[m_drawCol]), m_line.thickness, m_line.pattern, m_line.userPattern, pts, 5, m_pDC);
#ifdef DEBUG_RIP
	printf("RIP_RECTANGLE: %d %d %d %d\n", x0, y0, x1, y1);
#endif
}

//RIP_ONE_PALETTE
void CRipData::ripOnePalette(CRipCommandData *command) { 

	int index   = command->m_data[0];
	int value	= command->m_data[1];
	m_pal[index]= value;
#ifdef DEBUG_RIP
	printf("RIP_ONE_PALETTE: %d %d\n", index, value);
#endif
}

//RIP_SET_PALETTE
void CRipData::ripSetPalette(CRipCommandData *command) { 

	for (int i=0; i<16; i++) {
		m_pal[i] = command->m_data[i];
	}
#ifdef DEBUG_RIP
	printf("RIP_SET_PALETTE: %d %d %d ...\n", pal[0], pal[1], pal[2]);
#endif
}

//RIP_WRITE_MODE
void CRipData::ripWriteMode(CRipCommandData *command) { 

	m_writeMode = (RipWriteMode)command->m_data[0];

    //00     Normal drawing mode (overwrite)
    //01     XOR (complimentary) mod
	/*! \todo xor'd write mode */
	if (m_writeMode==Xor) {
		::AfxMessageBox("using write xor'd write mode not implmenented.. email gz at tom.germano@snet.net with the rip attached");
	}
#ifdef DEBUG_RIP
	printf("RIP_WRITE_MODE: %d\n", m_writeMode);
#endif
}

//RIP_MOVE
void CRipData::ripMove(CRipCommandData *command) { 

	m_curX = command->m_data[0];
	m_curY = command->m_data[1];
	m_pDC->MoveTo(m_curX,m_curY);
#ifdef DEBUG_RIP
	printf("RIP_MOVE: %d %d\n", m_curX, m_curY);
#endif
}

//RIP_BAR bb
void CRipData::ripBar(CRipCommandData *command) { 

	CRect rc;
	rc.left		= command->m_data[0];
	rc.top		= command->m_data[1]-1;
	/*! \note !! */
	rc.right	= command->m_data[2];//+1;
	rc.bottom	= command->m_data[3]+1;//+1;

	m_pDC->SelectStockObject(NULL_PEN);
	m_pDC->SelectObject(m_brush);

	POINT pts[4];
	pts[0].x = rc.left;
	pts[0].y = rc.top;
	pts[1].x = rc.right;
	pts[1].y = rc.top;
	pts[2].x = rc.right;
	pts[2].y = rc.bottom;
	pts[3].x = rc.left;
	pts[3].y = rc.bottom;
	
	//m_pDC->Polygon(pts, 4);
	drawPoly(getCol(m_pal[m_fill.color]), pts, 4, m_pDC, &m_fill);
#ifdef DEBUG_RIP
	printf("RIP_BAR: %d %d %d %d\n", rc.left, rc.top, rc.right, rc.bottom);
#endif
}

//RIP_CIRCLE bb
void CRipData::ripCircle(CRipCommandData *command) { 

	int x		= command->m_data[0];
	int y		= command->m_data[1];
	int radius  = command->m_data[2];
	int ulx		= x-radius;
	int uly		= y-radius;
	int lrx		= x+radius;
	int lry		= y+radius;

	setPen();
	m_pDC->SelectObject(m_pen);
	m_pDC->SelectStockObject(NULL_BRUSH);
	m_pDC->Ellipse(ulx, uly, lrx, lry);
#ifdef DEBUG_RIP
	printf("RIP_CIRCLE: %d %d %d\n", x, y, radius);
#endif
}
							
//RIP_OVAL_PIE_SLICE bb
void CRipData::ripOvalPieSlice(CRipCommandData *command) { 

	int x		= command->m_data[0];
	int y		= command->m_data[1];
	int stAng	= command->m_data[2];
	int eAng	= command->m_data[3];
	int radx	= command->m_data[4];
	int rady	= command->m_data[5];

	int ulx		= x-radx;
	int uly		= y-rady;
	int lrx		= x+radx;
	int lry		= y+rady;
	int startx	= radx*cos(degToRad(stAng));
	int starty	= rady*sin(degToRad(stAng));
	int endx	= radx*cos(degToRad(eAng));
	int endy	= rady*sin(degToRad(eAng));

	setPen();
	m_pDC->SelectObject(m_pen);
	m_pDC->SelectObject(m_brush);
	m_pDC->Pie(ulx, uly, lrx, lry, startx, starty, endx, endy);
#ifdef DEBUG_RIP
	printf("RIP_OVAL_PIE_SLICE: %d %d %d %d %d %d\n", x, y, stAng, eAng, radx, rady);
#endif
}

//RIP_OVAL bb
void CRipData::ripOval(CRipCommandData *command) { 

	int x		= command->m_data[0];
	int y		= command->m_data[1];
	int stAng	= command->m_data[2];
	int eAng	= command->m_data[3];
	int radx	= command->m_data[4];
	int rady	= command->m_data[5];

	int ulx		= x-radx;
	int uly		= y-rady;
	int lrx		= x+radx;
	int lry		= y+rady;

	setPen();
	m_pDC->SelectObject(m_pen);
	m_pDC->SelectStockObject(NULL_BRUSH);
	m_pDC->Ellipse(ulx, uly, lrx, lry);
#ifdef DEBUG_RIP
	printf("RIP_OVAL: %d %d %d %d %d %d\n", x, y, stAng, eAng, radx, rady);
#endif
}

//RIP_FILLED_OVAL bb
void CRipData::ripFilledOval(CRipCommandData *command) { 

	int x		= command->m_data[0];
	int y		= command->m_data[1];
	int radx	= command->m_data[2];
	int rady	= command->m_data[3];

	int ulx		= x-radx;
	int uly		= y-rady;
	int lrx		= x+radx;
	int lry		= y+rady;

	setPen();
	m_pDC->SelectObject(m_pen);
	m_pDC->SelectObject(m_brush);
	m_pDC->Ellipse(ulx, uly, lrx, lry);
#ifdef DEBUG_RIP
	printf("RIP_FILLED_OVAL: %d %d %d %d\n", x, y, radx, rady);
#endif
}

//RIP_POLYGON
void CRipData::ripPolygon(CRipCommandData *command) { 

	int numPoints = command->m_data[0];
	POINT first;
	first.x = command->m_data[1];
	first.y = command->m_data[2];

	setPen(m_line.pattern, m_line.userPattern);
	m_pDC->SelectObject(m_pen);
//	POINT pos = m_pDC->GetCurrentPosition();
	m_pDC->MoveTo(first);

	POINT pts[2];
	pts[0] = first;

	int offset = 0;
	for (int i=1; i<numPoints; i++) {
	//	POINT pt;
		pts[1].x = command->m_data[3+offset++];
		pts[1].y = command->m_data[3+offset++];
		//m_pDC->LineTo(pt);
		//m_pDC->MoveTo(pt);
		drawLine(getCol(m_pal[m_drawCol]), m_line.thickness, m_line.pattern, m_line.userPattern, pts, 2, m_pDC);
		pts[0] = pts[1];
	}	
	//m_pDC->LineTo(first);
	//m_pDC->MoveTo(pos);
	pts[1] = first;
	drawLine(getCol(m_pal[m_drawCol]), m_line.thickness, m_line.pattern, m_line.userPattern, pts, 2, m_pDC);
/*#ifdef DEBUG_RIP
	printf("RIP_POLYGON: %d %d\n", pt.x, pt.y);
#endif*/
}




//RIP_FILL_POLYGON 
void CRipData::ripFillPolygon(CRipCommandData *command) { 

	/*! Do not like this here */
	POINT pts[512];

	int i;
	int numPoints = command->m_data[0];
	int offset=1;
	for (i=0; i<numPoints; i++) {
		pts[i].x = command->m_data[offset++];
		pts[i].y = command->m_data[offset++];
	}

	drawPoly(getCol(m_pal[m_fill.color]), pts, numPoints, m_pDC);

	/*
	m_pDC->SelectStockObject(NULL_PEN);
//	m_pDC->SelectObject(m_pen);
	m_pDC->SelectObject(m_brush);
	m_pDC->Polygon(pts, numPoints);
	
	if (m_drawCol==0)
		m_drawCol = m_fill.color;
	
	//Draw outline if it isn't black
	//if (m_drawCol!=0 || (m_drawCol==0 && m_fill.color==0)) {

		//setPen(m_line.pattern, m_line.userPattern);
		//m_pDC->SelectObject(m_pen);
		//m_pDC->SelectStockObject(NULL_BRUSH);

	drawLine(getCol(m_pal[m_drawCol]), m_line.thickness, m_line.pattern, m_line.userPattern, pts, numPoints, m_pDC);
	/*
		POINT pt = m_pDC->GetCurrentPosition();
		//m_pDC->MoveTo(pts[0]);
		for (i=1; i<numPoints; i++) {
			//m_pDC->LineTo(pts[i]);
			//m_pDC->MoveTo(pts[i]);
			drawLine(getCol(m_pal[m_drawCol]), m_line.thickness, m_line.pattern, m_line.userPattern, &pts[i-1], 2, m_pDC);
		}	
		//m_pDC->LineTo(pts[0]);
		//m_pDC->MoveTo(pt);
		
		POINT pts2[2];
		pts2[0] = pts[0];
		pts2[1] = pts[numPoints - 1];
		drawLine(getCol(m_pal[m_drawCol]), m_line.thickness, m_line.pattern, m_line.userPattern, pts2, 2, m_pDC);
		
	//}
		*/
	//*
	POINT pts2[2];
	if (m_drawCol!=0) {
		for (i=0; i<numPoints; i++) {
			pts2[0] = pts[i];
			pts2[1] = pts[(i+1)%numPoints];
			drawLine(getCol(m_pal[m_drawCol]), m_line.thickness, m_line.pattern, m_line.userPattern, pts2, 2, m_pDC);
			//drawLine(getCol(m_pal[m_fill.color]), m_line.thickness, m_line.pattern, m_line.userPattern, pts2, 2, m_pDC);
		}
	}
	//*/

#ifdef DEBUG_RIP
	printf("RIP_FILL_POLYGON: %d %d\n", pts[0].x, pts[0].y);
#endif
}

//RIP_POLYLINE bb
void CRipData::ripPolyline(CRipCommandData *command) { 

	POINT pts[512];
	int numPoints = command->m_data[0];
	int offset = 1;
	for (int i=0; i<numPoints; i++) {
		pts[i].x = command->m_data[offset++];
		pts[i].y = command->m_data[offset++];
	}

	setPen(m_line.pattern, m_line.userPattern);
	m_pDC->SelectObject(m_pen);
	//m_pDC->Polyline(pts, numPoints);
	drawLine(getCol(m_pal[m_drawCol]), m_line.thickness, m_line.pattern, m_line.userPattern, pts, numPoints, m_pDC);
#ifdef DEBUG_RIP
	printf("RIP_POLYLINE: %d %d\n", pts[0].x, pts[0].y);
#endif
}

//RIP_FILL bb
void CRipData::ripFill(CRipCommandData *command) { 

	int x		= command->m_data[0];
	int y		= command->m_data[1];
	int border	= command->m_data[2];
	m_pDC->SelectObject(m_brush);
	m_pDC->FloodFill(x, y, getCol(m_pal[border]));
#ifdef DEBUG_RIP
	printf("RIP_FILL: %d %d %d\n", x, y, border);
#endif
}

//RIP_LINE_STYLE bb
void CRipData::ripLineStyle(CRipCommandData *command) { 

	m_line.pattern		= command->m_data[0];
	m_line.userPattern  = command->m_data[1];
	m_line.thickness	= command->m_data[2];
#ifdef DEBUG_RIP
	printf("RIP_LINE_STYLE: %d %d %d\n", m_line.pattern, m_line.userPattern, m_line.thickness);
#endif
}

//RIP_FILL_STYLE bb
void CRipData::ripFillStyle(CRipCommandData *command) { 

	m_fill.pattern	= command->m_data[0];
	m_fill.color	= command->m_data[1];
	setBrush();
#ifdef DEBUG_RIP
	printf("RIP_FILL_STYLE: %d %d\n", m_fill.pattern, m_fill.color);
#endif
}


//RIP_BEZIER bb
void CRipData::ripBezier(CRipCommandData *command) { 

	POINT pts[4];
	int offset = 0;
	for (int i=0; i<4; i++) {
		pts[i].x = command->m_data[offset++];
		pts[i].y = command->m_data[offset++];
	}
	int count = command->m_data[offset++];

	setPen(m_line.pattern, m_line.userPattern);
	m_pDC->SelectObject(m_pen);

	/*! does not work properly! */
	//pDC->PolyBezier(pts, 4);

	bool first = true;
	/*! \note set third parm TO 256 FOR precision, 64 FOR speed */
	//int nSteps = 256;
	int nSteps = count;
	double step = 1.f/(double)(nSteps);
	
	double t=0,tm3,t2,t2m3,t3,t3m3;
	double nc1,nc2,nc3,nc4;
	POINT old, cur, drawPts[2];

	for (i=0; i<nSteps; i++) {

        t  +=	step;
		tm3	= t*3.f; 
		t2	= t*t; 
		t2m3= t2*3.f; 
		t3	= t2*t; 
		t3m3= t3*3.f;
		
		nc1	= 1-tm3+t2m3-t3;
		nc2	= t3m3-2.f*t2m3+tm3;
		nc3	= t2m3-t3m3;
		nc4	= t3;

		double x = nc1*pts[0].x + nc2*pts[1].x + nc3*pts[2].x + nc4*pts[3].x;
		double y = nc1*pts[0].y + nc2*pts[1].y + nc3*pts[2].y + nc4*pts[3].y;
		cur.x = (x-floor(x))>0.5?ceil(x):floor(x);
		cur.y = (y-floor(y))>0.5?ceil(y):floor(y);
		if (first) {
			drawPts[0] = pts[0];
			drawPts[1] = cur;			
			first = false;
		} else {
			drawPts[0] = old;
			drawPts[1] = cur;		   
		}

		//m_pDC->Polyline(drawPts, 2);
		drawLine(getCol(m_pal[m_drawCol]), m_line.thickness, m_line.pattern, m_line.userPattern, drawPts, 2, m_pDC);
		old.x = cur.x;
		old.y = cur.y;	  
	}	

#ifdef DEBUG_RIP
	printf("RIP_BEZIER: %d %d %d %d\n", pts[0].x, pts[0].y, pts[1].x, pts[1].y);
#endif
}

//RIP_ARC bb
void CRipData::ripArc(CRipCommandData *command) { 

	int x		= command->m_data[0];
	int y		= command->m_data[1];
	int stAng	= command->m_data[2];
	int eAng	= command->m_data[3];
	int radius	= command->m_data[4];

	setPen();
	m_pDC->SelectObject(m_pen);
	m_pDC->AngleArc(x, y, radius, stAng, eAng);
#ifdef DEBUG_RIP
	printf("RIP_ARC: %d %d %d %d %d\n", x, y, stAng, eAng, radius);
#endif
}

//RIP_OVAL_ARC bb
void CRipData::ripOvalArc(CRipCommandData *command) { 

	int x		= command->m_data[0];
	int y		= command->m_data[1];
	int stAng	= command->m_data[2];
	int eAng	= command->m_data[3];
	int radx	= command->m_data[4];
	int rady	= command->m_data[5];

	int ulx		= x-radx;
	int uly		= y-rady;
	int lrx		= x+radx;
	int lry		= y+rady;
	int startx	= radx*cos(degToRad(stAng));
	int starty	= rady*sin(degToRad(stAng));
	int endx	= radx*cos(degToRad(eAng));
	int endy	= rady*sin(degToRad(eAng));

	setPen();
	m_pDC->SelectObject(m_pen);
	m_pDC->Arc(ulx, uly, lrx, lry, startx, starty, endx, endy);
#ifdef DEBUG_RIP
	printf("RIP_OVAL_ARC: %d %d %d %d %d %d\n", x, y, stAng, eAng, radx, rady);
#endif
}

//RIP_PIE_SLICE bb
void CRipData::ripPieSlice(CRipCommandData *command) { 

	int x		= command->m_data[0];
	int y		= command->m_data[1];
	int stAng	= command->m_data[2];
	int eAng	= command->m_data[3];
	int radius	= command->m_data[4];

	int ulx		= x-radius;
	int uly		= y-radius;
	int lrx		= x+radius;
	int lry		= y+radius;
	int startx	= radius*cos(degToRad(stAng));
	int starty	= radius*sin(degToRad(stAng));
	int endx	= radius*cos(degToRad(eAng));
	int endy	= radius*sin(degToRad(eAng));

	setPen();
	m_pDC->SelectObject(m_pen);
	m_pDC->SelectObject(m_brush);
	m_pDC->Pie(ulx, uly, lrx, lry, startx, starty, endx, endy);
#ifdef DEBUG_RIP
	printf("RIP_PIE_SLICE: %d %d %d %d %d\n", x, y, stAng, eAng, radius);
#endif
}

//RIP_FILL_PATTERN 
void CRipData::ripFillPattern(CRipCommandData *command) { 

//	char c[8];
	for (int i=0; i<8; i++)
		m_fill.userPattern[i] = command->m_data[i];

	m_fill.userColor = command->m_data[8];
#ifdef DEBUG_RIP
	printf("RIP_FILL_PATTERN: %d %d %d %d\n", c[0], c[1], c[2], c[3]);
#endif
}

