/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

// package includes

// extern includes
#include <caosGL/core/math.h>

#include <caosGL/gfx/cMatrix.h>

using namespace caosGL::gfx;

namespace caosGL {
	namespace gfx {
		/**
		 *<br> class:		cMatrix
		 *<br> namespace:	caosGL::gfx::threeD
		 *<br> inherits:	<none>
		 *<br> implements:	<none>
		 *<br> purpose:		4x4 cMatrix for storage & manipulation of transformations in scene graph. 
		 *					Provides basic maths operations, IO and via osg::Object reference counting.
		 *					You can directly load the cMatrix with OpenGL's LoadcMatrixf () function via
		 *					the public member _mat as the cMatrix is stored in the OpenGL format.
		 *					Caution: The disadvantage of this feature is, that the cMatrix access is
		 *					'transposed' if you compare it with the standard C/C++ 2d-array-access
		 *					convention . I.e. _mat[i][j] accesses the ith column of the jth row in the
		 *					4x4 cMatrix.
		 */

		static log4cpp::Category & cat = log4cpp::Category::getInstance ("caosGL::gfx::cMatrix");

		#define square(x) ((x)* (x))
		#define DEG2RAD(x) ((x)*M_PI/180.0)
		#define RAD2DEG(x) ((x)*180.0/M_PI)

		typedef struct quaternion_
		{
			tFloat x ;
			tFloat y ;
			tFloat z ;
			tFloat w ;
		} quaternion ;

		/* C = a (row).b (row) */

		#define matrix_inner_product(a,b,row,col,C) \
			{ \
				 (C)[row][col] = (a)[row][0] * (b)[0][col] + \
					 (a)[row][1] * (b)[1][col] + \
					 (a)[row][2] * (b)[2][col] + \
					 (a)[row][3] * (b)[3][col]; \
			}

		/* C = a.b */

		#define matrix_mult(a,b,C) \
			{ \
				matrix_inner_product (a, b, 0, 0, C); \
				matrix_inner_product (a, b, 0, 1, C); \
				matrix_inner_product (a, b, 0, 2, C); \
				matrix_inner_product (a, b, 0, 3, C); \
				matrix_inner_product (a, b, 1, 0, C); \
				matrix_inner_product (a, b, 1, 1, C); \
				matrix_inner_product (a, b, 1, 2, C); \
				matrix_inner_product (a, b, 1, 3, C); \
				matrix_inner_product (a, b, 2, 0, C); \
				matrix_inner_product (a, b, 2, 1, C); \
				matrix_inner_product (a, b, 2, 2, C); \
				matrix_inner_product (a, b, 2, 3, C); \
				matrix_inner_product (a, b, 3, 0, C); \
				matrix_inner_product (a, b, 3, 1, C); \
				matrix_inner_product (a, b, 3, 2, C); \
				matrix_inner_product (a, b, 3, 3, C); \
			}

		static tVoid quaternion_matrix (quaternion *q, tFloat mat[4][4])
		{
			/* copied from Shoemake/ACM SIGGRAPH 89 */
			tFloat xs, ys, zs, wx, wy, wz, xx, xy, xz, yy, yz, zz ;

			xs = q->x + q->x;
			ys = q->y + q->y;
			zs = q->z + q->z;

			wx = q->w * xs ; wy = q->w * ys ; wz = q->w * zs ;
			xx = q->x * xs ; xy = q->x * ys ; xz = q->x * zs ;
			yy = q->y * ys ; yz = q->y * zs ; zz = q->z * zs ;

			mat[0][0] = 1.0 - (yy + zz) ;
			mat[0][1] = xy - wz ;
			mat[0][2] = xz + wy ;
			mat[1][0] = xy + wz ;
			mat[1][1] = 1.0 - (xx + zz) ;
			mat[1][2] = yz - wx ;
			mat[2][0] = xz - wy ;
			mat[2][1] = yz + wx ;
			mat[2][2] = 1.0 - (xx + yy) ;

			mat[0][3] = 0.0;
			mat[1][3] = 0.0;
			mat[2][3] = 0.0;

			mat[3][0] = 0.0;
			mat[3][1] = 0.0;
			mat[3][2] = 0.0;
			mat[3][3] = 1.0;
		}


		cMatrix::cMatrix ()
		{
			makeIdent ();
		}


		cMatrix::cMatrix (const cMatrix & matrix)
		{
			memcpy (_mat,matrix._mat,sizeof (_mat));
		}


		cMatrix & cMatrix::operator = (const cMatrix & matrix)
		{
			if ( &matrix==this) return *this;
			memcpy (_mat,matrix._mat,sizeof (_mat));
			return *this;
		}


		cMatrix::cMatrix (
		tFloat a00, tFloat a01, tFloat a02, tFloat a03,
		tFloat a10, tFloat a11, tFloat a12, tFloat a13,
		tFloat a20, tFloat a21, tFloat a22, tFloat a23,
		tFloat a30, tFloat a31, tFloat a32, tFloat a33)
		{
			_mat[0][0] = a00;
			_mat[0][1] = a01;
			_mat[0][2] = a02;
			_mat[0][3] = a03;

			_mat[1][0] = a10;
			_mat[1][1] = a11;
			_mat[1][2] = a12;
			_mat[1][3] = a13;

			_mat[2][0] = a20;
			_mat[2][1] = a21;
			_mat[2][2] = a22;
			_mat[2][3] = a23;

			_mat[3][0] = a30;
			_mat[3][1] = a31;
			_mat[3][2] = a32;
			_mat[3][3] = a33;
		}


		cMatrix::~cMatrix ()
		{
		}


		tVoid cMatrix::makeIdent ()
		{
			_mat[0][0] = 1.0f;
			_mat[0][1] = 0.0f;
			_mat[0][2] = 0.0f;
			_mat[0][3] = 0.0f;

			_mat[1][0] = 0.0f;
			_mat[1][1] = 1.0f;
			_mat[1][2] = 0.0f;
			_mat[1][3] = 0.0f;

			_mat[2][0] = 0.0f;
			_mat[2][1] = 0.0f;
			_mat[2][2] = 1.0f;
			_mat[2][3] = 0.0f;

			_mat[3][0] = 0.0f;
			_mat[3][1] = 0.0f;
			_mat[3][2] = 0.0f;
			_mat[3][3] = 1.0f;
		}

		tVoid cMatrix::set (const tFloat* m)
		{
			 _mat[0][0] = m[0];
			 _mat[0][1] = m[1];
			 _mat[0][2] = m[2];
			 _mat[0][3] = m[3];

			 _mat[1][0] = m[4];
			 _mat[1][1] = m[5];
			 _mat[1][2] = m[6];
			 _mat[1][3] = m[7];

			 _mat[2][0] = m[8];
			 _mat[2][1] = m[9];
			 _mat[2][2] = m[10];
			 _mat[2][3] = m[11];

			 _mat[3][0] = m[12];
			 _mat[3][1] = m[13];
			 _mat[3][2] = m[14];
			 _mat[3][3] = m[15];
		}

		tVoid cMatrix::set (const tDouble* m)
		{
			 _mat[0][0] = m[0];
			 _mat[0][1] = m[1];
			 _mat[0][2] = m[2];
			 _mat[0][3] = m[3];

			 _mat[1][0] = m[4];
			 _mat[1][1] = m[5];
			 _mat[1][2] = m[6];
			 _mat[1][3] = m[7];

			 _mat[2][0] = m[8];
			 _mat[2][1] = m[9];
			 _mat[2][2] = m[10];
			 _mat[2][3] = m[11];

			 _mat[3][0] = m[12];
			 _mat[3][1] = m[13];
			 _mat[3][2] = m[14];
			 _mat[3][3] = m[15];
//				 _mat[0][0] = m[0];
//				 _mat[1][0] = m[1];
//				 _mat[2][0] = m[2];
//				 _mat[3][0] = m[3];
//
//				 _mat[0][1] = m[4];
//				 _mat[1][1] = m[5];
//				 _mat[2][1] = m[6];
//				 _mat[3][1] = m[7];
//
//				 _mat[0][2] = m[8];
//				 _mat[1][2] = m[9];
//				 _mat[2][2] = m[10];
//				 _mat[3][2] = m[11];
//
//				 _mat[0][3] = m[12];
//				 _mat[1][3] = m[13];
//				 _mat[2][3] = m[14];
//				 _mat[3][3] = m[15];
		}


		tVoid cMatrix::set (
						 tFloat a00, tFloat a01, tFloat a02, tFloat a03,
						 tFloat a10, tFloat a11, tFloat a12, tFloat a13,
						 tFloat a20, tFloat a21, tFloat a22, tFloat a23,
						 tFloat a30, tFloat a31, tFloat a32, tFloat a33)
		{
			_mat[0][0] = a00;
			_mat[0][1] = a01;
			_mat[0][2] = a02;
			_mat[0][3] = a03;

			_mat[1][0] = a10;
			_mat[1][1] = a11;
			_mat[1][2] = a12;
			_mat[1][3] = a13;

			_mat[2][0] = a20;
			_mat[2][1] = a21;
			_mat[2][2] = a22;
			_mat[2][3] = a23;

			_mat[3][0] = a30;
			_mat[3][1] = a31;
			_mat[3][2] = a32;
			_mat[3][3] = a33;
		}

		tVoid cMatrix::copy (const cMatrix & matrix)
		{
			memcpy (_mat,matrix._mat,sizeof (_mat));
		}


		tVoid cMatrix::makeScale (tFloat sx, tFloat sy, tFloat sz)
		{
			makeIdent ();
			_mat[0][0] = sx;
			_mat[1][1] = sy;
			_mat[2][2] = sz;
		}


		tVoid cMatrix::preScale (tFloat sx, tFloat sy, tFloat sz, const cMatrix & m)
		{
			cMatrix transMat;
			transMat.makeScale (sx, sy, sz);
			mult (transMat,m);
		}


		tVoid cMatrix::postScale (const cMatrix & m, tFloat sx, tFloat sy, tFloat sz)
		{
			cMatrix transMat;
			transMat.makeScale (sx, sy, sz);
			mult (m,transMat);
		}


		tVoid cMatrix::preScale (tFloat sx, tFloat sy, tFloat sz)
		{
			cMatrix transMat;
			transMat.makeScale (sx, sy, sz);
			preMult (transMat);
		}


		tVoid cMatrix::postScale (tFloat sx, tFloat sy, tFloat sz)
		{
			cMatrix transMat;
			transMat.makeScale (sx, sy, sz);
			postMult (transMat);
		}


		tVoid cMatrix::makeTrans (tFloat tx, tFloat ty, tFloat tz)
		{
			makeIdent ();
			_mat[3][0] = tx;
			_mat[3][1] = ty;
			_mat[3][2] = tz;
		}


		tVoid cMatrix::preTrans (tFloat tx, tFloat ty, tFloat tz, const cMatrix & m)
		{
			cMatrix transMat;
			transMat.makeTrans (tx, ty, tz);
			mult (transMat,m);
		}


		tVoid cMatrix::postTrans (const cMatrix & m, tFloat tx, tFloat ty, tFloat tz)
		{
			cMatrix transMat;
			transMat.makeTrans (tx, ty, tz);
			mult (m,transMat);
		}


		tVoid cMatrix::preTrans (tFloat tx, tFloat ty, tFloat tz)
		{
			_mat[3][0] = (tx * _mat[0][0]) + (ty * _mat[1][0]) + (tz * _mat[2][0]) + _mat[3][0];
			_mat[3][1] = (tx * _mat[0][1]) + (ty * _mat[1][1]) + (tz * _mat[2][1]) + _mat[3][1];
			_mat[3][2] = (tx * _mat[0][2]) + (ty * _mat[1][2]) + (tz * _mat[2][2]) + _mat[3][2];
			_mat[3][3] = (tx * _mat[0][3]) + (ty * _mat[1][3]) + (tz * _mat[2][3]) + _mat[3][3];
		}


		tVoid cMatrix::postTrans (tFloat tx, tFloat ty, tFloat tz)
		{
			cMatrix transMat;
			transMat.makeTrans (tx, ty, tz);
			postMult (transMat);
		}

		tVoid cMatrix::makeRot (const cVector & old_vec, const cVector & new_vec)
		{
			/* dot product == cos (angle old_vec<>new_vec). */
			tFloat d = new_vec * old_vec;
			if (d < 0.9999)
			{
				tFloat angle = acos (d);
				cVector rot_axis = new_vec ^ old_vec;
				makeRot (RAD2DEG (angle),
						 rot_axis.x, rot_axis.y, rot_axis.z);
			}
			else
				makeIdent ();
		}

		tVoid cMatrix::makeRot (tFloat deg, tFloat x, tFloat y, tFloat z)
		{
			tFloat __mat[4][4];
			quaternion q;
			tFloat d = sqrtf (square (x) + square (y) + square (z));

			if (d == 0)
				return;

			tFloat sin_HalfAngle = sinf (DEG2RAD (deg/2));
			tFloat cos_HalfAngle = cosf (DEG2RAD (deg/2));

			q.x = sin_HalfAngle * (x/d);
			q.y = sin_HalfAngle * (y/d);
			q.z = sin_HalfAngle * (z/d);
			q.w = cos_HalfAngle;

			quaternion_matrix ( &q, __mat);

			for (int i=0;i<4;++i)
			{
				for (int j=0;j<4;++j)
				{
					_mat[i][j]=__mat[i][j];
				}
			}
		}


		tVoid cMatrix::preRot (tFloat deg, tFloat x, tFloat y, tFloat z, const cMatrix & m)
		{
			cMatrix rotMat;
			rotMat.makeRot (deg, x, y, z);
			mult (rotMat,m);
		}


		tVoid cMatrix::postRot (const cMatrix & m, tFloat deg, tFloat x, tFloat y, tFloat z)
		{
			cMatrix rotMat;
			rotMat.makeRot (deg, x, y, z);
			mult (m,rotMat);
		}


		tVoid cMatrix::preRot (tFloat deg, tFloat x, tFloat y, tFloat z)
		{
			quaternion q;
			tFloat __mat[4][4];
			tFloat res_mat[4][4];

			tFloat d = sqrtf (square (x) + square (y) + square (z));

			if (d == 0)
				return;

			tFloat sin_HalfAngle = sinf (DEG2RAD (deg/2));
			tFloat cos_HalfAngle = cosf (DEG2RAD (deg/2));

			q.x = sin_HalfAngle * (x/d);
			q.y = sin_HalfAngle * (y/d);
			q.z = sin_HalfAngle * (z/d);
			q.w = cos_HalfAngle;

			quaternion_matrix ( &q, __mat);
			matrix_mult (__mat, _mat, res_mat);
			memcpy (_mat, res_mat, sizeof (_mat));
		}


		tVoid cMatrix::postRot (tFloat deg, tFloat x, tFloat y, tFloat z)
		{
			quaternion q;
			tFloat __mat[4][4];
			tFloat res_mat[4][4];

			tFloat d = sqrtf (square (x) + square (y) + square (z));

			if (d == 0)
				return;

			tFloat sin_HalfAngle = sinf (DEG2RAD (deg/2));
			tFloat cos_HalfAngle = cosf (DEG2RAD (deg/2));

			q.x = sin_HalfAngle * (x/d);
			q.y = sin_HalfAngle * (y/d);
			q.z = sin_HalfAngle * (z/d);
			q.w = cos_HalfAngle;

			quaternion_matrix ( &q, __mat);
			matrix_mult (_mat, __mat , res_mat);
			memcpy (_mat, res_mat, sizeof (_mat));
		}


		tVoid cMatrix::setTrans (tFloat tx, tFloat ty, tFloat tz)
		{
			_mat[3][0] = tx;
			_mat[3][1] = ty;
			_mat[3][2] = tz;
		}


		tVoid cMatrix::setTrans (const cVector & v)
		{
			_mat[3][0] = v.x;
			_mat[3][1] = v.y;
			_mat[3][2] = v.z;
		}


		tVoid cMatrix::preMult (const cMatrix & m)
		{
			cMatrix tm;
			matrix_mult (m._mat, _mat, tm._mat);
			*this = tm;
		}


		tVoid cMatrix::postMult (const cMatrix & m)
		{
			cMatrix tm;
			matrix_mult (_mat, m._mat, tm._mat);
			*this = tm;
		}


		tVoid cMatrix::mult (const cMatrix & lhs,const cMatrix & rhs)
		{
			if ( &lhs==this || &rhs==this)
			{
				cMatrix tm;
				matrix_mult (lhs._mat, rhs._mat, tm._mat);
				*this = tm;
			}
			else
			{
				matrix_mult (lhs._mat, rhs._mat, _mat);
			}
		}


		cMatrix cMatrix::operator * (const cMatrix & m) const
		{
			cMatrix tm;
			matrix_mult (_mat,m._mat, tm._mat);
			return tm;
		}


		tBool cMatrix::invert (const cMatrix & invm)
		{
			if ( &invm==this) {
				cMatrix tm (invm);
				return invert (tm);
			}

			// code lifted from VR Juggler.
			// not cleanly added, but seems to work.  RO.

			const tFloat*  a = reinterpret_cast<const tFloat*> (invm._mat);
			tFloat*  b = reinterpret_cast<tFloat*> (_mat);

			int     n = 4;
			int     i, j, k;
			int     r[ 4], c[ 4], row[ 4], col[ 4];
			tFloat  m[ 4][ 4*2], pivot, max_m, tmp_m, fac;

			/* Initialization */
			for (i = 0; i < n; i ++)
			{
				r[ i] = c[ i] = 0;
				row[ i] = col[ i] = 0;
			}

			/* Set working matrix */
			for (i = 0; i < n; i++)
			{
				for (j = 0; j < n; j++)
				{
					m[ i][ j] = a[ i * n + j];
					m[ i][ j + n] = (i == j) ? 1.0 : 0.0 ;
				}
			}

			/* Begin of loop */
			for (k = 0; k < n; k++)
			{
				/* Choosing the pivot */
				for (i = 0, max_m = 0; i < n; i++)
				{
					if (row[ i]) continue;
					for (j = 0; j < n; j++)
					{
						if (col[ j]) continue;
						tmp_m = fabs (m[ i][j]);
						if (tmp_m > max_m)
						{
							max_m = tmp_m;
							r[ k] = i;
							c[ k] = j;
						}
					}
				}
				row[ r[k] ] = col[ c[k] ] = 1;
				pivot = m[ r[ k] ][ c[ k] ];

				if (fabs (pivot) <= 1e-20)
				{
					cat.info ("*** pivot = %f in mat_inv. ***\n");
					//exit (0);
					return false;
				}

				/* Normalization */
				for (j = 0; j < 2*n; j++)
				{
					if (j == c[ k])
						m[ r[ k]][ j] = 1.0;
					else
						m[ r[ k]][ j] /=pivot;
				}

				/* Reduction */
				for (i = 0; i < n; i++)
				{
					if (i == r[ k])
						continue;

					for (j=0, fac = m[ i][ c[k]];j < 2*n; j++)
					{
						if (j == c[ k])
							m[ i][ j] =0.0;
						else
							m[ i][ j] -=fac * m[ r[k]][ j];
					}
				}
			}

			/* Assign invers to a matrix */
			for (i = 0; i < n; i++)
				for (j = 0; j < n; j++)
					row[ i] = (c[ j] == i) ? r[j] : row[ i];

			for (i = 0; i < n; i++)
				for (j = 0; j < n; j++)
					b[ i * n +  j] = m[ row[ i]][j + n];

			return true;                 // It worked
		}
	}
}

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof