/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>
#include <caosGL/core/math.h>

// package includes
//#include <some.h>

// extern includes
#include <caosGL/core/cParser.h>
#include <math.h>

#include <caosGL/gfx/cAnimCurveReader.h>

#define READ_AMOUNT 8192

using namespace caosGL::core;

namespace caosGL {
	namespace gfx {
		/**
		 *<br> class:		cAnimCurveReader
		 *<br> namespace:	caosGL::gfx
		 *<br> inherits:	<none>
		 *<br> implements:	<none>
		 *<br> purpose:		it's job is to read the animation curves from a stream
		 *
		 */

		/********************************************************************************************/
		cAnimCurveReader::cAnimCurveReader () {
		}

		/********************************************************************************************/
		cAnimCurveReader::~cAnimCurveReader () {
		}
		
		// function definitions
//		static cAnimCurve * assembleAnimCurve (string name, vector<cReadKey*> &keys, tBool isWeighted, tBool useOldSmooth);
		static cAnimCurve::eInfinityType wordAsInfinityType (string type);
		static cReadKey::eTangentType wordAsTangentType (string type);
		static tFloat getFrameRate (string s) {
			if (s.find ("film")!=-1) {
				return 24.0;
			} else if (s.find ("ntsc")!=-1) {
				return 30.0;
			} else if (s.find ("pal")!=-1) {
				return 25.0;
			} else if (s.find ("game")!=-1) {
				return 15.0;
			}
			return 25.0;
		}

		static tFloat getAngularConversion (string s) {
			if (s.find ("deg")!=-1) {
				return dDegRad;
			} else if (s.find ("rad")!=-1) {
				return 1.0;
			}
			return 1.0;
		}

		/********************************************************************************************/
		map<string,cEvaluatable*> * cAnimCurveReader::readCurves (istream * _stream) {
			// used...
			tBool			endOfKeys;
			tBool			endOfCurve;
			cAnimCurve::eInfinityType	preInfinity;
			cAnimCurve::eInfinityType	postInfinity;
			tBool			isWeighted;
			tInt			numKeys;
			tFloat			unitConversion;
			tBool			continueReading = true;
			tFloat			frameRate = 24.0;	/* film */
			tFloat			animVersion;
			tFloat			angularConversion = dDegRad;
			tInt			numCurves;
			map<string,cEvaluatable*> * _curves = new map<string,cEvaluatable*>;
			vector<cReadKey*> keys; 
			string			nameBuffer;
			string			attributeBuffer;
			string			buf;
			istream			& stream = *_stream; // get a local, non-pointer var
												 // for easier use.
			
			numCurves = 0;

			/* Read the header section */

			/****** animVersion ******/				
			stream >> buf;
			while (buf.find ("animVersion")==-1) {
				stream.ignore (READ_AMOUNT,'\n');
				stream >> buf;
			}
			stream >> animVersion;

			/****** timeUnit ******/
			stream >> buf;
			while (buf.find ("timeUnit")==-1) {
				stream.ignore (READ_AMOUNT,'\n');
				stream >> buf;
			}
			stream >> buf;
			frameRate = getFrameRate (buf);
			
			/****** angularUnit ******/
			stream >> buf;
			while (buf.find ("angularUnit")==-1) {
				stream.ignore (READ_AMOUNT,'\n');
				stream >> buf;
			}
			stream >> buf;
			angularConversion = getAngularConversion (buf);
			
			/****** anim, animData ******/
			continueReading = stream.good ();
			while (continueReading) {
				stream >> buf;
				continueReading = stream.good ();
				/* anim */
				if (buf.find ("anim")!=-1 && buf.find ("animData")==-1) {
					/* full attribute name */
					stream >> buf;
					continueReading = stream.good ();
					if (continueReading) {
						/* short attribute name */
						stream >> buf;
						continueReading = stream.good ();
						if (continueReading) {
							attributeBuffer = buf;
							/* object name */
							stream >> buf;
							continueReading = stream.good ();
							if (continueReading) {
								nameBuffer = buf;
							}
						}
						// eat the rest of the line...
						stream.ignore (READ_AMOUNT,'\n');
					}
				}
				/* animData */
				else if (buf.find ("animData")!=-1) {
					endOfCurve = false;
					numKeys = 0;
					isWeighted = true;
					preInfinity = cAnimCurve::eInfinityConstant;
					postInfinity = cAnimCurve::eInfinityConstant;
					unitConversion = 1.0;
					/* read the parameters for the animation curve */
					while (continueReading && !endOfCurve) {
						stream >> buf;
						continueReading = stream.good ();
						if (!continueReading) break;
						if (buf[0]=='}') {
							endOfCurve = true;
						} else if (buf.find ("preInfinity")!=-1) {
							stream >> buf;
							preInfinity = wordAsInfinityType (buf);
						} else if (buf.find ("postInfinity")!=-1) {
							stream >> buf;
							postInfinity = wordAsInfinityType (buf);
						} else if (buf.find ("weighted")!=-1) {
							tInt i;
							stream >> i;
							isWeighted = (i == 1);
						} else if (buf.find ("output")!=-1) {
							stream >> buf;
							continueReading = stream.good ();
							if (!continueReading) break;
							if (buf.find ("angular")!=-1)
								unitConversion = angularConversion;
						} else if (buf.find ("keys")!=-1) {
							/* read the list of keys */
							endOfKeys = false;
							while (continueReading && !endOfKeys) {
								stream >> buf;
								continueReading = stream.good ();
								if (!continueReading) break;
								if (buf[0]=='}') {
									endOfKeys = true;
								} else if (buf[0]!='{') {
									cReadKey * key = new cReadKey ();
									key->time = (tFloat)atof ((const char *)buf.c_str ()) / frameRate;
									stream >> (key->value);
									key->value *= unitConversion;
									string lBuf;
									stream >> lBuf;
									key->inTangentType = wordAsTangentType (lBuf);
									stream >> lBuf;
									key->outTangentType = wordAsTangentType (lBuf);
									tInt i;
									stream >> i;
									stream >> i;
									if (animVersion >= 1.1) {
										stream >> i;
									}
									if (key->inTangentType == cReadKey::eTangentFixed) {
										stream >> (key->inAngle);
										key->inAngle *= angularConversion;
										stream >> (key->inWeight);
									}
									if (key->outTangentType == cReadKey::eTangentFixed) {
										stream >> (key->outAngle);
										key->outAngle *= angularConversion;
										stream >> (key->outWeight);
									}
									keys.push_back (key);
									stream.ignore (READ_AMOUNT,'\n');
								}
							}
						} else {
							/* Eat the rest of the line */
							stream.ignore (READ_AMOUNT,'\n');
						}
					}
					if (!keys.empty ()) {
						string curveName;
						curveName.append (nameBuffer);curveName.append (".");curveName.append (attributeBuffer);
						cAnimCurve * animCurve = assembleAnimCurve (curveName,keys,isWeighted,false);
						if (animCurve != cNULL) {
							animCurve->preInfinity (preInfinity);
							animCurve->postInfinity (postInfinity);
							_curves->insert (map<string,cEvaluatable*>::value_type (animCurve->name (), animCurve));
						}
						keys.clear ();
					}
				
				} else if (continueReading) {
					stream.ignore (READ_AMOUNT,'\n');
				}
			}
		
			return (_curves);
		}

		/********************************************************************************************/
		map <string, cEvaluatable *> * cAnimCurveReader::readCurves (TiXmlElement * node) {
			map <string, cEvaluatable *> * _curves = new map <string, cEvaluatable *>;
			tFloat frameRate = 25;
			tFloat angularConversion = 1.0;
	
			for (TiXmlAttribute * attr = node->FirstAttribute (); attr ; attr = attr->Next ()) {
				string attrName = attr->Name ();

				if (attrName == "timeUnit") {
					frameRate = getFrameRate (attr->Value ());
				} else if (attrName == "angularUnit") {
					angularConversion = getAngularConversion (attr->Value ());
				}
			}

			for (TiXmlElement * curve = node->FirstChildElement ("curve"); 
					curve; curve = curve->NextSiblingElement ("curve")) {
				vector <cReadKey *> keys;
				string name;
				string obj;
				string attrib;
				tBool weighted;
				cAnimCurve::eInfinityType preInfinity;
				cAnimCurve::eInfinityType postInfinity;
				tFloat start = -HUGE;
				tFloat end =    HUGE;

				for (TiXmlAttribute * attr = curve->FirstAttribute (); attr ; attr = attr->Next ()) {
					string attrName = attr->Name ();

					if (attrName == "name") {
						name = attr->Value ();
					} else if (attrName == "obj") {
						obj = attr->Value ();
					} else if (attrName == "attrib") {
						attrib = attr->Value ();
					} else if (attrName == "weighted") {
						weighted = cParser::parseBool (attr->Value ());
					} else if (attrName == "preInfinity") {
						preInfinity = wordAsInfinityType (attr->Value ());
					} else if (attrName == "postInfinity") {
						postInfinity = wordAsInfinityType (attr->Value ());
					} else if (attrName == "start") {
						start = cParser::parseDouble (attr->Value ());
					} else if (attrName == "end") {
						end = cParser::parseDouble (attr->Value ());
					}
				}
				for (TiXmlElement * keyNode = curve->FirstChildElement ("key"); 
						keyNode; keyNode = keyNode->NextSiblingElement ("key")) {
					cReadKey * key = new cReadKey ();
					for (TiXmlAttribute * attr = keyNode->FirstAttribute (); attr ; attr = attr->Next ()) {
						string attrName = attr->Name ();

						if (attrName == "time") {
							key->time = cParser::parseFloat (attr->Value ());
						} else if (attrName == "value") {
							key->value = cParser::parseDouble (attr->Value ());
						} else if (attrName == "inTangentType") {
							key->inTangentType = wordAsTangentType (attr->Value ());
						} else if (attrName == "outTangentType") {
							key->outTangentType = wordAsTangentType (attr->Value ());
						} else if (attrName == "inTangent") {
							key->inAngle = cParser::parseDouble (attr->Value ());
						} else if (attrName == "inWeight") {
							key->inWeight = cParser::parseDouble (attr->Value ());
						} else if (attrName == "outTangent") {
							key->outAngle = cParser::parseDouble (attr->Value ());
						} else if (attrName == "outWeight") {
							key->outWeight = cParser::parseDouble (attr->Value ());
						}
					}
					keys.push_back (key);
				}

				cAnimCurve * animCurve = assembleAnimCurve (name,keys,weighted,false);
				if (animCurve != cNULL) {
					animCurve->preInfinity (preInfinity);
					animCurve->postInfinity (postInfinity);
					animCurve->set ('obj', obj);
					animCurve->set ('attr', attrib);
					animCurve->set ('strt', start);
					animCurve->set ('end', end);
					_curves->insert (map <string, cEvaluatable *>::value_type (animCurve->name (), animCurve));
				}
				keys.clear ();
			}

			return _curves;
		}

		/********************************************************************************************/
		/*
		//	Function Name:
		//		assembleAnimCurve
		//
		//	Description:
		//		A static helper function to assemble an EtCurve animation curve
		//	from a linked list of heavy-weights keys
		//
		//  Input Arguments:
		//		EtReadKey *firstKey			The linked list of keys
		//		EtInt numKeys				The number of keyss
		//		EtBoolean isWeighted		Whether or not the curve has weighted tangents
		//		EtBoolean useOldSmooth		Whether or not to use pre-Maya2.0 smooth
		//									tangent computation
		//
		//  Return Value:
		//		EtCurve *animCurve			The assembled animation curve
		//
		//	Note:
		//		This function will also free the memory used by the heavy-weight keys
		*/
		cAnimCurve * cAnimCurveReader::assembleAnimCurve (string name, vector<cReadKey*> &keys, tBool isWeighted, tBool useOldSmooth) {
			cReadKey *	key = cNULL;
			cReadKey *	prevKey = cNULL;
			cReadKey *	nextKey = cNULL;
			cKey *		thisKey;
			tInt		index = 0;
			tFloat		py, ny, dx;
			tBool		hasSmooth;
			tFloat		length;
			tFloat		inLength, outLength, inSlope, outSlope;
			tFloat		inTanX, inTanY, outTanX, outTanY;
			tFloat		inTanXs, inTanYs, outTanXs, outTanYs;
			// cAnimCurve proxy
			tInt			_numKeys = keys.size ();
			tBool			_isWeighted = isWeighted;
			tBool			_isStatic = true;
			cAnimCurve::eInfinityType _preInfinity = cAnimCurve::eInfinityConstant;
			cAnimCurve::eInfinityType _postInfinity = cAnimCurve::eInfinityConstant;

			/* evaluate cache */
			cKey*			_lastKey = cNULL;	
			tInt			_lastIndex = -1;	
			tInt			_lastInterval = -1;	
			tBool			_isStep = false;	
			cKey*			_keyList;	


			
			/* make sure we have useful information */
			if (keys.empty ()) {
				return (cAnimCurve*)cNULL;
			}
			
			/* allocate some memory for the animation curve */
//				animCurve = new cAnimCurve ();
			_keyList = new cKey [keys.size ()];
			
			/* initialise the animation curve parameters */
			_numKeys = keys.size ();
			_isWeighted = isWeighted;
			_isStatic = true;
			_preInfinity = cAnimCurve::eInfinityConstant;
			_postInfinity = cAnimCurve::eInfinityConstant;
			
			/* initialise the cache */
			_lastKey = cNULL;
			_lastIndex = -1;
			_lastInterval = -1;
			_isStep = false;
			
			/* compute tangents */
			vector<cReadKey*>::iterator it = keys.begin();
			nextKey = *it;
			for (;it != keys.end();) {
				it++;
				prevKey = key;
				key = nextKey;
				nextKey = *it;
				if (it==keys.end ())
					nextKey = cNULL;
				
				/* construct the final cKey (light-weight key) */
				thisKey = &(_keyList [index++]);
				thisKey->time = key->time;
				thisKey->value = key->value;
				
				/* compute the in-tangent values */
				/* cReadKey::eTangentClamped */
				if (key->inTangentType == cReadKey::eTangentClamped && prevKey != cNULL) {
					py = prevKey->value - key->value;
					if (py < 0.0) py = -py;
					ny = (nextKey == cNULL ? py : nextKey->value - key->value);
					if (ny < 0.0) ny = -ny;
					if ((ny <= 0.05) || (py <= 0.05)) {
						key->inTangentType = cReadKey::eTangentFlat;
					}
				}
				hasSmooth = false;
				switch (key->inTangentType) {
				case cReadKey::eTangentFixed:
					inTanX = key->inWeight * cos (key->inAngle) * 3.0;
					inTanY = key->inWeight * sin (key->inAngle) * 3.0;
					break;
				case cReadKey::eTangentLinear:
					if (prevKey == cNULL) {
						inTanX = 1.0;
						inTanY = 0.0;
					}
					else {
						inTanX = key->time - prevKey->time;
						inTanY = key->value - prevKey->value;
					}
					break;
				case cReadKey::eTangentFlat:
					if (prevKey == cNULL) {
						inTanX = (nextKey == cNULL ? 0.0 : nextKey->time - key->time);
						inTanY = 0.0;
					}
					else {
						inTanX = key->time - prevKey->time;
						inTanY = 0.0;
					}
					break;
				case cReadKey::eTangentStep:
					inTanX = 0.0;
					inTanY = 0.0;
					break;
				case cReadKey::eTangentSlow:
				case cReadKey::eTangentFast:
					key->inTangentType = cReadKey::eTangentSmooth;
					if (prevKey == cNULL) {
						inTanX = 1.0;
						inTanY = 0.0;
					}
					else {
						inTanX = key->time - prevKey->time;
						inTanY = key->value - prevKey->value;
					}
					break;
				case cReadKey::eTangentSmooth:
				case cReadKey::eTangentClamped:
					key->inTangentType = cReadKey::eTangentSmooth;
					hasSmooth = true;
					break;
				}
				
				/* compute the out-tangent values */
				/* cReadKey::eTangentClamped */
				if ((key->outTangentType == cReadKey::eTangentClamped) && (nextKey != cNULL)) {
					ny = nextKey->value - key->value;
					if (ny < 0.0) ny = -ny;
					py = (prevKey == cNULL ? ny : prevKey->value - key->value);
					if (py < 0.0) py = -py;
					if ((ny <= 0.05) || (py <= 0.05)) {
						key->outTangentType = cReadKey::eTangentFlat;
					}
				}
				switch (key->outTangentType) {
				case cReadKey::eTangentFixed:
					outTanX = key->outWeight * cosf (key->outAngle) * 3.0;
					outTanY = key->outWeight * sinf (key->outAngle) * 3.0;
					break;
				case cReadKey::eTangentLinear:
					if (nextKey == cNULL) {
						outTanX = 1.0;
						outTanY = 0.0;
					}
					else {
						outTanX = nextKey->time - key->time;
						outTanY = nextKey->value - key->value;
					}
					break;
				case cReadKey::eTangentFlat:
					if (nextKey == cNULL) {
						outTanX = (prevKey == cNULL ? 0.0 : key->time - prevKey->time);
						outTanY = 0.0;
					}
					else {
						outTanX = nextKey->time - key->time;
						outTanY = 0.0;
					}
					break;
				case cReadKey::eTangentStep:
					outTanX = 0.0;
					outTanY = 0.0;
					break;
				case cReadKey::eTangentSlow:
				case cReadKey::eTangentFast:
					key->outTangentType = cReadKey::eTangentSmooth;
					if (nextKey == cNULL) {
						outTanX = 1.0;
						outTanY = 0.0;
					}
					else {
						outTanX = nextKey->time - key->time;
						outTanY = nextKey->value - key->value;
					}
					break;
				case cReadKey::eTangentSmooth:
				case cReadKey::eTangentClamped:
					key->outTangentType = cReadKey::eTangentSmooth;
					hasSmooth = true;
					break;
				}
				
				/* compute smooth tangents (if necessary) */
				if (hasSmooth) {
					if (useOldSmooth && _isWeighted) {
						/* pre-Maya 2.0 smooth tangents */
						if ((prevKey == cNULL) && (nextKey != cNULL)) {
							outTanXs = nextKey->time - key->time;
							outTanYs = nextKey->value - key->value;
							inTanXs = outTanXs;
							inTanYs = outTanYs;
						}
						else if ((prevKey != cNULL) && (nextKey == cNULL)) {
							outTanXs = key->time - prevKey->time;
							outTanYs = key->value - prevKey->value;
							inTanXs = outTanXs;
							inTanYs = outTanYs;
						}
						else if ((prevKey != cNULL) && (nextKey != cNULL)) {
							/* There is a CV before and after this one */
							/* Find average of the adjacent in and out tangents */
							inTanXs = key->time - prevKey->time;
							inTanYs = key->value - prevKey->value;
							outTanXs = nextKey->time - key->time;
							outTanYs = nextKey->value - key->value;
							
							if (inTanXs > 0.01) {
								inSlope = inTanYs / inTanXs;
							}
							else {
								inSlope = 0.0;
							}
							inLength = (inTanXs * inTanXs) + (inTanYs * inTanYs);
							if (outTanXs > 0.01) {
								outSlope = outTanYs / outTanXs;
							}
							else {
								outSlope = 0.0;
							}
							outLength = (outTanXs * outTanXs) + (outTanYs * outTanYs);
							
							if (!dSMALL(inLength) || dSMALL(outLength)) {
								inLength = sqrt (inLength);
								outLength = sqrt (outLength);
								outTanYs = ((inSlope * outLength) + (outSlope * inLength)) / (inLength + outLength);
								inTanYs = outTanYs * inTanXs;
								outTanYs *= outTanXs;
								/*
								// Set the In and Out tangents, at that keyframe, to be the
								// smaller (in length) off the two.
								*/
								inLength = (inTanXs * inTanXs) + (inTanYs * inTanYs);
								outLength = (outTanXs * outTanXs) + (outTanYs * outTanYs);
								if (inLength < outLength) {
									outTanXs = inTanXs;
									outTanYs = inTanYs;
								}
								else {
									inTanXs = outTanXs;
									inTanYs = outTanYs;
								}
							}
						}
						else {
							inTanXs = 1.0;
							inTanYs = 0.0;
							outTanXs = 1.0;
							outTanYs = 0.0;
						}
					}
					else {
						/* Maya 2.0 smooth tangents */
						if ((prevKey == cNULL) && (nextKey != cNULL)) {
							outTanXs = nextKey->time - key->time;
							outTanYs = nextKey->value - key->value;
							inTanXs = outTanXs;
							inTanYs = outTanYs;
						}
						else if ((prevKey != cNULL) && (nextKey == cNULL)) {
							outTanXs = key->time - prevKey->time;
							outTanYs = key->value - prevKey->value;
							inTanXs = outTanXs;
							inTanYs = outTanYs;
						}
						else if ((prevKey != cNULL) && (nextKey != cNULL)) {
							/* There is a CV before and after this one*/
							/* Find average of the adjacent in and out tangents. */
							
							dx = nextKey->time - prevKey->time;
							if (dx < 0.0001) {
								outTanYs = dMaxTan;
							}
							else {
								outTanYs = (nextKey->value - prevKey->value) / dx;
							}
							
							outTanXs = nextKey->time - key->time;
							inTanXs = key->time - prevKey->time;
							inTanYs = outTanYs * inTanXs;
							outTanYs *= outTanXs;
						}
						else {
							inTanXs = 1.0;
							inTanYs = 0.0;
							outTanXs = 1.0;
							outTanYs = 0.0;
						}
					}
					if (key->inTangentType == cReadKey::eTangentSmooth) {
						inTanX = inTanXs;
						inTanY = inTanYs;
					}
					if (key->outTangentType == cReadKey::eTangentSmooth) {
						outTanX = outTanXs;
						outTanY = outTanYs;
					}
				}
				/* make sure the computed tangents are valid */
				if (_isWeighted) {
					if (inTanX < 0.0) inTanX = 0.0;
					if (outTanX < 0.0) outTanX = 0.0;
				}
				else {
					if (inTanX < 0.0) {
						inTanX = 0.0;
					}
					length = sqrt ((inTanX * inTanX) + (inTanY * inTanY));
					if (!dSMALL (length)) {	/* zero lengths can come from step tangents */
						inTanX /= length;
						inTanY /= length;
					}
					if (dSMALL(inTanX) && !dSMALL(inTanY)) {
						inTanX = 0.0001f;
						inTanY = (inTanY < 0.0 ? -1.0 : 1.0) * (inTanX * dMaxTan);
					}
					if (outTanX < 0.0) {
						outTanX = 0.0;
					}
					length = sqrt ((outTanX * outTanX) + (outTanY * outTanY));
					if (!dSMALL (length)) {	/* zero lengths can come from step tangents */
						outTanX /= length;
						outTanY /= length;
					}
					if (dSMALL(outTanX) && !dSMALL(outTanY)) {
						outTanX = 0.0001f;
						outTanY = (outTanY < 0.0 ? -1.0 : 1.0) * (outTanX * dMaxTan);
					}
				}
				
				thisKey->inTanX = inTanX;
				thisKey->inTanY = inTanY;
				thisKey->outTanX = outTanX;
				thisKey->outTanY = outTanY;
				
				/*
				// check whether or not this animation curve is static (i.e. all the
				// key values are the same)
				*/
				if (_isStatic) {
					if ((prevKey != cNULL) && (prevKey->value != key->value)) {
						_isStatic = false;
					}
					else if (!dSMALL(inTanY) || !dSMALL(outTanY)) {
						_isStatic = false;
					}
				}
			
			}
			if (_isStatic) {
				if ((prevKey != cNULL) && (key != cNULL) && (prevKey->value != key->value)) {
					_isStatic = false;
				}
			}
	

			// erase all the elements in the heavyweight key vector
//				keys.erase (keys.begin ());
			cAnimCurve * ac = new cAnimCurve (name,cNULL);
			ac->init (_numKeys, _isWeighted, _isStatic, _preInfinity, _postInfinity,
					  _lastKey, _lastIndex, _lastInterval, _isStep, _keyList);
			return ac;
		}



		/* Infinity type words */
		#define dWordInfinityConstant		("constant")
		#define dWordInfinityLinear			("linear")
		#define dWordInfinityCycle			("cycle")
		#define dWordInfinityCycleRelative	("cycleRelative")
		#define dWordInfinityOscillate		("oscillate")
		
		/********************************************************************************************/
		/*
		//	Function Name:
		//		wordAsInfinityType
		//
		//	Description:
		//		A static helper function to translate a string into an infinity type
		//
		//  Input Arguments:
		//		string						The string to translate
		//
		//  Return Value:
		//		cAnimCurve::tInfinityType	The translated infinity type
		//										(acInfinityConstant by default)
		*/
		static cAnimCurve::eInfinityType wordAsInfinityType (string type) {
			if (type.find (dWordInfinityConstant)!=-1) {
				return (cAnimCurve::eInfinityConstant);
			} 
			if (type.find (dWordInfinityLinear)!=-1) {
				return (cAnimCurve::eInfinityLinear);
			}
			if (type.find (dWordInfinityCycleRelative)!=-1) {
				return (cAnimCurve::eInfinityCycleRelative);
			}
			if (type.find (dWordInfinityCycle)!=-1) {
				return (cAnimCurve::eInfinityCycle);
			}
			if (type.find (dWordInfinityOscillate)!=-1) {
				return (cAnimCurve::eInfinityOscillate);
			}
			return (cAnimCurve::eInfinityConstant);
		}

		/* Tangent type words */
		#define dWordTangentFixed		("fixed")
		#define dWordTangentLinear		("linear")
		#define dWordTangentFlat		("flat")
		#define dWordTangentSmooth		("spline")
		#define dWordTangentStep		("step")
		#define dWordTangentSlow		("slow")
		#define dWordTangentFast		("fast")
		#define dWordTangentClamped		("clamped")
		
		/********************************************************************************************/
		/*
		//	Function Name:
		//		wordAsTangentType
		//
		//	Description:
		//		A static helper function to translate a string into a tangent type
		//
		//  Input Arguments:
		//		EtByte *type				The string to translate
		//
		//  Return Value:
		//		EtTangentType type			The translated tangent type (kTangentSmooth
		//										by default)
		*/
		static cReadKey::eTangentType wordAsTangentType (string type) {
			if (type.find (dWordTangentFixed)!=-1) {
				return (cReadKey::eTangentFixed);
			}
			if (type.find (dWordTangentLinear)!=-1) {
				return (cReadKey::eTangentLinear);
			}
			if (type.find (dWordTangentFlat)!=-1) {
				return (cReadKey::eTangentFlat);
			}
			if (type.find (dWordTangentSmooth)!=-1) {
				return (cReadKey::eTangentSmooth);
			}
			if (type.find (dWordTangentStep)!=-1) {
				return (cReadKey::eTangentStep);
			}
			if (type.find (dWordTangentSlow)!=-1) {
				return (cReadKey::eTangentSlow);
			}
			if (type.find (dWordTangentFast)!=-1) {
				return (cReadKey::eTangentFast);
			}
			if (type.find (dWordTangentClamped)!=-1) {
				return (cReadKey::eTangentClamped);
			}
			return (cReadKey::eTangentSmooth);
		}
	} // namespace gfx
} // namespace caosGL

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof