/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

// package includes
#include <caosGL/gfx/cState.h>
#include <caosGL/gfx/c3DLoader.h>
#include <caosGL/gfx/cCamera.h>

// extern includes
#include <caosGL/core/DynGL.h>
#include <caosGL/core/cParser.h>
#include <caosGL/gfx/cTime.h>

#include <caosGL/gfx/c3DEngine.h>

using namespace caosGL::gfx;

namespace caosGL {
	namespace gfx {
		/**
		 *<br> class:		c3DEngine
		 *<br> namespace:	caosGL::gfx
		 *<br> inherits:	caosGL::gfx::cBaseNode
		 *<br> implements:	<none>
		 *<br> purpose:		Implements the functionality of a 3d engine. Basically, this engine sucks 
		 *                  big time. This is how it "works":
		 *<br>              - it constructs the scene as a tree.
		 *<br>              - at scene compile time, it extracts the cameras and lights, along with
		 *                    with their transform nodes.
		 *<br>              - at draw time, it first activates de camera, then the lights, and then 
		 *                    does 2 passes of the scene, first drawing the opaque objects, then
		 *                    the transparent ones.
		 *<br>              See? I told you it sucked!
		 *
		 */

		static vector <tAttr> attributeNames;
		static log4cpp::Category& cat = log4cpp::Category::getInstance ("caosGL::gfx::c3DEngine");

		/********************************************************************************************/
		c3DEngine::c3DEngine (const string & n, cGroup * f) : super (n,f) {
			init ();
		}

		/********************************************************************************************/
		c3DEngine::~c3DEngine () {
		}

		/********************************************************************************************/
		tVoid c3DEngine::setScene (cGroup * rN){
			rootNode = rN;
		}

		/********************************************************************************************/
		tVoid c3DEngine::loadCaos (const string & file) {
			setScene (c3DLoader::instance ()->loadCaos (file, name ()));
		}

		/********************************************************************************************/
		tBool c3DEngine::traverse (tBool transparent) {
//**/			cat.debug ("*********** start of traverse: %s", (transparent?"y":"n"));
			return traverse (rootNode->getNodes (), transparent, cTime::elapsed ());
		}

		/********************************************************************************************/
		tBool c3DEngine::traverse (const tNodeList * nodeList, tBool transparent, tFloat t) {
			tBool ret = false;
			for (tNodeList::const_iterator it = nodeList->begin (); it != nodeList->end (); it++) {
				cBaseNode * bn = (*it);
				
				if (!bn->visit (t))
					continue;

//**/				cat.debug ("Visiting node %s, type %s", bn->name ().c_str (), bn->getTypeName ().c_str ());
//**/				cat.debug ("Node info: %s", bn->toString().c_str());

				iDrawable * dr = dynamic_cast <iDrawable *> (bn);
				if (dr && dr->transparent() == transparent) {
					dr->draw (t);
				}

				cGroup * gr = dynamic_cast <cGroup *> (bn);
				if (gr) {
					traverse (gr->getNodes (), transparent, t);
				}

//**/				cat.debug ("Leaving node %s", bn->name ().c_str ());

				bn->leave ();
				ret = true;
			}
			return ret;
		}

		/********************************************************************************************/
		tBool c3DEngine::compile (const tNodeList * nodeList) {
			tBool ret = false;
			for (tNodeList::const_iterator it = nodeList->begin (); it != nodeList->end (); it++) {
				cBaseNode * bn = (*it);
				cGroup * gr = dynamic_cast <cGroup *> (bn);
				if (gr) {
					compile (gr->getNodes ());
				}
				
				bn->compile ();

				ret = true;
			}
			return ret;
		}

		/********************************************************************************************/
		tBool c3DEngine::findCameras () {
			findCameras(rootNode->getNodes ());
			for (map<string, cCamera*>::const_iterator it = cameras.begin(); it != cameras.end (); it++) {
				cCamera * cam = it->second;
				cGroup * trsfrm = cam->father ();
				cGroup * fthr = trsfrm->father ();
				fthr->removeNode (trsfrm->name ());
				cam->trasform (dynamic_cast<cTransform *> (trsfrm));
			}
			return cameras.size() > 0;
		}

		/********************************************************************************************/
		tBool c3DEngine::findCameras (const tNodeList * nodeList) {
			for (tNodeList::const_iterator it = nodeList->begin (); it != nodeList->end (); it++) {
				cBaseNode * bn = (*it);
				cCamera * cam = dynamic_cast <cCamera *> (bn);
				if (cam) {
					camera = cam;
					cameras.insert(make_pair (cam->name (), cam));
					cat.debug ("Found camera %s", camera->name().c_str());
					return true;
				}
				cGroup * gr = dynamic_cast <cGroup *> (bn);
				if (gr) {
					findCameras (gr->getNodes ());
				}
			}
			return false;
		}
		
		/********************************************************************************************/
		tBool c3DEngine::findLights () {
			findLights (rootNode->getNodes ());
			tInt i = 0;
			for (vector <cLight *>::const_iterator it = lights.begin(); it != lights.end (); it++) {
				cLight * light = *it;
				light->set ('num', i++);
			}
			return lights.size() > 0;
		}

		/********************************************************************************************/
		tBool c3DEngine::findLights (const tNodeList * nodeList) {
			for (tNodeList::const_iterator it = nodeList->begin (); it != nodeList->end (); it++) {
				cBaseNode * bn = (*it);
				cLight * light = dynamic_cast <cLight *> (bn);
				if (light) {
					lights.push_back (light);
					//cat.debug ("Found light %s", light->name ().c_str ());
					return true;
				}
				cGroup * gr = dynamic_cast <cGroup *> (bn);
				if (gr) {
					findLights (gr->getNodes ());
				}
			}
			return false;
		}
		
		// from iDrawable
		/********************************************************************************************/
		tBool c3DEngine::draw (const tFloat time) {
			camera->visit (time);
			// reset these things. god this is horrible!!
			// this is in a badly needed need of a rewrite (like this sentence!)
			glColor4ub (255,255,255,255);
			if (_lighted)
				cState::enable (cState::oGL_LIGHTING);
			else
				cState::disable (cState::oGL_LIGHTING);

			glDepthMask (1);
			cState::enable (cState::oGL_DEPTH_TEST);
			traverse (false);
			glDepthMask (0);
			traverse (true);
			glDepthMask (1);
			camera->leave();
			return true;
		}

		// from iDrawable
		/********************************************************************************************/
		tBool c3DEngine::transparent () const {
			return false;
		}

		// from iDrawable
		/********************************************************************************************/
		tVoid c3DEngine::transparent (const tBool trasnp) {
		}

		// from cBaseNode
		/********************************************************************************************/
		tBool c3DEngine::compile () {
			compile (rootNode->getNodes ());
			findCameras ();
			_lighted = findLights ();
			rootNode->printTree();
			if (cameraName != "") {
				set ('cam', cameraName);
			}
			return true;
		}

		// from cBaseNode
		/********************************************************************************************/
		tBool c3DEngine::visit (tFloat t) {
			if (!cBaseNode::visit (t)) return false;
			evaluateAll (t);
			return true;
		}

		// from cBaseNode
		/********************************************************************************************/
		tVoid c3DEngine::leave () {
			return;
		}

		// from cBaseNode
		/********************************************************************************************/
		tBool c3DEngine::init () {
			_lighted = false;
			cameraName = "";
			#define ATTRIB(n,t,v,d) ATTRIB_INIT_VAR(n,v)
			#include c3DEngine_attribsFile
			#undef ATTRIB
			return true;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool c3DEngine::set (const tDWord key, const string & value) {
			if (super::set (key, value)) return true;
			switch (key) {
				#define ATTRIB(n,t,v,d) ATTRIB_SET_S(n)
				#include c3DEngine_attribsFile
				#undef ATTRIB
			case 'file': {
				file = value;
				loadCaos (file);
			} return true;
			case 'cam': {
				cameraName = value;
				map <string, cCamera*>::const_iterator it = cameras.find (value);
				if (it != cameras.end()) {
					camera = it->second;
					cameraName = camera->name();
				}
			} return true;
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool c3DEngine::set (const tDWord key, const tFloat & value) {
			if (super::set (key, value)) return true;
			switch (key) {
				#define ATTRIB(n,t,v,d) ATTRIB_SET_N(n)
				#include c3DEngine_attribsFile
				#undef ATTRIB
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool c3DEngine::get (const tDWord key, string & value) const {
			if (super::get (key, value)) return true;
			switch (key) {
				#define ATTRIB(n,t,v,d) ATTRIB_GET(n)
				#include c3DEngine_attribsFile
				#undef ATTRIB
			case 'file': value = file; return true;
			case 'cam': value = camera?camera->name():""; return true;
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const vector <tAttr> * c3DEngine::getAttributeNames () const {
			if (attributeNames.size () == 0) {
				const vector <tAttr> * v = super::getAttributeNames ();
				attributeNames.insert (attributeNames.begin (), v->begin (), v->end ());
				#define ATTRIB(n,t,v,d) ATTRIB_ATTRIBNAMES(n,d)
				#include c3DEngine_attribsFile
				#undef ATTRIB
				attributeNames.push_back(make_pair('file',string("3D file")));
				attributeNames.push_back(make_pair('cam',string("Active camera")));
			}
			return &attributeNames;
		}

		// from cBaseNode
		/********************************************************************************************/
		const string c3DEngine::getTypeName () const { return "caosGL::gfx::c3DEngine"; }
	}
}

// for node creation
#include <caosGL/core/cRegistry.h>
#include <caosGL/gfx/cNodeCreator.h>

class c3DEngineNodeCreator : public cNodeCreator {
public:
	c3DEngineNodeCreator () {
		name ("caosGL::gfx::c3DEngine");
	}
	cBaseNode * createNode (const string n, cGroup * f) {
		return new caosGL::gfx::c3DEngine (n,f);
	}
};
caosGL::core::cRegisterNodeCreator <c3DEngineNodeCreator> c3DEngineNodeCreatorInstance;

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof